/** @file scim_x11_frontend_setup.cpp
 * implementation of Setup Module of X11FrontEnd.
 */

/*
 * Smart Common Input Method
 * 
 * Copyright (c) 2002 James Su <suzhe@turbolinux.com.cn>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 *
 * $Id: scim_x11_frontend_setup.cpp,v 1.11 2004/08/24 14:42:20 suzhe Exp $
 *
 */

#define Uses_SCIM_CONFIG_PATH
#define Uses_SCIM_CONFIG_BASE

#include <iostream>

#include <gtk/gtk.h>

#include "scim_private.h"
#include "scim.h"
#include "scimkeyselection.h"

using namespace scim;

#define scim_module_init x11_frontend_setup_LTX_scim_module_init
#define scim_module_exit x11_frontend_setup_LTX_scim_module_exit

#define scim_setup_module_create_ui       x11_frontend_setup_LTX_scim_setup_module_create_ui
#define scim_setup_module_get_category    x11_frontend_setup_LTX_scim_setup_module_get_category
#define scim_setup_module_get_name        x11_frontend_setup_LTX_scim_setup_module_get_name
#define scim_setup_module_get_description x11_frontend_setup_LTX_scim_setup_module_get_description
#define scim_setup_module_load_config     x11_frontend_setup_LTX_scim_setup_module_load_config
#define scim_setup_module_save_config     x11_frontend_setup_LTX_scim_setup_module_save_config
#define scim_setup_module_query_changed   x11_frontend_setup_LTX_scim_setup_module_query_changed

#define SCIM_CONFIG_FRONTEND_X11_ONTHESPOT             "/FrontEnd/X11/OnTheSpot"
#define SCIM_CONFIG_FRONTEND_X11_BROKEN_WCHAR          "/FrontEnd/X11/BrokenWchar"
#define SCIM_CONFIG_FRONTEND_X11_DYNAMIC               "/FrontEnd/X11/Dynamic"
#define SCIM_CONFIG_FRONTEND_X11_SERVER_NAME           "/FrontEnd/X11/ServerName"
#define SCIM_CONFIG_FRONTEND_X11_PANEL_PROGRAM         "/FrontEnd/X11/PanelProgram"

static GtkWidget * create_setup_window ();
static void        load_config (const ConfigPointer &config);
static void        save_config (const ConfigPointer &config);
static bool        query_changed ();

// Module Interface.
extern "C" {
    void scim_module_init (void)
    {
    }

    void scim_module_exit (void)
    {
    }

    GtkWidget * scim_setup_module_create_ui (void)
    {
        return create_setup_window ();
    }

    String scim_setup_module_get_category (void)
    {
        return String ("FrontEnd");
    }

    String scim_setup_module_get_name (void)
    {
        return String (_("X Window"));
    }

    String scim_setup_module_get_description (void)
    {
        return String (_("A FrontEnd module for X11R6 Window System, using XIM Protocol."));
    }

    void scim_setup_module_load_config (const ConfigPointer &config)
    {
        load_config (config);
    }

    void scim_setup_module_save_config (const ConfigPointer &config)
    {
        save_config (config);
    }

    bool scim_setup_module_query_changed ()
    {
        return query_changed ();
    }
} // extern "C"

// Internal data structure
struct KeyboardConfigData
{
    const char *key;
    const char *label;
    const char *title;
    const char *tooltip;
    GtkWidget  *entry;
    GtkWidget  *button;
    String      data;
};

// Internal data declaration.
static bool __config_on_the_spot           = false;

static bool __have_changed                 = false;

static GtkWidget * __widget_on_the_spot           = 0;

static GtkTooltips * __widget_tooltips            = 0;

// Declaration of internal functions.

static KeyboardConfigData __config_keyboards [] =
{
    {
        // key
        SCIM_CONFIG_FRONTEND_KEYS_TRIGGER,
        // label
        N_("_Trigger:"),
        // title
        N_("Select the trigger keys"),
        // tooltip
        N_("The key events to turn on/off SCIM input method. "
           "Click on the button on the right to edit it."),
        // entry
        NULL,
        // button
        NULL,
        // data
        "Control+space"
    },
    {
        // key
        SCIM_CONFIG_FRONTEND_KEYS_NEXT_FACTORY,
        // label
        N_("_Next input method:"),
        // title
        N_("Select the next input method keys"),
        // tooltip
        N_("The key events to switch to the next input method. "
           "Click on the button on the right to edit it."),
        // entry
        NULL,
        // button
        NULL,
        // data
        "Control+Alt+Down,"
        "Control+Shift_R,"
        "Control+Shift_L"
    },
    {
        // key
        SCIM_CONFIG_FRONTEND_KEYS_PREVIOUS_FACTORY,
        // label
        N_("_Previous input method:"),
        // title
        N_("Select the previous input method keys"),
        // tooltip
        N_("The key events to switch to the previous input method. "
           "Click on the button on the right to edit it."),
        // entry
        NULL,
        // button
        NULL,
        // data
        "Control+Alt+Up,"
        "Shift+Control_R,"
        "Shift+Control_L"
    },
    {
        // key
        SCIM_CONFIG_FRONTEND_KEYS_SHOW_FACTORY_MENU,
        // label
        N_("_Show input method menu:"),
        // title
        N_("Select the show input method menu keys"),
        // tooltip
        N_("The key events to show the input method menu. "
           "Click on the button on the right to edit it."),
        // entry
        NULL,
        // button
        NULL,
        // data
        "Control+Alt+l,"
        "Control+Alt+m,"
        "Control+Alt+s,"
        "Control+Alt+Right,"
    },
    {
        // key
        NULL,
        // label
        NULL,
        // title
        NULL,
        // tooltip
        NULL,
        // entry
        NULL,
        // button
        NULL,
        // data
        ""
    },
};

// Declaration of internal functions.
static void
on_default_editable_changed          (GtkEditable     *editable,
                                      gpointer         user_data);

static void
on_default_toggle_button_toggled     (GtkToggleButton *togglebutton,
                                      gpointer         user_data);

static void
on_default_key_selection_clicked     (GtkButton       *button,
                                      gpointer         user_data);


static void
setup_widget_value ();

// Function implementations.
static GtkWidget *
create_setup_window ()
{
    static GtkWidget *window = 0;

    if (!window) {
        GtkWidget *table;
        GtkWidget *frame;
        GtkWidget *hbox;
        GtkWidget *label;
        int i;

        __widget_tooltips = gtk_tooltips_new ();

        // Create the toplevel box.
        window = gtk_vbox_new (FALSE, 0);
        gtk_widget_show (window);

        // Create the ToolBar setup block.
        frame = gtk_frame_new (_("XIM Settings"));
        gtk_widget_show (frame);
        gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
        gtk_box_pack_start (GTK_BOX (window), frame, FALSE, FALSE, 0);

        hbox = gtk_hbox_new (FALSE, 0);
        gtk_widget_show (hbox);
        gtk_container_add (GTK_CONTAINER (frame), hbox);
        gtk_container_set_border_width (GTK_CONTAINER (hbox), 4);

        __widget_on_the_spot = gtk_check_button_new_with_mnemonic (_("_On The Spot"));
        gtk_widget_show (__widget_on_the_spot);
        gtk_box_pack_start (GTK_BOX (hbox), __widget_on_the_spot, FALSE, FALSE, 4);

        // Create the Lookup Table setup block.
        frame = gtk_frame_new (_("Keyboard"));
        gtk_widget_show (frame);
        gtk_container_set_border_width (GTK_CONTAINER (frame), 4);
        gtk_box_pack_start (GTK_BOX (window), frame, FALSE, FALSE, 0);

        table = gtk_table_new (3, 3, FALSE);
        gtk_widget_show (table);
        gtk_container_add (GTK_CONTAINER (frame), table);
        gtk_table_set_row_spacings (GTK_TABLE (table), 4);
        gtk_table_set_col_spacings (GTK_TABLE (table), 8);

        for (i = 0; __config_keyboards [i].key; ++ i) {
            label = gtk_label_new (NULL);
            gtk_label_set_text_with_mnemonic (GTK_LABEL (label), _(__config_keyboards[i].label));
            gtk_widget_show (label);
            gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
            gtk_misc_set_padding (GTK_MISC (label), 4, 0);
            gtk_table_attach (GTK_TABLE (table), label, 0, 1, i, i+1,
                              (GtkAttachOptions) (GTK_FILL),
                              (GtkAttachOptions) (GTK_FILL), 4, 4);

            __config_keyboards [i].entry = gtk_entry_new ();
            gtk_widget_show (__config_keyboards [i].entry);
            gtk_table_attach (GTK_TABLE (table), __config_keyboards [i].entry, 1, 2, i, i+1,
                              (GtkAttachOptions) (GTK_FILL|GTK_EXPAND),
                              (GtkAttachOptions) (GTK_FILL), 4, 4);
            gtk_entry_set_editable (GTK_ENTRY (__config_keyboards[i].entry), FALSE);

            __config_keyboards[i].button = gtk_button_new_with_label ("...");
            gtk_widget_show (__config_keyboards[i].button);
            gtk_table_attach (GTK_TABLE (table), __config_keyboards[i].button, 2, 3, i, i+1,
                              (GtkAttachOptions) (GTK_FILL),
                              (GtkAttachOptions) (GTK_FILL), 4, 4);
            gtk_label_set_mnemonic_widget (GTK_LABEL (label), __config_keyboards[i].button);
        }

        // Connect all signals.
        g_signal_connect ((gpointer) __widget_on_the_spot, "toggled",
                          G_CALLBACK (on_default_toggle_button_toggled),
                          &__config_on_the_spot);

        for (i = 0; __config_keyboards [i].key; ++ i) {
            g_signal_connect ((gpointer) __config_keyboards [i].button, "clicked",
                              G_CALLBACK (on_default_key_selection_clicked),
                              &(__config_keyboards [i]));
            g_signal_connect ((gpointer) __config_keyboards [i].entry, "changed",
                              G_CALLBACK (on_default_editable_changed),
                              &(__config_keyboards [i].data));
        }

        // Set all tooltips.
        gtk_tooltips_set_tip (__widget_tooltips, __widget_on_the_spot,
                              _("If this option is checked, "
                                "the OnTheSpot input style will be used "
                                "when the client supports it."), NULL);

        for (i = 0; __config_keyboards [i].key; ++ i) {
            gtk_tooltips_set_tip (__widget_tooltips, __config_keyboards [i].entry,
                                  _(__config_keyboards [i].tooltip), NULL);
        }

        setup_widget_value ();
    }

    return window;
}

static void
setup_widget_value ()
{
    if (__widget_on_the_spot) {
        gtk_toggle_button_set_active (
            GTK_TOGGLE_BUTTON (__widget_on_the_spot),
            __config_on_the_spot);
    }

    for (int i = 0; __config_keyboards [i].key; ++ i) {
        if (__config_keyboards [i].entry) {
            gtk_entry_set_text (
                GTK_ENTRY (__config_keyboards [i].entry),
                __config_keyboards [i].data.c_str ());
        }
    }
}

static void
load_config (const ConfigPointer &config)
{
    if (!config.null ()) {
        __config_on_the_spot = 
            config->read (String (SCIM_CONFIG_FRONTEND_X11_ONTHESPOT),
                          __config_on_the_spot);

        for (int i = 0; __config_keyboards [i].key; ++ i) {
            __config_keyboards [i].data =
                config->read (String (__config_keyboards [i].key),
                              __config_keyboards [i].data);
        }

        setup_widget_value ();

        __have_changed = false;
    }
}

static void
save_config (const ConfigPointer &config)
{
    if (!config.null ()) {
        config->write (String (SCIM_CONFIG_FRONTEND_X11_ONTHESPOT),
                       __config_on_the_spot);

        for (int i = 0; __config_keyboards [i].key; ++ i) {
            config->write (String (__config_keyboards [i].key),
                          __config_keyboards [i].data);
        }

        __have_changed = false;
    }
}

static bool
query_changed ()
{
    return __have_changed;
}

static void
on_default_editable_changed (GtkEditable *editable,
                             gpointer     user_data)
{
    String *str = static_cast <String *> (user_data);

    if (str) {
        *str = String (gtk_entry_get_text (GTK_ENTRY (editable)));
        __have_changed = true;
    }
}

static void
on_default_toggle_button_toggled (GtkToggleButton *togglebutton,
                                  gpointer         user_data)
{
    bool *toggle = static_cast<bool*> (user_data);

    if (toggle) {
        *toggle = gtk_toggle_button_get_active (togglebutton);
        __have_changed = true;
    }
}

static void
on_default_key_selection_clicked (GtkButton *button,
                                  gpointer   user_data)
{
    KeyboardConfigData *data = static_cast <KeyboardConfigData *> (user_data);

    if (data) {
        GtkWidget *dialog = scim_key_selection_dialog_new (_(data->title));
        gint result;

        scim_key_selection_dialog_set_keys (
            SCIM_KEY_SELECTION_DIALOG (dialog),
            data->data.c_str ());

        result = gtk_dialog_run (GTK_DIALOG (dialog));

        if (result == GTK_RESPONSE_OK) {
            const gchar *keys = scim_key_selection_dialog_get_keys (
                            SCIM_KEY_SELECTION_DIALOG (dialog));

            if (!keys) keys = "";

            if (String (keys) != data->data)
                gtk_entry_set_text (GTK_ENTRY (data->entry), keys);
        }

        gtk_widget_destroy (dialog);
    }
}

/*
vi:ts=4:nowrap:expandtab
*/
