/***************************************************************************
    smb4kbrowserwidget.h  -  Network browse widget class for Smb4K.
                             -------------------
    begin                : Sam Feb 22 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef SMB4KBROWSERWIDGET_H
#define SMB4KBROWSERWIDGET_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qstringlist.h>

// KDE includes
#include <klistview.h>
#include <kconfig.h>

// application specific includes
#include "../core/smb4knetworkitems.h"


/**
 * This is the browser widget class of Smb4K.
 */
class Smb4KBrowserWidgetItem;
class Smb4KBrowserWidget : public KListView
{
  Q_OBJECT

  friend class Smb4KView;
  friend class Smb4KBrowserWidgetItem;

  public:
    /**
     * The constructor.
     */
    Smb4KBrowserWidget( QWidget *parent = 0, const char *name = 0 );
    /**
     * The destructor.
     */
    ~Smb4KBrowserWidget();

    /**
      * This function is called, when the user tries to open an item in the list view.
      * Takes the selected item and a bool as arguments. For more information
      * look at the class documentation of KListView.
      */
    void setOpen( QListViewItem *item, bool on );
    /**
      * This function inserts a host item into the browser window.
      */
    void insertItem( Smb4KHostItem *item );
    /**
      * This function reads the options for the browser widget.
      */
    void readOptions();
    
  protected slots:
    /**
     * Is called, when a list view item is to be mounted.
     */
    void slotItemExecuted( QListViewItem *item );
    /**
     * Is called whenever a QListViewItem is expanded or collapsed.
     */
    void slotItemExpandedCollapsed( QListViewItem *item );
    /**
     * Is called, when the selection in the list view is changed. (Is
     * connected to the selectionChanged() signal.
     */
    void slotSelectionChanged( QListViewItem *item );
    /**
     * This slot is invoked by the rightButtonPressed() signal.
     */
    void slotRightButtonPressed( QListViewItem *item, const QPoint &point, int col );
    /**
     * Is called, if the user clicked somewhere in the list view.
     */
    void slotItemClicked( QListViewItem *item );
    /**
     * Takes the workgroups/masters list and puts it into the list
     * view.
     */
    void slotWorkgroups( const QValueList<Smb4KWorkgroupItem *> &list );
    /**
     * Takes a list of the members of a workgroup and inserts them into the
     * list view as children of the workgroup item.
     */
    void slotMembers( const QValueList<Smb4KHostItem *> &list );
    /**
     * Takes the shares of a queried host and inserts them as children of
     * the host item.
     */
    void slotShares( const QValueList<Smb4KShareItem *> &list );
    /**
     * Mounts a share.
     */
    void slotMountShare();
    /**
     * Opens the askpass dialod.
     */
    void slotAskPass();
    /**
     * Opens the preview dialog.
     */
    void slotPreview();
    /**
     * Adds a bookmark to the list of bookmarks.
     */
    void slotAddBookmark();
    /**
     * Takes an Smb4KHostItem, extracts the IP address of the host and adds 
     * it to the appropriate item in the KListView.
     * @param item Smb4KHostItem for which the IP address should be entered
     */
    void slotAddIPAddress( Smb4KHostItem *item );
    /**
     * Opens the print dialog for a printer share.
     */
    void slotPrint();
    /**
     * This slot marks the mounted shares. It is called by the 
     * Smb4KMounter::importedShares signal.
     */
    void slotMarkShares();

  private:
    /**
     * The KConfig object.
     */
    KConfig *m_config;
    /**
     * This value is TRUE, if the hidden shares are to be shown, otherwise
     * it is FALSE.
     */
    bool m_hidden;
    /**
     * This value is TRUE, if the IPC$ shares are to be shown, otherwise
     * it is FALSE.
     */
    bool m_ipc;
    /**
     * This value is TRUE, if the ADMIN$ shares are to be shown, otherwise
     * it is false.
     */
    bool m_admin;
    /**
     * This value is TRUE, if the printer shares are to be shown, otherwise
     * it is false.
     */
    bool m_printer;
    /**
     * This function shows a dialog, that asks for a login name, if the user
     * wants to log into a 'homes' share.
     */
    const QString specifyUser( const QString &host );
    /**
     * Enumeration for the columns in the list view.
     */
    enum Columns{ Network = 0, Type = 1, IP = 2, Comment = 3 };
};


/**
 * This class cares about the items that are inserted into the browser widget.
 */

class Smb4KBrowserWidget;
class Smb4KBrowserWidgetItem : public QListViewItem
{
  public:
    /**
     * The constructor for toplevel (workgroup) items.
     */
    Smb4KBrowserWidgetItem( QListView *parent, Smb4KWorkgroupItem *item );
    /**
     * The constructor for the workgroup member items.
     */
    Smb4KBrowserWidgetItem( QListViewItem *parent, Smb4KHostItem *item );
    /**
     * The constructor for the share items.
     */
    Smb4KBrowserWidgetItem( QListViewItem *parent, Smb4KShareItem *item );
    /**
     * The destructor.
     */
    virtual ~Smb4KBrowserWidgetItem();
    /**
     * Returns a pointer to the Smb4KWorkgoupItem object.
     */
    Smb4KWorkgroupItem *getWorkgroupItem() const { return m_workgroup; }
    /**
     * Returns a pointer to the Smb4KHostItem object.
     */
    Smb4KHostItem *getHostItem() const { return m_host; }
    /**
     * Returns a pointer to the Smb4KShareItem.
     */
    Smb4KShareItem *getShareItem() const { return m_share; }
    /**
     * Tells the Smb4KBrowserWidgetItem if the share is already mounted or not and sets the 
     * folder icon accordingly (open folder == mounted, closed folder == not mounted). 
     * @param mounted Has to be TRUE if the share is mounted and FALSE otherwise.
     * This function will only work with shares. Workgroup and host items are ignored.
     */
    void setMounted( bool mounted );
    /**
     * Returns TRUE if the share is mounted and FALSE otherwise.
     */
    bool isMounted() { return m_mounted; }
     
  protected:
    /**
     * Reimplemented from QListViewItem.
     */
    void paintCell( QPainter *p, const QColorGroup &cg, int column, int width, int align );
  
  private:
    /**
     * The Smb4KWorkgroupItem object.
     */
    Smb4KWorkgroupItem *m_workgroup;
    /**
     * The Smb4KHostItem object.
     */
    Smb4KHostItem *m_host;
    /**
     * The Smb4KShareItem object.
     */
    Smb4KShareItem *m_share;
    /**
     * Is TRUE if the share is mounted and FALSE otherwise.
     * It will only be set, if Smb4KBrowserWidgetItem::depth() == 2,
     * i. e. it is a share.
     */
    bool m_mounted;
};

#endif
