/*
	SRG - Squid Report Generator
	Report by site
	Copyright 2005 University of Waikato

	This file is part of SRG.

	SRG is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	SRG is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with SRG; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

*/

#include "UserReport.h"
#include "SiteReport.h"
#include "LocationReport.h"
#include "srg.h"
#include "prototypes.h"

SiteReport::~SiteReport() {

	list<LocationReport*>::const_iterator iter;
	for (iter=locations.begin(); iter != locations.end(); iter++) {
		delete (*iter);
	}

}

void SiteReport::process_line(const log_line *line) {

	if (srg.debug)
		fprintf(stderr, "In SiteReport::process_line for site '%s'\n",
				mName);

	/* Work out the location of this line */
	char *location = strdup(line->request->location);

	LocationReport *theLocation = (LocationReport *)findLocation(location);
	if (theLocation==NULL) {
		if (srg.debug)
			fprintf(stderr, "Site Location not found in %s - "
					"creating %s\n", mName, location);
		/* The location does not exist. Create it. */
		theLocation = new LocationReport(location);
		locations.push_back(theLocation);
	}

	/* Process the line */
	theLocation->process_line(line);
	
	free(location);
	
}

void SiteReport::generate_report(const char *basename, const char *user) {

	char *t = NULL;
	char *basepath = NULL;
	char *filename = NULL;
	FILE *outfile = NULL;
	
	/* Create the output filename */
	t = md5this(mName);
	asprintf(&basepath, "%s/%s", basename, t);
	free(t);
	asprintf(&t, "%s/%s", srg.outputDir, basepath);
	if (mkdir(t, 0755) == -1 && errno != EEXIST) {
		fprintf(stderr, "Error (%s) creating directory in " 
				"site report for %s: %s\n", strerror(errno), 
				mName, t);
		exit(1);
	}
	if (srg.debug)
		fprintf(stderr, "Creating site report in %s\n", t);
	free(t);
		
	/* Create the index file */
	asprintf(&filename, "%s/%s/%s", srg.outputDir, basepath, 
			srg.indexfname);
	outfile = fopen(filename, "w");
	if(outfile==NULL) {
		fprintf(stderr,"SiteReport: Cannot open output file: %s\n",
				filename);
		exit(1);
	}
	free(filename);

	/* Header & Title */
	if (srg.groupBy > 0) {
		html_header(outfile, "../../../");
	} else {
		html_header(outfile, "../../");
	}
        
	fprintf(outfile, "<!-- SRG %s (%s) Generated SiteReport -->\n", 
		version, HOME_URL);

	/* Misc Stats */
	fprintf(outfile, "<center><table cellpadding=2 cellspacing=2>");
	fprintf(outfile, "<tr><td class=\"bodyText\">Period:</td><td class=\""
			"bodyText\">%d %s %d",
			localtime(&srg.startTime)->tm_mday, 
			month_names[localtime(&srg.startTime)->tm_mon], 
			localtime(&srg.startTime)->tm_year+1900);
	fprintf(outfile, " - %d %s %d</td></tr>", 
			localtime(&srg.endTime)->tm_mday, 
			month_names[localtime(&srg.endTime)->tm_mon], 
			localtime(&srg.endTime)->tm_year+1900);
	if (srg.groupBy > 0) {
		fprintf(outfile, "<tr><td class=\"bodyText\">Group:</td><td "
				"class=\"bodyText\">%s</td></tr>", user);
	}
	fprintf(outfile, "<tr><td align=right class=\"bodyText\">Site:</td><td"
			" align=left class=\"bodyText\">&nbsp;<a href=\""
			"http://%s/\">%s</a></td></tr>", mName, mName);
	fprintf(outfile, "</table></center>");

	/* Notices Row */
	fprintf(outfile, "<div align=\"center\" id=\"srg-message\">"
			"&nbsp;</div>\n");

	/* Main Table */
	fprintf(outfile, "<center><table cellpadding=4 cellspacing=0 "
			"id=\"srgtable\"><thead><tr>"
			"<th>LOCATION</th><th>HITS</th><th>BYTES</th><th>"
			"BYTES %%</th><th>HIT</th><th>MISS</th>");
	if (srg.showtimes)
		fprintf(outfile, "<th>TIME%%</th><th>TIME(ms)</th>");	
	if (srg.showrates)
		fprintf(outfile, "<th>RATE (kB/s)</th>");
	fprintf(outfile, "</tr></thead>\n");

	list<LocationReport*>::const_iterator iter;
	summary_info locationstats;
	float percentin = 0;
	float percentout = 0;
	float timespent = 0;
	float bytespercent = 0;
	summary_info thissitestats = getStats();
	float hitspercentot = 0;
	float missespercentot = 0;
	int rows = 0;

	/* Initialise the Summary Stats Structure */
	locationstats.connects = 0;
	locationstats.bytesTransferred = 0;
	locationstats.hits = 0;
	locationstats.misses = 0;
	locationstats.bytesHit = 0;
	locationstats.bytesMissed = 0;
	locationstats.timeSpent = 0;
	locationstats.deniedHits = 0;

	locations.sort(LessByBytesTransferred<LocationReport>());

	for (iter=locations.begin(); iter != locations.end(); iter++) {
		locationstats = (*iter)->getStats();
		if (srg.locationStats) {
			(*iter)->generate_report(basepath, user, mName);
		}
		char *location = (*iter)->getName();

		if (locationstats.hits+locationstats.misses == 0) {
			percentin = -1;
			percentout = -1;
		} else {
			hitspercentot += percentin = 
				(float)(100*locationstats.hits) / 
				(locationstats.hits+locationstats.misses);
			missespercentot += percentout = 
				(float)(100*locationstats.misses) / 
				(locationstats.hits+locationstats.misses);
		}
		if (thissitestats.timeSpent == 0) {
			timespent = 100;
		} else {
			timespent = (float)(100*locationstats.timeSpent) /
				thissitestats.timeSpent;
		}
		if (thissitestats.bytesTransferred == 0) {
			bytespercent = 100;
		} else { 
			bytespercent = (float)
				(100*locationstats.bytesTransferred) / 
				thissitestats.bytesTransferred;
		}
		
		fprintf(outfile, "<tr%s>", 
				((rows%2)==0) ? " class=\"highlightRow\"" : "");
		char *locationhref = NULL;
		char *locationname = NULL;
		if (srg.locationStats) {
			/* Link to location report */
			t = md5this(location);
			asprintf(&locationhref, "%s.%s", t, 
					srg.outputMode == OUTPUT_PHP ? "php" : "html");
			free(t);
			char tmpname[41];
			snprintf(tmpname, 40, "%s", location);
			asprintf(&locationname, "%s%s", tmpname, 
					(strlen(location)>40) ? "&#133;" : "");
		} else {
			asprintf(&locationhref, "http://%s%s", mName,
					(strcasecmp(location, "^index^")==0) ?
					"/" : location);
			char tmpname[41];
			snprintf(tmpname, 40, "%s", location);
			asprintf(&locationname, "%s%s", tmpname, 
					(strlen(location)>40) ? "&#133;" : "");
		}
		fprintf(outfile, "<td class=\"bodyText\"><a href=\"%s\">"
				"%s</a></td>", locationhref, locationname);
		free(locationhref);
		free(locationname);
		t = FormatOutput(locationstats.connects);
		fprintf(outfile, "<td class=\"cellNum\">%s</td>", t);
		free(t);
		t = FormatOutput(locationstats.bytesTransferred);
		fprintf(outfile, "<td class=\"cellNum\">%s</td>", t);
		free(t);
		fprintf(outfile, "<td class=\"cellNum\">%2.2f%%</td>", 
				bytespercent);
		if (percentin != -1) {
			fprintf(outfile, "<td class=\"cellNum\">%2.2f%%</td>"
					"<td class=\"cellNum\">%2.2f%%</td>",
					percentin, percentout);
		} else {
			fprintf(outfile, "<td class=\"cellNum\">-</td>"
					"<td class=\"cellNum\">-</td>");
		}
		if (srg.showtimes) {
			t = FormatOutput(locationstats.timeSpent);
			fprintf(outfile, "<td class=\"cellNum\">%2.2f%%</td>"
					"<td class=\"cellNum\">%s</td>",
					timespent, t);
			free(t);
		}
		if (srg.showrates) {
			if (locationstats.timeSpent == 0) {
				fprintf(outfile, "<td class=\"cellNum\">"
						"-</td>");
			} else {
				fprintf(outfile, "<td class=\"cellNum\">"
					"%.2f</td>", 
					(float)locationstats.bytesTransferred/
					(float)locationstats.timeSpent);
			}
		}
		fprintf(outfile, "</tr>\n");
		rows++;
	}

	if (rows == 0) {
		percentin = percentout = -1;
	} else {
		percentin = hitspercentot / rows;
		percentout = missespercentot / rows;
	}

	/* Totals Row */
	fprintf(outfile, "<tfoot><tr><th class=\"cellText\">Totals:</th>");
	t = FormatOutput(thissitestats.connects);
	fprintf(outfile, "<th class=\"cellNum\">%s</th>", t);
	free(t);
	t = FormatOutput(thissitestats.bytesTransferred);
	fprintf(outfile, "<th class=\"cellNum\">%s</th>", t);
	free(t);
	fprintf(outfile, "<th class=\"cellNum\">100%%</th>");
	if (percentin != -1) {
		fprintf(outfile, "<th align=\"center\" class=\"cellNum\">"
				"%2.2f%%</th><th align=\"center\" class=\""
				"cellNum\">%2.2f%%</th>", percentin, 
				percentout);
	} else {
		fprintf(outfile, "<th align=\"center\" class=\"cellNum\">"
				"-</th><th align=\"center\" class=\""
				"cellNum\">-</th>");
	}
	if (srg.showtimes) {
		t = FormatOutput(thissitestats.timeSpent);
		fprintf(outfile, "<th class=\"cellNum\">100%%</th><th class=\""
				"cellNum\">%s</th>", t);
		free(t);
	}
	if (srg.showrates) {
		if (locationstats.timeSpent == 0) {
			fprintf(outfile, "<th class=\"cellNum\">-</th>");
		} else {
			fprintf(outfile, "<th class=\"cellNum\">%2.0f</th>",
					(float)thissitestats.bytesTransferred/
					(float)thissitestats.timeSpent);
		}
	}
	fprintf(outfile, "</tr></tfoot>\n\t</table><br>");

	if (srg.authenticate) {
		fprintf(outfile, "\n\t<?php } else {\n report_error(\""
				"Could not authenticate user\");\n}?>");
	}

	fprintf(outfile, "\n\t<br><a href=\"../%s\">Back</a> to sites page",
			srg.indexfname);

	/* Finish off the HTML */
	if (srg.groupBy > 0) {
		html_footer(outfile, "../../../");
	} else {
		html_footer(outfile, "../../");
	}
	fclose(outfile);
	
	free(basepath);

}

void SiteReport::updateStats() {

	zeroStats();

	list<LocationReport*>::const_iterator iter;

	for (iter=locations.begin(); iter != locations.end(); iter++) {
		summary_info locationStats = (*iter)->getStats();
		stats.connects += locationStats.connects;
		stats.bytesTransferred += locationStats.bytesTransferred;
		stats.timeSpent += locationStats.timeSpent;
		stats.hits += locationStats.hits;
		stats.bytesHit += locationStats.bytesHit;
		stats.misses += locationStats.misses;
		stats.bytesMissed += locationStats.bytesMissed;
		stats.deniedHits += locationStats.deniedHits;
	}
	

}

LocationReport *SiteReport::findLocation(char * location) {

	list<LocationReport*>::const_iterator iter;

	/* Iterate through list and compare each element. */
	for (iter=locations.begin(); iter != locations.end(); iter++){
		if(strcasecmp((*iter)->getName(),location) ==0)
			return (LocationReport *)(*iter);
	}

	return NULL;
}

// Get statistics about this report
summary_info SiteReport::getStats() {

	summary_info returnInfo;

	updateStats();

	returnInfo = stats;

	return returnInfo;

}
