/*
 * ui.c - a module for UI
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1996-1997 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#if defined(HAVE_STRING_H) || defined(STDC_HEADERS)
# include <string.h>
#else
# include <strings.h>
#endif
#ifdef HAVE_MALLOC_H
# include <malloc.h>
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#include <sys/types.h>
#include <sys/param.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/StringDefs.h>
#include <X11/Xresource.h>
#include <X11/Xos.h>
#include <Xm/Xm.h>
#include <Xm/Separator.h>

#include "cf-xmdvi.h"
#include "dvi-2_6.h"
#include "dev.h"
#include "defs.h"
#include "resource.h"
#include "paper.h"
#include "window.h"
#include "ui.h"
#include "key.h"
#include "../X11/x11-defs.h"

Public int       x_ui_command;

Private Display  *xdisp = NULL;
Private Widget   x_toplevel;
Private XEvent   x_event;
Private Visual   *x_visual;
Private int      x_depth;
Private Colormap x_cmap;

Private void    put_pixmap(int,int,DVI_PIXMAP_RGB,long,long);
Private void    put_graymap(int,DVI_GRAYMAP,long,long);
Private void    x_rectangle(long,long,long,long,GC);
Private void    x_paper_frame(void);




Public int
x_ui_loop(void)
{
  int         r, page;
  PAGE_CACHE  pc; 

  ui_set_page_jump_range();
  key_reset();

  for (;;){
    while (x_ui_command == UI_CMD_NONE){
      XtNextEvent(&x_event);
      XtDispatchEvent(&x_event);
    }

    switch (x_ui_command){
    case UI_CMD_NONE:
      continue;
    default:
      fprintf(stderr, "%s: Unknown UI command: %d\n", 
	      "XMDVI Internal Error", x_ui_command);
      exit(1);
    case UI_CMD_QUIT:
      return x_ui_command;
    case UI_CMD_RELOAD:
    case UI_CMD_LOAD:
      if (DviFileNameNew == NULL){
	fprintf(stderr, "%s: New DVI file name is unknown.\n", 
		"XMDVI Internal Error");
	exit(1);
      }
      return x_ui_command;
    case UI_CMD_REDRAW:
      if (XVAR(DviDev,page_cache) != NULL){
	pc = x_get_cache(DviCurrentPage);
	pc->page = -1;
      }
      break;
    case UI_CMD_PAGE:
      break;
    }
    x_ui_command = UI_CMD_NONE;
    if (DviFile == NULL)
      continue;

    /*
     * Note: 'DviCurrentPage' may be modified during page draw by
     *  device polling. Thus, we must not use 'DviCurrentPage'.
     */
    page = DviCurrentPage;
    pc = x_get_cache(page);

    if (pc->page == DviCurrentPage){
      XCopyArea(xdisp, pc->pix, 
		XVAR(DviDev,view), XVAR(DviDev,gc_copy), 
		0, 0, XVAR(DviDev,page_width), XVAR(DviDev,page_height), 
		0, 0);
      if (pc->cont == NULL){
	continue;
      }
    } else {
      if (pc->cont != NULL){
	DVI_FREE_CONTEXT(DviFile, DviDev, pc->cont);
	pc->cont = NULL;
      }
    }

    x_cursor_change(CURSOR_DRAWING);
    pc->page = DviCurrentPage;
    if (pc->cont == NULL){
      XFillRectangle(xdisp, XVAR(DviDev,view), 
		     XVAR(DviDev,gc_clear), 0, 0, 
		     XVAR(DviDev,page_width), XVAR(DviDev,page_height));
      XFillRectangle(xdisp, pc->pix,
		     XVAR(DviDev,gc_pc_clear), 0, 0, 
		     XVAR(DviDev,page_width), XVAR(DviDev,page_height));
      r = DVI_DRAW_PAGE(DviFile, DviDev, DviCurrentPage, DviShrinkFactor);
    } else {
      r = DVI_CONTINUE_TO_CONTEXT(DviFile, DviDev, pc->cont, DviShrinkFactor);
      DVI_FREE_CONTEXT(DviFile, DviDev, pc->cont);
      pc->cont = NULL;
    }

    if (r == DVI_DRAW_INTERRUPTED){
      pc->cont = DVI_GET_CURRENT_CONTEXT(DviFile, DviDev);
    } else {
      x_paper_frame();
      x_cursor_change(CURSOR_READY);
    }
  }
}

Public int 
x_open_display(void) 
{
  int     argc = 1;
  char    *argv[] = { PROG_NAME, NULL };

  x_toplevel = XtInitialize(PROG_NAME, PROG_NAME_RESOURCE,
			    NULL, 0, &argc, argv);
  xdisp    = XtDisplay(x_toplevel);
  x_visual = DefaultVisual(xdisp, 0);
  x_depth  = DefaultDepth(xdisp, 0);
  x_cmap   = DefaultColormap(xdisp, 0);

  return 0;
}

Public int
x_visual_color_direct(void)
{
  switch (x_visual->class){
  case StaticGray:
  case GrayScale:
  case StaticColor:
  case PseudoColor:
    return 0;
  case TrueColor:
  case DirectColor:
    return 1;
  }
  return 0;
}

Public int
x_visual_color(void)
{
  switch (x_visual->class){
  case StaticGray:
  case GrayScale:
    return 0;
  case StaticColor:
  case PseudoColor:
  case TrueColor:
  case DirectColor:
    return 1;
  }
  return 0;
}

Public int
x_ui_make(void)
{
  XSTAT      xstat;
  XColor     xc, xc0;

  if ((xstat = calloc(1, sizeof(struct s_xstat))) == NULL)
    return -1;
  SET_XVAR(DviDev,xstat);
  XVAR(DviDev,xdisp) = XtDisplay(x_toplevel);

  XVAR(DviDev,page_cache)      = NULL;
  XVAR(DviDev,page_cache_size) = 0;

  XVAR(DviDev,page_width)   = 0;
  XVAR(DviDev,page_height)  = 0;
  XVAR(DviDev,offset_x)     = 0;
  XVAR(DviDev,offset_y)     = 0;

  XAllocNamedColor(XVAR(DviDev,xdisp), x_cmap, 
		   RES(DviDev,color_char), &xc, &xc0); 
  XVAR(DviDev,default_color_char) = XVAR(DviDev,color_char) = xc.pixel;
  XAllocNamedColor(XVAR(DviDev,xdisp), x_cmap, 
		   RES(DviDev,color_paper), &xc, &xc0); 
  XVAR(DviDev,default_color_paper) = XVAR(DviDev,color_paper) = xc.pixel;
  XAllocNamedColor(XVAR(DviDev,xdisp), x_cmap, 
		   RES(DviDev,color_frame), &xc, &xc0); 
  XVAR(DviDev,color_frame) = xc.pixel;

#if !defined(HAVE_OPENWIN)
  XInitImage(&XVAR(DviDev,image));
#endif
  XVAR(DviDev,image.format)           = XYBitmap;
  XVAR(DviDev,image.xoffset)          = 0;
  XVAR(DviDev,image.byte_order)       = MSBFirst;
  XVAR(DviDev,image.bitmap_unit)      = 8;
  XVAR(DviDev,image.bitmap_bit_order) = MSBFirst;
  XVAR(DviDev,image.bitmap_pad)       = 8;
  XVAR(DviDev,image.depth)            = 1;

  if (x_make_panel(x_toplevel) < 0)
    return -1;
  if (x_make_preview(x_toplevel) < 0)
    return -1;

  return 0;
}

Public void
x_bell(void)
{
  XBell(xdisp, 50);
  XFlush(xdisp);
}

Public void
x_alloc_page_cache(double shrink, int w, int h)
{
  XGCValues   gcval;
  PAGE_CACHE  pc; 
  int         p;
  double      r;

  if (XVAR(DviDev,page_cache) != NULL){
    for (p = 0; p < XVAR(DviDev,page_cache_size); p++){
      if (XVAR(DviDev,page_cache)[p].pix != 0)
	XFreePixmap(xdisp, XVAR(DviDev,page_cache)[p].pix);
      if (XVAR(DviDev,page_cache)[p].cont != NULL)
	DVI_FREE_CONTEXT(DviFile, DviDev, XVAR(DviDev,page_cache)[p].cont);
    }
    free(XVAR(DviDev,page_cache));
    XVAR(DviDev,page_cache) = NULL;
    XVAR(DviDev,page_cache_size) = 0;
    XFreeGC(xdisp, XVAR(DviDev,gc_pc_image_mask));
    XFreeGC(xdisp, XVAR(DviDev,gc_pc_image_put));
    XFreeGC(xdisp, XVAR(DviDev,gc_pc_image_rgb));
    XFreeGC(xdisp, XVAR(DviDev,gc_pc_rect));
    XFreeGC(xdisp, XVAR(DviDev,gc_pc_frame));
    XFreeGC(xdisp, XVAR(DviDev,gc_pc_clear));
  }
  
  r = shrink / RES(DviDev,shrink); 
  XVAR(DviDev,page_cache_size) 
    = RES(DviDev,page_cache) * r * r;
  if (XVAR(DviDev,page_cache_size) < 1)
    XVAR(DviDev,page_cache_size) = 1;
  if (XVAR(DviDev,page_cache_size) > 2 * RES(DviDev,page_cache))
    XVAR(DviDev,page_cache_size) = 2 * RES(DviDev,page_cache);

  XVAR(DviDev,page_cache) 
       = calloc(XVAR(DviDev,page_cache_size), sizeof(struct s_page_cache));
  if (XVAR(DviDev,page_cache) == NULL){
    fprintf(stderr, "No memory\n");
    exit(1);
  }

  for (p = 0; p < XVAR(DviDev,page_cache_size); p++){
    XVAR(DviDev,page_cache)[p].w           = w;
    XVAR(DviDev,page_cache)[p].h           = h;
    XVAR(DviDev,page_cache)[p].pix         = 0;   /* alloced on demand */
    XVAR(DviDev,page_cache)[p].page        = -1;
    XVAR(DviDev,page_cache)[p].cont        = NULL;
    XVAR(DviDev,page_cache)[p].pending_eps = 0;
    XVAR(DviDev,page_cache)[p].n_eps_figs  = 0;
  }

  pc = x_get_cache(DviCurrentPage);

  gcval.function   = GXandInverted;
  XVAR(DviDev,gc_pc_image_mask) 
    = XCreateGC(xdisp, pc->pix, GCFunction, &gcval);
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_image_mask), 0xffffffff);
  XSetBackground(xdisp, XVAR(DviDev,gc_pc_image_mask), 0x00000000);

  gcval.function = GXor;
  XVAR(DviDev,gc_pc_image_put)
    = XCreateGC(xdisp, pc->pix, GCFunction, &gcval);
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_image_put), 
		 XVAR(DviDev,default_color_char));
  XSetBackground(xdisp, XVAR(DviDev,gc_pc_image_put),
		 0x00000000);

  XVAR(DviDev,gc_pc_image_rgb) = XCreateGC(xdisp, pc->pix, 0, NULL);
  XVAR(DviDev,gc_pc_rect)      = XCreateGC(xdisp, pc->pix, 0, NULL);
  XVAR(DviDev,gc_pc_frame)     = XCreateGC(xdisp, pc->pix, 0, NULL);
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_frame), XVAR(DviDev,color_frame));
  XVAR(DviDev,gc_pc_clear)     = XCreateGC(xdisp, pc->pix, 0, NULL);
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_clear), XVAR(DviDev,color_paper));
}

PAGE_CACHE 
x_get_cache(int page)
{
  int         slot;
  PAGE_CACHE  pc; 

  slot = page % XVAR(DviDev,page_cache_size);
  pc = &XVAR(DviDev,page_cache)[slot];
  if (pc->pix == 0){
    pc->pix = XCreatePixmap(xdisp, DefaultRootWindow(xdisp), 
			    pc->w, pc->h, x_depth);
  } 
  return pc;
}

Public void
x_cancel_page_cache(void)
{
  int  p;

  if ((DviDev == NULL) || (XVAR(DviDev,page_cache) == NULL))
    return;
  for (p = 0; p < XVAR(DviDev,page_cache_size); p++){
    XVAR(DviDev,page_cache)[p].page        = -1;
    if (XVAR(DviDev,page_cache)[p].cont != NULL)
      DVI_FREE_CONTEXT(DviFile, DviDev, XVAR(DviDev,page_cache)[p].cont);
    XVAR(DviDev,page_cache)[p].cont        = NULL;
    XVAR(DviDev,page_cache)[p].pending_eps = 0;
    XVAR(DviDev,page_cache)[p].n_eps_figs  = 0;
  }
}

Public void
x_cancel_page_cache_by_eps_mode_change(void)
{
  int  p;

  if ((DviDev == NULL) || (XVAR(DviDev,page_cache) == NULL))
    return;
  for (p = 0; p < XVAR(DviDev,page_cache_size); p++){
    if (XVAR(DviDev,page_cache)[p].n_eps_figs > 0){
      XVAR(DviDev,page_cache)[p].page        = -1;
      if (XVAR(DviDev,page_cache)[p].cont != NULL)
	DVI_FREE_CONTEXT(DviFile, DviDev, XVAR(DviDev,page_cache)[p].cont);
      XVAR(DviDev,page_cache)[p].cont        = NULL;
      XVAR(DviDev,page_cache)[p].pending_eps = 0;
      XVAR(DviDev,page_cache)[p].n_eps_figs  = 0;
    }
  }
}


Public int
x_poll(void)
{
  while (XtPending()){
    XtNextEvent(&x_event);
    XtDispatchEvent(&x_event);
    if (x_ui_command != UI_CMD_NONE)
      return 1;
  }
  return 0;
}




/*
 * Color Change 
 */

Public void  
x_change_color_fg(double r, double g, double b)
{
  XColor        xc;
  unsigned long pix;

  if ((r < 0) || (g < 0) || (b < 0)){
    pix = XVAR(DviDev,default_color_char);
  } else {
    xc.red   = (unsigned int)(0xffff * r) & 0xffff;
    xc.green = (unsigned int)(0xffff * g) & 0xffff;
    xc.blue  = (unsigned int)(0xffff * b) & 0xffff;
    XAllocColor(xdisp, x_cmap, &xc);
    pix = xc.pixel;
  }
  XVAR(DviDev,color_char) = pix;
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_image_put), pix);
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_rect),  pix);
}

Public void  
x_change_color_bg(double r, double g, double b, int page_resume_flag)
{
  XColor        xc;
  PAGE_CACHE    pc; 
  unsigned long pix;

  if ((r < 0) || (g < 0) || (b < 0)){
    pix = XVAR(DviDev,default_color_paper);
  } else {
    xc.red   = (unsigned int)(0xffff * r) & 0xffff;
    xc.green = (unsigned int)(0xffff * g) & 0xffff;
    xc.blue  = (unsigned int)(0xffff * b) & 0xffff;
    XAllocColor(xdisp, x_cmap, &xc);
    pix = xc.pixel;
  }
  XVAR(DviDev,color_paper) = pix;

  XSetForeground(xdisp, XVAR(DviDev,gc_pc_clear), pix);
  XSetForeground(xdisp, XVAR(DviDev,gc_clear), pix);
  if (page_resume_flag == 0){
    pc = x_get_cache(DviCurrentPage);
    XFillRectangle(xdisp, pc->pix, 
		   XVAR(DviDev,gc_pc_clear), 0, 0, 
		   XVAR(DviDev,page_width), XVAR(DviDev,page_height));
    XFillRectangle(xdisp, XVAR(DviDev,view), 
		   XVAR(DviDev,gc_clear), 0, 0, 
		   XVAR(DviDev,page_width), XVAR(DviDev,page_height));
  }
}

Public void  
x_change_color_gray_fg(double gs)
{
  XColor        xc, xcfg, xcbg;
  unsigned long pix;

  if (gs < 0){
    pix = XVAR(DviDev,default_color_char);
  } else {
    xcfg.pixel = XVAR(DviDev,default_color_char);
    xcbg.pixel = XVAR(DviDev,default_color_paper);
    XQueryColor(xdisp, x_cmap, &xcfg);
    XQueryColor(xdisp, x_cmap, &xcbg);
    xc.red   = (unsigned int)(xcbg.red   - (xcbg.red   - xcfg.red)   * gs);
    xc.green = (unsigned int)(xcbg.green - (xcbg.green - xcfg.green) * gs);
    xc.blue  = (unsigned int)(xcbg.blue  - (xcbg.blue  - xcfg.blue)  * gs);
    XAllocColor(xdisp, x_cmap, &xc);
    pix = xc.pixel;
  }
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_image_put), pix);
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_rect),  pix);
  XVAR(DviDev,color_char) = pix;
}

Public void  
x_change_color_gray_bg(double gs, int page_resume_flag)
{
  XColor         xc, xc0, xc1;
  PAGE_CACHE     pc; 
  unsigned long  pix;

  if (gs < 0){
    pix = XVAR(DviDev,default_color_paper);
  } else {
    xc0.pixel = XVAR(DviDev,default_color_paper);
    xc1.pixel = XVAR(DviDev,default_color_char);
    XQueryColor(xdisp, x_cmap, &xc0);
    XQueryColor(xdisp, x_cmap, &xc1);
    xc.red   = (unsigned int)(xc0.red   - (xc0.red   - xc1.red)   * gs);
    xc.green = (unsigned int)(xc0.green - (xc0.green - xc1.green) * gs);
    xc.blue  = (unsigned int)(xc0.blue  - (xc0.blue  - xc1.blue)  * gs);
    XAllocColor(xdisp, x_cmap, &xc);
    pix = xc.pixel;
  }
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_image_put), pix);
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_rect),  pix);
  XVAR(DviDev,color_paper) = pix;

  XSetForeground(xdisp, XVAR(DviDev,gc_pc_clear), pix);
  XSetForeground(xdisp, XVAR(DviDev,gc_clear), pix);
  if (page_resume_flag == 0){
    pc = x_get_cache(DviCurrentPage);
    XFillRectangle(xdisp, pc->pix, 
		   XVAR(DviDev,gc_pc_clear), 0, 0, 
		   XVAR(DviDev,page_width), XVAR(DviDev,page_height));
    XFillRectangle(xdisp, XVAR(DviDev,view), 
		   XVAR(DviDev,gc_clear), 0, 0, 
		   XVAR(DviDev,page_width), XVAR(DviDev,page_height));
  }
}

Public void  
x_reset_colors(void)
{
  XVAR(DviDev,color_char) = XVAR(DviDev,default_color_char);
  XVAR(DviDev,color_paper) = XVAR(DviDev,default_color_paper);

  XSetForeground(xdisp, XVAR(DviDev,gc_pc_image_put), XVAR(DviDev,color_char));
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_rect), XVAR(DviDev,color_char));
  XSetForeground(xdisp, XVAR(DviDev,gc_pc_clear), XVAR(DviDev,color_paper));
  XSetForeground(xdisp, XVAR(DviDev,gc_clear), XVAR(DviDev,color_paper));
}




/*
 * Put Bitmap & Rectangle 
 */
Public void
x_put_bitmap(DVI_BITMAP bitmap, long x, long y)
{
  PAGE_CACHE pc; 

  if ((bitmap->width == 0) || (bitmap->height == 0)) 
    return;

  x += XVAR(DviDev,offset_x);
  y += XVAR(DviDev,offset_y);

  XVAR(DviDev,image.width)          = bitmap->width;
  XVAR(DviDev,image.height)         = bitmap->height;
  XVAR(DviDev,image.bytes_per_line) = bitmap->raster;
  XVAR(DviDev,image.data)           = (char*)bitmap->bitmap;

  pc = x_get_cache(DviCurrentPage);
  XPutImage(xdisp, pc->pix,
	    XVAR(DviDev,gc_pc_image_mask), &XVAR(DviDev,image), 
	    0, 0, x, y, bitmap->width, bitmap->height);
  XPutImage(xdisp, pc->pix,
	    XVAR(DviDev,gc_pc_image_put), &XVAR(DviDev,image), 
	    0, 0, x, y, bitmap->width, bitmap->height);
  XCopyArea(xdisp, pc->pix, XVAR(DviDev,view), 
	    XVAR(DviDev,gc_copy), 
	    x, y, bitmap->width, bitmap->height, x, y);
}

Public void
x_put_rectangle(long x, long y, long w, long h)
{
  long   xw, yh;

  x += XVAR(DviDev,offset_x);
  y += XVAR(DviDev,offset_y);
  if (h == 0)
    h = 1;
  if (w == 0)
    w = 1;
  xw = w;
  if (x+w > XVAR(DviDev,page_width))
    xw = XVAR(DviDev,page_width) - x;
  yh = h;
  if (y+h > XVAR(DviDev,page_height))
    yh = XVAR(DviDev,page_height) - y;

  x_rectangle(x, y, xw, yh, XVAR(DviDev,gc_pc_rect));
}

Private void 
x_rectangle(long x, long y, long w, long h, GC gc)
{
  PAGE_CACHE pc; 

  pc = x_get_cache(DviCurrentPage);
  XFillRectangle(xdisp, pc->pix, gc, x, y, w, h);
  XCopyArea(xdisp, pc->pix, XVAR(DviDev,view),
	    XVAR(DviDev,gc_copy), x, y, w, h, x, y);
}

Private void 
x_paper_frame(void)
{
  x_rectangle(0, 0, XVAR(DviDev,page_width), 1, 
	      XVAR(DviDev,gc_pc_frame));
  x_rectangle(0, 0, 1, XVAR(DviDev,page_height), 
	      XVAR(DviDev,gc_pc_frame));
  x_rectangle(0, XVAR(DviDev,page_height)-1, XVAR(DviDev,page_width), 1, 
	      XVAR(DviDev,gc_pc_frame));
  x_rectangle(XVAR(DviDev,page_width)-1, 0, 1, XVAR(DviDev,page_height),
	      XVAR(DviDev,gc_pc_frame));
}


Public void
x_put_pixmap_rgb(DVI_PIXMAP_RGB pixmap_rgb, long x, long y)
{
  switch (x_visual->class){
  default:
  case StaticGray:
  case GrayScale:
    break;
  case StaticColor:
  case PseudoColor:
    put_pixmap(256, 16, pixmap_rgb, x, y);
    break;
  case TrueColor:
  case DirectColor:
    put_pixmap(16*16*16, 18, pixmap_rgb, x, y);
    break;
  }
}

Private void
put_pixmap(int ncolors, int nlevels, DVI_PIXMAP_RGB pixmap, long x, long y)
{
  static unsigned long   *pix_table   = NULL;
  static char            *pix_alloced = NULL;
  static int             left_ncolors = -1;
  static int             last_ncolors = -1;
  static int             last_nlevels = -1;
  unsigned long   pix;
  XImage          *x_image_rgb;
  unsigned char   *buffer, *p;
  XColor          col;
  PAGE_CACHE      pc; 
  int             r, g, b, xxx, yyy, i, n;
  int 	          cr, cg, cb, cd, r0, b0, g0, d0;

  if (pixmap == NULL)
    return;
  if ((pixmap->width == 0) || (pixmap->height == 0)) 
    return;
  
  XFlush(xdisp);

  if ((last_ncolors != ncolors) || (last_nlevels != nlevels)){
    if (pix_table != NULL){
      free(pix_table);
      pix_table = NULL;
    }
    if (pix_alloced != NULL){
      free(pix_alloced);
      pix_alloced = NULL;
    }
  }
  if (pix_table == NULL){
    n = nlevels*nlevels*nlevels;
    pix_table   = (unsigned long*)calloc(n, sizeof(unsigned long));
    pix_alloced = (char*)malloc(n);
    if ((pix_table == NULL) || (pix_alloced == NULL)){
      if (pix_table != NULL)
	free(pix_table);
      if (pix_alloced != NULL)
	free(pix_alloced);
      pix_table   = NULL;
      pix_alloced = NULL;
      return;
    }
    for (i = 0; i < n; i++)
      pix_alloced[i] = 0;
    left_ncolors = ncolors;
    last_ncolors = ncolors;
    last_nlevels = nlevels;
  }

  buffer = (unsigned char*)malloc(x_depth * pixmap->height * pixmap->width);
  if (buffer == NULL)
    return;

  x_image_rgb
    = XCreateImage(xdisp, x_visual, x_depth, XYPixmap,
		   0, (char*)buffer,
		   pixmap->width, pixmap->height,
		   8, (pixmap->width+7)/8);

  col.flags = DoRed|DoGreen|DoBlue;
  for (yyy = 0; yyy < pixmap->height; yyy++){
    p = &pixmap->bitmap[yyy * pixmap->raster];
    for (xxx = 0; xxx < pixmap->width; xxx++, p += 3){
      r = ((nlevels-1) * p[0]) / pixmap->max;
      g = ((nlevels-1) * p[1]) / pixmap->max;
      b = ((nlevels-1) * p[2]) / pixmap->max;
      pix  = r | (g * nlevels) | (b * nlevels * nlevels);
      if (pix_alloced[pix] == 1){
	col.pixel = pix_table[pix];
      } else {
	if (left_ncolors == 0){
	  cr = cg = cb = nlevels - 1;
	  cd = (cr - r)*(cr - r) + (cg - g)*(cg - g) + (cb - b)*(cb - b);
	  col.pixel = 0;
	  for (i = 0; i < nlevels*nlevels*nlevels; i++){
	    if (pix_alloced[i] == 1){
	      r0 = i % nlevels;
	      g0 = (i/nlevels) % nlevels;
	      b0 = i/(nlevels*nlevels);
	      if (cd > (d0 = (r0-r)*(r0-r) + (g0-g)*(g0-g) + (b0-b)*(b0-b))){
		cr = r0;
		cg = g0; 
		cb = b0;
		col.pixel = pix_table[i];
		if (d0 == 0)
		  break;
	      }
	    }
	  }
	} else {
	  col.red   = (65535L * p[0]) / pixmap->max;
	  col.green = (65535L * p[1]) / pixmap->max;
	  col.blue  = (65535L * p[2]) / pixmap->max;
	  if (XAllocColor(xdisp, x_cmap, &col) == 0){
	    if (x_cmap == DefaultColormap(xdisp, 
					  DefaultScreen(xdisp))){
	      x_cmap = XCopyColormapAndFree(xdisp, x_cmap);
	    }
	    XAllocColor(xdisp, x_cmap, &col);
	  }
	  left_ncolors--;
	  pix_alloced[pix] = 1;
	}
	pix_table[pix] = col.pixel;
      }
      XPutPixel(x_image_rgb, xxx, yyy, col.pixel);
    }
  }

  x += XVAR(DviDev,offset_x);
  y += XVAR(DviDev,offset_y);

  pc = x_get_cache(DviCurrentPage);
  XPutImage(xdisp, pc->pix,
	    XVAR(DviDev,gc_pc_image_rgb), x_image_rgb, 
	    0, 0, x, y, pixmap->width, pixmap->height);
  XCopyArea(xdisp, pc->pix, XVAR(DviDev,view), 
	    XVAR(DviDev,gc_copy), 
	    x, y, pixmap->width, pixmap->height, x, y);

  XDestroyImage(x_image_rgb);

  XFlush(xdisp);
}


Public void
x_put_graymap(DVI_GRAYMAP graymap, long x, long y)
{
  switch (x_visual->class){
  default:
  case StaticGray:
  case GrayScale:
    put_graymap(16, graymap, x, y);
    break;
  case StaticColor:
  case PseudoColor:
    put_graymap(64, graymap, x, y);
    break;
  case TrueColor:
  case DirectColor:
    put_graymap(256, graymap, x, y);
    break;
  }
}

Private void
put_graymap(int nlevels, DVI_GRAYMAP graymap, long x, long y)
{
  static unsigned long   *pix_table   = NULL;
  static int             last_nlevels = -1;
  unsigned long   pix;
  XImage          *x_image_rgb;
  unsigned char   *buffer, *p;
  XColor          col;
  PAGE_CACHE      pc; 
  int             xxx, yyy, i;
  
  if (graymap == NULL)
    return;
  if ((graymap->width == 0) || (graymap->height == 0)) 
    return;
  
  buffer = (unsigned char*)malloc(x_depth * graymap->height * graymap->raster);
  if (buffer == NULL)
    return;

  x_image_rgb
    = XCreateImage(xdisp, x_visual, x_depth, XYPixmap,
		   0, (char*)buffer,
		   graymap->width, graymap->height,
		   8, (graymap->width+7)/8);

  if (last_nlevels != nlevels){
    if (pix_table != NULL){
      free(pix_table);
      pix_table = NULL;
    }
  }
  if (pix_table == NULL){
    pix_table = (unsigned long*)calloc(nlevels, sizeof(unsigned long));
    if (pix_table == NULL){
      XDestroyImage(x_image_rgb);
      return;
    }
  }
  last_nlevels = nlevels;

  for (i = 0; i < nlevels; i++)
    pix_table[i]  = ~0L;
  col.flags = DoRed|DoGreen|DoBlue;
  for (yyy = 0; yyy < graymap->height; yyy++){
    p = &graymap->bitmap[yyy * graymap->raster];
    for (xxx = 0; xxx < graymap->width; xxx++, p++){
      pix = (*p * (nlevels-1)) / graymap->max;
      if ((col.pixel = pix_table[pix]) == ~0L){
	col.red = col.green = col.blue = (65535L * (*p)) / graymap->max;
	if (XAllocColor(xdisp, x_cmap, &col) == 0){
	  if (x_cmap == DefaultColormap(xdisp, DefaultScreen(xdisp)))
	    x_cmap = XCopyColormapAndFree(xdisp, x_cmap);
	  XAllocColor(xdisp, x_cmap, &col);
	}
	pix_table[pix] = col.pixel;
      }
      XPutPixel(x_image_rgb, xxx, yyy, col.pixel);
    }
  }

  x += XVAR(DviDev,offset_x);
  y += XVAR(DviDev,offset_y);

  pc = x_get_cache(DviCurrentPage);
  XPutImage(xdisp, pc->pix,
	    XVAR(DviDev,gc_pc_image_rgb), x_image_rgb, 
	    0, 0, x, y, graymap->width, graymap->height);
  XCopyArea(xdisp, pc->pix, XVAR(DviDev,view), XVAR(DviDev,gc_copy), 
	    x, y, graymap->width, graymap->height, x, y);

  XDestroyImage(x_image_rgb);

  XFlush(xdisp);
}




Public Widget
x_p_make_hsep(Widget parent)
{
  Widget  xsep;
  Arg     args[2];
  int     i = 0;

  XtSetArg(args[i], XmNorientation, XmHORIZONTAL);  i++;
  XtSetArg(args[i], XmNseparatorType, XmSINGLE_LINE);  i++;
  xsep = XmCreateSeparator(parent, "hsep", args, i);
  XtManageChild(xsep);
  return xsep;
}

Public Widget
x_p_make_vsep(Widget parent)
{
  Widget  xsep;
  Arg     args[2];
  int     i = 0;

  XtSetArg(args[i], XmNorientation, XmVERTICAL);  i++;
  XtSetArg(args[i], XmNseparatorType, XmSINGLE_LINE);  i++;
  xsep = XmCreateSeparator(parent, "vsep", args, i);
  XtManageChild(xsep);
  return xsep;
}

Public void
x_p_put_top(Widget self)
{
  Arg     args[3];
  int     i = 0;

  XtSetArg(args[i], XmNtopAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNrightAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_FORM);  i++;
  XtSetValues(self, args, i);
}

Public void
x_p_put_below(Widget self, Widget top)
{
  Arg     args[4];
  int     i = 0;

  XtSetArg(args[i], XmNtopAttachment, XmATTACH_WIDGET);  i++;
  XtSetArg(args[i], XmNtopWidget, top);  i++;
  XtSetArg(args[i], XmNrightAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_FORM);  i++; 
  XtSetValues(self, args, i);
}

Public void
x_p_put_right(Widget self, Widget left)
{
  Arg     args[2];
  int     i = 0;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_WIDGET);  i++;
  XtSetArg(args[i], XmNleftWidget, left);  i++;
  XtSetValues(self, args, i);
}

Public void
x_p_put_left(Widget self, Widget right)
{
  Arg     args[2];
  int     i = 0;

  XtSetArg(args[i], XmNrightAttachment, XmATTACH_WIDGET);  i++;
  XtSetArg(args[i], XmNrightWidget, right);  i++;
  XtSetValues(self, args, i);
}

Public void
x_p_put_rightmost(Widget self)
{
  Arg     args[1];
  int     i = 0;

  XtSetArg(args[i], XmNrightAttachment, XmATTACH_FORM);  i++;
  XtSetValues(self, args, i);
}

Public void
x_p_put_leftmost(Widget self)
{
  Arg     args[1];
  int     i = 0;

  XtSetArg(args[i], XmNleftAttachment, XmATTACH_FORM); i++;
  XtSetValues(self, args, 1);
}

Public void
x_p_put_center(Widget self)
{
  Arg     args[3];
  int     i = 0;

  XtSetArg(args[i], XmNrightAttachment, XmATTACH_FORM);  i++;
  XtSetArg(args[i], XmNleftAttachment, XmATTACH_FORM);  i++;
  XtSetValues(self, args, i);
}



Private XtResource DefaultResource[] = {
  { "vflibcap",           "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, vflibcap),  XtRString, DEFAULT_VFLIBCAP },
  { "dpi",                "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_dpi),     XtRString, DEFAULT_DPI },
  { "kpathsea_mode",      "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, kpathsea_mode), XtRString, DEFAULT_KPATHSEA_MODE },
  { "orientation",        "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_orient), XtRString, DEFAULT_ORIENTATION },
  { "paperSize",          "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_paper),   XtRString, DEFAULT_PAPER },
  { "shrinkFactor",       "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_shrink), XtRString, DEFAULT_SHRINK_FACTOR },
  { "magnificationMin",   "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_mag_min), XtRString, DEFAULT_MAGNIFICATION_MIN },
  { "magnificationMax",   "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_mag_max), XtRString, DEFAULT_MAGNIFICATION_MAX },
  { "previewWindowWidth",  "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_prevwin_w), XtRString, DEFAULT_PREVWIN_WIDTH },
  { "previewWindowHeight", "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_prevwin_h), XtRString, DEFAULT_PREVWIN_HEIGHT },
  { "keymap",            "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_keymap), XtRString, DEFAULT_KEYMAP },
  { "offsetX",            "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_offset_x), XtRString, DEFAULT_OFFSET_X },
  { "offsetY",            "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_offset_y), XtRString, DEFAULT_OFFSET_Y },
  { "colorCharacter",     "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, color_char), XtRString, DEFAULT_PAPER_CHAR_COLOR },
  { "colorPaper",         "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, color_paper), XtRString, DEFAULT_PAPER_PAPER_COLOR },
  { "colorFrame",         "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, color_frame), XtRString, DEFAULT_PAPER_FRAME_COLOR },
  { "visualText",         "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_visual_text), XtRString, DEFAULT_VISUAL_TEXT },
  { "visualEPS",          "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_visual_eps), XtRString, DEFAULT_VISUAL_EPS },
  { "pageCacheSize",      "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_page_cache), XtRString, DEFAULT_PAGE_CACHE_SIZE },
  { "pollingInterval",    "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_poll), XtRString, DEFAULT_POLLING_INTERVAL },
  { "drawEPSFigures",     "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_draw_eps), XtRString, DEFAULT_DRAW_EPS_FIGS },
  { "epsStyle",           "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_eps_style), XtRString, DEFAULT_EPS_FIGURE_STYLE },
  { "gsPath",             "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, gs_path), XtRString, DEFAULT_GS_PATH },
  { "gsTimeOut",          "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_gs_timeout), XtRString, DEFAULT_GS_TIMEOUT },
  { "tempDirectory",      "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, temp_dir), XtRString, DEFAULT_TEMP_DIRECTORY },
  { "novice",             "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_novice), XtRString, DEFAULT_NOVICE },
  { "messageWindowSize",     "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_msg_window_size), XtRString, 
      DEFAULT_MESSAGE_WINDOW },
  { "messageWindowHistory",  "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_msg_history_size), XtRString, 
      DEFAULT_MESSAGE_HISTORY },
  { "lpList",             "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lplist), XtRString, DEFAULT_PRINTER_LIST },
  { "debugPrintSpec",             "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_debug_lpspec), XtRString, 
      DEFAULT_DEBUG_PRINT_SPEC },
  { "printMessageWindowColumns",   "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_print_msg_window_columns), XtRString, 
      DEFAULT_PRINT_MESSAGE_WINDOW_C },
  { "printMessageWindowRows",      "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_print_msg_window_rows), XtRString, 
      DEFAULT_PRINT_MESSAGE_WINDOW_R },
  { "printMessageWindowHistory",  "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, s_print_msg_history_size), XtRString, 
      DEFAULT_PRINT_MESSAGE_HISTORY },
  { "printFileSpecDVI",           "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpr_spec_dvi), XtRString, 
      DEFAULT_PRINTFILE_SPEC_DVI },
  { "printThisPageSpecDVI",       "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpr_page_spec_dvi), XtRString, 
      DEFAULT_PRINTPAGE_SPEC_DVI },
  { "printQueueSpecDVI",          "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpq_spec_dvi), XtRString, 
      DEFAULT_PRINTQUEUE_SPEC_DVI },
  { "printCancelSpecDVI",         "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lprm_spec_dvi), XtRString, 
      DEFAULT_REMOVE_QUEUE_SPEC_DVI },
  { "printFileSpecPS",            "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpr_spec_ps), XtRString, 
      DEFAULT_PRINTFILE_SPEC_PS },
  { "printThisPageSpecPS",        "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpr_page_spec_ps), XtRString, 
      DEFAULT_PRINTPAGE_SPEC_PS },
  { "printQueueSpecPS",           "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpq_spec_ps), XtRString, 
      DEFAULT_PRINTQUEUE_SPEC_PS },
  { "printCancelSpecPS",          "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lprm_spec_ps), XtRString, 
      DEFAULT_REMOVE_QUEUE_SPEC_PS },
  { "printFileSpecUSR1",          "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpr_spec_usr1), XtRString, 
      DEFAULT_PRINTFILE_SPEC_USR1 },
  { "printThisPageSpecUSR1",      "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpr_page_spec_usr1), XtRString, 
      DEFAULT_PRINTPAGE_SPEC_USR1 },
  { "printQueueSpecUSR1",         "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpq_spec_usr1), XtRString, 
      DEFAULT_PRINTQUEUE_SPEC_USR1 },
  { "printCancelSpecUSR1",        "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lprm_spec_usr1), XtRString, 
      DEFAULT_REMOVE_QUEUE_SPEC_USR1 },
  { "printFileSpecUSR2",          "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpr_spec_usr2), XtRString, 
      DEFAULT_PRINTFILE_SPEC_USR2 },
  { "printThisPageSpecUSR2",      "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpr_page_spec_usr2), XtRString, 
      DEFAULT_PRINTPAGE_SPEC_USR2 },
  { "printQueueSpecUSR2",         "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lpq_spec_usr2), XtRString, 
      DEFAULT_PRINTQUEUE_SPEC_USR2 },
  { "printCancelSpecUSR2",        "String", XtRString, sizeof(char*),
      XtOffset(RESOURCE, lprm_spec_usr2), XtRString, 
      DEFAULT_REMOVE_QUEUE_SPEC_USR2 }
};

Public XrmDatabase
x_init_application_resources(void)
{
  XrmDatabase  xrdb, xrdb1; 
  char         *xrm, *home, *xenv, *p;
  char         path[MAXPATHLEN], host[256];

  XrmInitialize();
  xrdb = NULL;

  sprintf(path, "%s/lib/X11/app-defaults/%s", XDIR, PROG_NAME_RESOURCE);
  xrdb = XrmGetFileDatabase(path);

  home = ".";
  if ((xenv = getenv("XAPPLRESDIR")) != NULL)
    sprintf(path, "%s/%s", xenv, PROG_NAME_RESOURCE);
  else {
    if ((home = (char*)getenv("HOME")) == NULL)
      home = ".";
    sprintf(path, "%s/.app-defaults/%s", home, PROG_NAME_RESOURCE);
  }
  xrdb1 = XrmGetFileDatabase(path); 
  if (xrdb1 != NULL)
    XrmMergeDatabases(xrdb1, &xrdb);

  if ((xrm = XResourceManagerString(xdisp)) != NULL){
    xrdb1 = XrmGetStringDatabase(xrm);
  } else {
    sprintf(path, "%s/.Xdefaults", home);
    xrdb1 = XrmGetFileDatabase(path);
  }
  if (xrdb1 != NULL)
    XrmMergeDatabases(xrdb1, &xrdb);

  if ((xenv = getenv("XENVIRONMENT")) != NULL){
    xrdb1 = XrmGetFileDatabase(xenv);
#if HAVE_GETHOSTNAME
  } else {
    gethostname(host, sizeof(host));
    host[sizeof(host)-1] = '\0';
    if ((p = strchr(host, '.')) != NULL)
      *p = '\0';
    sprintf(path, "%s/.Xdefaults-%s", home, host);
    xrdb1 = XrmGetFileDatabase(path);
#endif
  }
  if (xrdb1 != NULL)
    XrmMergeDatabases(xrdb1, &xrdb);

  return xrdb;
}

Public void
x_resource_set_defaults(RESOURCE res)
{
  XtGetApplicationResources(x_toplevel, res, DefaultResource, 
			    XtNumber(DefaultResource), NULL, 0);
}


/*EOF*/

