# $Id: iface.tcl,v 1.21 2004/07/15 21:56:10 aleksey Exp $

namespace eval ifacetk {
    variable options

    custom::defgroup IFace \
	[::msgcat::mc "Options for main interface."] \
	-group Tkabber -tag "Main Interface"

    custom::defvar options(use_tabbar) 1 \
	[::msgcat::mc \
	     "Use Tabbed Interface (you need to restart)."] \
	-group IFace -type boolean

    custom::defvar options(show_toolbar) 1 \
	[::msgcat::mc "Show Toolbar."] \
	-group IFace -type boolean \
	-command ifacetk::switch_toolbar

    custom::defvar options(show_presencebar) 1 \
	[::msgcat::mc "Show presence bar."] \
	-group IFace -type boolean \
	-command ifacetk::switch_presencebar

    if {$::tcl_platform(platform) != "unix"} {
	custom::defvar options(font) $::default_font \
	    [::msgcat::mc "Font to use in roster, chat windows etc."] \
	    -group IFace -type font \
	    -command ifacetk::switch_font
    }

    custom::defvar options(raise_new_tab) 1 \
	[::msgcat::mc "Raise new tab."] \
	-group IFace -type boolean

    custom::defvar options(message_numbers_in_tabs) 1 \
	[::msgcat::mc "Show number of unread messages in tab titles."] \
	-group IFace -type boolean

    custom::defvar options(update_title_delay) 600 \
	[::msgcat::mc "Delay between getting focus and updating window or tab title in milliseconds."] \
	-group IFace -type integer

    custom::defvar options(show_tearoffs) 1 \
	[::msgcat::mc "Show menu tearoffs when possible."] \
	-group IFace -type boolean

    custom::defvar options(closebuttonaction) close \
	[::msgcat::mc "What action does the close button."] \
	-group IFace -type radio -layout vertical \
	-values [list \
		     close    [::msgcat::mc "Close Tkabber"] \
		     nothing  [::msgcat::mc "Do nothing"] \
		     minimize [::msgcat::mc "Minimize"] \
		    ]

    variable toolbaricon

    set toolbaricon(groupchat) [Bitmap::get [pixmap tkabber glade-groupchat.gif]]
    set toolbaricon(adduser)   [Bitmap::get [pixmap tkabber glade-new-user.gif]]
    set toolbaricon(browser)   [Bitmap::get [pixmap tkabber jb.gif]]
    set toolbaricon(online)    [Bitmap::get [pixmap tkabber glade-online.gif]]
    set toolbaricon(offline)   [Bitmap::get [pixmap tkabber glade-offline.gif]]

    # TODO
    set ::searchicon(case)       [Bitmap::get [pixmap tkabber search_case.gif]]
    set ::searchicon(exact)      [Bitmap::get [pixmap tkabber search_exact.gif]]
    set ::searchicon(back)       [Bitmap::get [pixmap tkabber search_bk.gif]]
    set ::searchicon(forward)    [Bitmap::get [pixmap tkabber search_fw.gif]]

    proc isource {file} {
	set dir [file dirname [info script]]
	source [file join $dir $file]
    }

    isource ilogin.tcl
    isource iroster.tcl
    
    variable focused ""
    variable after_focused_id ""

    variable number_msg
    variable personal_msg
}

option add *errorForeground red widgetDefault

wm protocol . WM_DELETE_WINDOW {
    ifacetk::closebuttonproc
}

proc ifacetk::closebuttonproc {} { 
    variable options

    switch -- $options(closebuttonaction) {
        close {
	    terminate
        }
        nothing {}
        minimize {
	    wm iconify .
        }
    }
}

set xselection ""

proc EncodeTextSelection {txt offset len} {
    set elided ""
    if {[lcontain [$txt tag names] emphasized] && [$txt tag cget emphasized -elide]} {
	set elided emphasized
    } elseif {[lcontain [$txt tag names] nonemphasized] && [$txt tag cget nonemphasized -elide]} {
	set elided nonemphasized
    }
    if {[cequal $elided ""]} {
	set ::xselection [eval $txt get [$txt tag ranges sel]]
    } else {
	lassign [$txt tag ranges sel] selstart selstop
	set sel $selstart
	set ::xselection ""
	while {![cequal [set range [$txt tag nextrange $elided $sel $selstop]] ""]} {
	    append ::xselection [$txt get $sel [lindex $range 0]]
	    set sel [lindex $range 1]
	}
	append ::xselection [$txt get $sel $selstop]
    }
    encoding convertto [crange $::xselection $offset [expr $offset + $len - 1]]
}

proc EncodeEntrySelection {txt offset len} {
    if [$txt selection present] {
	set idx1 [$txt index sel.first]	
	set idx2 [$txt index sel.last]
	set ::xselection [string range [$txt get] $idx1 $idx2]
	encoding convertto \
	    [crange $::xselection $offset [expr $offset + $len - 1]]
    } else {
	set ::xselection ""
    }
}

if {[cequal $tcl_platform(platform) unix]} {
    proc ::tk::GetSelection {w {sel PRIMARY}} {
        if {![catch {selection get -displayof $w \
			-selection $sel -type UTF8_STRING} txt]} {
	    return $txt
	} elseif {![catch {selection get -displayof $w -selection $sel} txt]} {
	    if {[selection own] == ""} {
		return [encoding convertfrom $txt]
	    } else {
		return $::xselection
	    }
	} else {
	    return -code error "could not find default selection"
	}
    }

    bind Text  <Map> { selection handle %W "EncodeTextSelection %W" }
    bind Entry <Map> { selection handle %W "EncodeEntrySelection %W" }
    
    if {[info tclversion] < 8.4} {
	bind Text <ButtonRelease-2> {
	    if {![catch {::tk::GetSelection %W} sel]} {
		%W insert current $sel
	    }
	}
	bind Entry <ButtonRelease-2> {
	    if {![catch {::tk::GetSelection %W} sel]} {
		%W insert insert $sel
	    }
	}
    }
}
###############################################################################

proc ifacetk::load_descmenu {} {
    global descmenu

    set descmenu \
	[list "&Tkabber" {} {} $ifacetk::options(show_tearoffs) \
	     [list \
		  [list cascad [::msgcat::mc "Presence"] {} presence $ifacetk::options(show_tearoffs) \
		       [list \
			    [list command [::msgcat::mc "Online"] {} {} {} \
				 -command {set userstatus available}] \
			    [list command [::msgcat::mc "Free to chat"] {} {} {} \
				 -command {set userstatus chat}] \
			    [list command [::msgcat::mc "Away"] {} {} {} \
				 -command {set userstatus away}] \
			    [list command [::msgcat::mc "Extended away"] {} {} {} \
				 -command {set userstatus xa}] \
			    [list command [::msgcat::mc "Do not disturb"] {} {} {} \
				 -command {set userstatus dnd}] \
			    [list command [::msgcat::mc "Invisible"] {} {} {} \
				 -command {set userstatus invisible}] \
			    {separator} \
			    [list command [::msgcat::mc "Change priority..."] {} {} {} \
				 -command {change_priority_dialog}]]] \
		  [list command [::msgcat::mc "Log in..."] {} {} {Ctrl l} \
		       -command {ifacetk::login_dialog}] \
		  [list command [::msgcat::mc "Log out"] {} {} {Ctrl j} \
		       -command {logout_dialog}] \
		  [list command [::msgcat::mc "Log out with reason..."] {} {} {} \
		       -command {show_logout_dialog}] \
		  {separator} \
		  [list command [::msgcat::mc "Customize"] {} {} {} \
		       -command {custom::open_window Tkabber}] \
		  {separator} \
		  [list command [::msgcat::mc "Profile on"] {} {} {} -command {
		      profile -commands -eval on
		  }] \
		  [list command [::msgcat::mc "Profile report"] {} {} {} -command {
		      profile off profil
		      profrep profil real profresults
		  }] \
		  {separator} \
		  [list command [::msgcat::mc "Quit"] {} {} {} \
		       -command {quit}] \
		 ] \
	     "&[::msgcat::mc Services]" {} {} $ifacetk::options(show_tearoffs) \
	     [list \
		  [list command [::msgcat::mc "Send message..."] {} {} {} \
		       -command {message::send_dialog}] \
		  $::roster_main_menu \
		  [list command [::msgcat::mc "Browser"] {} {} {} \
		       -command {browser::open}] \
		  [list command [::msgcat::mc "Discovery"] {} {} {} \
		       -command {disco::browser::open_win ""}] \
		  [list command [::msgcat::mc "Join group..."] {} {} {}\
		       -command {join_group_dialog}] \
		  [list cascad [::msgcat::mc "Sound"] {} {} $ifacetk::options(show_tearoffs) \
		       [list \
			    [list checkbutton [::msgcat::mc "Mute"] {} {} {} \
				 -variable sound::options(mute)]]] \
		  [list cascad [::msgcat::mc "Chats"] {} chats $ifacetk::options(show_tearoffs) \
		       [list \
			    [list checkbutton [::msgcat::mc "Generate event messages"] {} {} {} \
				 -variable muc::options(gen_events)] \
			    [list checkbutton [::msgcat::mc "Smart autoscroll"] {} {} {} \
				 -variable chat::options(smart_scroll)] \
			    [list checkbutton [::msgcat::mc "Stop autoscroll"] {} {} {} \
				 -variable chat::options(stop_scroll)] \
			    [list checkbutton [::msgcat::mc "Emphasize"] {} {} {} \
				 -variable chat::options(emphasize)]]] \
		  [list cascad [::msgcat::mc "Privacy rules"] {} {} $ifacetk::options(show_tearoffs) \
		       [list \
			    [list command [::msgcat::mc "Manually edit rules..."] {} {} {} \
				 -command {privacy::request_lists}] \
			    [list command [::msgcat::mc "Edit invisible list..."] {} {} {} \
				 -command {privacy::edit_special_list invisible}] \
			    [list command [::msgcat::mc "Edit ignore list..."] {} {} {} \
				 -command {privacy::edit_special_list ignore}] \
			    [list checkbutton [::msgcat::mc "Activate lists at startup"] {} {} {} \
				 -variable privacy::options(activate_at_startup)] \
			    {separator} \
			    [list command [::msgcat::mc "Message filters..."] {} {} {} \
				 -command {filters::open}]]] \
		  [list command [::msgcat::mc "Change password..."] {} {} {} \
		       -command {change_password_dialog}] \
		  [list command [::msgcat::mc "Message archive"] {} {} {} \
		       -command {message_archive::show_archive}] \
		  [list command [::msgcat::mc "Show user info..."] {} {} {} \
		       -command {userinfo::show_info_dialog}] \
		  [list command [::msgcat::mc "Edit my info..."] {} {} {} \
		       -command {
			    set connid [jlib::route ""]
			    userinfo::open [jlib::connection_bare_jid $connid] \
				-editable 1 -connection $connid
			}] \
		  [list cascad [::msgcat::mc "Avatar"] {} {} $ifacetk::options(show_tearoffs) \
		       [list \
			    [list checkbutton [::msgcat::mc "Announce"] {} {} {} \
				 -variable avatar::options(announce)] \
			    [list checkbutton [::msgcat::mc "Allow downloading"] {} {} {} \
				 -variable avatar::options(share)] \
			    [list command [::msgcat::mc "Send to server"] {} {} {} \
				 -command {avatar::store_on_server}]]] \
		  {separator} \
		  [list cascad [::msgcat::mc "Admin tools"] {} admin $ifacetk::options(show_tearoffs) \
		       [list \
			    [list command [::msgcat::mc "Send broadcast message..."] {} {} {} \
				 -command {eval {message::send_dialog \
						     -to "$loginconf(server)/announce/online"}}] \
			    [list command [::msgcat::mc "Send message of the day..."] {} {} {} \
				 -command {eval {message::send_dialog \
						     -to "$loginconf(server)/announce/motd"}}] \
			    [list command [::msgcat::mc "Update message of the day..."] {} {} {} \
				 -command {eval {message::send_dialog \
						     -to "$loginconf(server)/announce/motd/update"}}] \
			    [list command [::msgcat::mc "Delete message of the day"] {} {} {} \
				 -command {eval {message::send_msg \
						     "$loginconf(server)/announce/motd/delete" -type normal}}]]] \
		 ] \
	     "&[::msgcat::mc Help]" {} {} $ifacetk::options(show_tearoffs) \
	     [list \
		  [list command [::msgcat::mc "Quick help"] {} {} {} \
		       -command ifacetk::quick_help_window] \
		  [list command [::msgcat::mc "About"] {} {} {} \
		       -command ifacetk::about_window] \
		 ] \
	    ]
    
    if {![info exists enable_profiling] || !$enable_profiling || \
	    ([clength [info commands profile]] == 0)} {
	set tmpmenu {}
	set eatP 0
	foreach menu [lindex $descmenu 4] {
	    if {[::msgcat::mc "Profile on"] == [lindex $menu 1] || \
		    [::msgcat::mc "Profile report"] == [lindex $menu 1]} {
		set eatP 1
	    } elseif {$eatP} {
		set eatP 0
	    } else {
		lappend tmpmenu $menu
	    }
	}
	set descmenu [lreplace $descmenu 4 4 $tmpmenu]
    }
    
    set descmenu [menuload $descmenu]
}

###############################################################################

proc ifacetk::quick_help_window {} {
    help_window [::msgcat::mc "Quick Help"] \
"\n[::msgcat::mc {Main window:}]
    $::tk_modify-L\t\t\t[::msgcat::mc {Log in}]
    $::tk_modify-J\t\t\t[::msgcat::mc {Log out}]
\n[::msgcat::mc Tabs:]
    $::tk_modify-F4\t\t\t[::msgcat::mc {Close tab}]
    $::tk_modify-PgUp/Down\t\t[::msgcat::mc {Previous/Next tab}]
    $::tk_modify-Alt-PgUp/Down\t[::msgcat::mc {Move tab left/right}]
    Alt-\[1-9,0\]\t\t[::msgcat::mc {Switch to tab number 1-9,10}]
    $::tk_modify-R\t\t\t[::msgcat::mc {Hide/Show roster}]
\n[::msgcat::mc Chats:]
    TAB\t\t\t[::msgcat::mc {Complete nickname}]
    $::tk_modify-Up/Down\t\t[::msgcat::mc {Previous/Next history message}]
    Alt-E\t\t\t[::msgcat::mc {Show emoticons}]
    $::tk_modify-Z\t\t\t[::msgcat::mc {Undo}]
    $::tk_modify-Shift-Z\t\t[::msgcat::mc {Redo}]
    Alt-PgUp/Down\t\t[::msgcat::mc {Scroll chat window up/down}]
    [::msgcat::mc {Right mouse	}]\t[::msgcat::mc {Correct word}]
"
}

proc ifacetk::about_window {} {
    global version

    help_window [::msgcat::mc "About"] "
Tkabber $version

Copyright \u00a9 2002-2004 [::msgcat::mc {Alexey Shchepin}]
\n[::msgcat::mc Authors:]
    [::msgcat::mc {Alexey Shchepin}]
    [::msgcat::mc {Marshall T. Rose}]
    [::msgcat::mc {Sergei Golovan}]
    [::msgcat::mc {Michail Litvak}]

http://tkabber.jabberstudio.org/"
}

proc ifacetk::help_window {title message} {
    set mainlogo [Bitmap::get [pixmap tkabber mainlogo.gif]]
    Dialog .m -anchor e -separator yes -title $title -image $mainlogo \
	    -side bottom -modal local -default 0 -cancel 0
    .m add -text [::msgcat::mc "OK"]
    set frame [.m getframe]
    message $frame.msg -text $message
    pack $frame.msg -side left
    .m draw
    destroy .m
}

proc ifacetk::switch_font {args} {
    variable options

    set opts [lassign $options(font) family size]
    set args [list -family $family -size $size]
    set bold 0
    set italic 0
    set underline 0
    set overstrike 0
    foreach opt $opts {
	switch -- $opt {
	    bold { 
		lappend args -weight bold
		set bold 1
	    }
	    italic {
		lappend args -slant italic
		set italic 1
	    }
	    underline {
		lappend args -underline 1
		set underline 1
	    }
	    overstrike {
		lappend args -overstrike 1
		set overstrike 1
	    }
		
	}
    }
    if {!$bold} {
	lappend args -weight normal
    }
    if {!$italic} {
	lappend args -slant roman
    }
    if {!$underline} {
	lappend args -underline 0
    }
    if {!$overstrike} {
	lappend args -overstrike 0
    }

    eval [list font configure font] $args
    define_fonts
    roster::redraw_after_idle
}

proc ifacetk::show_ssl_info {} {
    global ssl_certificate_fields

    if {[winfo exists .ssl_info]} {
	destroy .ssl_info
    }

    if {[lempty [set msg_list [ssl_info]]]} return

    Dialog .ssl_info -title [::msgcat::mc "SSL Info"] -separator 1 -anchor e \
	-default 0 -cancel 0 -modal none
    .ssl_info add -text [::msgcat::mc "Close"] -command "destroy .ssl_info"
    set fr [.ssl_info getframe]

    if {[llength $msg_list] == 2} {
	lassign $msg_list server msg
	set page [frame $fr.page]
	set title [format [::msgcat::mc "%s SSL Certificate Info"] $server]
	grid [label $page.title -text $title] -row 0 -column 0 -sticky ew
	grid [message $page.info -aspect 50000 -text $msg] \
	    -row 1 -column 0 -sticky ew
	pack $page -expand yes -fill both -padx 1m -pady 1m
    } else {
	set nb [NoteBook $fr.nb]
	pack $nb -expand yes -fill both -padx 0m -pady 0m

	set i 0
	foreach {server msg} $msg_list {
	    set page [$nb insert end page$i -text $server]
	    set title [format [::msgcat::mc "%s SSL Certificate Info"] $server]
	    grid [label $page.title -text $title] -row 0 -column 0 -sticky ew
	    grid [message $page.info -aspect 50000 -text $msg] \
		-row 1 -column 0 -sticky ew
	    incr i
	}
	$nb compute_size
	$nb raise page0
    }
    .ssl_info draw
}

proc ifacetk::update_ssl_ind {args} {
    global use_tls
    variable ssl_ind

    if {!$use_tls} return

    lassign [update_ssl_info] state fg balloon

    if {![cequal $fg warning]} {
	label .fake_label
	set fg [.fake_label cget -foreground]
	destroy .fake_label
    } else {
	set fg [option get $ssl_ind errorForeground Label]
    }
    $ssl_ind configure -state $state
    if {[cequal $state normal]} {
	$ssl_ind configure -foreground $fg
    }
    DynamicHelp::register $ssl_ind balloon $balloon
}

hook::add connected_hook [namespace current]::ifacetk::update_ssl_ind
hook::add disconnected_hook [namespace current]::ifacetk::update_ssl_ind

proc ifacetk::add_toolbar_button {path icon command helptext} {
    $path add -image $icon \
	-highlightthickness 0 -takefocus 0 -relief link -bd $::tk_borderwidth \
	-padx 1 -pady 1 -command $command -helptext $helptext
}

proc ifacetk::online_icon {args} {
    variable toolbaricon

    if {$roster::show_only_online} {
	return $toolbaricon(online)
    } else {
	return $toolbaricon(offline)
    }
}

proc ifacetk::set_toolbar_icon {path index script args} {
    if {![winfo exists $path]} {
	return
    }
    set image [eval $script]
    $path itemconfigure $index -image $image
}

proc ifacetk::create_main_window {} {
    global usetabbar
    global user_status_list
    global use_tls
    global descmenu
    variable mf
    variable rosterwidth
    variable rw
    variable nw
    variable ssl_ind
    variable toolbaricon
    variable main_window_title
    variable options

    set main_window_title "Tkabber"
    wm title . $main_window_title
    wm iconname . $main_window_title
    wm group . .

    bind all <<PrevWindow>> {focus [Widget::focusPrev %W]}

    load_descmenu
    set mf [MainFrame .mainframe -menu $descmenu -textvariable status]
    unset descmenu

    if {$use_tls} {
	set ssl_ind [$mf addindicator -text "SSL" -state disabled]
	$ssl_ind configure -relief flat
	DynamicHelp::register $ssl_ind balloon [::msgcat::mc "Disconnected"]
	bind $ssl_ind <1> [namespace current]::show_ssl_info
    }

    pack $mf -expand yes -fill both

    bind $mf <Destroy> {
	jlib::disconnect
	quit
    }

    set bbox [ButtonBox [$mf addtoolbar].bbox -spacing 0 -padx 1 -pady 1]

    add_toolbar_button $bbox $toolbaricon(adduser) {message::send_subscribe_dialog ""} \
	[::msgcat::mc "Add new user..."]
    add_toolbar_button $bbox $toolbaricon(browser) {disco::browser::open_win ""} \
	[::msgcat::mc "Service Discovery"]
    add_toolbar_button $bbox $toolbaricon(groupchat) join_group_dialog \
	[::msgcat::mc "Join group..."]

    add_toolbar_button $bbox [ifacetk::online_icon] \
	[namespace current]::roster::switch_only_online \
	[::msgcat::mc "Toggle showing offline users"]
    trace variable [namespace current]::roster::show_only_online w \
	[list [namespace current]::set_toolbar_icon $bbox [$bbox index end] \
	      [namespace current]::online_icon]

    if {![cequal [info commands ::ssj::sign:toggleP] ""]} {
	add_toolbar_button $bbox [message::signed:icon] ssj::sign:toggleP \
	[::msgcat::mc "Toggle signing"]
	ssj::signed:trace \
	    [list [namespace current]::set_toolbar_icon $bbox [$bbox index end] \
		  message::signed:icon]
    }
    if {![cequal [info commands ::ssj::encrypt:toggleP] ""]} {
	add_toolbar_button $bbox [message::encrypted:icon] ssj::encrypt:toggleP \
	[::msgcat::mc "Toggle encryption (when possible)"]
	ssj::encrypted:trace \
	    [list [namespace current]::set_toolbar_icon $bbox [$bbox index end] \
		  message::encrypted:icon]
    }

    pack $bbox -side left -anchor w
    switch_toolbar

    if {$options(use_tabbar)} {
        set usetabbar 1
    } else {
        set usetabbar 0
    }

    set ww 0
    foreach {status str} $user_status_list {
	if {[string length $str] > $ww} {
	    set ww [string length $str]
	}
    }

    frame .presence
    menubutton .presence.button -menu .presence.button.menu -relief $::tk_relief \
	-textvariable userstatusdesc -direction above -width $ww \
	-state disabled 

    bind [entry .presence.status -textvariable textstatus] \
	<Return> {set userstatus $userstatus}
    if {$usetabbar} {
	pack .presence.button -side left
	pack .presence.status -side left -fill x -expand yes
    } else {
	pack .presence.button -side top -anchor w
	pack .presence.status -side top -fill x -expand yes -anchor w
    }
    pack .presence -side bottom -anchor w -in [$mf getframe] -fill x

    if {[winfo exists [set m .presence.button.menu]]} {
	destroy $m
    }
    menu $m -tearoff $ifacetk::options(show_tearoffs)
    foreach {status str} $user_status_list {
	if {![cequal $status unavailable]} {
	    $m add command -label $str \
	    -command [list set userstatus $status]
	}
    }

    switch_presencebar

    init_toolbar_popup_menu

    catch {
	bind [$mf gettoolbar 0] <3> [list tk_popup .toolbar_popup %X %Y]
	bind $mf.status <3> [list tk_popup .toolbar_popup %X %Y]
    }

    set rosterwidth [option get . mainRosterWidth [winfo class .]]
    if {$rosterwidth == ""} {
	set rosterwidth [winfo pixels . 4c]
    }

    if {$usetabbar} {
	set pw [PanedWin [$mf getframe].pw -side bottom -pad 2 -width 8]
	pack $pw -fill both -expand yes
	set rw [$pw add -minsize 0 -weight 0]
	set nw [$pw add -minsize 32 -weight 1]

	roster::create .roster -width $rosterwidth -height 300 \
	    -popup [namespace current]::roster::popup_menu \
	    -grouppopup [namespace current]::roster::group_popup_menu
	pack .roster -expand yes -fill both -side left -in $rw

	NoteBook .nb -width 400
	pack .nb -side right -in $nw -fill both -expand yes
	[winfo parent $rw] configure -width $rosterwidth

	bind . <Control-Key-r>      [list [namespace current]::collapse_roster]
	bind . <Control-Prior>      [list [namespace current]::tab_move .nb -1]
	bind . <Control-Next>       [list [namespace current]::tab_move .nb 1]
	bind . <Control-Meta-Prior> [list [namespace current]::current_tab_move .nb -1]
	bind . <Control-Alt-Prior>  [list [namespace current]::current_tab_move .nb -1]
	bind . <Control-Meta-Next>  [list [namespace current]::current_tab_move .nb  1]
	bind . <Control-Alt-Next>   [list [namespace current]::current_tab_move .nb  1]
	bind . <Control-Key-F4> {
	    if {[.nb raise] != ""} {
		eval destroy [pack slaves [.nb getframe [.nb raise]]]
		.nb delete [.nb raise] 1
		ifacetk::tab_move .nb 0
	    }
	}

	for {set i 1} {$i < 10} {incr i} {
	    bind . <Meta-Key-$i> [list [namespace current]::tab_raise_by_number .nb $i]
	    bind . <Alt-Key-$i>  [list [namespace current]::tab_raise_by_number .nb $i]
	}
	bind . <Meta-Key-0> [list [namespace current]::tab_raise_by_number .nb 10]
	bind . <Alt-Key-0>  [list [namespace current]::tab_raise_by_number .nb 10]

	set m [menu .tabsmenu -tearoff 0]
	$m add command -label [::msgcat::mc "Close"] -accelerator $::tk_close \
	    -command {
		if {[.nb raise] != ""} {
		    eval destroy [pack slaves [.nb getframe $curmenutab]]
		    .nb delete $curmenutab 1
		    ifacetk::tab_move .nb 0
		}
	    }
	$m add command -label [::msgcat::mc "Close other tabs"] \
	    -command {
		foreach tab [.nb pages] {
		    if {$tab != $curmenutab} {
			eval destroy [pack slaves [.nb getframe $tab]]
			.nb delete $tab 1
		    }
		}
		ifacetk::tab_move .nb 0
	    }
	$m add command -label [::msgcat::mc "Close all tabs"] \
	    -command {
		foreach tab [.nb pages] {
		    eval destroy [pack slaves [.nb getframe $tab]]
		    .nb delete $tab 1
		}
	    }

	.nb bindtabs <3> {ifacetk::tab_menu %X %Y}

	DragSite::register .nb.c -draginitcmd [namespace current]::draginitcmd
	DropSite::register .nb.c -dropovercmd [namespace current]::dropovercmd \
	    -dropcmd [namespace current]::dropcmd -droptypes {NoteBookPage}

	set geometry [option get . geometry [winfo class .]]
	if {$geometry == ""} {
	    set geometry 788x550
	}
    } else {
	roster::create .roster -width $rosterwidth -height 300 \
	    -popup [namespace current]::roster::popup_menu \
	    -grouppopup [namespace current]::roster::group_popup_menu
	pack .roster -expand yes -fill both -side left -in [$mf getframe]
	set geometry [option get . geometry [winfo class .]]
	if {$geometry == ""} {
	    set geometry 200x350
	}
    }

    wm geometry . $geometry

    if {$usetabbar} {
	bind . <FocusIn> [list [namespace current]::get_focus .]
	bind . <FocusOut> [list [namespace current]::loose_focus .]
    }

    define_alert_colors
}

proc ifacetk::draginitcmd {target x y top} {
    set c .nb.c

    set tags [$c gettags current]
    if {[lcontain $tags page]} {
	# page name
	set data [string range [lindex $tags 1] 2 end]
	return [list NoteBookPage {move} $data]
    } else {
	return {}
    }
}

proc ifacetk::dropovercmd {target source event X Y op type data} {
    set c .nb.c

    set x [expr {$X-[winfo rootx $c]}]
    set y [expr {$Y-[winfo rooty $c]}]
    set xc [$c canvasx $x]
    set yc [$c canvasy $y]

    set tags [$c gettags [lindex [$c find closest $xc $yc] 0]]
    if {[lcontain $tags page]} {
	DropSite::setcursor based_arrow_down
	return 3
    } else {
	DropSite::setcursor dot
	return 2
    }
}

proc ifacetk::dropcmd {target source X Y op type data} {
    set c .nb.c

    set x [expr {$X-[winfo rootx $c]}]
    set y [expr {$Y-[winfo rooty $c]}]
    set xc [$c canvasx $x]
    set yc [$c canvasy $y]

    set tags [$c gettags [lindex [$c find closest $xc $yc] 0]]
    if {[lcontain $tags page]} {
	# page name
	set data1 [string range [lindex $tags 1] 2 end]
	.nb move $data [.nb index $data1]
    }
}

proc ifacetk::init_toolbar_popup_menu {} {
    set m .toolbar_popup

    if {[winfo exists $m]} {
	destroy $m
    }

    menu $m -tearoff 0

    $m add checkbutton -label [::msgcat::mc "Show toolbar"] \
	-variable [namespace current]::options(show_toolbar)
    $m add checkbutton -label [::msgcat::mc "Show presence bar"] \
	-variable [namespace current]::options(show_presencebar)
}

proc ifacetk::destroy_win {path} {
    global usetabbar

    if {$usetabbar} {
	set page [ifacetk::nbpage $path]
	eval destroy [pack slaves [.nb getframe $page]]
	.nb delete $page 1
	ifacetk::tab_move .nb 0
    } else {
	destroy $path
    }
}

hook::add finload_hook [namespace current]::ifacetk::create_main_window 1

proc ifacetk::allow_presence_change {connid} {
    .presence.button configure -state normal
}

proc ifacetk::disallow_presence_change {connid} {
    if {[llength [jlib::connections]] == 0} {
	.presence.button configure -state disabled
    }
}

hook::add connected_hook [namespace current]::ifacetk::allow_presence_change
hook::add disconnected_hook [namespace current]::ifacetk::disallow_presence_change

proc ifacetk::on_open_chat_window {chatid type} {
    variable number_msg
    variable personal_msg

    set number_msg($chatid) 0
    set personal_msg($chatid) 0
}

proc ifacetk::on_close_chat_window {chatid} {
    variable number_msg
    variable personal_msg

    unset number_msg($chatid)
    unset personal_msg($chatid)
}

hook::add open_chat_pre_hook [namespace current]::ifacetk::on_open_chat_window
hook::add close_chat_post_hook [namespace current]::ifacetk::on_close_chat_window

proc ifacetk::nbpage {path} {
    return [crange [win_id tab $path] 1 end]
}

proc ifacetk::update_chat_title {chatid} {
    global usetabbar
    variable options
    variable number_msg
    variable personal_msg

    set cw $chat::opened($chatid)

    if {$usetabbar} {
	set tabtitle $chat::chats(tabtitlename,$chatid)
	if {$options(message_numbers_in_tabs) && ($number_msg($chatid) > 0)} {
	    append tabtitle " ($number_msg($chatid))"
	}
	.nb itemconfigure [nbpage $cw] -text $tabtitle
    } else {
	if {$personal_msg($chatid)} {
	    set star "*"
	} else {
	    set star ""
	}
	if {$number_msg($chatid) > 0} {
	    set title "($number_msg($chatid)$star) $chat::chats(titlename,$chatid)"
	} else {
	    set title $chat::chats(titlename,$chatid)
	}
	wm title $cw $title
	wm iconname $cw $title
    }
}

proc ifacetk::update_chat_titles {} {

    foreach chatid [chat::opened] {
	lassign [chat::window_titles $chatid] \
	    ::chat::chats(tabtitlename,$chatid) \
	    ::chat::chats(titlename,$chatid)
	ifacetk::update_chat_title $chatid
    }
}

proc ifacetk::update_main_window_title {} {
    global usetabbar
    variable main_window_title
    variable number_msg
    variable personal_msg

    if {!$usetabbar} return

    set star ""
    set messages 0
    foreach chatid [chat::opened] {
	incr messages $number_msg($chatid)
	if {$personal_msg($chatid)} {
	    set star "*"
	}
    }
    if {$messages} {
	set title "($messages$star) $main_window_title"
    } else {
	set title $main_window_title
    }
    wm title . $title
    wm iconname . $title
}

proc ifacetk::chat_window_is_active {chatid} {
    global usetabbar
    variable focused

    set cw [chat::winid $chatid]
    set w [winfo toplevel $cw]
    if {[cequal $w $focused] && \
	    (!$usetabbar || ([cequal [.nb raise] [nbpage $cw]]))} {
	return 1
    } else {
	return 0
    }
}

proc ifacetk::add_number_of_messages_to_title {chatid from type body extras} {
    global usetabbar
    variable focused
    variable number_msg
    variable personal_msg

    if {[chat_window_is_active $chatid]} return

    if {$type == "chat"} {
	incr number_msg($chatid)
	set personal_msg($chatid) 1
    } elseif {$type == "groupchat"} {
	set jid [chat::get_jid $chatid]
	set myjid [chat::our_jid $chatid]
	set mynick [chat::get_nick $myjid $type]
	if {![cequal $myjid $from]} {
	    incr number_msg($chatid)
	}
	if {![cequal $jid $from] && ![cequal $myjid $from] && \
		[lindex [check_message $mynick $body] 0]} {
	    set personal_msg($chatid) 1
	}
    }
    update_chat_title $chatid
    update_main_window_title 
}

hook::add draw_message_hook [namespace current]::ifacetk::add_number_of_messages_to_title 18

proc ifacetk::set_main_window_title_on_connect {connid} {
    variable main_window_title

    set main_window_title "[jlib::connection_jid $connid] - Tkabber"
    wm title . $main_window_title
    wm iconname . $main_window_title
}

proc ifacetk::set_main_window_title_on_disconnect {connid} {
    variable main_window_title

    if {[llength [jlib::connections]] == 0} {
	set main_window_title "Tkabber"
    } else {
	set main_window_title \
	    "[jlib::connection_jid [lindex [jlib::connections] end]] - Tkabber"
    }
    wm title . $main_window_title
    wm iconname . $main_window_title
}

hook::add connected_hook \
    [namespace current]::ifacetk::set_main_window_title_on_connect
hook::add disconnected_hook \
    [namespace current]::ifacetk::set_main_window_title_on_disconnect

proc add_win {args} {
    eval ifacetk::add_win $args
}

proc ifacetk::add_win {path args} {
    global usetabbar
    variable options

    set title ""
    set tabtitle ""
    set class ""
    set raisecmd ""
    set type ""

    foreach {attr val} $args {
	switch -- $attr {
	    -title    {set title    $val}
	    -tabtitle {set tabtitle $val}
	    -class    {set class    $val}
	    -raisecmd {set raisecmd $val}
	    -type     {set type     $val}
	    default   {error "Unknown option $attr"}
	}
    }

    if {$usetabbar} {
	set page [nbpage $path]
	set f [.nb insert end $page -text $tabtitle -raisecmd $raisecmd]
	frame $path -class $class
	pack $path -expand yes -fill both -in $f
	#tkwait visibility $path
	set ::tabcolors($page) ""
	if {$options(raise_new_tab) || [llength [.nb pages]] == 1} {
	    after idle [list catch [list .nb raise $page]]
	}
    } else {
	toplevel $path -class $class -relief flat -bd 2m
	wm title $path $title
	wm iconname $path $title
	set geometry [option get $path ${type}geometry $class]
	if {$geometry != ""} {
	    wm geometry $path $geometry
	}
	bind $path <FocusIn>  [list [namespace current]::get_focus $path]
	bind $path <FocusOut> [list [namespace current]::loose_focus $path]
    }
}

proc ifacetk::get_focus {path} {
    variable focused
    variable after_focused_id
    variable options

    if {![winfo exists $path]} return
    if {[winfo toplevel $path] != $path} return
    if {![cequal $path .] && ![info exists chat::chat_id($path)]} return

    if {$focused != $path} {
	if {$after_focused_id != ""} {
	    after cancel $after_focused_id
	}
	set after_focused_id \
	    [after $options(update_title_delay) [list [namespace current]::set_title $path]]
	set focused $path
    }
}

proc ifacetk::set_title {path} {
    global usetabbar
    variable number_msg
    variable personal_msg

    if {![winfo exists $path]} return
    
    if {$usetabbar} {
	if {[set p [.nb raise]] != ""} {
	    tab_set_updated $p
	}
	update_main_window_title
    } else {
	if {[info exists chat::chat_id($path)]} {
	    set chatid $chat::chat_id($path)
	    set number_msg($chatid) 0
	    set personal_msg($chatid) 0
	    update_chat_title $chatid
	}
    }
}

proc ifacetk::loose_focus {path} {
    variable focused
    variable after_focused_id

    if {![winfo exists $path]} return
    if {[winfo toplevel $path] != $path} return

    if {($focused == $path) && ($after_focused_id != "")} {
	after cancel $after_focused_id
	set after_focused_id ""
    }
    set focused ""
    balloon::destroy
}

proc ifacetk::tab_move {nb shift} {
    set len [llength [$nb pages]]

    if {$len > 0} {
	set newidx [expr [$nb index [$nb raise]] + $shift]

	if {$newidx < 0} {
	    set newidx [expr $len - 1]
	} elseif {$newidx >= $len} {
	    set newidx 0
	}

	$nb see [lindex [$nb pages] $newidx]
	$nb raise [lindex [$nb pages] $newidx]
    }
}

proc ifacetk::current_tab_move {nb shift} {
    set len [llength [$nb pages]]

    if {$len > 0} {
	set newidx [expr [$nb index [$nb raise]] + $shift]

	if {$newidx < 0} {
	    set newidx [expr $len - 1]
	} elseif {$newidx >= $len} {
	    set newidx 0
	}

	$nb move [$nb raise] $newidx
	$nb see [$nb raise]
    }
}

proc ifacetk::tab_raise_by_number {nb num} {
    set pages [$nb pages]
    incr num -1
    set page [lindex $pages $num]
    if {$page != ""} {
	$nb raise $page
    }
}

proc ifacetk::define_alert_colors {} {
    global usetabbar
    global alert_colors

    if {$usetabbar} {
	option add *NoteBook.alertColor0 Black widgetDefault
	option add *NoteBook.alertColor1 DarkBlue widgetDefault
	option add *NoteBook.alertColor2 Blue widgetDefault
	option add *NoteBook.alertColor3 Red widgetDefault

	set alert_colors [list \
		[option get .nb alertColor0 NoteBook] \
		[option get .nb alertColor1 NoteBook] \
		[option get .nb alertColor2 NoteBook] \
		[option get .nb alertColor3 NoteBook]]
	custom::defvar alert_colors $alert_colors \
	    [::msgcat::mc "Colors for tab alert levels."] \
	    -group IFace -type string
    }
}

proc tab_set_updated {args} {
    eval ifacetk::tab_set_updated $args
}

array set ::alert_lvls {info 1 error 1 server 1 message 2 mesg_to_user 3}

proc ifacetk::tab_set_updated {path {updated 0} {level ""}} {
    global usetabbar
    global tabcolors
    global alert_lvls alert_colors
    variable focused
    variable number_msg
    variable personal_msg

    if {!$usetabbar} return

    set page [nbpage $path]
    if {[.nb index $page] < 0} {
	set page $path
	set path [pack slaves [.nb getframe $page]]
    }

    set st [wm state .]
    if {(![cequal $st normal] && ![cequal $st zoomed]) || \
	    ([cequal [focus -displayof .] ""])} {
	set backgroundP 1
    } else {
	set backgroundP 0
    }
    if {(!$updated) && $backgroundP} {
	set updated 1
	set level message
    }

    if {$updated && \
	    (($page != [.nb raise]) && \
		[info exists alert_lvls($level)] || \
		$backgroundP)} {
	set lvlnum $alert_lvls($level)
    } else {
	set lvlnum 0
    }

    if {!$updated || $tabcolors($page) < $lvlnum} {
	set color [lindex $alert_colors $lvlnum]
	.nb itemconfigure $page -foreground $color
	set tabcolors($page) $lvlnum
	hook::run tab_set_updated $path $updated $level
    }
    if {[info exists chat::chat_id($path)]} {
	if {!$updated || ($lvlnum == 0)} {
	    set chatid $chat::chat_id($path)
	    set number_msg($chatid) 0
	    set personal_msg($chatid) 0
	    update_chat_title $chatid
	    update_main_window_title
	}
    }
}

proc ifacetk::tab_menu {x y page} {
    global curmenutab
    set curmenutab $page
    tk_popup .tabsmenu $x $y    
}

###############################################################################

proc ifacetk::collapse_roster {} {
    variable rw
    variable nw
    variable rosterwidth

    set r [[winfo parent $rw] cget -width]
    set n [[winfo parent $nw] cget -width]
    if {$r > 0} {
	set rosterwidth $r
	[winfo parent $rw] configure -width 0
	[winfo parent $nw] configure -width [expr {$r + $n}]
    } else {
	[winfo parent $rw] configure -width $rosterwidth
	[winfo parent $nw] configure -width [expr {$r + $n - $rosterwidth}]
    }
}

###############################################################################

proc ifacetk::switch_toolbar {args} {
    variable options
    variable mf
    $mf showtoolbar 0 $options(show_toolbar)
}

proc ifacetk::switch_presencebar {args} {
    variable options
    variable mf

    if {[winfo exists .presence]} {
	if {$options(show_presencebar)} {
	    pack .presence -side bottom -anchor w \
		-in [.mainframe getframe] -fill x
	} else {
	    pack forget .presence
	}
    }
}

proc ifacetk::deiconify {} {
    global usetabbar tcl_platform

    if {[cequal $tcl_platform(platform) "windows"] && $usetabbar} {
	if {[cequal [option get . mainwindowstate Tkabber] "zoomed"]} {
	    wm state . zoomed
	}
    }
    wm deiconify .
}

hook::add finload_hook [namespace current]::ifacetk::deiconify 99

