// inst.c - 6502 Microprocessor Instructions Table
//
// Copyright (c) 2002, Timothy M. Stark
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
// TIMOTHY M STARK BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
// IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
// CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
// Except as contained in this notice, the name of Timothy M Stark shall not
// be used in advertising or otherwise to promote the sale, use or other 
// dealings in this Software without prior written authorization from
// Timothy M Stark.

#include "emu/defs.h"
#include "cpu/m6502/m6502.h"

#define ADC  "Add with Carry"
#define AND  "Logical AND"
#define ASL  "Arithmetic Shift Left"
#define BCC  "Branch if Carry Clear"
#define BCS  "Branch if Carry Set"
#define BEQ  "Branch if Equal"
#define BIT  "Bit Test"
#define BMI  "Branch if Minus"
#define BNE  "Branch if Not Equal"
#define BPL  "Branch if Plus"
#define BRK  "Break"
#define BVC  "Branch if Overflow Clear"
#define BVS  "Branch if Overflow Set"
#define CLC  "Clear Carry Flag"
#define CLD  "Clear Decimal Mode"
#define CLI  "Clear Interrupt Disable"
#define CLV  "Clear Overflow Flag"
#define CMP  "Compare"
#define CPX  "Compare Index X Register"
#define CPY  "Compare Index Y Register"
#define DEC  "Decrement"
#define DEX  "Decrement Index X Register"
#define DEY  "Decrement Index Y Register"
#define EOR  "Exclusive OR"
#define INC  "Increment"
#define INX  "Increment Index X Register"
#define INY  "Increment Index Y Register"
#define JMP  "Jump"
#define JSR  "Jump to Subroutine"
#define LDA  "Load Accumulator"
#define LDX  "Load Index X Register"
#define LDY  "Load Index Y Register"
#define LSR  "Logical Shift Right"
#define NOP  "No Operation"
#define ORA  "Logical Inclusive OR"
#define PHA  "Push Accumulator"
#define PHP  "Push Status Register"
#define PLA  "Pull Accumulator"
#define PLP  "Pull Status Register"
#define ROL  "Rotate Left"
#define ROR  "Rotate Right"
#define RTI  "Return from Interrupt"
#define RTS  "Return from Subroutine"
#define SBC  "Subtract with Carry"
#define SEC  "Set Carry Flag"
#define SED  "Set Decimcal Mode"
#define SEI  "Set Interrupt Disable"
#define STA  "Store Accumulator"
#define STX  "Store Index X Register"
#define STY  "Store Index Y Register"
#define TAX  "Transfer Accumulator to Index X Register"
#define TAY  "Transfer Accumulator to Index Y Register"
#define TSX  "Transfer Stack to Index X Register"
#define TXA  "Transfer Index X Register to Accumulator"
#define TXS  "Transfer Index X Register to Stack"
#define TYA  "Transfer Index Y Register to Accumulator"

#define ABS  1
#define ABX  2
#define ABY  3
#define ACC  4
#define IMM  5
#define IMP  6
#define IDX  7
#define IDY  8
#define IND  9
#define REL  10
#define ZPG  11
#define ZPX  12
#define ZPY  13

#define OP02 1

M65_INST m65_Inst[] =
{
	{ // ADC - Add with Carry
		"ADC", ADC,
		OP02,
		8,
		{ // Operand List
			{ ABS, 0x6D, 3, 4 },
			{ ABX, 0x7D, 3, 4 },
			{ ABY, 0x79, 3, 4 },
			{ IDX, 0x61, 2, 6 },
			{ IDY, 0x71, 2, 5 },
			{ IMM, 0x69, 2, 2 },
			{ ZPG, 0x65, 2, 3 },
			{ ZPX, 0x75, 2, 4 },
		}
	},

	{ // AND - Logical AND
		"AND", AND,
		OP02,
		8,
		{ // Operand List
			{ ABS, 0x2D, 3, 4 },
			{ ABX, 0x3D, 3, 4 },
			{ ABY, 0x39, 3, 4 },
			{ IDX, 0x21, 2, 6 },
			{ IDY, 0x31, 2, 5 },
			{ IMM, 0x29, 2, 2 },
			{ ZPG, 0x25, 2, 3 },
			{ ZPX, 0x35, 2, 4 }
		}
	},

	{ // ASL - Arithmetic Shift Left
		"ASL", ASL,
		OP02,
		5,
		{ // Operand List
			{ ABS, 0x0E, 3, 6 },
			{ ABX, 0x1E, 3, 7 },
			{ ACC, 0x0A, 1, 2 },
			{ ZPG, 0x06, 2, 5 },
 			{ ZPX, 0x16, 2, 6 }
		}
	},

	{ // BCC - Branch if Carry Clear
		"BCC", BCC,
		OP02,
		1,
		{ // Operand List
			{ REL, 0x90, 2, 2 }
		}
	},

	{ // BCS - Branch if Carry Set
		"BCS", BCS,
		OP02,
		1,
		{ // Operand List
			{ REL, 0xB0, 2, 2 }
		}
	},
	
	{ // BEQ - Branch if Equal
		"BEQ", BEQ,
		OP02,
		1,
		{ // Operand List
			{ REL, 0xF0, 2, 2 }
		}
	},
	
	{ // BIT - Bit Test
		"BIT", BIT,
		OP02,
		2,
		{ // Operand List
			{ ABS, 0x2C, 3, 4 },
			{ ZPG, 0x24, 2, 3 }
		}
	},

	{ // BMI - Branch if Minus
		"BMI", BMI,
		OP02,
		1,
		{ // Operand List
			{ REL, 0x30, 2, 2 }
		}
	},
	
	{ // BNE - Branch if Not Equal
		"BNE", BNE,
		OP02,
		1,
		{ // Operand List
			{ REL, 0xD0, 2, 2 }
		}
	},
	
	{ // BPL - Branch if Plus
		"BPL", BPL,
		OP02,
		1,
		{ // Operand List
			{ REL, 0x10, 2, 2 }
		}
	},
	
	{ // BRK - Break Command
		"BRK", BRK,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x00, 1, 7 }
		}
	},
	
	{ // BVC - Branch if Overflow Clear
		"BVC", BVC,
		OP02,
		1,
		{ // Operand List
			{ REL, 0x50, 2, 2 }
		}
	},
	
	{ // BVS - Branch if Overflow Set
		"BVS", BVS,
		OP02,
		1,
		{ // Operand List
			{ REL, 0x70, 2, 2 }
		}
	},

	{ // CLC - Clear Carry
		"CLC", CLC,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x18, 1, 2 }
		}
	},

	{ // CLD - Clear Decimal Code
		"CLD", CLD,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0xD8, 1, 2 }
		}
	},
	
	{ // CLI - Clear Interrupt Disable
		"CLI", CLI,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x58, 1, 2 }
		}
	},
	
	{ // CLV - Clear Overflow
		"CLV", CLV,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0xB8, 1, 2 }
		}
	},
	
	{ // CMP - Compare
		"CMP", CMP,
		OP02,
		8,
		{ // Operand List
			{ ABS, 0xCD, 3, 4 },
			{ ABX, 0xDD, 3, 4 },
			{ ABY, 0xD9, 3, 4 },
			{ IDX, 0xC1, 2, 6 },
			{ IDY, 0xD1, 2, 5 },
			{ IMM, 0xC9, 2, 2 },
			{ ZPG, 0xC5, 2, 3 },
			{ ZPX, 0xD5, 2, 4 },
		}
	},

	{ // CPX - Compare with Index X Register
		"CPX", CPX,
		OP02,
		3,
		{ // Operand List
			{ ABS, 0xEC, 3, 4 },
			{ IMM, 0xE0, 2, 2 },
			{ ZPG, 0xE4, 2, 3 },
		}
	},

	{ // CPY - Compare with Index Y Register
		"CPY", CPY,
		OP02,
		3,
		{ // Operand List
			{ ABS, 0xCC, 3, 4 },
			{ IMM, 0xC0, 2, 2 },
			{ ZPG, 0xC4, 2, 3 },
		}
	},

	{ // DEC - Decrement
		"DEC", DEC,
		OP02,
		4,
		{ // Operand List
			{ ABS, 0xCE, 3, 6 },
			{ ABX, 0xDE, 3, 7 },
			{ ZPG, 0xC6, 2, 5 },
			{ ZPX, 0xD6, 2, 6 },
		}
	},

	{ // DEX - Decrement with Index X Register
		"DEX", DEX,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0xCA, 1, 2 },
		}
	},

	{ // DEY - Decrement with Index Y Register
		"DEY", DEY,
		OP02,
		1,
		{ // Operand List
			{ IMM, 0x88, 1, 2 },
		}
	},

	{ // EOR - Exclusive OR
		"EOR", EOR,
		OP02,
		8,
		{ // Operand List
			{ ABS, 0x4D, 3, 4 },
			{ ABX, 0x5D, 3, 4 },
			{ ABY, 0x59, 3, 4 },
			{ IDX, 0x41, 2, 6 },
			{ IDY, 0x51, 2, 5 },
			{ IMM, 0x49, 2, 2 },
			{ ZPG, 0x45, 2, 3 },
			{ ZPX, 0x55, 2, 4 },
		}
	},

	{ // INC - Increment
		"INC", INC,
		OP02,
		4,
		{ // Operand List
			{ ABS, 0xEE, 3, 6 },
			{ ABX, 0xFE, 3, 7 },
			{ ZPG, 0xE6, 2, 5 },
			{ ZPX, 0xF6, 2, 6 },
		}
	},

	{ // INX - Increment with Index X Register
		"INX", INX,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0xE8, 1, 2 },
		}
	},

	{ // INY - Increment with Index Y Register
		"INY", INY,
		OP02,
		1,
		{ // Operand List
			{ IMM, 0xC8, 1, 2 },
		}
	},

	{ // JMP - Jump
		"JMP", JMP,
		OP02,
		2,
		{ // Operand List
			{ ABS, 0x4C, 3, 3 },
			{ IND, 0x6C, 3, 5 },
		}
	},

	{ // JSR - Jump to Subroutine
		"JSR", JSR,
		OP02,
		1,
		{ // Operand List
			{ ABS, 0x20, 3, 6 },
		}
	},

	{ // LDA - Load Accumulator
		"LDA", LDA,
		OP02,
		8,
		{ // Operand List
			{ ABS, 0xAD, 3, 4 },
			{ ABX, 0xBD, 3, 4 },
			{ ABY, 0xB9, 3, 4 },
			{ IDX, 0xA1, 2, 6 },
			{ IDY, 0xB1, 2, 5 },
			{ IMM, 0xA9, 2, 2 },
			{ ZPG, 0xA5, 2, 3 },
			{ ZPX, 0xB5, 2, 4 },
		}
	},

	{ // LDX - Load Index X Register
		"LDX", LDX,
		OP02,
		5,
		{ // Operand List
			{ ABS, 0xAE, 3, 4 },
			{ ABY, 0xBE, 3, 4 },
			{ IMM, 0xA2, 2, 2 },
			{ ZPG, 0xA6, 2, 3 },
			{ ZPY, 0xB6, 2, 4 },
		}
	},

	{ // LDY - Load Index Y Register
		"LDY", LDY,
		OP02,
		5,
		{ // Operand List
			{ ABS, 0xAC, 3, 4 },
			{ ABX, 0xBC, 3, 4 },
			{ IMM, 0xA0, 2, 2 },
			{ ZPG, 0xA4, 2, 3 },
			{ ZPX, 0xB4, 2, 4 },
		}
	},

	{ // LSR - Logical Shift Right
		"LSR", LSR,
		OP02,
		5,
		{ // Operand List
			{ ABS, 0x4E, 3, 6 },
			{ ABX, 0x5E, 3, 7 },
			{ ACC, 0x4A, 1, 2 },
			{ ZPG, 0x46, 2, 5 },
			{ ZPX, 0x56, 2, 6 },
		}
	},
	
	{ // NOP - No Operation
		"NOP", NOP,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0xEA, 1, 2 },
		}
	},

	{ // ORA - Logical OR Accumulator
		"ORA", ORA,
		OP02,
		8,
		{ // Operand List
			{ ABS, 0x0D, 3, 4 },
			{ ABX, 0x1D, 3, 4 },
			{ ABY, 0x19, 3, 4 },
			{ IDX, 0x01, 2, 6 },
			{ IDY, 0x11, 2, 5 },
			{ IMM, 0x09, 2, 2 },
			{ ZPG, 0x05, 2, 3 },
			{ ZPX, 0x15, 2, 4 },
		}
	},

	{ // PHA - Push Accumulator
		"PHA", PHA,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x48, 1, 3 },
		}
	},

	{ // PHP - Push Processor Status
		"PHP", PHP,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x08, 1, 3 },
		}
	},

	{ // PLA - Pull Accumulator
		"PLA", PLA,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x68, 1, 4 },
		}
	},

	{ // PLP - Pull Processor Status
		"PLP", PLP,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x28, 1, 4 },
		}
	},

	{ // ROL - Rotate Left
		"ROL", ROL,
		OP02,
		5,
		{ // Operand List
			{ ABS, 0x2E, 3, 6 },
			{ ABX, 0x3E, 3, 7 },
			{ ACC, 0x2A, 1, 2 },
			{ ZPG, 0x26, 2, 5 },
			{ ZPX, 0x36, 2, 6 },
		}
	},

	{ // ROR - Rotate Right
		"ROR", ROR,
		OP02,
		5,
		{ // Operand List
			{ ABS, 0x6E, 3, 6 },
			{ ABX, 0x7E, 3, 7 },
			{ ACC, 0x6A, 1, 2 },
			{ ZPG, 0x66, 2, 5 },
			{ ZPX, 0x76, 2, 6 },
		}
	},

	{ // RTI - Return from Interrupt
		"RTI", RTI,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x40, 1, 6 },
		}
	},

	{ // RTS - Return from Subroutine
		"RTS", RTS,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x60, 1, 5 },
		}
	},

	{ // SBC - Subtract with Carry
		"SBC", SBC,
		OP02,
		8,
		{ // Operand List
			{ ABS, 0xED, 3, 4 },
			{ ABX, 0xFD, 3, 4 },
			{ ABY, 0xF9, 3, 4 },
			{ IDX, 0xE1, 2, 6 },
			{ IDY, 0xF1, 2, 5 },
			{ IMM, 0xE9, 2, 2 },
			{ ZPG, 0xE5, 2, 3 },
			{ ZPX, 0xF5, 2, 4 },
		}
	},
	
	{ // SEC - Set Carry
		"SEC", SEC,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x38, 1, 2 },
		}
	},

	{ // SED - Set Decimal Mode
		"SED", SED,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0xF8, 1, 2 },
		}
	},

	{ // SEI - Set Interrupt Disable
		"SEI", SEI,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x78, 1, 2 },
		}
	},
	
	{ // STA - Store Accumulator
		"STA", STA,
		OP02,
		7,
		{ // Operand List
			{ ABS, 0x8D, 3, 4 },
			{ ABX, 0x9D, 3, 5 },
			{ ABY, 0x99, 3, 5 },
			{ IDX, 0x81, 2, 6 },
			{ IDY, 0x91, 2, 6 },
			{ ZPG, 0x85, 2, 3 },
			{ ZPX, 0x95, 2, 4 },
		}
	},

	{ // STX - Store Index X Register
		"STX", STX,
		OP02,
		3,
		{ // Operand List
			{ ABS, 0x8E, 3, 4 },
			{ ZPG, 0x86, 2, 3 },
			{ ZPY, 0x96, 2, 4 },
		}
	},

	{ // STY - Store Index Y Register
		"STY", STY,
		OP02,
		3,
		{ // Operand List
			{ ABS, 0x8C, 3, 4 },
			{ ZPG, 0x84, 2, 3 },
			{ ZPX, 0x94, 2, 4 },
		}
	},
	
	{ // TAX - Transfer Accumulator to Index X Register
		"TAX", TAX,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0xAA, 1, 2 },
		}
	},

	{ // TAY - Transfer Accumulator to Index Y Register
		"TAY", TAY,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0xA8, 1, 2 },
		}
	},

	{ // TSX - Transfer Stack to Index X Register
		"TSX", TSX,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0xBA, 1, 2 },
		}
	},

	{ // TXA - Transfer Index X Register to Accumulator
		"TXA", TXA,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x8A, 1, 2 },
		}
	},

	{ // TXS - Transfer Index X Register to Stack
		"TXS", TXS,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x9A, 1, 2 },
		}
	},

	{ // TYA - Transfer Index Y Register to Accumulator
		"TYA", TYA,
		OP02,
		1,
		{ // Operand List
			{ IMP, 0x98, 1, 2 },
		}
	},

	{ NULL } // Null Terminator
};

#ifdef MAIN
int main(int argc, char **argv)
{
	M65_INST *t[256];
	int idx1, idx2;
	
	memset(t, 0, sizeof(t));

	for (idx1 = 0; m65_Inst[idx1].Name; idx1++) {
		M65_INST *p = &m65_Inst[idx1];
		for (idx2 = 0; idx2 < p->nOperands; idx2++)
			t[p->Operands[idx2].opCode] = p;
	}

	for (idx1 = 0; idx1 < 256;) {
		for (idx2 = 0; idx2 < 16; idx1++, idx2++)
			printf("%s ", t[idx1] ? t[idx1]->Name : "   ");
		printf("\n");
	}

	exit(1);
}
#endif /* MAIN */
