#ifndef SIG_EVENT_HH
#define SIG_EVENT_HH

//---------------------------------------------------------------------------
// Copyright (c) 1995-1999 Ohio Board of Regents and the University of
// Cincinnati.  All Rights Reserved.

// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

//---------------------------------------------------------------------------

#include <warped/DeserializerManager.h>
#include "tyvis/VHDLEvent.hh"
#include "tyvis/VHDLKernel.hh"

/** class for sending signals around */
class SigEvent: public VHDLEvent {
public:
  SigEvent( const VHDLVTime &initSendTime,
	    const VHDLVTime &initReceiveTime,
	    const string &initSender,
	    const string &initReceiver,
	    SigType initType,
	    int initSigId,
	    int initSigSrc,
	    const VHDLVTime &initRejTime,
	    const ArrayInfo &initSBound,
	    const ArrayInfo &initDBound,
	    int initSenderSigId,
	    unsigned int initSequenceNumber,
	    TransactionType initTransType,
	    const VHDLData &initData ) :
    VHDLEvent( initSendTime, initReceiveTime, initSender, initReceiver ),
    myType( initType ),
    sigId( initSigId ),
    sigSrc( initSigSrc ),
    rejTime( initRejTime ),
    sBound( initSBound ),
    dBound( initDBound ),
    senderSigId( initSenderSigId ),
    sequenceNumber( initSequenceNumber ),
    transactionType( initTransType ),
    myData( initData.clone() ){}

  SigEvent( const VHDLVTime &initSendTime,
	    const VHDLVTime &initReceiveTime,
	    const string &initSender,
	    const string &initReceiver,
	    const VHDLData &initData ) : 
    VHDLEvent( initSendTime, initReceiveTime, initSender, initReceiver ),
    myType( EXPLICIT ),
    sigId( -1 ),
    sigSrc( -1 ),
    rejTime( VHDLVTime::getVHDLVTimeZero() ),
    sBound( nullInfo ),
    dBound( nullInfo ),
    senderSigId( -1 ),
    sequenceNumber( 0 ),
    transactionType( NORMALTRANSACTION ),
    myData( initData.clone() ){}
  
  explicit SigEvent( const SigEvent* s ) : 
    VHDLEvent( dynamic_cast<const VHDLVTime &>(s->getSendTime()),
	       dynamic_cast<const VHDLVTime &>(s->getReceiveTime()),
	       s->getSender(),
	       s->getReceiver() ),
    myType( EXPLICIT ),
    sigId( s->sigId ),
    sigSrc( s->sigSrc ),
    rejTime( s->rejTime ),
    sBound( nullInfo ),
    dBound( nullInfo ),
    senderSigId( s->senderSigId ),
    sequenceNumber( 0 ),
    transactionType( NORMALTRANSACTION ),
    myData( s->getData().clone() ){}
  
  ~SigEvent(){
    delete myData;
  }

  void serialize( SerializedInstance *si ) const {
    VHDLEvent::serialize( si );
    si->addInt( myType );
    si->addInt( sigId );
    si->addInt( sigSrc );
    si->addSerializable( &rejTime );
    si->addSerializable( &sBound );
    si->addSerializable( &dBound );
    si->addInt( senderSigId );
    si->addUnsigned( sequenceNumber );
    si->addInt( transactionType );
    si->addSerializable( myData );
  }

  static Serializable *deserialize( SerializedInstance *si ){
    const VHDLVTime *sendTime = dynamic_cast<VHDLVTime *>(si->getSerializable());
    const VHDLVTime *recvTime = dynamic_cast<VHDLVTime *>(si->getSerializable());
    const string &sender = si->getString();
    const string &receiver = si->getString();
    const unsigned int eventId = si->getUnsigned();
    SigType type = SigType( si->getInt() );
    int sigId = si->getInt();
    int sigSrc = si->getInt();
    const VHDLVTime *rejTime =  dynamic_cast<VHDLVTime *>(si->getSerializable());
    const ArrayInfo *sBound = dynamic_cast<ArrayInfo *>(si->getSerializable());
    const ArrayInfo *dBound = dynamic_cast<ArrayInfo *>(si->getSerializable());
    int senderSigId = si->getInt();
    int sequenceNumber = si->getUnsigned();
    TransactionType transactionType = TransactionType( si->getInt() );
    const VHDLData *data = dynamic_cast<VHDLData *>(si->getSerializable());
    
    return new SigEvent( *sendTime, 
			 *recvTime,
			 sender,
			 receiver,
			 eventId,
			 type,
			 sigId,
			 sigSrc,
			 *rejTime,
			 *sBound,
			 *dBound,
			 senderSigId,
			 sequenceNumber,
			 transactionType,
			 *data );    
  }

  int getSigId() const { return sigId; }
  int getSigSrc() const { return sigSrc; }
  void setSigSrc( int newSigSrc ){ sigSrc = newSigSrc; }

  VHDLVTime getRejTime() const { return rejTime; }
  ArrayInfo getSourceBounds() const { return sBound; }
  ArrayInfo getDestBounds() const { return dBound; } 
  int getSenderSigId() const { return senderSigId; }
  unsigned int getSequenceNumber(){ return sequenceNumber; }
  void setTransactionType( TransactionType newType ){ transactionType = newType; }

  bool isCancelTransactionEvent() const { 
    if( transactionType == CANCELTRANSACTION ){ 
      return true; 
    }
    else{ 
      ASSERT( transactionType == NORMALTRANSACTION ); 
      return false; 
    }
  }

  bool isSigEvent() const { return true; }
  
  bool isNullTransaction(){ 
    if( myType == NULL_TRANSACTION ){
      return true;
    }
    else{
      return false;
    }
 }

  void execute( VHDLKernel *onProcess ){
    if( isNullTransaction() == false ){
      onProcess->updateDriver( this );
    }
    else{
      onProcess->disconnectDriver( this );
    }
  }

  static const string &getSigEventType(){
    static string sigEventType = "SigEventType";
    return sigEventType;
  }

  const string &getDataType() const {
    return getSigEventType();
  }

  const VHDLData &getData() const {
    return *myData;
  }

  static void registerDeserializer(){
    DeserializerManager::instance()->registerDeserializer( getSigEventType(),
							   &SigEvent::deserialize );
  }

protected:

private:
  /**
     Constructor used by deserializer.
  */
  SigEvent( const VHDLVTime &initSendTime,
	    const VHDLVTime &initReceiveTime,
	    const string &initSender,
	    const string &initReceiver,
	    const unsigned int eventId,
	    SigType initType,
	    int initSigId,
	    int initSigSrc,
	    const VHDLVTime &initRejTime,
	    const ArrayInfo &initSBound,
	    const ArrayInfo &initDBound,
	    int initSenderSigId,
	    unsigned int initSequenceNumber,
	    TransactionType initTransType,
	    const VHDLData &initData ) :
    VHDLEvent( initSendTime, initReceiveTime, initSender, initReceiver, eventId ),
    myType( initType ),
    sigId( initSigId ),
    sigSrc( initSigSrc ),
    rejTime( initRejTime ),
    sBound( initSBound ),
    dBound( initDBound ),
    senderSigId( initSenderSigId ),
    sequenceNumber( initSequenceNumber ),
    transactionType( initTransType ),
    myData( initData.clone() ){}


  SigType         myType;
  int             sigId;
  int             sigSrc;
  VHDLVTime       rejTime;
  ArrayInfo       sBound;
  ArrayInfo       dBound;
  int             senderSigId;
  unsigned        int sequenceNumber;
  TransactionType transactionType;
  const VHDLData  *myData;
};

inline ostream&
operator<<(ostream& os, const SigEvent& s) {
  os << static_cast<const Event &>(s)
     << "\ns.getSenderSigId(): " << s.getSenderSigId()
     << "\ns.getSigId(): " << s.getSigId() 
     << "\ns.getSigSrc(): " << s.getSigSrc() 
     << "\nrejTime: " << s.getRejTime() 
     << "\nbound: " << s.getSourceBounds()
     << "\ndbound: " << s.getDestBounds()
     << "\ndata: " << s.getData() << "\n";
  
  return os;
}

inline int
SigEventCompareRecv(const SigEvent* a, const SigEvent* b) {
  register int retVal;  
  if ( a->getReceiveTime() > b->getReceiveTime() ) {
    retVal = 1;
  }
  else if ( a->getReceiveTime() < b->getReceiveTime()) {
    retVal = -1;
  }
  else {
    retVal = 0;
  }
  return retVal;
}


#endif
