/** Copyright (c) 1995-1999 Ohio Board of Regents and the University of
    Cincinnati.  All Rights Reserved.

    You may modify, distribute, and use the software contained in this
    package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
    version 2, June 1991. A copy of this license agreement can be found
    in the file "LGPL", distributed with this archive. */

#ifndef SIGNALNETINFO_HH
#define SIGNALNETINFO_HH

#include <fstream>
#include <vector>
using std::fstream;
using std::vector;

#include "tyvis/vhdl.hh"
#include "tyvis/FanDest.hh"
#include "tyvis/ObjectBase.hh"
#include "tyvis/Block.hh"

class SourceBase;

class SignalNetinfo : public ObjectBase {
public:
  /** global signal ID global allocator variable.  This variable is kept
      globally for all signals and is increased for each time a signal is
      created. */
  static int globalSignalId;
  
  /** global signal ID.  A globally unique numeric identifier of a
      signal. */
  int id;

  /** fanout list size.  This variable holds the current number of entries in
      the fanout list (obj_ids). */
  int getFanOut() const { return fanoutTable.size(); }

  
  /** Add fan out list from another SignalNetinfo object.  Adds the fan out
      list of the given SignalNetinfo object to the fan out list of the current
      object. */
  SignalNetinfo &Add( const SignalNetinfo& sig );
  SignalNetinfo &Add( const SignalNetinfo *ptr );

  /** Add VHDLKernel object to the fan out list.  Adds the given VHDLKernel
      process object to the fan out list. */
  SignalNetinfo &Add( VHDLKernel * );
  SignalNetinfo &Add( VHDLKernel *, int sigid );
  SignalNetinfo &Add( const string &processName, int  sigid );

  void print(ostream&) const;
  virtual ObjectType getKind() const {return SIGNAL_NETINFO;};
  bool _is_signal() const;
  SignalNetinfo();
  SignalNetinfo(SignalNetinfo* ptr);
  virtual ~SignalNetinfo();

  const VHDLData& readVal() const;
  void updateVal(const VHDLData&);

  ObjectBase& operator=(const SignalNetinfo &s);
  ObjectBase& operator=(const ObjectBase& s);
  ObjectBase* clone() const;
  virtual void setResolutionFunctionId(int resolutionFnId);
  virtual void setTypeConversionFunctionId(int typeConversionFnId);

  bool driver_added_flag;

  void set_driver_added_flag(const bool);
  void set_driver_added_flag();
  bool get_driver_added_flag() const;
  bool get_sourcebase_delete_flag() const;
  void set_sourcebase_delete_flag(const bool);

  Block* getAdditionalDriverList();
  void setAdditionalDriverList(Block *);
  void addAdditionalDriver(VHDLType *driver);
  void addAdditionalDriver(Block *newDriverList);
  void dump_connectivity_info(ofstream&);
  void dump_driver_data(ofstream&);

  void setSource( SourceBase *newSource ){
    source = newSource;
  }

  SourceBase *getSource() const {    
    return source;
  }

  const vector<FanDest *> &getFanoutTable() const {
    return fanoutTable;
  }

  Block &getDownTypeConversionFnList(){
    return downTypeConversionFnList;
  }

  void addDownTypeConversionFn( void *newFunction ){
    downTypeConversionFnList.addElement( newFunction );
  }

private:
  /** fanout list.  This contains the simulation object IDs that this
      signal gets transmitted to.  This data should never change during
      a simulation, so it is allocated only once, during initState, and
      deleted after the simulation is over.  The copy constructor and
      operator= simply copy the pointer to the new object. */
  vector<FanDest *> fanoutTable;
  SourceBase *source;
  Block      downTypeConversionFnList;
  bool sourcebase_delete_flag;
  Block      *additionalDriverList;
  bool additionalDriverListDeleteFlag;
};

#endif

