#include "ArrayTypeTest.hh"
#include "tyvis/ArrayType.hh"
#include "tyvis/UniversalInteger.hh"
#include "tyvis/IntegerType.hh"
#include "tyvis/EnumerationType.hh"
#include "tyvis/std_standardPkg.hh"

void
ArrayTypeTest::setUp(){
  allOnes = initBitArray( "1111" );
  allZeros = initBitArray( "0000" );
  middleOnes = initBitArray( "0110" );
  middleZeros = initBitArray( "1001" );
}

void
ArrayTypeTest::tearDown(){
  const ArrayType *allOnesCheck = initBitArray( "1111" );
  const ArrayType *allZerosCheck = initBitArray( "0000" );
  const ArrayType *middleOnesCheck = initBitArray( "0110" );
  const ArrayType *middleZerosCheck = initBitArray( "1001" );

  CPPUNIT_ASSERT( *allOnesCheck == *allOnes );
  CPPUNIT_ASSERT( *allZerosCheck == *allZeros );
  CPPUNIT_ASSERT( *middleOnesCheck == *middleOnes );
  CPPUNIT_ASSERT( *middleZerosCheck == *middleZeros );

  delete allOnes;
  delete allZeros;
  delete middleOnes;
  delete middleZeros;
  delete allOnesCheck;
  delete allZerosCheck;
  delete middleOnesCheck;
  delete middleZerosCheck;
}

ArrayType *
ArrayTypeTest::initBitArray( const string &initPattern ){
  CPPUNIT_ASSERT( initPattern.length() == 4 );
  ArrayType *retval = new ArrayType( ObjectBase::VARIABLE,
				     *getBV4TypeInfo(),
				     -1 );
  initBitArray( *retval, initPattern );

  return retval;
}

void
ArrayTypeTest::initBitArray( ArrayType &toInit, const string &initPattern ){
  int stringIndex = 0;

  for( int arrayIndex = initPattern.length() - 1;
       arrayIndex >= 0; 
       arrayIndex--, stringIndex++ ){
    if( initPattern[stringIndex] == '0' ){
      toInit[arrayIndex] = EnumerationType::getEnumerationZero(); 
    }
    else{
      CPPUNIT_ASSERT( initPattern[stringIndex] == '1' );
      toInit[arrayIndex] = EnumerationType::getEnumerationOne();
    }
  }
}

void
ArrayTypeTest::testConstructor(){  
  ArrayType at1;
  CPPUNIT_ASSERT( !at1._is_alias() );

  ArrayType at2( ObjectBase::VARIABLE,
		 *getStringTypeInfo(),
		 -1 );
  CPPUNIT_ASSERT( at2.get_number_of_elements() == 100 );
}

void
ArrayTypeTest::testInsertion(){  
  ArrayType at( ObjectBase::VARIABLE,
		*getStringTypeInfo(),
		-1 );
  fillArray( at );
  for( int i = at.get_bounds(0).left();
       i < at.get_bounds(0).right();
       i++ ){
    CPPUNIT_ASSERT( at[i] == UniversalInteger(i) );
  }
}

void
ArrayTypeTest::testAnd(){
  CPPUNIT_ASSERT( allOnes->vhdlAnd( *allZeros ) == *allZeros );
  CPPUNIT_ASSERT( allZeros->vhdlAnd( *allOnes ) == *allZeros );
  CPPUNIT_ASSERT( allOnes->vhdlAnd( *middleOnes ) == *middleOnes );
  CPPUNIT_ASSERT( allOnes->vhdlAnd( *middleZeros ) == *middleZeros );
  CPPUNIT_ASSERT( allZeros->vhdlAnd( *middleOnes ) == *allZeros );
  CPPUNIT_ASSERT( allZeros->vhdlAnd( *middleZeros ) == *allZeros );
}

void
ArrayTypeTest::testNand(){
  CPPUNIT_ASSERT( allOnes->vhdlNand( *allZeros ) == *allOnes );
  CPPUNIT_ASSERT( allZeros->vhdlNand( *allOnes ) == *allOnes );
  CPPUNIT_ASSERT( allOnes->vhdlNand( *middleOnes ) == *middleZeros );
  CPPUNIT_ASSERT( allOnes->vhdlNand( *middleZeros ) == *middleOnes );
  CPPUNIT_ASSERT( allZeros->vhdlNand( *middleOnes ) == *allOnes );
  CPPUNIT_ASSERT( allZeros->vhdlNand( *middleZeros ) == *allOnes );
}

void
ArrayTypeTest::testOr(){
  CPPUNIT_ASSERT( allOnes->vhdlOr( *allZeros ) == *allOnes );
  CPPUNIT_ASSERT( allZeros->vhdlOr( *allOnes ) == *allOnes );
  CPPUNIT_ASSERT( allOnes->vhdlOr( *middleOnes ) == *allOnes );
  CPPUNIT_ASSERT( allOnes->vhdlOr( *middleZeros ) == *allOnes );
  CPPUNIT_ASSERT( allZeros->vhdlOr( *middleOnes ) == *middleOnes );
  CPPUNIT_ASSERT( allZeros->vhdlOr( *allZeros ) == *allZeros );
}

void
ArrayTypeTest::testNor(){
  CPPUNIT_ASSERT( allOnes->vhdlNor( *allZeros ) == *allZeros );
  CPPUNIT_ASSERT( allZeros->vhdlNor( *allOnes ) == *allZeros );
  CPPUNIT_ASSERT( allOnes->vhdlNor( *middleOnes ) == *allZeros );
  CPPUNIT_ASSERT( allOnes->vhdlNor( *middleZeros ) == *allZeros );
  CPPUNIT_ASSERT( allZeros->vhdlNor( *middleOnes ) == *middleZeros );
  CPPUNIT_ASSERT( allZeros->vhdlNor( *allZeros ) == *allOnes );
}

void
ArrayTypeTest::testXor(){
  CPPUNIT_ASSERT( allOnes->vhdlXor( *allZeros ) == *allOnes );
  CPPUNIT_ASSERT( allZeros->vhdlXor( *allOnes ) == *allOnes );
  CPPUNIT_ASSERT( allOnes->vhdlXor( *middleOnes ) == *middleZeros );
  CPPUNIT_ASSERT( allOnes->vhdlXor( *middleZeros ) == *middleOnes );
  CPPUNIT_ASSERT( allZeros->vhdlXor( *middleOnes ) == *middleOnes );
  CPPUNIT_ASSERT( allZeros->vhdlXor( *allZeros ) == *allZeros );
}

void
ArrayTypeTest::testXnor(){
  CPPUNIT_ASSERT( allOnes->vhdlXnor( *allZeros ) == *allZeros );
  CPPUNIT_ASSERT( allZeros->vhdlXnor( *allOnes ) == *allZeros );
  CPPUNIT_ASSERT( allOnes->vhdlXnor( *middleOnes ) == *middleOnes );
  CPPUNIT_ASSERT( allOnes->vhdlXnor( *middleZeros ) == *middleZeros );
  CPPUNIT_ASSERT( allZeros->vhdlXnor( *middleOnes ) == *middleZeros );
  CPPUNIT_ASSERT( allZeros->vhdlXnor( *allZeros ) == *allOnes );
}

void
ArrayTypeTest::testNot(){
  CPPUNIT_ASSERT( allOnes->vhdlNot() == *allZeros );
  CPPUNIT_ASSERT( allZeros->vhdlNot() == *allOnes );
  CPPUNIT_ASSERT( middleZeros->vhdlNot() == *middleOnes );
  CPPUNIT_ASSERT( middleOnes->vhdlNot() == *middleZeros );
}

void
ArrayTypeTest::testConcatenation(){
  // There are 4 forms of concatenation:
  // ( Array, Array ) = Array
  testConcatenationTwoArrays();
  // ( Array, Element ) = Array
  // ( Element, Array ) = Array
  // ( Element, Element ) = Array
}

void
ArrayTypeTest::testConcatenationTwoArrays(){
  const ArrayType left( ObjectBase::VARIABLE, *getStringTypeInfo(), -1 );
  const ArrayType right( ObjectBase::VARIABLE, *getStringTypeInfo(), -1 );

  fillArray( left );
  fillArray( right );

  // If both operands are one-dimensional arrays of the same type, the
  // result of the concatenation is a one-dimensional array type of this
  // same type whose length is the sum of the lengths of the operands.
  ArrayType retval = savantConcatenate( left, right );
  CPPUNIT_ASSERT( retval.get_number_of_elements() == 200 );
  for( int i = retval.get_bounds(0).left();
       i <= retval.get_bounds(0).right(); 
       i++ ){
    if( i <= left.get_bounds(0).right() ){
      CPPUNIT_ASSERT( retval[i] == UniversalInteger(i) );
    }
    else{
      CPPUNIT_ASSERT( retval[i] == UniversalInteger(i - 100 ) );      
    }
  }
}

void
ArrayTypeTest::testSLL(){
  // Shift zero places
  CPPUNIT_ASSERT( allOnes->vhdlSll( UniversalInteger(0) ) == *allOnes );
  CPPUNIT_ASSERT( allZeros->vhdlSll( UniversalInteger(0) ) == *allZeros );
  CPPUNIT_ASSERT( middleZeros->vhdlSll( UniversalInteger(0) ) == *middleZeros );
  CPPUNIT_ASSERT( middleOnes->vhdlSll( UniversalInteger(0) ) == *middleOnes );
  // Shift all zeros in
  CPPUNIT_ASSERT( allOnes->vhdlSll( UniversalInteger(4) ) == *allZeros );
  CPPUNIT_ASSERT( allOnes->vhdlSll( UniversalInteger(5) ) == *allZeros );
  CPPUNIT_ASSERT( allZeros->vhdlSll( UniversalInteger(4) ) == *allZeros );
  CPPUNIT_ASSERT( middleZeros->vhdlSll( UniversalInteger(4) ) == *allZeros );
  CPPUNIT_ASSERT( middleOnes->vhdlSll( UniversalInteger(4) ) == *allZeros );

  // Shift one place
  ArrayType *checkVal = initBitArray( "1110" );
  CPPUNIT_ASSERT( allOnes->vhdlSll( UniversalInteger(1) ) == *checkVal );
  delete checkVal;
  // Shift two places
  checkVal = initBitArray( "1100" );
  CPPUNIT_ASSERT( allOnes->vhdlSll( UniversalInteger(2) ) == *checkVal );
  delete checkVal;
  // Shift three places
  checkVal = initBitArray( "1000" );
  CPPUNIT_ASSERT( allOnes->vhdlSll( UniversalInteger(3) ) == *checkVal );
  delete checkVal;
}

void
ArrayTypeTest::testSRL(){
  // Shift zero places
  CPPUNIT_ASSERT( allOnes->vhdlSrl( UniversalInteger(0) ) == *allOnes );
  CPPUNIT_ASSERT( allZeros->vhdlSrl( UniversalInteger(0) ) == *allZeros );
  CPPUNIT_ASSERT( middleZeros->vhdlSrl( UniversalInteger(0) ) == *middleZeros );
  CPPUNIT_ASSERT( middleOnes->vhdlSrl( UniversalInteger(0) ) == *middleOnes );
  // Shift all zeros in
  CPPUNIT_ASSERT( allOnes->vhdlSrl( UniversalInteger(4) ) == *allZeros );
  CPPUNIT_ASSERT( allOnes->vhdlSrl( UniversalInteger(5) ) == *allZeros );
  CPPUNIT_ASSERT( allZeros->vhdlSrl( UniversalInteger(4) ) == *allZeros );
  CPPUNIT_ASSERT( middleZeros->vhdlSrl( UniversalInteger(4) ) == *allZeros );
  CPPUNIT_ASSERT( middleOnes->vhdlSrl( UniversalInteger(4) ) == *allZeros );

  // Shift one place
  ArrayType *checkVal = initBitArray( "0111" );
  CPPUNIT_ASSERT( allOnes->vhdlSrl( UniversalInteger(1) ) == *checkVal );
  delete checkVal;
  // Shift two places
  checkVal = initBitArray( "0011" );
  CPPUNIT_ASSERT( allOnes->vhdlSrl( UniversalInteger(2) ) == *checkVal );
  delete checkVal;
  // Shift three places
  checkVal = initBitArray( "0001" );
  CPPUNIT_ASSERT( allOnes->vhdlSrl( UniversalInteger(3) ) == *checkVal );
  delete checkVal;
}

const ArrayTypeInfo *
ArrayTypeTest::buildBV4TypeInfo(){
  ArrayTypeInfo *retval = new ArrayTypeInfo( 1,
					     &SavantbitType_info,
					     false,
					     false,
					     -1,
					     new IntegerTypeInfo( 3, ArrayInfo::downto, 0 )
					     );
					     
  return retval;
}

const ArrayTypeInfo *
ArrayTypeTest::getBV4TypeInfo(){
  static const ArrayTypeInfo *bv4TypeInfo = buildBV4TypeInfo();
  return bv4TypeInfo;
}

const ArrayTypeInfo *
ArrayTypeTest::buildStringTypeInfo(){
  ArrayTypeInfo *retval = new ArrayTypeInfo( 1,
					     &SavantcharacterType_info,
					     false,
					     false,
					     -1,
					     new IntegerTypeInfo( 0, 
								  ArrayInfo::to,
								  99 )
					     );
					     
  return retval;
}

const ArrayTypeInfo *
ArrayTypeTest::getStringTypeInfo(){
  static const ArrayTypeInfo *stringTypeInfo = buildStringTypeInfo();
  return stringTypeInfo;
}

void 
ArrayTypeTest::fillArray( const ArrayType &toFill ){
  for( int i = toFill.get_bounds(0).left();
       i <= toFill.get_bounds(0).right();
       i++ ){
    toFill[i] = IntegerType( ObjectBase::VARIABLE,
			     i );
  }
}
