/* ==================================================== ======== ======= *
 *
 *  unatima.hpp  [Native Layer: platform dependent implementation]
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _unatima_hpp_
#define	_unatima_hpp_
//pragma ident	"@(#)unatima.hh	ubit:03.06.04"
#include <ubit/config.h>
#include <ubit/unatgraph.hpp>


/** native image (encapsulates the corresponding X resources).
 * can have a "transparent background" (i.e. be shaped)
 * (!! mais seul XPM Reader lit images transp !!)
 * NB:
 * - ximashape == null ==> opaque image
 * - ximashape != null ==> transparent background
 */
class UNatIma {
public:
  static const char* const GIF;
  static const char* const JPG;
  static const char* const XPM;
  static const char* const XPM_DATA;

  UNatIma(UNatDisp*, UX_Image xima, UX_Image ximashape);
  /**< constructor.
   * Note:
   * - ximashape == null => opaque image
   * - ximashape != null => shaped image (so called "transparent" background) 
   */

  ~UNatIma();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  u_dim getWidth()   const;
  u_dim getHeight()  const;

  const UX_Image getXImage()      const {return xima;}
  const UX_Image getXImageShape() const {return ximashape;}

  bool  isRealized() const;
  bool  isShaped()   const {return ximashape != null;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  static int readImage(UNatDisp*, const char* filename, const char* filetype, 
		       UNatIma*& natima);
  /**< creates a NatIma from an image file.
   * 'filetype' can be null or any of the constants: UNatIma::GIF, JPEG, XPM
   * or XPM_DATA. If 'filetype' is null, the type is deduced from the filename 
   * suffix. This fct returns 'natima' and the UFilestat loading status.
   */

  UNatIma* createScaledImage(float xscale, float yscale);
  UNatIma* createScaledImage(UNatDisp* to_nd, float xscale, float yscale);
  /**< 
   * creates a scaled copy of this image.
   * the second variant makes it possible to create the new image on
   * a different UNatDisp.
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  static UX_Image createEmptyImage(UNatDisp*, u_dim width, u_dim height, 
				   u_dim depth);
  /**< 
   * creates an empty XImage.
   * Notes:
   * - this function also allocates the XImage data field.
   * - size and depth must be > 0
   * - depth must be <= to the NatDisp depth
   * - returns null if the image could not be created
   */

  static UX_Image createScaledImage(UNatDisp* to_nd, UNatDisp* from_nd,
				    UX_Image from_ima, 
				    float xscale, float yscale);
  /**< creates a scaled copy of an XImage.
   * args:
   * - to_nd   : the UNatDisp of the XImage to be created
   * - from_nd : the UNatDisp of from_ima
   * - this functions converts image pixels according to both NatDisps
   * - returns null if the image could not be created
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  static bool blendImages(UNatDisp*, UX_Image xima1, UX_Image xima2, 
			  float alpha);
  static bool blendImage(UNatDisp*, UX_Image xima1, UX_Color pixel2,
			 float alpha);
  /**< image blending: the resulting XImage is stored in xima1.
   * returns false if images or natdisp are not compatibles
   * note that a trueColor Visual is requested for blending images */


  static unsigned long* allocXColors(UNatDisp *nd,
                                     const XColor* colors, int colorCount); 
  ///< allocates the xcolors and returns the conversion table (which must be deleted[] after use).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  /// type of File or Data Reader functions; returns UFilestat and natima (which can be null).
  typedef int (*Reader) (class UNatDisp*, const char *fpath, UNatIma*& natima);

  /// type of File or Data Writer functions; returns UFilestat.
  typedef int (*Writer) (class UNatDisp*, const char *fpath, const UNatIma* natima);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  static int xpmFileReader(UNatDisp*, const char *fpath, UNatIma*& natima);
  static int xpmDataReader(UNatDisp*, const char *xpm_data, UNatIma*& natima);
  static int gifFileReader(UNatDisp*, const char *fpath, UNatIma*& natima);
  static int jpegFileReader(UNatDisp*, const char *fpath,UNatIma*& natima); 
  ///< the avaibility of these readers depends on the compilation and linking options of the Ubit lib

#ifndef NO_DOC
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

protected:
  friend class UGraph;
  friend class UIma;
  friend class UPix;
  friend class UNatPix;

  UNatDisp *natdisp;
  UX_Image xima, ximashape;
  int      lscale;   //LOGICAL scale

#ifdef WITH_GL
  GLuint glIma;  // texture Identifiant
  unsigned char *imaData;
  bool createGLTexture(UNatDisp*);
#endif
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

class UNatPix {
  friend class UGraph;
  friend class UPix;
  UNatDisp *natdisp;
  UX_Pixmap xpix, xpixshape;
  u_dim width, height;
  int depth, lscale;   //lscale = LOGICAL scale (used by UPix)
#ifdef WITH_GL
  GLuint glIma;  // texture Identifiant
#endif

public:
  // xpixshape == None ==> opaque image
  // xpixshape != None ==> transparent background
  UNatPix(UNatDisp*, UX_Pixmap xpix, UX_Pixmap xpixshape, 
	  u_dim width, u_dim height, int depth);

  // creates the pixmaps from the images
  // ximashape == null ==> opaque image
  // ximashape != null ==> transparent background
  UNatPix(UNatDisp*, UX_Image xima, UX_Image ximashape);
  UNatPix(UNatDisp*, UNatIma*);
  ~UNatPix();

  void set(UX_Image xima, UX_Image ximashape);

  u_dim getWidth()  const {return width;}
  u_dim getHeight() const {return height;}
  bool isRealized() const {return xpix != None;}
  bool isShaped()   const {return xpixshape != None;}
  const UX_Pixmap getXPixmap()      const {return xpix;}
  const UX_Pixmap getXPixmapShape() const {return xpixshape;}
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
