/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "FSK441Decoder.h"

#include "common/FFT.h"
#include "common/SoundFile.h"
#include "common/Exception.h"

#include <cmath>

const int BUFFER_OFFSET = (FSK441_FFT_LENGTH - FSK441_SYMBOL_LENGTH) / 2;

const int BOTTOM_TONE0_BIN = (FSK441_TONE0_FREQ - 400) / FSK441_BIN_WIDTH;
const int MID_TONE0_BIN    = FSK441_TONE0_FREQ / FSK441_BIN_WIDTH;
const int TOP_TONE0_BIN    = (FSK441_TONE0_FREQ + 400) / FSK441_BIN_WIDTH;

const int BOTTOM_TONE1_BIN = BOTTOM_TONE0_BIN + FSK441_TONE_SPACING;
const int MID_TONE1_BIN    = MID_TONE0_BIN + FSK441_TONE_SPACING;
const int TOP_TONE1_BIN    = TOP_TONE0_BIN + FSK441_TONE_SPACING;

const int BOTTOM_TONE2_BIN = BOTTOM_TONE1_BIN + FSK441_TONE_SPACING;
const int MID_TONE2_BIN    = MID_TONE1_BIN + FSK441_TONE_SPACING;
const int TOP_TONE2_BIN    = TOP_TONE1_BIN + FSK441_TONE_SPACING;

const int BOTTOM_TONE3_BIN = BOTTOM_TONE2_BIN + FSK441_TONE_SPACING;
const int MID_TONE3_BIN    = MID_TONE2_BIN + FSK441_TONE_SPACING;
const int TOP_TONE3_BIN    = TOP_TONE2_BIN + FSK441_TONE_SPACING;

enum {
	FSK441_SEEKING,
	FSK441_LOCKED
};

int main(int argc, char **argv)
{
	if (argc < 4) {
		::fprintf(stderr, "Usage: FSK441Decoder <filename> <ratio> <ratio>\n");
		return 1;
	}

	wxString fileName  = wxString(argv[1]);
	double  startRatio = ::atof(argv[2]);
	double    endRatio = ::atof(argv[3]);

	try {
		CFSK441Decoder decoder(fileName, startRatio, endRatio);
		decoder.run();
	}
	catch (CException& ex) {
		::fprintf(stderr, "Error: %s\n", ex.getMessage().c_str());
		return 1;
	}
	catch (...) {
		::fprintf(stderr, "An exception has occurred\n");
		return 1;
	}

	return 0;
}

CFSK441Decoder::CFSK441Decoder(const wxString& fileName, double startRatio, double endRatio) :
m_fileName(fileName),
m_startRatio(startRatio),
m_endRatio(endRatio),
m_noise(),
m_DF(0),
m_audioData(NULL),
m_audioLen(0),
m_burstData()
{
	m_audioData = new double[FSK441_MAX_AUDIO_DATA];

	m_noise.addValue(0.2);
}

CFSK441Decoder::~CFSK441Decoder()
{
	delete[] m_audioData;
}

void CFSK441Decoder::run()
{
	CSoundFile* file = new CSoundFile(m_fileName, FSK441_SAMPLE_RATE);

	file->openRead();

	CFFT fft(FSK441_FFT_LENGTH);

	double audio[FSK441_SOUNDBUF_LENGTH];
	double in[FSK441_FFT_LENGTH];
	double bins[FSK441_FFT_LENGTH];

	int state = FSK441_SEEKING;

	while (true) {
		int len = FSK441_SOUNDBUF_LENGTH;
		file->read(audio, len);

		if (len <= 0)
			break;

		for (int i = 0; i < len; i++) {
			m_audioData[m_audioLen++] = audio[i];

			if (m_audioLen >= FSK441_SYMBOL_LENGTH && (m_audioLen % 5) == 0) {
				for (int i = 0; i < FSK441_FFT_LENGTH; i++)
					in[i] = 0.0;

				for (int i = 0; i < FSK441_SYMBOL_LENGTH; i++)
					in[i + BUFFER_OFFSET] = m_audioData[m_audioLen - FSK441_SYMBOL_LENGTH + i];

				fft.process(in, bins);

				switch (state) {
					case FSK441_SEEKING:
						state = seekBurst(bins);
						break;
					case FSK441_LOCKED:
						state = storeBurst(bins);
						break;
					default:
						::printf("Unknown state\n");
						state = FSK441_SEEKING;
						break;
				}
			}
		}
	}

	file->close();

	delete file;

	// End of the data and still receiving a burst, display the collected data
	if (state == FSK441_LOCKED)
		decodeBurst();
}

int CFSK441Decoder::seekBurst(double* bins)
{
	wxASSERT(bins != NULL);

	double max = bins[BOTTOM_TONE0_BIN];
	m_DF       = BOTTOM_TONE0_BIN - MID_TONE0_BIN;

	for (int i = BOTTOM_TONE0_BIN; i <= TOP_TONE3_BIN; i++)
		m_noise.addValue(bins[i]);

	for (int i = BOTTOM_TONE0_BIN + 1; i <= TOP_TONE0_BIN; i++) {
		if (bins[i] > max) {
			max  = bins[i];
			m_DF = i - MID_TONE0_BIN;
		}
	}

	for (int i = BOTTOM_TONE1_BIN; i <= TOP_TONE1_BIN; i++) {
		if (bins[i] > max) {
			max  = bins[i];
			m_DF = i - MID_TONE1_BIN;
		}
	}

	for (int i = BOTTOM_TONE2_BIN; i <= TOP_TONE2_BIN; i++) {
		if (bins[i] > max) {
			max  = bins[i];
			m_DF = i - MID_TONE2_BIN;
		}
	}

	for (int i = BOTTOM_TONE3_BIN; i <= TOP_TONE3_BIN; i++) {
		if (bins[i] > max) {
			max  = bins[i];
			m_DF = i - MID_TONE3_BIN;
		}
	}

	int tone0 = MID_TONE0_BIN + m_DF;
	int tone1 = MID_TONE1_BIN + m_DF;
	int tone2 = MID_TONE2_BIN + m_DF;
	int tone3 = MID_TONE3_BIN + m_DF;

	m_noise.subtractValue(bins[tone0]);
	m_noise.subtractValue(bins[tone1]);
	m_noise.subtractValue(bins[tone2]);
	m_noise.subtractValue(bins[tone3]);

	if ((max / m_noise.getAverage()) > m_startRatio) {
		m_burstData.setStartBurst();
		m_burstData.setDF(m_DF);

		m_burstData.setData(m_audioLen, bins[tone0], bins[tone1], bins[tone2], bins[tone3]);

		return FSK441_LOCKED;
	}

	return FSK441_SEEKING;
}

int CFSK441Decoder::storeBurst(double* bins)
{
	wxASSERT(bins != NULL);

	for (int i = BOTTOM_TONE0_BIN; i <= TOP_TONE3_BIN; i++)
		m_noise.addValue(bins[i]);

	int tone0 = MID_TONE0_BIN + m_DF;
	int tone1 = MID_TONE1_BIN + m_DF;
	int tone2 = MID_TONE2_BIN + m_DF;
	int tone3 = MID_TONE3_BIN + m_DF;

	double max = bins[tone0];

	if (bins[tone1] > max)
		max = bins[tone1];

	if (bins[tone2] > max)
		max = bins[tone2];

	if (bins[tone3] > max)
		max = bins[tone3];

	m_noise.subtractValue(bins[tone0]);
	m_noise.subtractValue(bins[tone1]);
	m_noise.subtractValue(bins[tone2]);
	m_noise.subtractValue(bins[tone3]);

	if ((max / m_noise.getAverage()) > m_endRatio) {
		m_burstData.setData(m_audioLen, bins[tone0], bins[tone1], bins[tone2], bins[tone3]);
		return FSK441_LOCKED;
	}

	decodeBurst();

	return FSK441_SEEKING;
}

void CFSK441Decoder::decodeBurst()
{
	double len = double(m_burstData.getLength()) / double(FSK441_SAMPLE_RATE);

	double strength = m_burstData.getStrength(m_noise.getAverage());

	if (len < 0.02 || strength < 6.0)
		return;

	bool valid = m_burstData.processBurst();

	if (valid) {
		wxString    text = m_burstData.getMessage();
		double startTime = double(m_burstData.getStartTime()) / double(FSK441_SAMPLE_RATE);
		int           df = m_burstData.getDF() * FSK441_BIN_WIDTH;

		::printf("Start:%.1fs Length:%.0fms DF:%dHz Strength:%.0fdB Text:%s\n", startTime, len * 1000.0, df, strength, text.mb_str());
	}
}
