/*-
# X-BASED TRIANGLES
#
#  Triangles.c
#
###
#
#  Copyright (c) 1994 - 2005	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Triangles */

#if defined(USE_RPLAY) || defined(USE_NAS) || defined(USE_VMSPLAY) || defined(USE_ESOUND) || defined(WINVER) || defined(DEF_PLAY)
#define USE_SOUND
extern void playSound(char * filename);
#endif
#include "TrianglesP.h"

#ifndef PICTURE
#if 1
#define PICTURE ""
#else
#ifdef WINVER
#define PICTURE "picture"
#else
#ifdef HAVE_XPM
#define PICTURE "./mandrill.xpm"
#else
#define PICTURE "./mandrill.xbm"
#endif
#endif
#endif
#endif

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wtriangles.ini"
#endif

#define SECTION "setup"
#else
#include "picture.h"

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static Boolean SetValuesTriangles(Widget current, Widget request, Widget renew);
static void QuitTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void DestroyTriangles(Widget old);
static void ResizeTriangles(TrianglesWidget w);
static void SizeTriangles(TrianglesWidget w);
static void InitializeTriangles(Widget request, Widget renew);
static void ExposeTriangles(Widget renew, XEvent * event, Region region);
static void HideTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void SelectTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void ReleaseTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeTrianglesMaybe(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void RandomizeTriangles2(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void GetTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void WriteTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void ClearTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void UndoTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void SolveTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void IncrementTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void DecrementTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void SpeedTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void SlowTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void SoundTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void EnterTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void LeaveTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void MoveTrianglesTl(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void MoveTrianglesTr(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void MoveTrianglesLeft(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void MoveRightTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void MoveTrianglesBl(TrianglesWidget w, XEvent * event, char **args, int nArgs);
static void MoveTrianglesBr(TrianglesWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsTriangles[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <KeyPress>r: Randomize()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 <KeyPress>@: Sound()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListTriangles[] =
{
	{(char *) "Quit", (XtActionProc) QuitTriangles},
	{(char *) "Hide", (XtActionProc) HideTriangles},
	{(char *) "MoveTl", (XtActionProc) MoveTrianglesTl},
	{(char *) "MoveTr", (XtActionProc) MoveTrianglesTr},
	{(char *) "MoveLeft", (XtActionProc) MoveTrianglesLeft},
	{(char *) "MoveRight", (XtActionProc) MoveRightTriangles},
	{(char *) "MoveBl", (XtActionProc) MoveTrianglesBl},
	{(char *) "MoveBr", (XtActionProc) MoveTrianglesBr},
	{(char *) "Select", (XtActionProc) SelectTriangles},
	{(char *) "Release", (XtActionProc) ReleaseTriangles},
	{(char *) "Randomize", (XtActionProc) RandomizeTriangles},
	{(char *) "RandomizeMaybe", (XtActionProc) RandomizeTrianglesMaybe},
	{(char *) "Randomize2", (XtActionProc) RandomizeTriangles2},
	{(char *) "Get", (XtActionProc) GetTriangles},
	{(char *) "Write", (XtActionProc) WriteTriangles},
	{(char *) "Clear", (XtActionProc) ClearTriangles},
	{(char *) "Undo", (XtActionProc) UndoTriangles},
	{(char *) "Solve", (XtActionProc) SolveTriangles},
	{(char *) "Increment", (XtActionProc) IncrementTriangles},
	{(char *) "Decrement", (XtActionProc) DecrementTriangles},
	{(char *) "Speed", (XtActionProc) SpeedTriangles},
	{(char *) "Slow", (XtActionProc) SlowTriangles},
	{(char *) "Sound", (XtActionProc) SoundTriangles},
	{(char *) "Enter", (XtActionProc) EnterTriangles},
	{(char *) "Leave", (XtActionProc) LeaveTriangles}
};

static XtResource resourcesTriangles[] =
{
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(TrianglesWidget, triangles.username),
	 XtRString, (caddr_t) "guest"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(TrianglesWidget, triangles.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(TrianglesWidget, triangles.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(TrianglesWidget, triangles.frameColor),
	 XtRString, (caddr_t) "cyan" /* XtDefaultForeground*/},
	{XtNtileColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(TrianglesWidget, triangles.tileColor),
	 XtRString, (caddr_t) "gray75" /*XtDefaultForeground*/},
	{XtNtileBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(TrianglesWidget, triangles.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultBackground*/},
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(TrianglesWidget, core.width),
	 XtRString, (caddr_t) "200"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(TrianglesWidget, core.height),
	 XtRString, (caddr_t) "173"},
	{XtNsize, XtCSize, XtRInt, sizeof (int),
	 XtOffset(TrianglesWidget, triangles.size),
	 XtRString, (caddr_t) "4"}, /* DEFAULTTILES */
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(TrianglesWidget, triangles.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(TrianglesWidget, triangles.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNinstall, XtCInstall, XtRBoolean, sizeof (Boolean),
	 XtOffset(TrianglesWidget, triangles.install),
	 XtRString, (caddr_t) "FALSE"},
	{XtNpicture, XtCPicture, XtRString, sizeof (String),
	 XtOffset(TrianglesWidget, triangles.picture),
	 XtRString, (caddr_t) PICTURE},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(TrianglesWidget, triangles.delay),
	 XtRString, (caddr_t) "10"}, /* DEFAULTDELAY */
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(TrianglesWidget, triangles.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNbumpSound, XtCBumpSound, XtRString, sizeof (String),
	 XtOffset(TrianglesWidget, triangles.bumpSound),
	 XtRString, (caddr_t) BUMPSOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(TrianglesWidget, triangles.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNbase, XtCBase, XtRInt, sizeof (int),
	 XtOffset(TrianglesWidget, triangles.base),
	 XtRString, (caddr_t) "10"}, /* DEFAULTBASE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(TrianglesWidget, triangles.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(TrianglesWidget, triangles.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(TrianglesWidget, triangles.menu),
	 XtRString, (caddr_t) "-1"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(TrianglesWidget, triangles.select),
	 XtRCallback, (caddr_t) NULL}
};

TrianglesClassRec trianglesClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Triangles",	/* class name */
		sizeof (TrianglesRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeTriangles,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListTriangles,	/* actions */
		XtNumber(actionsListTriangles),		/* num actions */
		resourcesTriangles,	/* resources */
		XtNumber(resourcesTriangles),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyTriangles,	/* destroy */
		(XtWidgetProc) ResizeTriangles,		/* resize */
		(XtExposeProc) ExposeTriangles,		/* expose */
		(XtSetValuesFunc) SetValuesTriangles,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsTriangles,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass trianglesWidgetClass = (WidgetClass) & trianglesClassRec;

#ifndef HAVE_USLEEP
#if !defined( VMS ) || defined( XVMSUTILS ) || ( __VMS_VER >= 70000000 )
#ifdef USE_XVMSUTILS
#include <X11/unix_time.h>
#endif
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#if HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif
#endif
#endif
#if defined(SYSV) || defined(SVR4)
#ifdef LESS_THAN_AIX3_2
#include <sys/poll.h>
#else /* !LESS_THAN_AIX3_2 */
#include <poll.h>
#endif /* !LESS_THAN_AIX3_2 */
#endif /* defined(SYSV) || defined(SVR4) */

/* not static in case usleep found in system include */
int
usleep(unsigned int usec)
{
#if (defined (SYSV) || defined(SVR4)) && !defined(__hpux)
#if defined(HAVE_NANOSLEEP)
	{
		struct timespec rqt;

		rqt.tv_nsec = 1000 * (usec % (unsigned int) 1000000);
		rqt.tv_sec = usec / (unsigned int) 1000000;
		return nanosleep(&rqt, NULL);
	}
#else
	(void) poll(
#if defined(__cplusplus) || defined(c_plusplus)
		(pollfd *) /* guess */
#else
		(void *)
#endif
		0, (int) 0, usec / 1000);	/* ms resolution */
#endif
#else
#ifdef VMS
	long timadr[2];

	if (usec != 0) {
		timadr[0] = -usec * 10;
		timadr[1] = -1;

		sys$setimr(4, &timadr, 0, 0, 0);
		sys$waitfr(4);
	}
#else
	struct timeval time_out;

#if 0
	/* (!defined(AIXV3) && !defined(__hpux)) */
	extern int select(int, fd_set *, fd_set *, fd_set *, struct timeval *);

#endif

	time_out.tv_usec = usec % (unsigned int) 1000000;
	time_out.tv_sec = usec / (unsigned int) 1000000;
	(void) select(0, (void *) 0, (void *) 0, (void *) 0, &time_out);
#endif
#endif
	return 0;
}
#endif

void
SetTriangles(TrianglesWidget w, int reason)
{
	trianglesCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(TrianglesWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->triangles.fontInfo) {
		XUnloadFont(XtDisplay(w), w->triangles.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->triangles.fontInfo);
	}
	if ((w->triangles.fontInfo = XLoadQueryFont(display,
			w->triangles.font)) == NULL) {
		(void) sprintf(buf,
			"Cannot open %s font.\nAttempting %s font as alternate\n",
			w->triangles.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->triangles.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Cannot open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->triangles.fontInfo) {
		w->triangles.digitOffset.x = XTextWidth(w->triangles.fontInfo, "8", 1)
			/ 2;
		w->triangles.digitOffset.y = w->triangles.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->triangles.digitOffset.x = 3;
		w->triangles.digitOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "triangles.log"
#endif

static Point triangleUnit[MAXORIENT][ROWTYPES + 1] =
{
	{
		{0, 0},
		{-1, -1},
		{2, 0},
		{-1, 1}},
	{
		{0, 0},
		{1, 1},
		{-2, 0},
		{1, -1}}
};
static Point triangleList[MAXORIENT][ROWTYPES + 1];
#ifdef RHOMBUS
static Point rhombusUnit[ROWTYPES][ROWTYPES + 2] =
{
	{
		{0, 0},
		{-2, 0},
		{1, 1},
		{2, 0},
		{-1, -1}},
	{
		{0, 0},
		{-1, 1},
		{2, 0},
		{1, -1},
		{-2, -0}},
	{
		{0, 0},
		{-1, 1},
		{1, 1},
		{1, -1},
		{-1, -1}}
};
static Point rhombusList[ROWTYPES][ROWTYPES + 2];
#endif

#if (!defined(WINVER) || (WINVER <= 0x030a)) /* if X or WINDOWS 3.1 or less */
static void
Sleep(unsigned int cMilliseconds)
{
#if (defined(WINVER) && (WINVER <= 0x030a))
	unsigned long time_out = GetTickCount() + cMilliseconds;

	while (time_out > GetTickCount());
#else
	(void) usleep(cMilliseconds * 1000);
#endif
}
#endif

void
intCat(char ** string, const char * var1, const int var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + 21))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%d", var1, var2);
}

void
stringCat(char ** string, const char * var1, const char * var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + strlen(var2) + 1))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%s", var1, var2);
}

static void
CheckTiles(TrianglesWidget w)
{
	char *buf1 = NULL, *buf2 = NULL;

	if (w->triangles.size < MINTILES) {
		intCat(&buf1,
			"Number of triangles on an edge out of bounds, use at least ",
			MINTILES);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTTILES);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->triangles.size = DEFAULTTILES;
	}
	if (w->triangles.delay < 0) {
		intCat(&buf1, "Delay can not be negative (",
			w->triangles.delay);
		stringCat(&buf2, buf1, "), taking absolute value");
		free(buf1);
		DISPLAY_WARNING(buf2);
		free(buf2);
		w->triangles.delay = -w->triangles.delay;
	}
	if (w->triangles.base < MINBASE || w->triangles.base > MAXBASE) {
		intCat(&buf1, "Base out of bounds, use ", MINBASE);
		stringCat(&buf2, buf1, "..");
		free(buf1);
		intCat(&buf1, buf2, MAXBASE);
		free(buf2);
		stringCat(&buf2, buf1, ", defaulting to ");
		free(buf1);
		intCat(&buf1, buf2, DEFAULTBASE);
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->triangles.base = DEFAULTBASE;
	}
}

/* This is fast for small i, a -1 is returned for negative i. */
static int
Sqrt(const int i)
{
	int j = 0;

	while (j * j <= i)
		j++;
	return (j - 1);
}

static int
ToPosition(int row, int trbl, int tlbr)
{
	return (row * row + row + trbl - tlbr);
}

static int
ToOrient(int row, int trbl, int tlbr)
{
	return ((trbl + tlbr == row) ? UP : DOWN);
}

int
Row(const int pos)
{
	return Sqrt(pos);
}

/* Passing row so there is no sqrt calculation again */
int
TrBl(const int pos, const int posRow)
{
	return ((pos - posRow * posRow) / 2);
}

int
TlBr(const int pos, int const posRow)
{
	return ((posRow * posRow + 2 * posRow - pos) / 2);
}

static int
CartesianX(TrianglesWidget w, int pos, int row)
{
	return (pos - row * row - row + w->triangles.size) *
		w->triangles.offset.x / 2 +
		w->triangles.delta.x / 2 + w->triangles.puzzleOffset.x;
}

static int
CartesianY(TrianglesWidget w, int row, int orient)
{
	return ((orient == UP) ? 0 : w->triangles.tileSize.y) +
		row * w->triangles.offset.y + w->triangles.delta.y + 2 +
		w->triangles.puzzleOffset.y;
}

static int
TileNFromSpace(TrianglesWidget w, int n, int rowType, int orient, int direction)
{
	int pos = w->triangles.spacePosition[orient];

	if (rowType == ROW) /* This one is easy */
		return (pos + ((direction == UP) ? -n : n));
	else {
		int row = Row(pos);
		int trbl = TrBl(pos, row);
		int tlbr = TlBr(pos, row);
		int offset1, offset2;

		if (direction == UP) {
			offset1 = -(n / 2 + (n & 1));
			offset2 = -(n / 2);
		} else {
			offset1 = n / 2;
			offset2 = n / 2 + (n & 1);
		}
		if (rowType == TRBL)
			return ((orient == DOWN) ?
			  ToPosition(row + offset1, trbl, tlbr + offset2) :
			  ToPosition(row + offset2, trbl, tlbr + offset1));
		else if (rowType == TLBR)
			return ((orient == DOWN) ?
			  ToPosition(row + offset1, trbl + offset2, tlbr) :
			  ToPosition(row + offset2, trbl + offset1, tlbr));
		else	/* rowType == ROW */
			return (pos + ((direction == UP) ? -n : n));
	}
}

Boolean
CheckSolved(const TrianglesWidget w)
{
	int i;

	for (i = 1; i < w->triangles.sizeSize - 1; i++)
		if (w->triangles.tileOfPosition[i - 1] != i)
			return FALSE;
	return TRUE;
}

static Boolean
bresenhamLine(int startX, int startY, int endX, int endY,
		int * pixX, int * pixY, int * pixNo)
{
	int pix = 0;
	int ex = endX - startX;
	int ey = endY - startY;
	int dx, dy, error;

	(*pixNo)++;
	if (ex > 0) {
		dx = 1;
	} else if (ex < 0) {
		dx = -1;
		ex = -ex;
	} else {
		dx = 0;
	}
	if (ey > 0) {
		dy = 1;
	} else if (ey < 0) {
		dy = -1;
		ey = -ey;
	} else {
		dy = 0;
	}
	*pixX = startX;
	*pixY = startY;
	if (ex > ey) {
		error = 2 * ey - ex;
		while (pix != *pixNo) {
			if (error >= 0) {
				error -= 2 * ex;
				*pixY += dy;
			}
			error += 2 * ey;
			*pixX += dx;
			pix++;
			if (*pixX == endX)
				return True;
		}
		return False;
	} else {
		error = 2 * ex - ey;
		while (pix != *pixNo) {
			if (error >= 0) {
				error -= 2 * ey;
				*pixX += dx;
			}
			error += 2 * ex;
			*pixY += dy;
			pix++;
			if (*pixY == endY)
				return True;
		}
		return False;
	}
}

static int
int2String(TrianglesWidget w, char *buf, int number, int base, Boolean capital)
{
	int digit, mult = base, last, position;
	int a, i, j, s, r;

	if (capital) {
		a = 'A', i = 'I', j = 'J', s = 'S', r = 'R';
	} else {
		a = 'a', i = 'i', j = 'j', s = 's', r = 'r';
	}
	if (number < 0) {
		char *buff;

		intCat(&buff, "int2String: 0 > ", number);
		DISPLAY_WARNING(buff);
		free(buff);
		return 0;
	}
	last = 1;
	while (number >= mult) {
		last++;
		mult *= base;
	}
	for (position = 0; position < last; position++) {
		mult /= base;
		digit = number / mult;
		number -= digit * mult;
		buf[position] = digit + '0';
		if (buf[position] > '9') {	/* ASCII */
			buf[position] += (a - '9' - 1);
		} else if (buf[position] < '0') {	/* EBCDIC */
			buf[position] += (a - '9' - 1);
			if (buf[position] > i)
				buf[position] += (j - i - 1);
			if (buf[position] > r)
				buf[position] += (s - r - 1);
		}
	}
	buf[last] = '\0';
	return last;
}

#ifdef WINVER
static void
POLYGON(TrianglesWidget w, Pixmap dr, GC color, GC lineColor,
		const POINT * poly, int n, Boolean origin)
{
	/* CoordModePrevious -> CoordModeOrigin */
	POINT *temp = NULL;
	int pt;

	if (!origin) {
		if (!(temp = (POINT *) malloc(sizeof (POINT) * n))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		temp[0] = poly[0];
		for (pt = 1; pt < n; pt++) {
			temp[pt].x = temp[pt - 1].x + poly[pt].x,
			temp[pt].y = temp[pt - 1].y + poly[pt].y;
		}
	}
	if (dr)
		w->triangles.hOldBitmap =
			(HBITMAP) SelectObject(w->triangles.memDC, dr);
	w->triangles.hPen = CreatePen(PS_SOLID, 1, lineColor);
	w->triangles.hOldPen = (HPEN) SelectObject(DC(w, dr),
		w->triangles.hPen);
	w->triangles.hBrush = CreateSolidBrush(color);
	w->triangles.hOldBrush = (HBRUSH) SelectObject(DC(w, dr),
		w->triangles.hBrush);
	(void) Polygon(DC(w, dr), (origin) ? poly : temp, n);
	(void) SelectObject(DC(w, dr), w->triangles.hOldBrush);
	(void) DeleteObject(w->triangles.hBrush);
	(void) SelectObject(DC(w, dr), w->triangles.hOldPen);
	(void) DeleteObject(w->triangles.hPen);
	if (dr)
		(void) SelectObject(w->triangles.memDC,
			w->triangles.hOldBitmap);
	if (!origin) {
		free(temp);
	}
}

static void
POLYLINE(TrianglesWidget w, Pixmap dr, GC color, const POINT * poly, int n,
		Boolean origin)
{
	/* CoordModePrevious -> CoordModeOrigin */
	POINT *temp = NULL;
	int pt;

	if (!origin) {
		if (!(temp = (POINT *) malloc(sizeof (POINT) * n))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		temp[0] = poly[0];
		for (pt = 1; pt < n; pt++) {
			temp[pt].x = temp[pt - 1].x + poly[pt].x,
			temp[pt].y = temp[pt - 1].y + poly[pt].y;
		}
	}
	if (dr)
		w->triangles.hOldBitmap =
			(HBITMAP) SelectObject(w->triangles.memDC, dr);
	w->triangles.hPen = CreatePen(PS_SOLID, 1, color);
	w->triangles.hOldPen = (HPEN) SelectObject(DC(w, dr),
		w->triangles.hPen);
	(void) Polyline(DC(w, dr), (origin) ? poly : temp, n);
	(void) SelectObject(DC(w, dr), w->triangles.hOldPen);
	(void) DeleteObject(w->triangles.hPen);
	if (dr)
		(void) SelectObject(w->triangles.memDC,
			w->triangles.hOldBitmap);
	if (!origin) {
		free(temp);
	}
}
#endif

static void
fill3DTriangle(TrianglesWidget w, Pixmap dr, GC gc, GC darkerGC, GC brighterGC,
	Point * list, Boolean raised, Boolean orient)
{
	GC currentGC = (raised) ? gc : darkerGC;
	Point tempList[4];
	int i;

	for (i = 0; i < 3; i++) {
		tempList[i].x = ((i == 0) ? 0 : tempList[i - 1].x) + list[i].x;
		tempList[i].y = ((i == 0) ? 0 : tempList[i - 1].y) + list[i].y;
	}
	tempList[3].x = tempList[0].x;
	tempList[3].y = tempList[1].y;

	POLYGON(w, dr, currentGC, currentGC, list, 3, False);
	currentGC = (raised ^ orient) ? ((raised) ? brighterGC : gc) :
		darkerGC;
	DRAWLINE(w, dr, currentGC,
		tempList[0].x, tempList[0].y, tempList[1].x, tempList[1].y);
	DRAWLINE(w, dr, currentGC,
		tempList[1].x, tempList[1].y, tempList[2].x, tempList[2].y);
	if (orient) {
		DRAWLINE(w, dr, currentGC,
			tempList[0].x, tempList[0].y + 1,
			tempList[1].x - 1, tempList[1].y);
		DRAWLINE(w, dr, currentGC,
			tempList[1].x - 1, tempList[1].y - 1,
			tempList[2].x + 1, tempList[2].y - 1);
	} else {
		DRAWLINE(w, dr, currentGC,
			tempList[0].x, tempList[0].y - 1,
			tempList[1].x + 1, tempList[1].y);
		DRAWLINE(w, dr, currentGC,
			tempList[1].x + 1, tempList[1].y + 1,
			tempList[2].x - 1, tempList[2].y + 1);
	}
	currentGC = (raised ^ orient) ? darkerGC :
		((raised) ? brighterGC : gc);
	DRAWLINE(w, dr, currentGC,
		tempList[2].x, tempList[2].y, tempList[0].x, tempList[0].y);
	if (orient) {
		DRAWLINE(w, dr, currentGC,
			tempList[2].x + 1, tempList[2].y,
			tempList[0].x, tempList[0].y + 1);
	} else {
		DRAWLINE(w, dr, currentGC,
			tempList[2].x - 1, tempList[2].y,
			tempList[0].x, tempList[0].y - 1);
	}
}

static void
drawShadow(TrianglesWidget w, Pixmap dr, GC gc, int startX, int startY,
	Point * list, Boolean orient)
{
	Point tempList[3];
	int i;

	for (i = 0; i < 3; i++) {
		tempList[i].x = ((i == 0) ? startX : tempList[i - 1].x) + list[i].x;
		tempList[i].y = ((i == 0) ? startY : tempList[i - 1].y) + list[i].y;
	}
	if (orient) {
		DRAWLINE(w, dr, gc,
			tempList[2].x, tempList[2].y,
			tempList[0].x, tempList[0].y);
		DRAWLINE(w, dr, gc,
			tempList[2].x + 1, tempList[2].y,
			tempList[0].x, tempList[0].y + 1);
	} else {
		DRAWLINE(w, dr, gc,
			tempList[0].x, tempList[0].y,
			tempList[1].x, tempList[1].y);
		DRAWLINE(w, dr, gc,
			tempList[1].x, tempList[1].y,
			tempList[2].x, tempList[2].y);
	}
}

static void
DrawTile(TrianglesWidget w, int pos, int orient, Boolean blank, Boolean erase,
		int pressedOffset, int offsetX, int offsetY)
{
	Pixmap *dr;
	Pixmap adr = 0;
	int dx, dy, row;
	GC tileGC, borderGC;

	/*dr = &(w->triangles.bufferTiles[pressedOffset]);*/
	dr = &adr;
	if (erase) {
		tileGC = w->triangles.inverseGC;
		borderGC = w->triangles.inverseGC;
	} else {
		tileGC = w->triangles.tileGC;
		borderGC = w->triangles.borderGC;
	}
	row = Row(pos);
	dx = CartesianX(w, pos, row) + offsetX + pressedOffset;
	dy = CartesianY(w, row, orient) + offsetY + pressedOffset;
	triangleList[orient][0].x = dx;
	triangleList[orient][0].y = dy;
	if (blank) {
		POLYGON(w, *dr, tileGC, borderGC, triangleList[orient], 3,
			False);
	} else {
		if (pressedOffset != 0) {
			drawShadow(w, *dr, w->triangles.tileDarkerGC,
				-pressedOffset, -pressedOffset,
				triangleList[orient], orient == UP);
		}
		fill3DTriangle(w, *dr, tileGC, w->triangles.tileDarkerGC,
			w->triangles.tileBrighterGC,
			triangleList[orient],
			pressedOffset == 0, orient == UP);
	}
	if (!blank) {
		int i = 0, digitOffsetX = 0, digitOffsetY = 0;
		int tile = w->triangles.tileOfPosition[pos];
		char buf[65];

		(void) int2String(w, buf, tile, w->triangles.base, True);
		while (tile >= 1) {
			tile /= w->triangles.base;
			digitOffsetX += w->triangles.digitOffset.x;
			i++;
		}
		digitOffsetY = (orient == UP) ? w->triangles.digitOffset.y +
			w->triangles.tileSize.y / 8 +
			2 * w->triangles.delta.y :
			w->triangles.digitOffset.y - w->triangles.tileSize.y +
			- w->triangles.tileSize.y / 7 +
			w->triangles.delta.y - 2;
		DRAWTEXT(w, *dr, borderGC,
			dx - digitOffsetX,
			dy + w->triangles.tileSize.y / 2 + digitOffsetY,
			buf, i);
	}
}

static void
DrawAllBufferedTiles(const TrianglesWidget w)
{
	int k, side = UP, fin = 1, step = 1;

	for (k = 0; k < w->triangles.sizeSize; k++) {
		DrawTile(w, k, side, (w->triangles.tileOfPosition[k] <= 0),
			(w->triangles.tileOfPosition[k] <= 0), FALSE, 0, 0);
		if (fin == k + 1) {
			side = UP;
			step += 2;
			fin += step;
		} else
			side = !side;
	}
}

void
DrawAllTiles(const TrianglesWidget w)
{
	int k, side = UP, fin = 1, step = 1;

	for (k = 0; k < w->triangles.sizeSize; k++) {
		DrawTile(w, k, side, (w->triangles.tileOfPosition[k] <= 0),
			(w->triangles.tileOfPosition[k] <= 0), FALSE, 0, 0);
		if (fin == k + 1) {
			side = UP;
			step += 2;
			fin += step;
		} else
			side = (side == UP) ? DOWN : UP;
	}
}

static int
MovableTile(TrianglesWidget w)
{
	int rowType = TRIANGLES_BLOCKED, l;

	/* Are the spaces in a "row" with the mouse click?
	   (If two, then one clicked on a space). */
	for (l = 0; l < ROWTYPES; l++) {
		if (w->triangles.currentRow[l] == w->triangles.spaceRow[DOWN][l] &&
		w->triangles.currentRow[l] == w->triangles.spaceRow[UP][l]) {
			if (rowType == TRIANGLES_BLOCKED) {
				rowType = l;
			} else {
				return TRIANGLES_SPACE;
			}
		}
	}
	return rowType;
}

#ifdef RHOMBUS
static void
DrawRhombus(TrianglesWidget w, int pos, int rowType, int offsetX, int offsetY)
{
	int dx, dy, k = Row(pos);

	dy = k * w->triangles.offset.y + w->triangles.delta.y + 2 +
		w->triangles.puzzleOffset.y - 2;
	dx = (pos - k * k - k + w->triangles.size) * w->triangles.offset.x / 2 +
		w->triangles.delta.x / 2 + w->triangles.puzzleOffset.x - 1;
	dx += offsetX;
	dy += offsetY;
	rhombusList[rowType][0].x = dx;
	rhombusList[rowType][0].y = dy;
	XFillPolygon(XtDisplay(w), XtWindow(w), w->triangles.inverseGC,
		rhombusList[rowType], 4,
		Convex, CoordModePrevious);
	XDrawLines(XtDisplay(w), XtWindow(w), w->triangles.inverseGC,
		rhombusList[rowType], 5,
		CoordModePrevious);
}
#endif

#ifdef RHOMBUS
/* Rhombus type is determined by its narrowist "width" to reuse constants */
static int
RhombusType(TrianglesWidget w)
{
	if (w->triangles.spaceRow[0][ROW] != w->triangles.spaceRow[1][ROW])
		return ROW;
	if (w->triangles.spaceRow[0][TRBL] != w->triangles.spaceRow[1][TRBL])
		return TRBL;
	if (w->triangles.spaceRow[0][TLBR] != w->triangles.spaceRow[1][TLBR])
		return TLBR;
	DISPLAY_WARNING("Rhombus Type: unknown");
	return -1;
}
#endif

#ifdef ANIMATE
static int
CountTiles(TrianglesWidget w, int dir, int rowType)
{
	switch (rowType) {
	case TLBR:
		return w->triangles.spaceRow[dir][TRBL] -
			w->triangles.currentRow[TRBL] +
			w->triangles.spaceRow[(dir == UP) ? DOWN : UP][ROW] -
			w->triangles.currentRow[ROW];
	case TRBL:
		return w->triangles.spaceRow[dir][TLBR] -
			w->triangles.currentRow[TLBR] +
			w->triangles.spaceRow[(dir == UP) ? DOWN : UP][ROW] -
			w->triangles.currentRow[ROW];
	case ROW:
		return w->triangles.spaceRow[dir][TRBL] -
			w->triangles.currentRow[TRBL] +
			w->triangles.currentRow[TLBR] -
			w->triangles.spaceRow[(dir == UP) ? DOWN : UP][TLBR];
	default:
		{
			char *buf;

			intCat(&buf, "CountTiles: rowType ", rowType);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
	return 0;
}

static int
DirTiles(TrianglesWidget w, int dir, int rowType)
{
	switch (rowType) {
	case TLBR:
		return (dir == UP) ? TL : BR;
	case TRBL:
		return (dir == UP) ? TR : BL;
	case ROW:
		return (dir == UP) ? LEFT : RIGHT;
	default:
		{
			char *buf;

			intCat(&buf, "DirTiles: rowType ", rowType);
			DISPLAY_WARNING(buf);
			free(buf);
		}
	}
	return dir;
}

static void
AnimateSlide(TrianglesWidget w, int dir, int fast, Boolean logMoves)
{
	int aTile, numTiles, direction = dir, rowType;
	int fillBeginPos, fillBeginPos2, fillBeginRow;
	int fillBeginOrient, fillBeginOrientOp;
	int pos, pos2, posNext, orient, orientNext;

	rowType = MovableTile(w);
	if (rowType < 0) {
		char *buf;

		intCat(&buf, "AnimateSlide: rowType ", rowType);
		DISPLAY_WARNING(buf);
		free(buf);
		numTiles = 0;
	} else {
		numTiles = CountTiles(w, dir, rowType);
		direction = DirTiles(w, dir, rowType);
	}
	if (numTiles < 0)
		numTiles = -numTiles;
	if (dir == UP)
		orient = (w->triangles.spacePosition[UP] +
			((rowType == TRBL) ? 2 : 0) >
			w->triangles.spacePosition[DOWN]) ? UP : DOWN;
	else
		orient = (w->triangles.spacePosition[UP] +
			((rowType == TRBL) ? 1 : 0) <
			w->triangles.spacePosition[DOWN]) ? UP : DOWN;
	fillBeginPos = TileNFromSpace(w, numTiles, rowType, orient,
		(dir == UP) ? DOWN : UP);
	fillBeginRow = Row(fillBeginPos);
	fillBeginOrient = ToOrient(fillBeginRow,
		TrBl(fillBeginPos, fillBeginRow),
		TlBr(fillBeginPos, fillBeginRow));
	fillBeginOrientOp = (fillBeginOrient == UP) ? DOWN : UP;
	fillBeginPos2 = w->triangles.spacePosition[fillBeginOrientOp];
#ifdef RHOMBUS
	{
		int inc = 0;
		int gapI = 0, moveI = 0;
	 	int gapJ = 0, moveJ = 0;

		gapI = w->triangles.tileSize.x * fast / w->triangles.numSlices;
		moveI = w->triangles.tileSize.x + w->triangles.delta.x;
		gapJ = 0;
		moveJ = w->triangles.tileSize.y + w->triangles.delta.y;
		if (gapI == 0)
			gapI++;
		for (inc = 0; inc < moveI + gapI; inc += gapI) {
			if (inc > moveI)
				inc = moveI;
			for (aTile = 0; aTile < numTiles; aTile++) {
				posNext = TileNFromSpace(w, aTile + 1, rowType,
					orient, !dir);
				orientNext = (aTile & 1) ? orient : !orient;
			/* Calculate deltas */
		 	DrawTile(w, posNext, orientNext, False, False, FALSE,
				1, 1);
		  	/* Erase old slivers */
		  }
#ifndef WINVER
		  XFlush(XtDisplay(w));
#endif
		  Sleep((unsigned int) (w->triangles.delay / fast));
	}
#else
	{
		int pixNo = 0;
		int pixX = 0, pixY = 0;
		int fromdx = 0, fromdy = 0;
		int todx = 0, tody = 0;
		int dx, dy, row;
		int firstRow, firstOrient;
		int *rowPos;
		int i, k;
		Boolean slideDone = False;

		if (!(rowPos = (int *)
				malloc(sizeof (int) * (numTiles + 2)))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		rowPos[2] = TileNFromSpace(w, 1, rowType, orient,
			(dir == UP) ? DOWN : UP);
		firstRow = Row(rowPos[2]);
		firstOrient = ToOrient(firstRow,
			TrBl(rowPos[2], firstRow),
			TlBr(rowPos[2], firstRow));
		rowPos[0] = w->triangles.spacePosition[firstOrient];
		rowPos[1] = w->triangles.spacePosition[(firstOrient == UP) ?
			DOWN : UP];
		for (i = 1; i < numTiles; i++) {
			rowPos[i + 2] = TileNFromSpace(w,
				i + 1, rowType, orient,
				(dir == UP) ? DOWN : UP);
		}
		row = Row(rowPos[2]);
		fromdx = CartesianX(w, rowPos[2], row);
		fromdy = CartesianY(w, row, firstOrient);
		row = Row(rowPos[0]);
		todx = CartesianX(w, rowPos[0], row);
		tody = CartesianY(w, row, firstOrient);
		dx = todx - fromdx;
		dy = tody - fromdy;
		i = 0;
		while (!slideDone) {
			for (k = 2; k < numTiles + 2; k++)
				DrawTile(w, rowPos[k],
					(firstOrient + k) & 1,
					True, True, FALSE, pixX, pixY);
			slideDone = bresenhamLine(0, 0, dx, dy,
				&pixX, &pixY, &pixNo);
			for (k = 2; k < numTiles + 2; k++)
				DrawTile(w, rowPos[k],
					(firstOrient + k) & 1,
					False, False, FALSE, pixX, pixY);
#ifndef WINVER
			XFlush(XtDisplay(w));
#endif
			if (i % 8 == 0)
				Sleep((unsigned int) (w->triangles.delay /
					fast));
			i++;
		}
		free(rowPos);
	}
#endif
	for (aTile = 0; aTile < numTiles; aTile++) {
		if (logMoves) {
			SetTriangles(w, TRIANGLES_MOVED);
			PutMove(direction);
		}
		posNext = TileNFromSpace(w, aTile + 1, rowType, orient,
			(dir == UP) ? DOWN : UP);
		orientNext = ((aTile & 1) != 0) ? orient :
			(orient == UP) ? DOWN : UP;
		if (orientNext == fillBeginOrientOp)
			fillBeginPos2 = posNext;
	}
	pos2 = w->triangles.spacePosition[orient];
	pos = w->triangles.spacePosition[(orient == UP) ? DOWN : UP];
	for (aTile = 0; aTile < numTiles; aTile++) {
		posNext = TileNFromSpace(w, aTile + 1, rowType, orient,
			(dir == UP) ? DOWN : UP);
		w->triangles.tileOfPosition[pos] =
			w->triangles.tileOfPosition[posNext];
		pos = pos2;
		pos2 = posNext;
	}
	w->triangles.tileOfPosition[fillBeginPos] =
		w->triangles.spacePosition[fillBeginOrient];
	w->triangles.tileOfPosition[fillBeginPos2] =
		w->triangles.spacePosition[fillBeginOrientOp];
	w->triangles.spacePosition[fillBeginOrient] = fillBeginPos;
	w->triangles.spacePosition[fillBeginOrientOp] = fillBeginPos2;
	if (fillBeginOrient == UP) {
		w->triangles.tileOfPosition[fillBeginPos] = 0;
		w->triangles.tileOfPosition[fillBeginPos2] = -1;
	} else {
		w->triangles.tileOfPosition[fillBeginPos] = -1;
		w->triangles.tileOfPosition[fillBeginPos2] = 0;
	}
	w->triangles.spaceRow[fillBeginOrient][ROW] = Row(fillBeginPos);
	w->triangles.spaceRow[fillBeginOrient][TRBL] = TrBl(fillBeginPos,
		w->triangles.spaceRow[fillBeginOrient][ROW]);
	w->triangles.spaceRow[fillBeginOrient][TLBR] = TlBr(fillBeginPos,
		w->triangles.spaceRow[fillBeginOrient][ROW]);
	w->triangles.spaceRow[fillBeginOrientOp][ROW] = Row(fillBeginPos2);
	w->triangles.spaceRow[fillBeginOrientOp][TRBL] = TrBl(fillBeginPos2,
		w->triangles.spaceRow[fillBeginOrientOp][ROW]);
	w->triangles.spaceRow[fillBeginOrientOp][TLBR] = TlBr(fillBeginPos2,
		w->triangles.spaceRow[fillBeginOrientOp][ROW]);
#ifdef RHOMBUS
	DrawRhombus(w, w->triangles.spacePosition[UP], RhombusType(w), 1, 1);
#endif
}
#endif

static void
ResetTiles(TrianglesWidget w)
{
	int i;

	w->triangles.sizeSize = w->triangles.size * w->triangles.size;
	if (w->triangles.tileOfPosition)
		(void) free((void *) w->triangles.tileOfPosition);
	if (!(w->triangles.tileOfPosition = (int *)
			malloc(sizeof (int) * w->triangles.sizeSize))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}
	if (startPosition)
		(void) free((void *) startPosition);
	if (!(startPosition = (int *)
			malloc(sizeof (int) * w->triangles.sizeSize))) {
		DISPLAY_ERROR("Not enough memory, exiting.");
	}

	w->triangles.spacePosition[UP] = w->triangles.sizeSize - 1;
	w->triangles.spaceRow[UP][TRBL] = w->triangles.size - 1; /* i */
	w->triangles.spaceRow[UP][TLBR] = 0; /* j */
	w->triangles.spaceRow[UP][ROW] = w->triangles.size - 1; /* k */
	if (w->triangles.size > 1) {
		w->triangles.spacePosition[DOWN] = w->triangles.sizeSize - 2;
		w->triangles.spaceRow[DOWN][TRBL] = w->triangles.size - 2; /* i */
		w->triangles.spaceRow[DOWN][TLBR] = 0; /* j */
		w->triangles.spaceRow[DOWN][ROW] = w->triangles.size - 1; /* k */
		w->triangles.tileOfPosition[w->triangles.sizeSize - 2] = -1;
	}
	w->triangles.tileOfPosition[w->triangles.sizeSize - 1] = 0;
	for (i = 1; i < w->triangles.sizeSize - 1; i++)
		w->triangles.tileOfPosition[i - 1] = i;
	FlushMoves(w);
	w->triangles.currentPosition = -1;
	w->triangles.started = FALSE;
}

static void
EraseFrame(const TrianglesWidget w, Pixmap dr)
{
	FILLRECTANGLE(w, dr, w->triangles.inverseGC,
		0, 0, w->core.width, w->core.height);
}

static void
DrawFrame(TrianglesWidget w, Pixmap dr, Boolean focus)
{
	int sumX, sumY, sumX2, offsetX, offsetY;
	Point tempList[5];
	GC gc = (focus) ? w->triangles.frameGC : w->triangles.borderGC;

	offsetX = w->triangles.puzzleOffset.x;
	offsetY = w->triangles.puzzleOffset.y;
	sumX = w->triangles.size * w->triangles.offset.x +
		w->triangles.delta.x + 1;
	sumY = w->triangles.size * w->triangles.offset.y +
		w->triangles.delta.y + 1;
	sumX2 = sumX / 2 + offsetX;
	sumX += offsetX;
	sumY += offsetY;
	if (offsetY > 0)
		FILLRECTANGLE(w, dr, w->triangles.borderGC,
			0, 0, w->core.width, offsetY);
	if (sumY < w->core.height)
		FILLRECTANGLE(w, dr, w->triangles.borderGC,
			0, sumY + 1, w->core.width, w->core.height - sumY);
	if (offsetX > 0)
		FILLRECTANGLE(w, dr, w->triangles.borderGC,
			0, 0, offsetX, w->core.height);
	if (sumX - 2 < w->core.width)
		FILLRECTANGLE(w, dr, w->triangles.borderGC,
			sumX - 2, 0, w->core.width - sumX + 2, w->core.height);
	tempList[0].x = offsetX;
	tempList[0].y = offsetY;
	tempList[1].x = sumX2 - 1;
	tempList[1].y = offsetY;
	tempList[2].x = offsetX;
	tempList[2].y = sumY;
	tempList[3].x = offsetX;
	tempList[3].y = offsetY;
	POLYGON(w, dr, w->triangles.borderGC, w->triangles.borderGC,
		tempList, 3, True);
	tempList[0].x = sumX - 2;
	tempList[0].y = offsetY;
	tempList[1].x = sumX - 2;
	tempList[1].y = sumY;
	tempList[2].x = sumX2 - 2;
	tempList[2].y = offsetY;
	tempList[3].x = sumX - 2;
	tempList[3].y = offsetY;
	POLYGON(w, dr, w->triangles.borderGC, w->triangles.borderGC,
		tempList, 3, True);
	tempList[0].x = sumX2 - 2;
	tempList[0].y = offsetY;
	tempList[1].x = sumX - 2;
	tempList[1].y = sumY;
	tempList[2].x = offsetX;
	tempList[2].y = sumY;
	tempList[3].x = sumX2 - 1;
	tempList[3].y = offsetY;
	tempList[4].x = sumX2 - 2;
	tempList[4].y = offsetY;
	POLYLINE(w, dr, gc, tempList, 5, True);
}

static void
MoveNoTiles(TrianglesWidget w)
{
	SetTriangles(w, TRIANGLES_IGNORE);
}

static void
MoveTiles(TrianglesWidget w, int from, int orient, int fast)
{
#ifdef ANIMATE
	if (fast != INSTANT && w->triangles.delay > 0) {
		w->triangles.currentPosition = from;
		w->triangles.currentRow[ROW] = Row(from);
		w->triangles.currentRow[TRBL] =
			TrBl(from, w->triangles.currentRow[ROW]);
		w->triangles.currentRow[TLBR] =
			TlBr(from, w->triangles.currentRow[ROW]);
		w->triangles.currentPositionOrient = ToOrient(w->triangles.currentRow[ROW],
			w->triangles.currentRow[TRBL],
			w->triangles.currentRow[TLBR]);
		if (w->triangles.currentPosition <
			w->triangles.spacePosition[w->triangles.currentPositionOrient]) {
			AnimateSlide(w, DOWN, fast, False);
		} else if (w->triangles.currentPosition >
			w->triangles.spacePosition[w->triangles.currentPositionOrient]) {
			AnimateSlide(w, UP, fast, False);
		} else
			return;
		w->triangles.currentPosition = -1;
	} else
#endif
	{
		int tempTile = w->triangles.tileOfPosition[from];

		w->triangles.tileOfPosition[from] =
			w->triangles.tileOfPosition[w->triangles.spacePosition[orient]];
		w->triangles.tileOfPosition[w->triangles.spacePosition[orient]] =
			tempTile;
		DrawTile(w, w->triangles.spacePosition[orient], orient,
			False, False, FALSE, 0, 0);
		w->triangles.spacePosition[orient] = from;
		w->triangles.spaceRow[orient][ROW] = Row(from);
		w->triangles.spaceRow[orient][TRBL] =
			TrBl(from, w->triangles.spaceRow[orient][ROW]);
		w->triangles.spaceRow[orient][TLBR] =
			TlBr(from, w->triangles.spaceRow[orient][ROW]);
		DrawTile(w, w->triangles.spacePosition[orient], orient,
			True, True, FALSE, 0, 0);
	}
#ifdef USE_SOUND
	if (w->triangles.sound) {
		playSound((char *) BUMPSOUND);
	}
#endif
}

static int
MoveTilesDir(TrianglesWidget w, int direction, int fast)
{
	int orient;

	switch (direction) {
		case TR:
			if (w->triangles.spaceRow[UP][TRBL] == w->triangles.spaceRow[DOWN][TRBL]
			    && w->triangles.spaceRow[UP][ROW] != w->triangles.size - 1) {
				orient = (w->triangles.spacePosition[UP] + 2 >
				w->triangles.spacePosition[DOWN]) ? UP : DOWN;
				MoveTiles(w, TileNFromSpace(w, 1, TRBL, orient, DOWN),
					!orient, fast);
				return TRUE;
			}
			break;
		case RIGHT:
			if (w->triangles.spaceRow[UP][ROW] == w->triangles.spaceRow[DOWN][ROW]
			    && w->triangles.spaceRow[UP][TRBL] != 0) {
				orient = (w->triangles.spacePosition[UP] <
				w->triangles.spacePosition[DOWN]) ? UP : DOWN;
				MoveTiles(w, TileNFromSpace(w, 1, ROW, orient, UP),
					!orient, fast);
				return TRUE;
			}
			break;
		case BR:
			if (w->triangles.spaceRow[UP][TLBR] == w->triangles.spaceRow[DOWN][TLBR]
			    && w->triangles.spaceRow[UP][TRBL] != 0) {
				orient = (w->triangles.spacePosition[UP] <
				w->triangles.spacePosition[DOWN]) ? UP : DOWN;
				MoveTiles(w, TileNFromSpace(w, 1, TLBR, orient, UP),
					!orient, fast);
				return TRUE;
			}
			break;
		case BL:
			if (w->triangles.spaceRow[UP][TRBL] == w->triangles.spaceRow[DOWN][TRBL]
			    && w->triangles.spaceRow[UP][TLBR] != 0) {
				orient = (w->triangles.spacePosition[UP] + 1 <
				w->triangles.spacePosition[DOWN]) ? UP : DOWN;
				MoveTiles(w, TileNFromSpace(w, 1, TRBL, orient, UP),
					!orient, fast);
				return TRUE;
			}
			break;
		case LEFT:
			if (w->triangles.spaceRow[UP][ROW] == w->triangles.spaceRow[DOWN][ROW]
			    && w->triangles.spaceRow[UP][TLBR] != 0) {
				orient = (w->triangles.spacePosition[UP] >
				w->triangles.spacePosition[DOWN]) ? UP : DOWN;
				MoveTiles(w, TileNFromSpace(w, 1, ROW, orient, DOWN),
					!orient, fast);
				return TRUE;
			}
			break;
		case TL:
			if (w->triangles.spaceRow[UP][TLBR] == w->triangles.spaceRow[DOWN][TLBR]
			    && w->triangles.spaceRow[UP][ROW] != w->triangles.size - 1) {
				orient = (w->triangles.spacePosition[UP] >
				w->triangles.spacePosition[DOWN]) ? UP : DOWN;
				MoveTiles(w, TileNFromSpace(w, 1, TLBR, orient, DOWN),
					!orient, fast);
				return TRUE;
			}
			break;
		default:
			{
				char *buf;

				intCat(&buf, "MoveTilesDir: direction ",
					direction);
				DISPLAY_WARNING(buf);
				free(buf);
			}
	}
	return FALSE;
}

int
MoveTrianglesDir(TrianglesWidget w, const int direction, const int fast)
{
	if (MoveTilesDir(w, direction, fast)) {
		SetTriangles(w, TRIANGLES_MOVED);
		PutMove(direction);
		return TRUE;
	}
	return FALSE;
}

#ifndef WINVER
static
#endif
int
MoveTriangles(TrianglesWidget w, const int direction)
{
	if (CheckSolved(w)) {
		MoveNoTiles(w);
		return FALSE;
	}
	if (!MoveTrianglesDir(w, direction, NORMAL)) {
		SetTriangles(w, TRIANGLES_BLOCKED);
		return FALSE;
	}
	if (CheckSolved(w)) {
		SetTriangles(w, TRIANGLES_SOLVED);
	}
	return TRUE;
}

static int
ExchangeTiles(TrianglesWidget w, int pos1, int pos2)
{
	int tempTile;

	if (w->triangles.tileOfPosition[pos1] <= 0)
		return FALSE;
	else if (w->triangles.tileOfPosition[pos2] <= 0)
		return FALSE;
	tempTile = w->triangles.tileOfPosition[pos1];
	w->triangles.tileOfPosition[pos1] = w->triangles.tileOfPosition[pos2];
	w->triangles.tileOfPosition[pos2] = tempTile;
	return TRUE;
}

static void
DiscreteMoves(TrianglesWidget w, int dir, int fast)
{
	int rowType, orient, next;
	char *buf;

	rowType = MovableTile(w);
	if (rowType < 0) {
		intCat(&buf, "DiscreteMoves: rowType ", rowType);
		DISPLAY_WARNING(buf);
		free(buf);
		DISPLAY_WARNING(buf);
	}
	if (dir == DOWN) {
		orient = (w->triangles.spacePosition[UP] +
			((rowType == TRBL) ? 1 : 0) <
			w->triangles.spacePosition[DOWN]) ? UP : DOWN;
		next = TileNFromSpace(w, 1, rowType, orient, UP);
		orient = (orient == UP) ? DOWN : UP;
		MoveTiles(w, next, orient, fast);
		SetTriangles(w, TRIANGLES_MOVED);
		switch (rowType) {
		case TLBR:
			PutMove(BR);
			return;
		case TRBL:
			PutMove(BL);
			return;
		case ROW:
			PutMove(RIGHT);
			return;
		default:
			{
				intCat(&buf,
					"DiscreteMoves: rowType ",
					rowType);
				DISPLAY_WARNING(buf);
				free(buf);
				return;
			}
		}
	} else {
		orient = (w->triangles.spacePosition[UP] +
			((rowType == TRBL) ? 2 : 0) >
			w->triangles.spacePosition[DOWN]) ? UP : DOWN;
		next = TileNFromSpace(w, 1, rowType, orient, DOWN);
		orient = (orient == UP) ? DOWN : UP;
		MoveTiles(w, next, orient, fast);
		SetTriangles(w, TRIANGLES_MOVED);
		switch (rowType) {
		case TLBR:
			PutMove(TL);
			return;
		case TRBL:
			PutMove(TR);
			return;
		case ROW:
			PutMove(LEFT);
			return;
		default:
			{
				intCat(&buf,
					"DiscreteMoves: rowType ",
					rowType);
				DISPLAY_WARNING(buf);
				free(buf);
				return;
			}
		}
	}
}

static int
PositionToTile(TrianglesWidget w, int x, int y, int *row, int *trbl, int *tlbr)
{
	int sumX, sumY, sumX2;

	sumX = w->triangles.size * w->triangles.offset.x + w->triangles.delta.x;
	sumY = w->triangles.size * w->triangles.offset.y + w->triangles.delta.y + 1;
	sumX2 = sumX / 2;
	x -= w->triangles.puzzleOffset.x;
	y -= w->triangles.puzzleOffset.y;
	if (x * (sumY + w->triangles.delta.y) + y * (sumX2 + 1) <
			(sumX2 + 1) * (sumY + w->triangles.delta.y) ||
			x * (sumY + w->triangles.delta.y) - y * (sumX2 - 1) >
			(sumX2 - 1) * (sumY + w->triangles.delta.y) ||
			y > sumY - w->triangles.delta.y)
		return -1;
	*row = (y - w->triangles.delta.y) / w->triangles.offset.y;
	*trbl = (x - sumX2 - 1 + *row * w->triangles.offset.x / 2) /
		w->triangles.offset.x;
	*trbl += ((x - (*trbl + 1) * w->triangles.offset.x) *
		(sumY + w->triangles.delta.y) + y * (sumX2 + 1)) /
		((sumX2 + 1) * (sumY + w->triangles.delta.y));
	*tlbr = (-x + sumX2 - 1 + *row * w->triangles.offset.x / 2) /
		w->triangles.offset.x;
	*tlbr += 1 + ((-x - (*tlbr + 1) * w->triangles.offset.x) *
		(sumY + w->triangles.delta.y) + y * (sumX2 - 1)) /
		((sumX2 - 1) * (sumY + w->triangles.delta.y));
	if (*row >= 0 && *trbl >= 0 && *tlbr >= 0 &&
			*row < w->triangles.size &&
			*trbl < w->triangles.size &&
			*tlbr < w->triangles.size) {
		return ToPosition(*row, *trbl, *tlbr);
	} else
		return -1;
}

static void
SelectTiles(TrianglesWidget w)
{
	if (w->triangles.currentPosition <
	    w->triangles.spacePosition[w->triangles.currentPositionOrient]) {
#ifdef ANIMATE
		if (w->triangles.delay > 0) {
			AnimateSlide(w, DOWN, NORMAL, True);
#ifdef USE_SOUND
			if (w->triangles.sound) {
				playSound((char *) BUMPSOUND);
			}
#endif
		} else
#endif
		{
			while (w->triangles.currentPosition <
			       w->triangles.spacePosition[w->triangles.currentPositionOrient]) {

				DiscreteMoves(w, DOWN, NORMAL);
			}
		}
	} else {		/*w->triangles.currentPosition >
				   w->triangles.spacePosition[w->triangles.currentPositionOrient] */
#ifdef ANIMATE
		if (w->triangles.delay > 0) {
			AnimateSlide(w, UP, NORMAL, True);
#ifdef USE_SOUND
			if (w->triangles.sound) {
				playSound((char *) BUMPSOUND);
			}
#endif
		} else
#endif
		{
			while (w->triangles.currentPosition >
			       w->triangles.spacePosition[w->triangles.currentPositionOrient]) {
				DiscreteMoves(w, UP, NORMAL);
			}
		}
	}
	if (CheckSolved(w)) {
		SetTriangles(w, TRIANGLES_SOLVED);
	}
}

static void
RandomizeTiles(TrianglesWidget w)
{
	if (w->triangles.currentPosition >= 0)
		return;
	w->triangles.cheat = False;
	/* First interchange tiles but only with
	   other tiles of the same orientation */
	if (w->triangles.size > 2) {
		int currentPos, randomPos, randomRow;
		int currentOrient = UP, randomOrient = DOWN;
		int step = 1, fin = 1;

		for (currentPos = 0; currentPos < w->triangles.sizeSize; currentPos++) {
			randomPos = currentPos;
			while (currentPos == randomPos || currentOrient != randomOrient) {
				randomPos = NRAND(w->triangles.sizeSize);
				randomRow = Row(randomPos);
				randomOrient = !((randomRow + TrBl(randomPos, randomRow) +
					TlBr(randomPos, randomRow)) % 2);
			}
			(void) ExchangeTiles(w, currentPos, randomPos);
			if (fin == currentPos + 1) {
				currentOrient = UP;
				step += 2;
				fin += step;
			} else
				currentOrient = !currentOrient;
		}
		DrawAllTiles(w);
	}
	/* Now move the spaces around randomly */
	if (w->triangles.size > 1) {
		int big = w->triangles.sizeSize + NRAND(2);
		int lastDirection = -1;
		int randomDirection;

		SetTriangles(w, TRIANGLES_RESET);

		if (w->triangles.size == 2)
			big *= big;

#ifdef DEBUG
		big = 3;
#endif

		if (big > 1000)
			big = 1000;
		while (big--) {
			randomDirection = NRAND(COORD);

#ifdef DEBUG
			sleep(1);
#endif

			if ((randomDirection + COORD / 2) % COORD != lastDirection) {
				if (MoveTrianglesDir(w, randomDirection, INSTANT))
					lastDirection = randomDirection;
				else
					big++;
			}
		}
		FlushMoves(w);
		SetTriangles(w, TRIANGLES_RANDOMIZE);
	}
	if (CheckSolved(w)) {
		SetTriangles(w, TRIANGLES_SOLVED);
	}
}

static void
GetTiles(TrianglesWidget w)
{
	FILE *fp;
	int c, i, size, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	FlushMoves(w);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &size);
	if (size >= MINTILES) {
		for (i = w->triangles.size; i < size; i++) {
			SetTriangles(w, TRIANGLES_INC);
		}
		for (i = w->triangles.size; i > size; i--) {
			SetTriangles(w, TRIANGLES_DEC);
		}
	} else {
		stringCat(&buf1, name, " corrupted: size ");
		intCat(&buf2, buf1, size);
		free(buf1);
		stringCat(&buf1, buf2, " should be between ");
		free(buf2);
		intCat(&buf2, buf1, MINTILES);
		free(buf1);
		stringCat(&buf1, buf2, " and MAXINT");
		free(buf2);
		DISPLAY_WARNING(buf1);
		free(buf1);
	}
#ifdef WINVER
	ResetTiles(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	ScanStartPosition(fp, w);
	SetTriangles(w, TRIANGLES_RESTORE);
	SetStartPosition(w);
	ScanMoves(fp, w, moves);
	(void) fclose(fp);
	(void) printf("%s: size %d, moves %d.\n", name, size, moves);
	free(lname);
	free(fname);
	w->triangles.cheat = True; /* Assume the worst. */
}

static void
WriteTiles(TrianglesWidget w)
{
	FILE *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "size%c %d\n", SYMBOL, w->triangles.size);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
	PrintStartPosition(fp, w);
	PrintMoves(fp);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
ClearTiles(TrianglesWidget w)
{
	if (w->triangles.currentPosition >= 0)
		return;
	ResetTiles(w);
	DrawAllTiles(w);
	SetTriangles(w, TRIANGLES_RESET);
}

static void
UndoTiles(TrianglesWidget w)
{
	if (MadeMoves() && w->triangles.currentPosition < 0) {
		int direction;

		GetMove(&direction);
		direction = (direction + (COORD / 2)) % COORD;
		if (MoveTilesDir(w, direction, DOUBLE)) {
			SetTriangles(w, TRIANGLES_UNDO);
		} else {
			char *buf1, *buf2;

			intCat(&buf1, "Move ", direction);
			stringCat(&buf2, buf1, " can not be made");
			free(buf1);
			DISPLAY_WARNING(buf2);
			free(buf2);
		}
	}
}

static void
SolveTiles(TrianglesWidget w)
{
	if (CheckSolved(w) || w->triangles.currentPosition >= 0)
		return;
	SetTriangles(w, TRIANGLES_SOLVE_MESSAGE);
}

static void
IncrementTiles(TrianglesWidget w)
{
	SetTriangles(w, TRIANGLES_INC);
}

static void
DecrementTiles(TrianglesWidget w)
{
	if (w->triangles.size <= MINTILES)
		return;
	SetTriangles(w, TRIANGLES_DEC);
}

static void
SpeedTiles(TrianglesWidget w)
{
	w->triangles.delay -= 5;
	if (w->triangles.delay < 0)
		w->triangles.delay = 0;
#ifdef HAVE_MOTIF
	SetTriangles(w, TRIANGLES_SPEED);
#endif
}

static void
SlowTiles(TrianglesWidget w)
{
	w->triangles.delay += 5;
#ifdef HAVE_MOTIF
	SetTriangles(w, TRIANGLES_SPEED);
#endif
}

static void
SoundTiles(TrianglesWidget w)
{
	w->triangles.sound = !w->triangles.sound;
}

#define FACTOR 0.7
#ifdef WINVER
#define MAXINTENSITY 0xFF
static int
brighter(const int light)
{
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);
	int temp = light;

	if (temp < i)
		temp = i;
	return MIN(temp / FACTOR, MAXINTENSITY);
}

static int
darker(const int light)
{
	 return (int) (light * FACTOR);
}


static void
SetValuesTriangles(TrianglesWidget w)
{
	struct tagColor {
		int red, green, blue;
	} color;
	char szBuf[80];

	w->triangles.size = GetPrivateProfileInt(SECTION, "size",
		DEFAULTTILES, INIFILE);
	w->triangles.base = GetPrivateProfileInt(SECTION, "base",
		DEFAULTBASE, INIFILE);
	w->triangles.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULTMONO, INIFILE);
	w->triangles.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverse",
		DEFAULTREVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->triangles.frameGC = RGB(color.red, color.green, color.blue);
	/* gray75 */
	(void) GetPrivateProfileString(SECTION, "tileColor", "191 191 191",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->triangles.tileGC = RGB(color.red, color.green, color.blue);
	w->triangles.tileBrighterGC = RGB(brighter(color.red),
		brighter(color.green), brighter(color.blue));
	w->triangles.tileDarkerGC = RGB(darker(color.red),
		darker(color.green), darker(color.blue));
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "tileBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->triangles.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->triangles.inverseGC = RGB(color.red, color.green, color.blue);
	(void) GetPrivateProfileString(SECTION, "picture", PICTURE,
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->triangles.picture, szBuf);
		w->triangles.picture[80] = 0;
	w->triangles.delay = GetPrivateProfileInt(SECTION, "delay",
		10, INIFILE);
	w->triangles.sound = (BOOL)
		GetPrivateProfileInt(SECTION, "sound", 0, INIFILE);
	(void) GetPrivateProfileString(SECTION, "bumpSound", BUMPSOUND,
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->triangles.bumpSound, szBuf);
	(void) GetPrivateProfileString(SECTION, "name", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->triangles.username, szBuf);
	 w->triangles.username[80] = 0;
}

void
DestroyTriangles(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

#else

#define MAXINTENSITY 0xFFFF

static Pixel
brighter(TrianglesWidget w, Pixel pixel)
{
	XColor color;
	int i = (int) ((1 - FACTOR) * MAXINTENSITY);

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
	if (color.red < i)
		color.red = i;
	if (color.green < i)
		color.green = i;
	if (color.blue < i)
		color.blue = i;
	color.red = (unsigned short) MIN(color.red / FACTOR, MAXINTENSITY);
	color.green = (unsigned short) MIN(color.green / FACTOR, MAXINTENSITY);
	color.blue = (unsigned short) MIN(color.blue / FACTOR, MAXINTENSITY);
	if (XAllocColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color))
		return color.pixel;
	return pixel;
}

static Pixel
darker(TrianglesWidget w, Pixel pixel)
{
	XColor color;

	color.pixel = pixel;
	XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
		&color);
	color.red = (unsigned short) (color.red * FACTOR);
	color.green = (unsigned short) (color.green * FACTOR);
	color.blue = (unsigned short) (color.blue * FACTOR);
	if (XAllocColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color))
		return color.pixel;
	return pixel;
}

static void
SetAllColors(TrianglesWidget w)
{
	XGCValues values;
	XtGCMask valueMask;

	valueMask = GCForeground | GCBackground;
	if (w->triangles.reverse) {
		values.foreground = w->triangles.foreground;
		values.background = w->triangles.background;
	} else {
		values.foreground = w->triangles.background;
		values.background = w->triangles.foreground;
	}
	if (w->triangles.inverseGC)
		XtReleaseGC((Widget) w, w->triangles.inverseGC);
	w->triangles.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->triangles.mono) {
		if (w->triangles.reverse) {
			values.foreground = w->triangles.background;
			values.background = w->triangles.foreground;
		} else {
			values.foreground = w->triangles.foreground;
			values.background = w->triangles.background;
		}
	} else {
		values.foreground = w->triangles.frameColor;
		values.background = w->triangles.borderColor;
	}
	if (w->triangles.frameGC)
		XtReleaseGC((Widget) w, w->triangles.frameGC);
	w->triangles.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->triangles.mono) {
		if (w->triangles.reverse) {
			values.foreground = w->triangles.background;
			values.background = w->triangles.foreground;
		} else {
			values.foreground = w->triangles.foreground;
			values.background = w->triangles.background;
		}
	} else {
		values.foreground = w->triangles.tileColor;
		values.background = w->triangles.borderColor;
	}
	if (w->triangles.tileGC)
		XtReleaseGC((Widget) w, w->triangles.tileGC);
	w->triangles.tileGC = XtGetGC((Widget) w, valueMask, &values);
	if (!w->triangles.mono) {
		values.foreground = brighter(w, w->triangles.tileColor);
	}
	if (w->triangles.tileBrighterGC)
		XtReleaseGC((Widget) w, w->triangles.tileBrighterGC);
	w->triangles.tileBrighterGC = XtGetGC((Widget) w, valueMask, &values);
	if (!w->triangles.mono) {
		values.foreground = darker(w, w->triangles.tileColor);
	}
	if (w->triangles.tileDarkerGC)
		XtReleaseGC((Widget) w, w->triangles.tileDarkerGC);
	w->triangles.tileDarkerGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->triangles.mono) {
		if (w->triangles.reverse) {
			values.foreground = w->triangles.foreground;
			values.background = w->triangles.background;
		} else {
			values.foreground = w->triangles.background;
			values.background = w->triangles.foreground;
		}
	} else {
		values.foreground = w->triangles.borderColor;
		values.background = w->triangles.tileColor;
	}
	if (w->triangles.borderGC)
		XtReleaseGC((Widget) w, w->triangles.borderGC);
	w->triangles.borderGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->triangles.fontInfo)
		XSetFont(XtDisplay(w), w->triangles.borderGC,
			w->triangles.fontInfo->fid);
}

static Boolean
SetValuesTriangles(Widget current, Widget request, Widget renew)
{
	TrianglesWidget c = (TrianglesWidget) current, w = (TrianglesWidget) renew;
	Boolean redraw = FALSE;
	Boolean redrawTiles = FALSE;

	CheckTiles(w);
	if (w->triangles.font != c->triangles.font ||
			w->triangles.borderColor != c->triangles.borderColor ||
			w->triangles.reverse != c->triangles.reverse ||
			w->triangles.mono != c->triangles.mono) {
		loadFont(w);
		SetAllColors(w);
		redrawTiles = True;
	} else if (w->triangles.background != c->triangles.background ||
			w->triangles.foreground != c->triangles.foreground ||
			w->triangles.borderColor != c->triangles.borderColor ||
			w->triangles.tileColor != c->triangles.tileColor) {
		SetAllColors(w);
		redrawTiles = True;
	}
	if (w->triangles.size != c->triangles.size ||
			w->triangles.base != c->triangles.base) {
		SizeTriangles(w);
		redraw = True;
	} else if (w->triangles.offset.x != c->triangles.offset.x ||
			w->triangles.offset.y != c->triangles.offset.y) {
		ResizeTriangles(w);
		redraw = True;
	}
	if (w->triangles.delay != c->triangles.delay) {
		w->triangles.numSlices = ((w->triangles.delay < MAXSLICES) ?
			w->triangles.delay + 1 : MAXSLICES);
	}
	if (w->triangles.menu != -1) {
		int menu = w->triangles.menu;

		w->triangles.menu = -1;
		switch (menu) {
		case 0:
			GetTiles(w);
			break;
		case 1:
			WriteTiles(w);
			break;
		case 3:
			ClearTiles(w);
			break;
		case 4:
			UndoTiles(w);
			break;
		case 5:
			RandomizeTiles(w);
			break;
		case 6:
			SolveTiles(w);
			break;
		case 7:
			IncrementTiles(w);
			break;
		case 8:
			DecrementTiles(w);
			break;
		case 9:
			SpeedTiles(w);
			break;
		case 10:
			SlowTiles(w);
			break;
		case 11:
			SoundTiles(w);
			break;
		default:
			break;
		}
	}
	if (redrawTiles && !redraw && XtIsRealized(renew) && renew->core.visible) {
		EraseFrame(c, 0);
		if (w->triangles.focus)
			DrawFrame(w, 0, True);
		DrawAllTiles(w);
	}
	return (redraw);
}

static void
QuitTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
#ifdef WINVER
	if (w->triangles.freeDC) {
		if (w->triangles.bufferTiles != NULL) {
			DeleteObject(w->triangles.bufferTiles);
		}
		DeleteDC(w->triangles.memDC);
		w->triangles.memDC = NULL;
	}
#else
	Display *display = XtDisplay(w);
	int sel;

	if (w->triangles.colormap != None) {
		XInstallColormap(display, w->triangles.oldColormap);
		XFreeColormap(display, w->triangles.colormap);
	}
	for (sel = 0; sel < 2; sel++)
		if (w->triangles.bufferTiles[sel] != None)
			XFreePixmap(display, w->triangles.bufferTiles[sel]);
	if (w->triangles.fontInfo) {
		XUnloadFont(display, w->triangles.fontInfo->fid);
		XFreeFont(display, w->triangles.fontInfo);
	}
	XtCloseDisplay(display);
#endif
	exit(0);
}

static void
DestroyTriangles(Widget old)
{
	TrianglesWidget w = (TrianglesWidget) old;

	XtReleaseGC(old, w->triangles.tileGC);
	XtReleaseGC(old, w->triangles.borderGC);
	XtReleaseGC(old, w->triangles.tileBrighterGC);
	XtReleaseGC(old, w->triangles.tileDarkerGC);
	XtReleaseGC(old, w->triangles.frameGC);
	XtReleaseGC(old, w->triangles.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->triangles.select);
}
#endif

static void
ResizeTiles(TrianglesWidget w)
{
	int i, j;
	int sel;

#ifdef WINVER
	if (w->triangles.memDC == NULL) {
		w->triangles.memDC = CreateCompatibleDC(w->core.hDC);
		if (w->triangles.memDC == NULL) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->triangles.bufferTiles[sel] != NULL) {
			DeleteObject(w->triangles.bufferTiles[sel]);
				w->triangles.bufferTiles[sel] = NULL;
		}
		if (!(w->triangles.picture && *(w->triangles.picture))) {
			if ((w->triangles.bufferTiles[sel] =
			    CreateCompatibleBitmap(w->core.hDC,
			    w->core.width, w->core.height)) == NULL) {
				DISPLAY_ERROR("Not enough memory, exiting.");
			}
		}
	}
#else
	Display *display = XtDisplay(w);
	Window window = XtWindow(w);
	XWindowAttributes xgwa;

	(void) XGetWindowAttributes(display, window, &xgwa);
	if (w->triangles.oldColormap == None) {
		w->triangles.mono = (xgwa.depth < 2 || w->triangles.mono);
		w->triangles.oldColormap = xgwa.colormap;
	}
	for (sel = 0; sel < 2; sel++) {
		if (w->triangles.bufferTiles[sel] != None) {
			XFreePixmap(display, w->triangles.bufferTiles[sel]);
			w->triangles.bufferTiles[sel] = None;
		}
		if ((w->triangles.bufferTiles[sel] = XCreatePixmap(display,
			window, w->core.width, w->core.height,
			xgwa.depth)) == None) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
	}
#endif
	if (w->triangles.picture && *(w->triangles.picture)) {
#ifdef WINVER
		for (sel = 0; sel < 2; sel++) {
			w->triangles.bufferTiles[sel] =
				LoadBitmap(w->core.hInstance,
				w->triangles.picture);
		}
#else
		if (w->triangles.image != NULL) {
			destroyImage(&(w->triangles.image),
				&(w->triangles.graphicsFormat));
		}
		if (!getImage(display, window,
				xgwa.visual, w->triangles.oldColormap, xgwa.depth,
				&(w->triangles.image), w->triangles.picture,
				w->triangles.install, &(w->triangles.graphicsFormat),
				&(w->triangles.colormap))) {
			w->triangles.picture = NULL;
		} else if (w->triangles.image == NULL) {
			w->triangles.picture = NULL;
		}
#endif
	}
#ifndef WINVER
	if (!(w->triangles.picture && *(w->triangles.picture)) &&
	    !fixedColors(xgwa.visual, xgwa.depth, w->triangles.install) &&
	    w->triangles.colormap == None) {
		w->triangles.colormap = XCreateColormap(display, window,
			xgwa.visual, AllocNone);
	}
	SetAllColors(w);
	for (sel = 0; sel < 2; sel++) {
		FILLRECTANGLE(w, w->triangles.bufferTiles[sel],
			w->triangles.inverseGC,
			0, 0, w->core.width, w->core.height);
		if ((w->triangles.picture && *(w->triangles.picture))) {

			(void) XPutImage(display, w->triangles.bufferTiles[sel],
				w->triangles.inverseGC, w->triangles.image, 0, 0,
				0, 0,
				MIN(w->triangles.image->width, w->core.width),
				MIN(w->triangles.image->height, w->core.height));
		}
	}
#endif
	for (j = 0; j < MAXORIENT; j++)
		for (i = 0; i <= ROWTYPES; i++) {
			triangleList[j][i].x = (w->triangles.tileSize.x / 2) *
				triangleUnit[j][i].x;
			triangleList[j][i].y = w->triangles.tileSize.y *
				triangleUnit[j][i].y;
		}
#ifdef RHOMBUS
	for (j = 0; j < ROWTYPES; j++)
		for (i = 0; i <= ROWTYPES + 1; i++) {
			rhombusList[j][i].x = ((w->triangles.tileSize.x + w->triangles.delta.x) / 2 ) *
				rhombusUnit[j][i].x;
			rhombusList[j][i].y = (w->triangles.tileSize.y + w->triangles.delta.y - 1) *
				rhombusUnit[j][i].y;
		}
#endif
	if (!(w->triangles.picture && *(w->triangles.picture))) {
		DrawAllBufferedTiles(w);
	}
}

#ifndef WINVER
static
#endif
void
ResizeTriangles(TrianglesWidget w)
{
	double sqrt_3 = 1.73205080756887729352744634150587237;
	Point tempSize;
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	w->triangles.delta.x = 5;
	w->triangles.delta.y = 3;
	w->triangles.offset.x = MAX(((int) w->core.width -
		w->triangles.delta.x) / w->triangles.size, 0);
	w->triangles.offset.y = MAX(((int) w->core.height -
		2 * w->triangles.delta.y) / w->triangles.size, 0);
	tempSize.x = (int) ((double) w->triangles.offset.y * 2.0 / sqrt_3);
	tempSize.y = (int) ((double) w->triangles.offset.x * sqrt_3 / 2.0);
	if (tempSize.y < w->triangles.offset.y)
		w->triangles.offset.y = tempSize.y;
	else /* tempSize.x <= w->triangles.offset.x */
		w->triangles.offset.x = tempSize.x;
	w->triangles.puzzleSize.x = w->triangles.offset.x * w->triangles.size +
		w->triangles.delta.x + 2;
	w->triangles.puzzleSize.y = w->triangles.offset.y * w->triangles.size +
		w->triangles.delta.y + 2;
	w->triangles.puzzleOffset.x = ((int) w->core.width -
		w->triangles.puzzleSize.x + 2) / 2;
	w->triangles.puzzleOffset.y = ((int) w->core.height -
		w->triangles.puzzleSize.y + 2) / 2;
	w->triangles.tileSize.x = MAX(w->triangles.offset.x -
		w->triangles.delta.x, 0);
	w->triangles.tileSize.y = MAX(w->triangles.offset.y -
		w->triangles.delta.y, 0);
}

#ifndef WINVER
static
#endif
void
SizeTriangles(TrianglesWidget w)
{
	ResetTiles(w);
	ResizeTriangles(w);
}

#ifndef WINVER
static
#endif
void
InitializeTriangles(
#ifdef WINVER
TrianglesWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
#ifdef WINVER
	SetValuesTriangles(w);
	brush = CreateSolidBrush(w->triangles.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
	w->triangles.bufferTiles[0] = NULL;
	w->triangles.bufferTiles[1] = NULL;
#else
	TrianglesWidget w = (TrianglesWidget) renew;

	(void) SRAND(getpid());
	w->triangles.bufferTiles[0] = None;
	w->triangles.bufferTiles[1] = None;
	w->triangles.colormap = None;
	w->triangles.oldColormap = None;
	w->triangles.fontInfo = NULL;
	w->triangles.tileGC = NULL;
	w->triangles.borderGC = NULL;
	w->triangles.tileBrighterGC = NULL;
	w->triangles.tileDarkerGC = NULL;
	w->triangles.frameGC = NULL;
	w->triangles.inverseGC = NULL;
#endif
	w->triangles.focus = False;
	loadFont(w);
	w->triangles.tileOfPosition = NULL;
	CheckTiles(w);
	InitMoves();
	w->triangles.numSlices = ((w->triangles.delay < MAXSLICES) ?
		w->triangles.delay + 1 : MAXSLICES);
	w->triangles.cheat = False;
	SizeTriangles(w);
}

#ifndef WINVER
static
#endif
void
ExposeTriangles(
#ifdef WINVER
TrianglesWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
#ifndef WINVER
	TrianglesWidget w = (TrianglesWidget) renew;

	if (!w->core.visible)
		return;
#endif
	ResizeTiles(w);
	EraseFrame(w, 0);
	DrawFrame(w, 0, w->triangles.focus);
	DrawAllTiles(w);
}

#ifndef WINVER
static
#endif
void
HideTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SetTriangles(w, TRIANGLES_HIDE);
}

#ifndef WINVER
static
#endif
void
SelectTriangles(TrianglesWidget w
#ifdef WINVER
, const int x, const int y
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
	int pos, row, trbl, tlbr, rowType;
#ifndef WINVER
	int x = event->xbutton.x, y = event->xbutton.y;
#endif

	pos = PositionToTile(w, x, y, &row, &trbl, &tlbr);
	if (pos >= 0) {
		if (CheckSolved(w)) {
			MoveNoTiles(w);
			w->triangles.currentPosition = -1;
			return;
		}
		w->triangles.currentPosition = pos;
		w->triangles.currentRow[ROW] = row;
		w->triangles.currentRow[TRBL] = trbl;
		w->triangles.currentRow[TLBR] = tlbr;
		w->triangles.currentPositionOrient = ToOrient(row, trbl, tlbr);
		rowType = MovableTile(w);
		if (rowType < 0) {
			DrawTile(w, w->triangles.currentPosition,
				w->triangles.currentPositionOrient,
				rowType == TRIANGLES_SPACE, False, TRUE, 0, 0);
#ifndef WINVER
			XFlush(XtDisplay(w));
#endif
			Sleep(100);
			DrawTile(w, w->triangles.currentPosition,
				w->triangles.currentPositionOrient,
				True, True, TRUE, 0, 0);
			if (rowType != TRIANGLES_SPACE)
				DrawTile(w, w->triangles.currentPosition,
					w->triangles.currentPositionOrient,
					False, False, FALSE, 0, 0);
			SetTriangles(w, rowType);
			w->triangles.currentPosition = -1;
			return;
		}
		DrawTile(w, w->triangles.currentPosition,
			w->triangles.currentPositionOrient, False, False, TRUE,
				0, 0);
	} else
		w->triangles.currentPosition = -1;
}

#ifndef WINVER
static
#endif
void
ReleaseTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	if (w->triangles.currentPosition < 0)
		return;
	DrawTile(w, w->triangles.currentPosition,
		w->triangles.currentPositionOrient, True, True, TRUE, 0, 0);
	DrawTile(w, w->triangles.currentPosition,
		w->triangles.currentPositionOrient, False, False, FALSE, 0, 0);
	SelectTiles(w);
	w->triangles.currentPosition = -1;
}

#ifndef WINVER
static
#endif
void
RandomizeTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	RandomizeTiles(w);
}

#ifndef WINVER
static void
RandomizeTrianglesMaybe(TrianglesWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->triangles.started)
		RandomizeTiles(w);
#ifdef HAVE_MOTIF
	else {
		SetTriangles(w, TRIANGLES_RANDOMIZE_QUERY);
	}
#endif
}

static void
RandomizeTriangles2(TrianglesWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->triangles.started)
#endif
		RandomizeTiles(w);
}
#endif

#ifndef WINVER
static
#endif
void
GetTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	GetTiles(w);
}

#ifndef WINVER
static
#endif
void
WriteTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	WriteTiles(w);
}

#ifndef WINVER
static
#endif
void
ClearTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ClearTiles(w);
}

#ifndef WINVER
static
#endif
void
UndoTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	UndoTiles(w);
}

#ifndef WINVER
static
#endif
void
SolveTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SolveTiles(w);
}

#ifndef WINVER
static
#endif
void
IncrementTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	IncrementTiles(w);
}

#ifndef WINVER
static
#endif
void
DecrementTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	DecrementTiles(w);
}

#ifndef WINVER
static
#endif
void
SpeedTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SpeedTiles(w);
}

#ifndef WINVER
static
#endif
void
SlowTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SlowTiles(w);
}

#ifndef WINVER
static
#endif
void
SoundTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SoundTiles(w);
}

#ifndef WINVER
static
#endif
void
EnterTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->triangles.focus = True;
	DrawFrame(w, 0, w->triangles.focus);
}

#ifndef WINVER
static
#endif
void
LeaveTriangles(TrianglesWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->triangles.focus = False;
	DrawFrame(w, 0, w->triangles.focus);
}

#ifndef WINVER
static void
MoveTrianglesTl(TrianglesWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveTriangles(w, TL);
}

static void
MoveTrianglesTr(TrianglesWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveTriangles(w, TR);
}

static void
MoveTrianglesLeft(TrianglesWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveTriangles(w, LEFT);
}

static void
MoveRightTriangles(TrianglesWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveTriangles(w, RIGHT);
}

static void
MoveTrianglesBl(TrianglesWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveTriangles(w, BL);
}

static void
MoveTrianglesBr(TrianglesWidget w, XEvent * event, char **args, int nArgs)
{
	(void) MoveTriangles(w, BR);
}
#endif
