// -*- C++ -*-

//=============================================================================
/**
 * @file SL3_SecurityCurrent.h
 *
 * SL3_SecurityCurrent.h,v 1.1 2004/07/31 13:06:27 ossama Exp
 *
 * @author Ossama Othman <ossama@dre.vanderbilt.edu>
 */
//=============================================================================


#ifndef TAO_SL3_SECURITY_CURRENT_H
#define TAO_SL3_SECURITY_CURRENT_H

#include "ace/pre.h"

#include "orbsvcs/orbsvcs/Security/security_export.h"

#if !defined (ACE_LACKS_PRAGMA_ONCE)
# pragma once
#endif /* ACE_LACKS_PRAGMA_ONCE */

#include "SL3_SecurityCurrent_Impl.h"

#include "orbsvcs/orbsvcs/SecurityLevel3C.h"

#include "tao/LocalObject.h"

#if defined(_MSC_VER)
#if (_MSC_VER >= 1200)
#pragma warning(push)
#endif /* _MSC_VER >= 1200 */
#pragma warning(disable:4250)
#endif /* _MSC_VER */


class TAO_ORB_Core;

namespace TAO
{
  namespace SL3
  {
    /**
     * @class SecurityCurrent
     *
     * @brief SecurityLevel3::SecurityCurrent implementation.
     *
     * Thread-specific information may be retrieved from the target
     * security service through this object.
     *
     * @note This SecurityCurrent implementation basically a variant
     *       of the bridge design pattern.  All operations are
     *       delegated on to concrete implementations.
     */
    class TAO_Security_Export SecurityCurrent
      : public virtual SecurityLevel3::SecurityCurrent,
        public virtual TAO_Local_RefCounted_Object
    {
    public:

      /// Constructor
      SecurityCurrent (size_t tss_slot, TAO_ORB_Core * oc);

      /**
       * @name SecurityLevel3::SecurityCurrent Methods
       *
       * Methods required by the SecurityLevel3::SecurityCurrent
       * interface.
       */
      //@{
      virtual SecurityLevel3::ClientCredentials_ptr client_credentials (
          ACE_ENV_SINGLE_ARG_DECL)
        ACE_THROW_SPEC ((CORBA::SystemException));

      virtual CORBA::Boolean request_is_local (ACE_ENV_SINGLE_ARG_DECL)
        ACE_THROW_SPEC ((CORBA::SystemException));
      //@}

      /// Return the TSS slot ID assigned to the "SecurityCurrent"
      /// object.
      /**
       * The concrete thread-specific storage SecurityCurrent
       * implementations will each use this slot ID.
       */
      size_t tss_slot (void) const;

    protected:

      /// Destructor
      /**
       * Protected destructor to enforce proper memory management
       * through the reference counting mechanism.
       */
      ~SecurityCurrent (void);

    private:

      /// Set the thread-specific storage
      /// SecurityLevel3::SecurityCurrent implementation.
      /**
       * The pointer is actually one to a concrete implementation provided
       * by the underlying security mechanism.  For example, SSLIOP
       * implements its own SSLIOP-specific operations corresponding
       * to the ones exposed by the SecurityCurrent interface.
       * Similarly, SECIOP would do the same.
       * @par
       * There is no function that places the implementation pointer
       * in TSS.  The underlying security mechanism does that.
       */
      SecurityCurrent_Impl * implementation (void);

    private:

      /**
       * @name Retricted Copying and Assignment
       *
       * Prevent copying through the copy constructor and the
       * assignment operator.
       */
      //@{
      SecurityCurrent (const SecurityCurrent &);
      void operator= (const SecurityCurrent &);
      //@}

    private:

      /// Thread-specific storage slot assigned to this object.
      const size_t tss_slot_;

      /// Pointer to the ORB Core corresponding to the ORB with which
      /// this object is registered.
      TAO_ORB_Core *orb_core_;

    };

  } // End SL3 namespace
}  // End TAO namespace


#if defined (__ACE_INLINE__)
# include "SL3_SecurityCurrent.inl"
#endif /* __ACE_INLINE__ */

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
#pragma warning(pop)
#endif /* _MSC_VER */

#include "ace/post.h"

#endif  /* TAO_SL3_SECURITY_CURRENT_H */
