// apt.cc
//
//  Copyright 1999-2002, 2004-2005 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.
//
//  Handles basic apt bookeeping that apt-pkg doesn't :)

#include "apt.h"
#include "../aptitude.h"


#include "config_signal.h"
#include "pkg_hier.h"
#include "rev_dep_iterator.h"
#include "tasks.h"
#include "undo.h"

#include <apt-pkg/configuration.h>
#include <apt-pkg/depcache.h>
#include <apt-pkg/error.h>
#include <apt-pkg/init.h>
#include <apt-pkg/pkgcachegen.h>
#include <apt-pkg/sourcelist.h>
#include <apt-pkg/version.h>

#include <fstream>

#include <assert.h>
#include <signal.h>
#include <unistd.h>
#include <sys/stat.h>
#include <errno.h>

using namespace std;

aptitudeCacheFile *apt_cache_file=NULL;
signalling_config *aptcfg=NULL;
pkgRecords *apt_package_records=NULL;
#if APT_PKG_MAJOR >= 3 && APT_PKG_MINOR >= 5
pkgSourceList *apt_source_list=NULL;
#endif
undo_list *apt_undos=NULL;
pkg_hier *user_pkg_hier=NULL;

string *pendingerr=NULL;
bool erroriswarning=false;

static Configuration *theme_config;
static Configuration *user_config;

SigC::Signal0<void> cache_closed, cache_reloaded, cache_reload_failed;
SigC::Signal0<void> hier_reloaded;

static void reload_user_pkg_hier()
{
  delete user_pkg_hier;
  user_pkg_hier=new pkg_hier;

  user_pkg_hier->input_file(PKGDATADIR "/function_groups");

  const char *cfgloc=getenv("HOME");
  if(cfgloc)
    {
      string user_hier=cfgloc+string("/.aptitude/function_pkgs");
      if(access(user_hier.c_str(), R_OK)==0)
	user_pkg_hier->input_file(user_hier);
      else
	user_pkg_hier->input_file(PKGDATADIR "/function_pkgs");
    }
}

void apt_preinit()
{
  signal(SIGPIPE, SIG_IGN);

  theme_config=new Configuration;
  user_config=new Configuration;

  ReadConfigFile(*theme_config, PKGDATADIR "/aptitude-defaults");

  pkgInitConfig(*_config);
  pkgInitSystem(*_config, _system);

  // Allow a user-specific customization file.
  const char *HOME=getenv("HOME");

  if(HOME)
    {
      string cfgloc=HOME;
      cfgloc+="/.aptitude/config";

      // This attempts to test whether the file is available
      if(access(cfgloc.c_str(), R_OK)==0)
	{
	  ReadConfigFile(*user_config, cfgloc);
	  ReadConfigFile(*_config, cfgloc);
	}
    }

  aptcfg=new signalling_config(user_config, _config, theme_config);

  apt_dumpcfg("Aptitude");

  apt_undos=new undo_list;
}

void apt_dumpcfg(const char *root)
{
  const char *HOME=getenv("HOME");

  if(!HOME)
    return;

  string cfgloc=HOME;
  cfgloc+="/.aptitude";

  if(mkdir(cfgloc.c_str(), 0700)<0 && errno!=EEXIST)
    {
      _error->Errno("mkdir", "%s", cfgloc.c_str());
      return;
    }

  cfgloc+="/config";

  ofstream f((cfgloc+".new").c_str());

  if(!f)
    {
      _error->Errno(_("Unable to open %s for writing"), cfgloc.c_str());
      return;
    }

  aptcfg->Dump(f);

  f.close();

  if(rename((cfgloc+".new").c_str(), cfgloc.c_str())!=0)
    {
      _error->Errno(_("Unable to replace %s with new configuration file"), cfgloc.c_str());
      return;
    }
}

// Revert back to the default set of options.  Is it a hack?  Hmmm...
void apt_revertoptions()
{
  Configuration *old_user_config=user_config;
  Configuration *old_config=_config;

  _config=new Configuration;

  user_config=new Configuration;

  // ick?
  pkgInitConfig(*_config);
  pkgInitSystem(*_config, _system);

  aptcfg->setcfg(user_config, _config, theme_config);

  delete old_user_config;
  delete old_config;
}

void apt_init(OpProgress *progress_bar, bool do_initselections,
	      const char *status_fname)
{
  if(!apt_cache_file)
    apt_reload_cache(progress_bar, do_initselections, status_fname);
}

void apt_reload_cache(OpProgress *progress_bar, bool do_initselections,
		      const char * status_fname)
{
  cache_closed();

  reset_tasks();

  if(apt_package_records)
    {
      delete apt_package_records;
      apt_package_records=NULL;
    }

  if(apt_cache_file)
    {
      delete apt_cache_file;
      apt_cache_file=NULL;
    }

#if APT_PKG_MAJOR >= 3 && APT_PKG_MINOR >= 5
  if(apt_source_list)
    {
      delete apt_source_list;
      apt_source_list=NULL;
    }
#endif

  aptitudeCacheFile *new_file=new aptitudeCacheFile;

#if APT_PKG_MAJOR >= 3 && APT_PKG_MINOR >= 5
  apt_source_list=new pkgSourceList;
  apt_source_list->ReadMainList();
#endif

  bool open_failed=!new_file->Open(*progress_bar, do_initselections,
				   (getuid()==0), status_fname)
    || _error->PendingError();

  if(open_failed && getuid()==0)
    {
      _error->Discard();

      open_failed=!new_file->Open(*progress_bar, do_initselections,
				  false, status_fname) ||
	_error->PendingError();

      if(!open_failed)
	_error->Warning(_("Warning: could not lock the cache file.  Opening in read-only mode"));
    }

  if(open_failed)
    {
      delete new_file;
      cache_reload_failed();
      return;
    }

  apt_cache_file=new_file;
  apt_package_records=new pkgRecords(*apt_cache_file);

  // Um, good time to clear our undo info.
  apt_undos->clear_items();

  load_tasks(*progress_bar);

  if(user_pkg_hier)
    {
      reload_user_pkg_hier();
      hier_reloaded();
    }

  cache_reloaded();
}

pkg_hier *get_user_pkg_hier()
{
  if(!user_pkg_hier)
    reload_user_pkg_hier();

  return user_pkg_hier;
}

pkg_action_state find_pkg_state(pkgCache::PkgIterator pkg)
{
  aptitudeDepCache::StateCache &state=(*apt_cache_file)[pkg];
  aptitudeDepCache::aptitude_state &extstate=(*apt_cache_file)->get_ext_state(pkg);

  if(state.InstBroken())
    return pkg_broken;
  else if(state.Delete())
    {
      if(extstate.remove_reason==aptitudeDepCache::manual)
	return pkg_remove;
      else if(extstate.remove_reason==aptitudeDepCache::unused)
	return pkg_unused_remove;
      else
	return pkg_auto_remove;
    }
  else if(state.Install())
    {
      if(!pkg.CurrentVer().end())
	{
	  if(state.iFlags&pkgDepCache::ReInstall)
	    return pkg_reinstall;
	  else if(state.Downgrade())
	    return pkg_downgrade;
	  else if(state.Upgrade())
	    return pkg_upgrade;
	  else
	    // FOO!  Should I abort here?
	    return pkg_install;
	}
      else if(extstate.install_reason!=aptitudeDepCache::manual)
	return pkg_auto_install;
      else
	return pkg_install;
    }

  else if(state.Status==1 &&
	  state.Keep())
    {
      if(!(state.Flags & pkgDepCache::AutoKept))
	return pkg_hold;
      else
	return pkg_auto_hold;
    }

  else if(state.iFlags&pkgDepCache::ReInstall)
    return pkg_reinstall;

  return pkg_unchanged;
}

bool pkg_obsolete(pkgCache::PkgIterator pkg)
{
  if(pkg.CurrentVer().end())
    return false;
  else
    {
      pkgCache::VerIterator ver=pkg.VersionList();
      ver++;

      if(!ver.end())
	return false;
      else // Ok, there's only one version.  Good.
	{
	  pkgCache::VerFileIterator files=pkg.VersionList().FileList();
	  if(!files.end())
	    {
	      files++;
	      if(!files.end())
		return false; // Nope, more than one file
	    }
	}
    }

  return true;
}

// This does not assume that the dependency is the first elements of
// its OR group.
void surrounding_or(pkgCache::DepIterator dep,
		    pkgCache::DepIterator &start,
		    pkgCache::DepIterator &end)
{
  bool found=false;

  start=dep.ParentVer().DependsList();
  end=start;

  while(!end.end() && !found)
    {
      start=end;

      while(end->CompareOp&pkgCache::Dep::Or)
	{
	  if(end==dep)
	    found=true;

	  ++end;
	}

      if(end==dep)
	found=true;

      ++end;
    }

  // If not, something is wrong with apt's cache.
  assert(found);
}

bool package_suggested(const pkgCache::PkgIterator &pkg)
{
  pkgDepCache::StateCache &state=(*apt_cache_file)[pkg];
  pkgCache::VerIterator candver=state.CandidateVerIter(*apt_cache_file);

  for(rev_dep_iterator d(pkg); !d.end(); ++d)
    if((*d)->Type==pkgCache::Dep::Suggests)
      {
	bool satisfied=false;

	pkgCache::DepIterator start,end;

	surrounding_or(*d, start, end);

	while(start!=end)
	  {
	    if(((*apt_cache_file)[start])&pkgDepCache::DepGInstall)
	      {
		satisfied=true;
		break;
	      }

	    ++start;
	  }

	if(!satisfied)
	  {
	    // Check whether the package doing the depending is going
	    // to be installed.
	    pkgCache::PkgIterator depender=(*d).ParentPkg();
	    pkgDepCache::StateCache &depstate=(*apt_cache_file)[depender];
	    pkgCache::VerIterator depinstver=depstate.InstVerIter(*apt_cache_file);

	    if(depender.CurrentVer().end() &&
	       depstate.Install() &&
	       !depinstver.end() &&
	       !candver.end() &&
	       _system->VS->CheckDep(candver.VerStr(),
				     (*d)->CompareOp, (*d).TargetVer()))
	      {
		if((*d).ParentVer()==depinstver)
		  return true;
	      }
	  }
      }

  return false;
}

bool package_recommended(const pkgCache::PkgIterator &pkg)
{
  pkgDepCache::StateCache &state=(*apt_cache_file)[pkg];
  pkgCache::VerIterator candver=state.CandidateVerIter(*apt_cache_file);

  for(rev_dep_iterator d(pkg); !d.end(); ++d)
    if((*d)->Type==pkgCache::Dep::Recommends)
      {
	bool satisfied=false;

	pkgCache::DepIterator start,end;

	surrounding_or(*d, start, end);

	while(start!=end)
	  {
	    if(((*apt_cache_file)[start])&pkgDepCache::DepGInstall)
	      {
		satisfied=true;
		break;
	      }

	    ++start;
	  }

	if(!satisfied)
	  {
	    // Check whether the package doing the depending is going
	    // to be installed or upgraded.
	    pkgCache::PkgIterator depender=(*d).ParentPkg();
	    pkgDepCache::StateCache &depstate=(*apt_cache_file)[depender];
	    pkgCache::VerIterator depinstver=depstate.InstVerIter(*apt_cache_file);

	    if(depstate.Install() &&
	       !candver.end() &&
	       _system->VS->CheckDep(candver.VerStr(),
				     (*d)->CompareOp, (*d).TargetVer()))
	      {
		if((*d).ParentVer()==depinstver)
		  return true;
	      }
	  }
      }

  return false;
}

#if APT_PKG_MAJOR >= 3 && APT_PKG_MINOR >= 5
bool package_trusted(const pkgCache::VerIterator &ver)
{
  for(pkgCache::VerFileIterator i = ver.FileList(); !i.end(); ++i)
    {
      pkgIndexFile *index;

      if(apt_source_list->FindIndex(i.File(), index) && !index->IsTrusted())
	return false;
    }

  return true;
}
#endif
