# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::tree::annotate;
use base 'AXP::Command::tree';

my $TERM_COLS = 80;

use Arch::Tree;

sub infoline {
	"show descriptions of ancestry revisions from logs"
}

sub optusage {
	"[options] file"
}

sub options {
	(
		$_[0]->tree_options,
		reverse  => { sh => 'r', desc => "sort from oldest to newest" },
		summary  => { sh => 's', desc => "show summary of each revision" },
		creator  => { sh => 'c', desc => "show creator of each revision" },
		date     => { sh => 'D', desc => "show date of each revision" },
		no_desc  => { desc => "do not assume the default -Dcs and auto -n" },
		no_full  => { sh => 'F', desc => "show short revision names" },
		filename => { sh => 'n', desc => "show historical file names (default: auto)" },
		no_email => { sh => 'E', desc => "don't show email of creator" },
		one_version => { sh => 'o', desc => "don't follow tags from other versions" },
		no_progress => { sh => 'P', desc => "don't show progress even if stderr is tty" },
		no_lines => { sh => 'L', desc => "don't show file lines, just revisions" },
		linenums => { sh => 'l', type => "=s", desc => "limit to line number(s), like: 12-24,50,75-" },
		match    => { sh => 'm', type => "=s", desc => "limit to lines matching RE, like '^sub '", arg => 'RE' },
		highlight => { sh => 'H', desc => "syntax-highlight the lines using markup" },
		group    => { sh => 'g', desc => "group lines (annotate once per group)" },
		delim    => { desc => 'specify delimiter (default: " | ")', type => "=s" },
		format   => { sh => 'f', desc => 'specify format (default: "%num%delim%line")', type => "=s" },
	)
}   
 
sub helptext {
	q{
		Produce annotated output of the given tree file, similarly to
		"cvs annotate", with a thorough configurability.

		All file lines are printed to stdout, prefixed by index of
		revision that last modified the line, and the indexed
		revision descriptions follow the file content.

		The options and the format of revision descriptions are
		similar to 'axp history' command, but the default is --desc.

		The format of the annotated lines is configurable using options
		--no-email, --highlight, --group, --delim D and --format F.
		The available format variables:
		    %num      - revision number (1 is normally the latest)
		    %delim    - delimiter
		    %line     - plain or highlight'd line
		    %version, %name, %summary, %date, %filename, %creator,
		    %email, %realname, %realname1, %realname2, %username
		The available width specifiers for any format variables:
		    %10var  %-10var  - left (right) pad with up to 10 spaces
		    %*20var %*-20var - cut at 20 first (last) characters
		    %-10*-20var - use 20 last characters, padded to at least 10
	}
}

sub format_line ($$$$$;$) {
	my ($format, $index, $revision_descs, $delim, $line, $no_num) = @_;

	my $num = $index;
	my $revision_desc;
	unless (defined $index) {
		$num = $no_num;
	} elsif ($index == -1) {
		$num = "";
	} else {
		$revision_desc = $revision_descs->[$index];
		$num = $revision_desc->{num};
	}
	my %values = (num => $num, delim => $delim, line => $line);

	$format =~ s/%(-?\d+)?(\*(-?)(\d+))?(?:{(\w+)}|(\w+))/
		my $name = $5 || $6;
		my $value = exists $values{$name}? $values{$name}:
			$revision_desc? defined $revision_desc->{$name}?
			$revision_desc->{$name}: "*no-$name*": "";
		$value = !$2 || $4 <= 3 || $4 > length($value)? $value: $3?
			"..." . substr($value, -$4 + 3, $4 - 3):
			substr($value, 0, $4 - 3) . "...";
		$1? sprintf("%$1s", $value): $value;
	/ge;
	return $format;
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	$opt{date} = $opt{creator} = $opt{summary} = 1
		unless $opt{no_desc} || $opt{date} || $opt{creator} || $opt{summary};

	my $tree = $self->tree;
	my $filepath = shift(@ARGV);
	warn "Post file-name parameters (@ARGV) are ignored\n" if @ARGV;

	my %args = (match_re => $opt{match});
	foreach (qw(one_version linenums highlight group)) {
		$args{$_} = $opt{$_} if $opt{$_};
	}

	my $nr = 0;  # print revision to fetch and its number
	if (!$opt{no_progress} && -t STDERR) {
		$args{prefetch_callback} = sub ($$) {
			print STDERR "\010 \010" x $TERM_COLS if $nr;
			my $revision = substr($_[0], 0, $TERM_COLS - 11);
			printf STDERR "%4d: %s ... ", ++$nr, $revision;
		};
	}

	my ($lines, $line_rd_indexes, $revision_descs) =
		$tree->get_annotate_revision_descs($filepath, %args);

	print STDERR "\010 \010" x $TERM_COLS if $nr;
	@$revision_descs = reverse @$revision_descs if $opt{reverse};
	goto REVISION_OUTPUT if $opt{no_lines};

	my $num = 0;
	my $digits = length(0 + @$revision_descs);
	my $no_num = sprintf "%${digits}s", '?';

	foreach my $revision_desc (@$revision_descs) {
		$opt{filename} ||= $_->{is_filepath_renamed}
			unless $opt{no_desc} || $opt{filename};
		$revision_desc->{realname} = $revision_desc->{creator};
		$revision_desc->{realname} =~ /^(.*?)(?: (.*))?$/;
		$revision_desc->{realname1} = $1 || "_none_";
		$revision_desc->{realname2} = $2 || "_none_";
		$revision_desc->{creator} .= " <" . $revision_desc->{email} . ">"
			unless $opt{no_email};
		$revision_desc->{num} = sprintf "%${digits}d", ++$num;
	}

	my $format = $opt{format} || "%num%delim%line";
	my $delim = $opt{delim} || " | ";
	for (my $i = 0; $i < @$lines; $i++) {
		my $index = $line_rd_indexes->[$i];
		my ($line, @rest_lines) = $lines->[$i];
		($line, @rest_lines) = @$line if $opt{group};
		$index = @$revision_descs - $index - 1 if $opt{reverse} && defined $index;
		print format_line($format, $index, $revision_descs, $delim, $line, $no_num), "\n";
		print format_line($format, -1, $revision_descs, $delim, $_), "\n"
			foreach @rest_lines;
	}

	print "-" x $TERM_COLS, "\n";

	REVISION_OUTPUT:
	$nr = 0;

	foreach my $revision_desc (@$revision_descs) {
		my $creator_line = "";
		$creator_line .= $revision_desc->{date} if $opt{date};
		if ($opt{creator}) {
			$creator_line .= "      " if $opt{date};
			$creator_line .= $revision_desc->{creator};
		}

		printf "%${digits}s: ", ++$nr unless $opt{no_lines};
		print "$revision_desc->{version}--" unless $opt{no_full};
		print $revision_desc->{name}, "\n";
		print "    ", $revision_desc->{filepath}, "\n" if $opt{filename};
		print "    $creator_line\n" if $creator_line;
		print "    ", $revision_desc->{summary}, "\n" if $opt{summary};
	}
}

1;
