/*
** Copyright 2003-2004 Double Precision, Inc.  See COPYING for
** distribution information.
*/

/*
** $Id: encode.c,v 1.3 2004/05/25 02:44:30 mrsam Exp $
*/
#include	"encode.h"
#include	<string.h>
#include	<stdlib.h>
#if	HAVE_LIBUNICODE
#include	"../unicode/unicode.h"
#endif

static int quoted_printable(struct libmail_encode_info *,
			    const char *, size_t);
static int base64(struct libmail_encode_info *,
		  const char *, size_t);
static int eflush(struct libmail_encode_info *,
		 const char *, size_t);

static const char *libmail_encode_autodetect(const char *charset, 
					     int (*func)(void *), void *arg)
{
	const char *encoding="7bit";
	int	l=0;
	int	longline=0;
	int c;
#if	HAVE_LIBUNICODE
	const struct unicode_info *ci = unicode_find(charset);
#endif

	while ((c = (*func)(arg)) != EOF)
	{
		unsigned char ch= (unsigned char)c;

		if (ch >= 0x80)
		{

#if	HAVE_LIBUNICODE
			if (!charset || !*charset)
				encoding="8bit";
			else if (ci && ci->flags & UNICODE_BODY_QUOPRI)
				encoding="quoted-printable";
			else if (!ci || ci->flags & UNICODE_BODY_BASE64)
				encoding="base64";
			else
				encoding="8bit";
#else
			encoding="8bit";
#endif
		}

		if (ch < 0x20 &&
		    ch != '\t' && ch != '\r' && ch != '\n')
		{
#if	HAVE_LIBUNICODE
			if (!charset || !*charset)
				;
			else if (ci && ci->flags & UNICODE_BODY_QUOPRI)
				encoding="quoted-printable";
			else if (!ci || ci->flags & UNICODE_BODY_BASE64)
				encoding="base64";
#else
			if (charset && *charset)
				encoding="quoted-printable";
#endif
                }

		if (ch == 0)
			return "base64";

		if (ch == '\n')	l=0;
		else if (++l > 990)
		{
			longline=1;
#if	HAVE_LIBUNICODE
			if (ci && ci->flags & UNICODE_BODY_QUOPRI)
				encoding="quoted-printable";
#else
			if (charset && *charset)
				encoding="quoted-printable";
#endif
		}

	}

	if (longline)
	{
#if	HAVE_LIBUNICODE
		if (ci && ci->flags & UNICODE_BODY_QUOPRI)
			encoding="quoted-printable";
		else
			encoding="base64";
#else
		if (charset && *charset)
			encoding="quoted-printable";
		else
			encoding="base64";
#endif
	}
	return encoding;
}

struct file_info {
	FILE *fp;
	off_t pos;
	off_t end;
};

static int read_file(void *arg)
{
int c;
struct file_info *fi = (struct file_info *)arg;
	if (fi->end >= 0 && fi->pos > fi->end)
		return EOF;
	c = getc(fi->fp);
	fi->pos++;
	return c;
}

static int read_string(void * arg)
{
int c;
unsigned char **strp = (unsigned char **)arg;
	if (**strp == 0)
		return EOF;
	c = (int)**strp;
	(*strp)++;
	return c;
}

const char *libmail_encode_autodetect_fp(FILE *fp, int okQp)
{
	if (okQp)
		return libmail_encode_autodetect_fppos(fp, "ISO-8859-1", 0, -1);
	else
		return libmail_encode_autodetect_fppos(fp, NULL, 0, -1);
}

const char *libmail_encode_autodetect_fppos(FILE *fp, const char *charset,
					    off_t start_pos, off_t end_pos)
{
struct file_info fi;
off_t orig_pos = ftell(fp);
off_t pos = orig_pos;
const char *rc;

	if (start_pos >= 0)
	{
		if (fseek(fp, start_pos, SEEK_SET) == (off_t)-1)
			return NULL;
		else
			pos = start_pos;
	}

	fi.fp = fp;
	fi.pos = pos;
	fi.end = end_pos;
	rc = libmail_encode_autodetect(charset, &read_file, &fi);
  
	if (fseek(fp, orig_pos, SEEK_SET) == (off_t)-1)
		return NULL;
	return rc;
}

const char *libmail_encode_autodetect_str(const char *str, const char *charset)
{
	return libmail_encode_autodetect(charset, &read_string, &str);
}


void libmail_encode_start(struct libmail_encode_info *info,
			  const char *transfer_encoding,
			  int (*callback_func)(const char *, size_t, void *),
			  void *callback_arg)
{
	info->output_buf_cnt=0;
	info->input_buf_cnt=0;

	switch (*transfer_encoding) {
	case 'q':
	case 'Q':
		info->encoding_func=quoted_printable;
		break;
	case 'b':
	case 'B':
		info->encoding_func=base64;
		break;
	default:
		info->encoding_func=eflush;
		break;
	}
	info->callback_func=callback_func;
	info->callback_arg=callback_arg;
}

int libmail_encode(struct libmail_encode_info *info,
		   const char *ptr,
		   size_t cnt)
{
	return ((*info->encoding_func)(info, ptr, cnt));
}

int libmail_encode_end(struct libmail_encode_info *info)
{
	int rc=(*info->encoding_func)(info, NULL, 0);

	if (rc == 0 && info->output_buf_cnt > 0)
	{
		rc= (*info->callback_func)(info->output_buffer,
					   info->output_buf_cnt,
					   info->callback_arg);
		info->output_buf_cnt=0;
	}

	return rc;
}

static int eflush(struct libmail_encode_info *info, const char *ptr, size_t n)
{
	while (n > 0)
	{
		size_t i;

		if (info->output_buf_cnt == sizeof(info->output_buffer))
		{
			int rc= (*info->callback_func)(info->output_buffer,
						       info->output_buf_cnt,
						       info->callback_arg);

			info->output_buf_cnt=0;
			if (rc)
				return rc;
		}

		i=n;

		if (i > sizeof(info->output_buffer) - info->output_buf_cnt)
			i=sizeof(info->output_buffer) - info->output_buf_cnt;

		memcpy(info->output_buffer + info->output_buf_cnt, ptr, i);
		info->output_buf_cnt += i;
		ptr += i;
		n -= i;
	}
	return 0;
}

static int base64_flush(struct libmail_encode_info *);

static int base64(struct libmail_encode_info *info,
		  const char *buf, size_t n)
{
	if (!buf)
	{
		int rc=0;

		if (info->input_buf_cnt > 0)
			rc=base64_flush(info);

		return rc;
	}

	while (n)
	{
		size_t	i;

		if (info->input_buf_cnt == sizeof(info->input_buffer))
		{
			int rc=base64_flush(info);

			if (rc != 0)
				return rc;
		}

		i=n;
		if (i > sizeof(info->input_buffer) - info->input_buf_cnt)
			i=sizeof(info->input_buffer) - info->input_buf_cnt;

		memcpy(info->input_buffer + info->input_buf_cnt,
		       buf, i);
		info->input_buf_cnt += i;
		buf += i;
		n -= i;
	}
	return 0;
}

static const char base64tab[]=
"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

static int base64_flush(struct libmail_encode_info *info)
{
	int	a=0,b=0,c=0;
	int	i, j;
	int	d, e, f, g;
	char	output_buf[ sizeof(info->input_buffer) / 3 * 4+1];

	for (j=i=0; i<info->input_buf_cnt; i += 3)
	{
		a=(unsigned char)info->input_buffer[i];
		b= i+1 < info->input_buf_cnt ?
			(unsigned char)info->input_buffer[i+1]:0;
		c= i+2 < info->input_buf_cnt ?
			(unsigned char)info->input_buffer[i+2]:0;

		d=base64tab[ a >> 2 ];
		e=base64tab[ ((a & 3 ) << 4) | (b >> 4)];
		f=base64tab[ ((b & 15) << 2) | (c >> 6)];
		g=base64tab[ c & 63 ];
		if (i + 1 >= info->input_buf_cnt)	f='=';
		if (i + 2 >= info->input_buf_cnt) g='=';
		output_buf[j++]=d;
		output_buf[j++]=e;
		output_buf[j++]=f;
		output_buf[j++]=g;
	}

	info->input_buf_cnt=0;

	output_buf[j++]='\n';
	return eflush(info, output_buf, j);
}

static const char xdigit[]="0123456789ABCDEF";

static int quoted_printable(struct libmail_encode_info *info,
			    const char *p, size_t n)
{
	char local_buf[256];
	int local_buf_cnt=0;

#define QPUT(c) do { if (local_buf_cnt == sizeof(local_buf)) \
                     { int rc=eflush(info, local_buf, local_buf_cnt); \
			local_buf_cnt=0; if (rc) return (rc); } \
			local_buf[local_buf_cnt]=(c); ++local_buf_cnt; } while(0)

	if (!p)
		return (0);

	while (n)
	{
		if (info->input_buf_cnt > 72 && *p != '\n')
		{
			QPUT('=');
			QPUT('\n');
			info->input_buf_cnt=0;
		}

		if ( *p == '\n')
			info->input_buf_cnt=0;
		else if (*p < ' ' || *p == '=' || *p >= 0x7F)
		{
			QPUT('=');
			QPUT(xdigit[ (*p >> 4) & 15]);
			QPUT(xdigit[ *p & 15 ]);
			info->input_buf_cnt += 3;
			p++;
			--n;
			continue;
		}
		else info->input_buf_cnt++;

		QPUT( *p);
		p++;
		--n;
	}

	if (local_buf_cnt > 0)
		return eflush(info, local_buf, local_buf_cnt);

	return 0;
}
