/**************************************************************************/
/*                                                                        */
/*  cpbk - a mirroring utility for backing up your files                  */
/*  Copyright (C) 1998 Kevin Lindsay <klindsay@mkintraweb.com>            */
/*  Copyright (C) 2001 Yuuki NINOMIYA <gm@debian.or.jp>                   */
/*                                                                        */
/*  This program is free software; you can redistribute it and/or modify  */
/*  it under the terms of the GNU General Public License as published by  */
/*  the Free Software Foundation; either version 2, or (at your option)   */
/*  any later version.                                                    */
/*                                                                        */
/*  This program is distributed in the hope that it will be useful,       */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         */
/*  GNU General Public License for more details.                          */
/*                                                                        */
/*  You should have received a copy of the GNU General Public License     */
/*  along with this program; if not, write to the                         */
/*  Free Software Foundation, Inc., 59 Temple Place - Suite 330,          */
/*  Boston, MA 02111-1307, USA.                                           */
/*                                                                        */
/**************************************************************************/

/* $Id: progress.c,v 1.4 2001/03/30 13:16:36 gm Exp $ */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include "intl.h"
#include "strlib.h"
#include "variable.h"
#include "proto.h"


static int whole_num_files;
static int current_num_files;
static off_t whole_num_size;
static off_t current_num_size;
static int win_col;


/* --- PUBLIC FUNCTIONS --- */

void update_progress_bar(void)
{
	static int pos = 0;

	if (config.suppress_progress || config.quiet || config.verbose) {
		return;
	}

	printf("\b%c", "-\\|/"[pos]);
	fflush(stdout);
	pos = (pos + 1) % 4;
}


void init_progress_info(Order *order)
{
	int i;
	struct winsize win;

	if (config.suppress_progress || config.quiet || config.verbose) {
		return;
	}

	whole_num_files = 0;
	current_num_files = 0;
	whole_num_size = 0;
	current_num_size = 0;

	ioctl(STDOUT_FILENO, TIOCGWINSZ, &win);
	win_col = win.ws_col;

	for (i = 0; order[i].action != END; i++) {
		if (order[i].action == NEW || order[i].action == UPDATE) {
			whole_num_files++;
			whole_num_size += (order[i].status.st_size / 1000);
		}
	}
}


void output_progress_info(Action act, const char *path_file, const char *dest_for_verbose, Order *order)
{
	char *process = NULL;
	char *message;
	int i;
	long rate;
	char bar[21];
	int num_digit;
	int size_digit;
	char digit_temp[30];

	if (config.verbose) {
		switch (act) {
		case NEW:
		case UPDATE:
			printf(_("Copy   : `%s' -> `%s'\n"), path_file, dest_for_verbose);
			break;
		case REMOVE:
		case RMDIR:
			printf(_("Remove : `%s'\n"), path_file);
			break;
		case MKDIR:
			printf(_("Make   : `%s'\n"), path_file);
			break;
		case CHMOD:
			printf(_("Change : `%s'\n"), path_file);
			break;
		default:
			internal_error(__FILE__, __LINE__);
		}
		return;
	}

	if (config.suppress_progress) {
		return;
	}

	switch (act) {
	case NEW:
	case UPDATE:
		process = _("Copying  : ");
		break;
	case REMOVE:
	case RMDIR:
		process = _("Removing : ");
		break;
	case MKDIR:
		process = _("Making   : ");
		break;
	case CHMOD:
		process = _("Changing : ");
		break;
	default:
		internal_error(__FILE__, __LINE__);
	}

	if (act == NEW || act == UPDATE) {
		current_num_files++;
		current_num_size += (order->status.st_size / 1000);
	}

	if (whole_num_size > 0) {
		rate = current_num_size * 100 / whole_num_size;
	} else {
		rate = 100;
	}

	for (i = 0; i < rate / 5; i++) {
		bar[i] = '*';
	}
	for (i = 0; i < 20 - (rate / 5); i++) {
		bar[19 - i] = '-';
	}
	bar[20] = '\0';

	sprintf(digit_temp, "%d", whole_num_files);
	num_digit = strlen(digit_temp);
	sprintf(digit_temp, "%ld", whole_num_size);
	size_digit = strlen(digit_temp);

	message = str_dup_printf(_(" %*d/%d files (%*ldKB/%ldKB) %3d%% [%s] %s%s"), num_digit, current_num_files, whole_num_files, size_digit, current_num_size, whole_num_size, rate, bar, process, path_file);
	
	if (win_col >= strlen(message)) {
		message = str_realloc(message, win_col + 1);
		for (i = strlen(message); i < win_col; i++) {
			message[i] = ' ';
		}
	}
	message[win_col] = '\0';

	printf("%s\r", message);
	fflush(stdout);

	free(message);
}
