/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: display.cc,v 1.35.2.7 2003/10/05 13:33:38 dheck Exp $
 */

/*
 * This file contains the code renders the graphics during the game
 * and in the editor.  This includes displaying the current landscape
 * with all its objects and the inventory at the bottom of the screen.
 */

#include "display.hh"
#include "objects.hh"
#include "video.hh"
#include "options.hh"

#include "px/sdl.hh"
#include "px/px.hh"
#include "px/tools.hh"

#include <vector>
#include <algorithm>
#include <functional>
#include <list>
#include <cmath>
#include <cassert>

using namespace std;
using namespace px;
using namespace display;
using namespace enigma;

#include "d_engine.hh"
#include "d_models.hh"
#include "d_statusbar.hh"

//----------------------------------------
// Local variables
//----------------------------------------

static DisplayFlags    display_flags = SHOW_ALL;

static GameDisplay   *gamedpy = 0;


//======================================================================
// DISPLAY ENGINE
//======================================================================

DisplayEngine::DisplayEngine (px::Screen *scr, int tilew, int tileh)
: m_tilew(tilew), m_tileh(tileh),
  m_screen (scr),
  m_offset(),
  m_area (scr->size()),
  m_width(0), m_height(0),
  m_redrawp(0,0)
{
}

DisplayEngine::~DisplayEngine()
{
    delete_sequence (m_layers.begin(), m_layers.end());
}

void
DisplayEngine::add_layer (DisplayLayer *l)
{
    l->set_engine (this);
    m_layers.push_back(l);
}

void
DisplayEngine::set_offset (const V2 &off)
{
    m_offset = off;
}

void
DisplayEngine::move_offset (const px::V2 &off)
{
    int oldx = int(m_offset[0] * m_tilew);
    int oldy = int(m_offset[1] * m_tileh);
    int newx = int(off[0] * m_tilew);
    int newy = int(off[1] * m_tileh);

    if (newx != oldx || newy != oldy)
    {
        Rect newarea(newx, newy, m_area.w, m_area.h);
        Rect oldarea(oldx, oldy, m_area.w, m_area.h);
        Rect common = intersect(newarea, oldarea);

        GC screengc (m_screen->get_surface());

        Rect blitrect(common.x-oldx+m_area.x,
                      common.y-oldy+m_area.y,
                      common.w, common.h);
        blit (screengc, 
              common.x-newx+m_area.x,
              common.y-newy+m_area.y,
              m_screen->get_surface(),
              blitrect);

        blitrect.x = common.x-newx+m_area.x;
        blitrect.y = common.y-newy+m_area.y;
        m_screen->update_rect(blitrect);

        m_offset = V2(newx/double(m_tilew), newy/double(m_tileh));

        RectList rl;
        rl.push_back (m_area);
        rl.sub (blitrect);

        for (RectList::iterator i=rl.begin(); i!=rl.end(); ++i)
        {
            Rect r = screen_to_world(*i);
            mark_redraw_area (r);
        }
    }
}

void
DisplayEngine::set_screen_area (const px::Rect & r)
{
    m_area = r;
}

void
DisplayEngine::new_world (int w, int h)
{
    m_width = w;
    m_height = h;
    m_offset = V2();
    m_redrawp.resize(w, h, true);

    for (unsigned i=0; i<m_layers.size(); ++i)
        m_layers[i]->new_world(w,h);
}


void
DisplayEngine::tick (double dtime)
{
    for_each (m_layers.begin(), m_layers.end(),
              bind2nd (mem_fun(&DisplayLayer::tick), dtime));
}

void
DisplayEngine::world_to_screen (const V2 & pos, int *x, int *y)
{
    *x = int((pos[0] - m_offset[0])*m_tilew) + get_area().x;
    *y = int((pos[1] - m_offset[1])*m_tileh) + get_area().y;
}

ScreenArea
DisplayEngine::world_to_screen (const WorldArea &a)
{
    int x, y;
    world_to_screen (V2(a.x, a.y), &x, &y);
    return ScreenArea (x, y, a.w*m_tilew, a.h*m_tileh);
}


WorldArea
DisplayEngine::screen_to_world (const ScreenArea &a)
{
    int sx = int(m_offset[0] * m_tilew) + a.x - get_area().x;
    int sy = int(m_offset[1] * m_tileh) + a.y - get_area().y;

    int x1 = Max(0, sx / m_tilew);
    int y1 = Max(0, sy / m_tileh);
    int x2 = Min(m_width, (sx + a.w+m_tilew-1) / m_tilew);
    int y2 = Min(m_height, (sy + +a.h+m_tileh-1) / m_tileh);

    return WorldArea (x1, y1, x2-x1, y2-y1);
}

V2
DisplayEngine::to_screen (const V2 &pos)
{
    return V2 ((pos[0] - m_offset[0])*m_tilew + get_area().x,
               (pos[1] - m_offset[1])*m_tileh + get_area().y);
}

V2
DisplayEngine::to_world (const V2 &pos)
{
    return V2((pos[0]-get_area().x)/m_tilew, (pos[1]-get_area().y)/m_tileh) + m_offset;
}

void
DisplayEngine::mark_redraw_area (const WorldArea &wa)
{
    int x2 = Min(m_width, wa.x+wa.w);
    int y2 = Min(m_height, wa.y+wa.h);
    for (int x=Max(0,wa.x); x <x2; x++)
        for (int y=Max(0,wa.y); y<y2; y++)
            m_redrawp(x,y) = true;
}

void
DisplayEngine::mark_redraw_screen()
{
    mark_redraw_area(screen_to_world(m_area));
}

void
DisplayEngine::draw_all (px::GC &gc)
{
    WorldArea wa = screen_to_world (get_area());
    int xpos, ypos;
    world_to_screen (V2(wa.x, wa.y), &xpos, &ypos);
    for (unsigned i=0; i<m_layers.size(); ++i) {
        clip(gc, get_area());
        m_layers[i]->draw (gc, wa, xpos, ypos);
        m_layers[i]->draw_onepass(gc);
    }
}

void
DisplayEngine::update_layer (DisplayLayer *l, WorldArea wa)
{
    GC gc(m_screen->get_surface());

    int x2 = wa.x+wa.w;
    int y2 = wa.y+wa.h;

//     ScreenArea sa = world_to_screen (WorldArea (x, y, 1, 1));
//     sa.intersect(get_area());
    clip(gc, get_area()); //sa);
    for (int x=wa.x; x<x2; x++)
        for (int y=wa.y; y<y2; y++)
            if (m_redrawp(x,y)) {
                int xpos, ypos;
                world_to_screen (V2(x, y), &xpos, &ypos);
                l->draw (gc, WorldArea(x,y,1,1), xpos, ypos);
            }

    l->draw_onepass (gc);

}

void
DisplayEngine::update_screen()
{
    GC gc(m_screen->get_surface());

    Rect area=get_area();
    clip(gc, area);
    WorldArea wa = screen_to_world (area);

    // Fill screen area not covered by world
    {
        RectList rl;
        rl.push_back (get_area());
        rl.sub (world_to_screen (WorldArea (0,0, m_width, m_height)));
        set_color (gc, 200, 0, 200);
        for (RectList::iterator i=rl.begin(); i!=rl.end(); ++i)
        {
            box (gc, *i);
            m_screen->update_rect (*i);
        }
    }

    for (unsigned i=0; i<m_layers.size(); ++i) {
        update_layer (m_layers[i], wa);
    }
    int x2 = wa.x+wa.w;
    int y2 = wa.y+wa.h;
    for (int x=wa.x; x<x2; x++)
        for (int y=wa.y; y<y2; y++)
            if (m_redrawp(x,y)) {
                m_redrawp(x,y) = false;
                m_screen->update_rect (world_to_screen (WorldArea (x, y, 1, 1)));
            }
}


//----------------------------------------------------------------------
// Model layer
//----------------------------------------------------------------------

void
ModelLayer::maybe_redraw_model(Model *m)
{
    WorldArea wa;
    if (m->has_changed(wa)) {
        get_engine()->mark_redraw_area(wa);
    }
}

void
ModelLayer::activate (Model *m)
{
    list<Model*> &am = m_active_models_new;
//    assert (find(am.begin(), am.end(), m) == am.end());
//     list<Model*>::iterator i = find(am.begin(), am.end(), (Model*) 0);
//     if (i == am.end())
//         am.push_back(m);
//     else
//         *i = m;
    am.push_back(m);
}

void
ModelLayer::deactivate (Model *m)
{
    list<Model*> &am = m_active_models;
    list<Model*>::iterator i = find(am.begin(), am.end(), m);
    if (i == am.end()) {
        m_active_models_new.remove(m);
    } else {
        //assert(i != am.end());
        *i = 0;
    }
//    delete m;
//    m_active_models.remove(m);
}

void
ModelLayer::new_world (int, int)
{
    m_active_models.clear();
    m_active_models_new.clear();
}

void
ModelLayer::tick (double dtime)
{
    ModelList &am = m_active_models;

    am.remove((Model*) 0);
    am.remove_if(mem_fun(&Model::is_garbage));

    // Append new active models to list
    am.splice (am.end(), m_active_models_new);

    /* for_each does not work; animation may remove itself during a
       tick.  This may happen for example when a model callback
       decides to replace the old model by another one.  */
    for (ModelList::iterator i=am.begin(); i!=am.end(); ++i) {
        if (Model *m = *i) {
            m->tick(dtime);

            // We have to check (*i) again because the list of active
            // models can change during a tick!
            if ((m = *i))
                maybe_redraw_model (m);
        }
    }
}


//----------------------------------------------------------------------
// Grid Layer
//----------------------------------------------------------------------

DL_Grid::DL_Grid()
: m_models (0, 0)
{
}

DL_Grid::~DL_Grid()
{
    delete_sequence (m_models.begin(), m_models.end());
}

void
DL_Grid::new_world (int w, int h)
{
    ModelLayer::new_world (w, h);
    delete_sequence (m_models.begin(), m_models.end());
    m_models.resize (w, h, 0);
}

void
DL_Grid::mark_redraw (int x, int y)
{
    get_engine()->mark_redraw_area (WorldArea (x, y, 2, 2));
}

void
DL_Grid::set_model (int x, int y, Model *m)
{
    if (!(x >= 0 && y >= 0 &&
          (unsigned)x<m_models.width() &&
          (unsigned)y<m_models.height()))
    {
        delete m;	// model is owned by DL_Grid!
        return;
    }

    if (m_models(x,y) != m) {
        if (Model *mm = m_models(x,y)) {
            mm->remove(this);
            delete mm;
        }
        m_models(x,y) = m;
        mark_redraw (x,y);
        if (m)
            m->expose (this, V2(x, y));
    }
}

Model *
DL_Grid::get_model (int x, int y)
{
    return m_models(x,y);
}

Model *DL_Grid::yield_model (int x, int y)
{
    Model *m = get_model (x, y);
    if (m)
        m->remove (this);
    m_models(x,y) = 0;
    mark_redraw (x,y);
    return m;
}


void
DL_Grid::draw (px::GC &gc, const WorldArea &a, int destx, int desty)
{
    int x2 = a.x+a.w;
    int y2 = a.y+a.h;
    int tilew = get_engine()->get_tilew();
    int tileh = get_engine()->get_tileh();
    int xpos = destx;
    for (int x=a.x; x<x2; ++x) {
        int ypos = desty;
        for (int y=a.y; y<y2; ++y) {
            if (Model *m = m_models(x,y))
                m->draw(gc, xpos, ypos);
            ypos += tileh;
        }
        xpos += tilew;
    }
}



//======================================================================
// SPRITES
//======================================================================

SpriteHandle::SpriteHandle() 
    : layer(0)
{
    id = DL_Sprites::MAGIC_SPRITEID;
}


void SpriteHandle::kill()
{
    if (layer) {
	layer->kill_sprite (id);
	layer = 0;
	id = DL_Sprites::MAGIC_SPRITEID;
    }
}

void SpriteHandle::move (const px::V2 &newpos) const
{
    if (layer)
        layer->move_sprite (id, newpos);
}

void SpriteHandle::replace_model (Model *m) const
{
    if (layer)
        layer->replace_sprite (id, m);
    else
        delete m;
}

Model *SpriteHandle::get_model () const
{
    return layer ? layer->get_model (id) : 0;
}

void SpriteHandle::set_callback (ModelCallback *cb) const
{
    if (Model *m = get_model())
        m->set_callback(cb);
}

//----------------------------------------------------------------------
// Sprite layer
//----------------------------------------------------------------------

DL_Sprites::DL_Sprites()
: numsprites(0), maxsprites(10000)
{}

DL_Sprites::~DL_Sprites()
{
    delete_sequence(sprites.begin(), sprites.end());
}

void
DL_Sprites::new_world (int w, int h)
{
    ModelLayer::new_world (w,h);
    delete_sequence (sprites.begin(), sprites.end());
    sprites.clear();
    numsprites = 0;
}

void
DL_Sprites::move_sprite (SpriteId id, const px::V2& newpos)
{
    Sprite *sprite = sprites[id];

    int oldx, oldy;
    int newx, newy;
    get_engine()->world_to_screen (newpos, &newx, &newy);
    get_engine()->world_to_screen (sprite->pos, &oldx, &oldy);

    if (oldx != newx || oldy != newy) {
	redraw_sprite_region(id);
	sprite->pos = newpos;
        if (Anim2d* anim = dynamic_cast<Anim2d*>(sprite->model))
            anim->move (newpos);
	redraw_sprite_region(id);
    }
}


SpriteId
DL_Sprites::add_sprite (Sprite *sprite)
{
    if (numsprites >= maxsprites) {
        delete sprite;
        return MAGIC_SPRITEID;
    }

    SpriteList &sl = sprites;
    SpriteId id = 0;

    // Find the first empty slot
    SpriteList::iterator i = find(sl.begin(), sl.end(), (Sprite*)0);
    if (i == sl.end()) {
        id = sl.size();
        sl.push_back(sprite);
    }
    else {
        id = distance(sl.begin(), i);
        *i = sprite;
    }
    if (Model *m = sprite->model)
        m->expose (this, V2 (sprite->pos[0], sprite->pos[1]));
    redraw_sprite_region(id);
    numsprites += 1;
    return id;
}

void
DL_Sprites::replace_sprite (SpriteId id, Model *m)
{
    Sprite *sprite = sprites[id];

    if (Model *old = sprite->model) {
        old->remove (this);
        delete old;
    }
    sprite->model = m;
    if (m) {
        m->expose (this, V2 (sprite->pos[0], sprite->pos[1]));
    }
    redraw_sprite_region(id);
}

void
DL_Sprites::kill_sprite (SpriteId id)
{
    if (Sprite *sprite = sprites[id]) {
        if (Model *m = sprite->model) {
            m->remove (this);
        }
        sprites[id] = 0;
        numsprites -= 1;
        delete sprite;
    }
}

void
DL_Sprites::draw (px::GC &/*gc*/, const WorldArea &/*a*/, int /*x*/, int /*y*/)
{}


void
DL_Sprites::draw_onepass (px::GC &gc) //, const WorldArea &a, int /*x*/, int /*y*/)
{
//    clip (gc, get_engine()->world_to_screen(a));
    draw_sprites (false, gc);
}

void
DL_Sprites::draw_sprites (bool shades, GC &gc)
{
    SpriteList &sl = sprites;

    for (unsigned i=0; i<sl.size(); ++i)
    {
        Sprite *s = sl[i];
        if (!s || !s->model)
            continue;

        if (s->model->is_garbage()  && s->layer==SPRITE_EFFECT) {
            // Only remove effect sprites -- actor sprites remain in
            // the world all the time
            kill_sprite (i);
        }
        else {
            int sx, sy;
            get_engine()->world_to_screen(s->pos, &sx, &sy);
            if (shades)
                s->model->draw_shadow(gc, sx, sy);
            else
                s->model->draw(gc, sx, sy);
        }
    }
}

void
DL_Sprites::redraw_sprite_region (SpriteId id)
{
    Sprite *s = sprites[id];
    int x1 = static_cast<int> (s->pos[0]-0.5);
    int y1 = static_cast<int> (s->pos[1]-0.5);
    int x2 = static_cast<int> (s->pos[0]+0.8);
    int y2 = static_cast<int> (s->pos[1]+0.8);
    get_engine()->mark_redraw_area (Rect (x1, y1, x2-x1+1, y2-y1+1)); //Rect(x, y, 3, 3));
}


//----------------------------------------------------------------------
// RUBBER BANDS
//----------------------------------------------------------------------

void
DL_Lines::draw_onepass (px::GC &gc)//, const WorldArea &a, int x, int y)
{
    DisplayEngine *engine = get_engine();

    set_color (gc, 240, 140, 20, 255);
    set_flags (gc.flags, GS_ANTIALIAS);

    for (LineMap::iterator i=m_rubbers.begin(); i!= m_rubbers.end(); ++i)
    {
        int x1, y1, x2, y2;
        engine->world_to_screen (i->second.start, &x1, &y1);
        engine->world_to_screen (i->second.end, &x2, &y2);

        line (gc, x1, y1, x2, y2);
    }
}

/* Mark the screen region occupied by a rubber band for redraw.
   Problem is: what region is that exactly?  What pixels on the screen
   will the line rasterizer touch?  Hard to tell, especially when
   anti-aliasing is used.

   This function constructs a list of rectangles that completely
   enclose the line by subdividing the line into n segments and
   constructing the bounding box for each of these segments.  To
   account for the (effective) finite width of the line, these boxes
   need to be enlarged by a small amount to make them overlap a bit.

   The number n of subdivision depends on the length of the line.  n=1
   would of course do, but we want to redraw as little of the screen
   as possible.
*/
void
DL_Lines::mark_redraw_line (const Line &r)
{
    const double maxboxsize = 1.0; //0.5;

    double w0 = r.start[0]-r.end[0];
    double h0 = r.start[1]-r.end[1];
    int n = int (max(abs(w0),abs(h0)) / maxboxsize)+1;

    double w = w0/n;
    double h = h0/n;

    double overlap = 0.5;

    double x = r.end[0];
    double y = r.end[1];

    WorldArea wa(int(x-overlap), int(y-overlap),
                 int(abs(w)+2*overlap+1), int(abs(h)+2*overlap+1));

    for (int i=0; i<n; ++i) {
        wa.x = int(x-overlap);
        wa.y = int(y-overlap);
//         if (wa.x != int(x) || wa.y !=int(y))
//             wa.w = wa.h = 2;
//         else
//             wa.w = wa.h = 1;

        get_engine()->mark_redraw_area (wa);

        x += w;
        y += h;
    }
}

RubberHandle
DL_Lines::add_line (const V2 &p1, const V2 &p2)
{
    m_rubbers[m_id] = Line(p1, p2);
    mark_redraw_line (m_rubbers[m_id]);
    return RubberHandle(this, m_id++);
}

void
DL_Lines::set_startpoint (unsigned id, const V2 &p1)
{
    mark_redraw_line (m_rubbers[id]);
    m_rubbers[id].start = p1;
    mark_redraw_line (m_rubbers[id]);
}

void
DL_Lines::set_endpoint (unsigned id, const V2 &p2)
{
    mark_redraw_line (m_rubbers[id]);
    m_rubbers[id].end = p2;
    mark_redraw_line (m_rubbers[id]);
}

void
DL_Lines::kill_line (unsigned id)
{
    mark_redraw_line (m_rubbers[id]);
    LineMap::iterator i=m_rubbers.find(id);
    if (i != m_rubbers.end())
        m_rubbers.erase(i);
}

RubberHandle::RubberHandle(DL_Lines *ll, unsigned id_)
: line_layer (ll), id(id_)
{
}

void RubberHandle::update_first(const V2 &p1)
{
    line_layer->set_startpoint (id, p1);
}

void RubberHandle::update_second(const V2 &p2)
{
    line_layer->set_endpoint (id, p2);
}

void RubberHandle::kill()
{
    line_layer->kill_line(id);
}


//----------------------------------------------------------------------
// SHADOWS
//----------------------------------------------------------------------

/*
** Drawing the shadows is a lot more difficult than drawing any of the
** other layers.  There are a couple of reasons for this:
**
** 1. Both Stones and actors cast a shadow.  Not a real problem, but
**    it makes the implementation more complex.
**
** 2. Shadows can overlap.  Not only can the shadows of stones and
**    actors overlap, but also the shadows of two adjacent stones can.
**    Since we are using alpha blending for the shadows, this means
**    that we cannot blit the invidual shadows to the screen, but we
**    have to use a intermediate buffer.
**
** 3. Performance is critical.  Drawing the shadows is time-consuming,
**    firstly because alpha blending is costly and secondly because of
**    the intermediate buffer.  So we should try to cache shadows *and*
**    avoid the buffer if possible.
**
** So, how do we approach these problems? We handle stone and actor
** shadows separately: The stone shadows do not change very often so
** it's easy to cache them, one tile at a time.  If there is no actor
** on this tile, we can blit the cached image directly to the screen,
** otherwise we have no choice but to use the buffer.
**
** The remaining problem is the shadow cache.  The easiest solution
** would be to use one huge image for the whole level and keep it in
** memory all the time.  This would consume roughly 20mb for a 100x100
** landscape, which is of course excessive, considering that there are
** rarely more than 40 different shadow tiles in each landscape.
**
** Instead, Enigma caches the most recently calculated shadow tiles in
** a linked list.  (If this should one day turn out to be too slow,
** it's still possible to resort to a hash table or something
** similar.)
*/

namespace display
{
    struct ImageQuad {
        Image *images[4];

        ImageQuad() { /* do not initialize fields. */ }

        ImageQuad (Image *i1, Image *i2, Image *i3, Image *i4) {
            images[0] = i1;
            images[1] = i2;
            images[2] = i3;
            images[3] = i4;
        }
        bool operator == (const ImageQuad &q) {
            return (images[0]==q.images[0] &&
                    images[1]==q.images[1] &&
                    images[2]==q.images[2] &&
                    images[3]==q.images[3]);
        }
        Image *operator[] (int idx) { return images[idx]; }
    };

    inline bool only_image_shadows (Model *models[4], ImageQuad &q) {
        int nimages=4;

        for (int i=0; i<4; ++i)
	{
            if (models[i] == 0)
	    {
                // No model at all? Take that as an image shadow (it's
                // static after all).
                q.images[i] = 0;
	    }
            else if (Model *shadow = models[i]->get_shadow())
            {
                if (ImageModel *im = dynamic_cast<ImageModel*>(shadow))
                    // We have a model with a static image shadow
                    q.images[i] = im->get_image();
                else
                    q.images[i] = 0, nimages--;
            }
            else
            {
                q.images[i] = 0; //, nimages--;
            }
        }
        return nimages==4;
    }


    struct StoneShadow {
        ImageQuad  images;
        Surface   *image;
        bool       in_cache;

        StoneShadow (ImageQuad iq, bool cached)
        : images(iq), image(0), in_cache(cached)
        {}
    };

    class StoneShadowCache : public px::Nocopy {
    public:
        StoneShadowCache(int tilew, int tileh);
        ~StoneShadowCache();

        StoneShadow *retrieve (Model *models[4]);
        void release (StoneShadow *s);
        void clear();
    private:
        typedef std::list<StoneShadow*> CacheList;

        // Variables
        size_t            m_max_size;		    // Max. number of different shadow tiles to cache
        CacheList         m_cache;
        int               m_tilew, m_tileh;
        vector<Surface *> m_surface_avail;

        // Private methods.
        Surface *new_surface ();
        StoneShadow *find_in_cache (const ImageQuad &images);

        void fill_image (StoneShadow *s);
        void fill_image (StoneShadow *sh, Model *models[4]);
    };
}

StoneShadowCache::StoneShadowCache(int tilew, int tileh)
: m_max_size(50), m_cache()
{
    m_tilew=tilew; m_tileh=tileh;
}

StoneShadowCache::~StoneShadowCache()
{
    clear();
}

void
StoneShadowCache::clear()
{
    for (CacheList::iterator i = m_cache.begin(); i!=m_cache.end(); ++i)
        delete (*i)->image;
    delete_sequence (m_cache.begin(), m_cache.end());
    m_cache.clear();
    delete_sequence (m_surface_avail.begin(), m_surface_avail.end());
    m_surface_avail.clear();
}

void
StoneShadowCache::fill_image (StoneShadow *sh)
{
    // Special case: no shadows at all:
    if (sh->images[0] == 0 && sh->images[1] == 0 &&
        sh->images[2] == 0 && sh->images[3] == 0)
    {
        sh->image = 0;
        return;
    }

    Surface *s = new_surface();
    GC gc(s);
    set_color (gc, 255,255,255);
    box(gc, s->size());

    if (Image *i = sh->images[0])
        i->draw (gc, -m_tilew, -m_tileh);
    if (Image *i = sh->images[1])
        i->draw (gc, 0, -m_tileh);
    if (Image *i = sh->images[2])
        i->draw (gc, -m_tilew, 0);
    if (Image *i = sh->images[3])
        i->draw (gc, 0, 0);

    sh->image = s;
}

void
StoneShadowCache::fill_image (StoneShadow *sh, Model *models[4])
{
    Surface *s = new_surface();
    GC gc(s);
    set_color (gc, 255,255,255);
    box(gc, s->size());
    if (models[0]) models[0]->draw_shadow (gc, -m_tilew, -m_tileh);
    if (models[1]) models[1]->draw_shadow (gc, 0, -m_tileh);
    if (models[2]) models[2]->draw_shadow (gc, -m_tilew, 0);
    if (models[3]) models[3]->draw_shadow (gc, 0,0);
    sh->image = s;
}


StoneShadow *
StoneShadowCache::find_in_cache (const ImageQuad &images)
{
    CacheList::iterator i=m_cache.begin();
    for (; i!=m_cache.end(); ++i) {
        if ((*i)->images == images) {
            StoneShadow *sh = *i;
            // Move entry to front of list
            m_cache.splice (m_cache.begin(), m_cache, i);
            return sh;
        }
    }
    return 0;
}

/*
 * Try to lookup the shadow created by the four models in `models[]' in the
 * shadow cache.  Of course, because shadows aren't easy in general, there
 * are a few complications.
 *
 * Only if the shadows of all four models are static
 *
 */
StoneShadow *
StoneShadowCache::retrieve (Model *models[4])
{
    StoneShadow *shadow = 0;

    ImageQuad images;

    // Only cache static stone shadows, i.e., those consisting
    // only of Image models.
    if (only_image_shadows (models, images)) {
        shadow = find_in_cache(images);
        if (!shadow) {
            shadow = new StoneShadow (images, true);
            fill_image (shadow);
            m_cache.push_front (shadow);
        }
    }
    else {
        shadow = new StoneShadow (images, false);
        fill_image (shadow, models);
    }
    return shadow;
}

void
StoneShadowCache::release (StoneShadow *s)
{
    if (s->in_cache) {
        // Image is in cache, no need to free anything
    }
    else {
        m_surface_avail.push_back(s->image);
        delete s;
    }
}


Surface *
StoneShadowCache::new_surface () {
    Surface *s = 0;
    if (m_surface_avail.empty()) {
#ifndef MACOSX
        SDL_Surface *ss = SDL_CreateRGBSurface(SDL_SWSURFACE|SDL_RLEACCEL,
                                               m_tilew, m_tileh, 16,
                                               0,0,0,0);
        s = Surface::make_surface (SDL_DisplayFormat(ss));
        SDL_FreeSurface(ss);
#else
        SDL_Surface *ss = SDL_CreateRGBSurface(SDL_SWSURFACE,
                                               m_tilew, m_tileh, 32,
                                               0,0,0,0);
        s = Surface::make_surface(ss);
#endif
    } else {
        s = m_surface_avail.back();
        m_surface_avail.pop_back();
    }
    return s;
}


//----------------------------------------
// DL_Shadows
//----------------------------------------

DL_Shadows::DL_Shadows (DL_Grid *grid, DL_Sprites *sprites)
: m_grid(grid), m_sprites(sprites)
{
    int tilew=32, tileh=32;
    m_cache = new StoneShadowCache(tilew, tileh);
#ifndef MACOSX
    SDL_Surface *ss = SDL_CreateRGBSurface(SDL_SWSURFACE,
                                           tilew, tileh,
                                           video::GetColorDepth(),
                                           0,0,0,0);
    SDL_SetColorKey(ss, SDL_SRCCOLORKEY,
                    SDL_MapRGB(ss->format, 255,255,255));
    SDL_SetAlpha(ss, SDL_SRCALPHA, 128);
    buffer = Surface::make_surface(ss);
#else
    SDL_Surface *ss = SDL_CreateRGBSurface(SDL_SWSURFACE,
                                           tilew, tileh, 32,
                                           0,0,0,0);
    SDL_SetAlpha(ss, SDL_SRCALPHA, 128);
    SDL_SetColorKey(ss, SDL_SRCCOLORKEY,
                    SDL_MapRGB(ss->format, 255,255,255));
    buffer = Surface::make_surface(ss);
#endif
}

DL_Shadows::~DL_Shadows()
{
    delete m_cache;
    delete buffer;
}

void
DL_Shadows::new_world(int /*w*/, int /*h*/)
{
    m_cache->clear();
}

void
DL_Shadows::draw (px::GC &gc, const WorldArea &a, int destx, int desty)
{
    int x2 = a.x+a.w;
    int y2 = a.y+a.h;
    int tilew = get_engine()->get_tilew();
    int tileh = get_engine()->get_tileh();
    int xpos = destx;
    for (int x=a.x; x<x2; ++x) {
        int ypos = desty;
        for (int y=a.y; y<y2; ++y) {
            draw (gc, xpos, ypos, x, y);
            ypos += tileh;
        }
        xpos += tilew;
    }
}

bool
DL_Shadows::has_actor (int x, int y)
{
    // Is any actor at least partially inside this field?
    for (unsigned i=0; i<m_sprites->sprites.size(); ++i) {
        Sprite *s = m_sprites->sprites[i];
        if (s && s->layer == SPRITE_ACTOR) {
            Rect r(int(s->pos[0]-0.3), int(s->pos[1]-0.3),2,2);
            if (r.contains(x,y)) {
                return true;
            }
        }
    }
    return false;
}

Model *
DL_Shadows::get_shadow_model(int x, int y)
{
    if (x >= 0 && y >= 0) {
        if (Model *m = m_grid->get_model(x,y))
            return m; //return m->get_shadow();
    }
    return 0;
}


void
DL_Shadows::draw(GC &gc, int xpos, int ypos, int x, int y)
{
    Model *models[4];
    models[0] = get_shadow_model (x-1, y-1);
    models[1] = get_shadow_model (x, y-1);
    models[2] = get_shadow_model (x-1, y);
    models[3] = get_shadow_model (x, y);

    StoneShadow *sh = m_cache->retrieve (models);

    bool hasActor = this->has_actor (x, y);
    if (hasActor || sh->image)
    {
        Surface *s = sh->image;
        if (hasActor) {
            GC gc2(buffer);
            if (s)
                blit (gc2, 0,0,s);
            else {
                set_color (gc2, 255, 255, 255);
                box (gc2, buffer->size());
            }
            for (unsigned i=0; i<m_sprites->sprites.size(); ++i) {
                if (Sprite *sp = m_sprites->sprites[i]) {
                    if (sp->model) {
                        int sx = int(sp->pos[0]*32) - x*32;
                        int sy = int(sp->pos[1]*32) - y*32;
                        sp->model->draw_shadow(gc2, sx, sy);
                    }
                }
            }
            blit(gc, xpos, ypos, buffer);
        }
        else {
            SDL_Surface *ss = s->get_surface();
            SDL_SetColorKey(ss, SDL_SRCCOLORKEY,
                            SDL_MapRGB(ss->format, 255,255,255));
            SDL_SetAlpha (ss, SDL_SRCALPHA, 128);
            blit (gc, xpos,ypos,s);
            SDL_SetAlpha (ss, 0, 0);
            SDL_SetColorKey (ss, 0,0);
        }
    }

    m_cache->release (sh);
}

void
DL_Shadows::shadow_blit (px::Surface *scr, int x, int y,
                         px::Surface *shadows, px::Rect r)
{
//     r.intersect(scr->size());
//     x=r.x;
//     y=r.y;

//     SDL_Surface *s = shadows->get_surface(); // source
//     SDL_Surface *d = scr->get_surface();

//     SDL_LockSurface(s);
//     SDL_LockSurface(d);

//     assert (s->format->BytesPerPixel==2 &&
//             d->format->BytesPerPixel==2);


//     /* This bitmask is used to erase the highest bit in every color
//        field.  Every color value can be effectively halved by (a)
//        shifting everything one bit to the right, and (b) applying the
//        mask to the resulting value. */
//     Uint32 mask;
//     mask  = (d->format->Rmask >> d->format->Rshift+1) << d->format->Rshift;
//     mask |= (d->format->Gmask >> d->format->Gshift+1) << d->format->Gshift;
//     mask |= (d->format->Bmask >> d->format->Bshift+1) << d->format->Bshift;

//     register Uint16 skey = (Uint16)shadow_ckey;
//     for (int h=r.h; h; --h)
//     {
//         Uint16 *sp = static_cast<Uint16*>(shadows->pixel_pointer(r.x, r.y));
//         Uint16 *dp = static_cast<Uint16*>(scr->pixel_pointer(x, y));
//         for (int w=r.w; w; --w)
//         {
//             if (*sp != skey)
//                 *dp = (*dp >> 1) & mask;
//             dp++;
//             sp++;
//         }
//         y++;
//         r.y++;
//     }

//     SDL_UnlockSurface(s);
//     SDL_UnlockSurface(d);
}


//----------------------------------------------------------------------
// Sprite following code
//----------------------------------------------------------------------

Follower::Follower (DisplayEngine *e)
: m_engine(e)
{
    ScreenArea gamearea = e->get_area();
    m_hoff = gamearea.w / e->get_tilew() - 1;
    m_voff = gamearea.h / e->get_tileh() - 1;
}

void
Follower::center(const px::V2 &point)
{
    set_offset(V2 (int(point[0] / m_hoff) * m_hoff,
		   int(point[1] / m_voff) * m_voff));
}

bool
Follower::set_offset (V2 offs)
{
    DisplayEngine *e = get_engine();
    offs[0] = max (offs[0], 0.0);
    offs[1] = max (offs[1], 0.0);
    offs[0] = min (double(e->get_width()-get_hoff()-1), offs[0]);
    offs[1] = min (double(e->get_height()-get_voff()-1), offs[1]);
    if (offs != e->get_offset()) {
	e->set_offset(offs);
	return true;
    }
    return false;
}

Follower_Screen::Follower_Screen(DisplayEngine *e)
: Follower(e)
{}


/* Determine whether the screen must be scrolled or not, and change
   the coordinate origin of the screen accordingly. */
void
Follower_Screen::tick(double, const px::V2 &point)
{
    DisplayEngine *engine = get_engine();
    ScreenArea gamearea = engine->get_area();
    int borderh = engine->get_tilew()/2;
    int borderv = engine->get_tileh()/2;

    while (true) {
        int sx, sy;
        engine->world_to_screen(point, &sx, &sy);

        V2 screenoff = engine->get_offset();

        if (sx < gamearea.x + borderh)
            screenoff[0] -= get_hoff();
        if (sy < gamearea.y + borderv)
            screenoff[1] -= get_voff();

        if (sx >= gamearea.x + gamearea.w-borderh)
            screenoff[0] += get_hoff();
        if (sy >= gamearea.y + gamearea.h-borderv)
            screenoff[1] += get_voff();

	if (set_offset(screenoff))
	    engine->mark_redraw_screen();
        else
            break;
    }
}



Follower_Scrolling::Follower_Scrolling(DisplayEngine *e)
: Follower (e),
  currently_scrolling(false),
  scrollspeed(0), resttime(0)
{}

void
Follower_Scrolling::center(const px::V2 &point)
{
    Follower::center(point);
    curpos = destpos = get_engine()->get_offset();
}


void
Follower_Scrolling::tick(double dtime, const px::V2 &point)
{
    DisplayEngine *engine   = get_engine();
    ScreenArea     gamearea = engine->get_area();
    int            tilew    = engine->get_tilew();
    int            tileh    = engine->get_tileh();
    const double   border   = 0.5;

    int sx, sy;
    engine->world_to_screen(point, &sx, &sy);

    bool scrollx_p = (sx < gamearea.x + tilew*border)
        || (sx >= gamearea.x + gamearea.w - tilew*border);

    bool scrolly_p = (sy < gamearea.y + tileh*border)
        || (sy >= gamearea.y + gamearea.h - tileh*border);

    if (scrollx_p || scrolly_p)
    {
        V2 olddest = destpos;
        V2 scrollpos = engine->get_offset();

        currently_scrolling = true;
        curpos = scrollpos;
        destpos = curpos + point - V2(scrollpos[0] + gamearea.w/tilew/2,
                                      scrollpos[1] + gamearea.h/tileh/2);
        destpos[0] = int(destpos[0]*tilew)/tilew;
        destpos[1] = int(destpos[1]*tileh)/tileh;
        destpos[0] = Clamp ((int)destpos[0], 0, engine->get_width()-gamearea.w/tilew);
        destpos[1] = Clamp ((int)destpos[1], 0, engine->get_height()-gamearea.h/tileh);
        if (!scrollx_p)
            destpos[0] = olddest[0];
        if (!scrolly_p)
            destpos[1] = olddest[1];

        scrollspeed = Max(40.0, length(destpos-curpos)/tilew*5);
        resttime = length(destpos - curpos)/scrollspeed;
        dir = normalize(destpos - curpos);
    }
    if (currently_scrolling) {
        resttime -= dtime;
        if (resttime <= 0) {
            engine->move_offset (destpos);
            currently_scrolling = false;
        } else {
            curpos += dir * scrollspeed*dtime;
            engine->move_offset (curpos);
        }
    }
}



//----------------------------------------------------------------------
// Editor / game display engine
//----------------------------------------------------------------------
CommonDisplay::CommonDisplay (const ScreenArea &a)
{
    DisplayEngine *engine = new DisplayEngine (video::GetScreen(), 32, 32);
    engine->set_screen_area (a);

    engine->add_layer (floor_layer = new DL_Grid);
    engine->add_layer (item_layer = new DL_Grid);
    sprite_layer = new DL_Sprites;
    stone_layer = new DL_Grid;
    shadow_layer = new DL_Shadows(stone_layer, sprite_layer);
    engine->add_layer (shadow_layer);
    engine->add_layer (sprite_layer);
    engine->add_layer (stone_layer);
    engine->add_layer (line_layer = new DL_Lines);
    engine->add_layer (effects_layer = new DL_Sprites);
    effects_layer->set_maxsprites(10);

    m_engine = engine;
}

CommonDisplay::~CommonDisplay()
{
    delete m_engine;
}


Model *
CommonDisplay::set_model (const GridLoc &l, Model *m)
{
    int x = l.pos.x, y=l.pos.y;

    switch (l.layer) {
    case GRID_FLOOR: floor_layer->set_model (x, y, m); break;
    case GRID_ITEMS: item_layer->set_model (x, y, m); break;
    case GRID_STONES:
        stone_layer->set_model (x, y, m);
//        shadow_layer->set_model (x, y, m);
//         shadow_layer->update (x, y);
        break;
    case GRID_COUNT: break;
    }
    return m;
}

Model *
CommonDisplay::get_model (const GridLoc &l)
{
    int x = l.pos.x, y=l.pos.y;
    switch (l.layer) {
    case GRID_FLOOR: return floor_layer->get_model (x, y);
    case GRID_ITEMS: return item_layer->get_model (x, y);
    case GRID_STONES: return stone_layer->get_model (x, y);
    case GRID_COUNT: return 0;
    }
    return 0;
}

Model *
CommonDisplay::yield_model (const GridLoc &l)
{
    int x = l.pos.x, y=l.pos.y;
    switch (l.layer) {
    case GRID_FLOOR: return floor_layer->yield_model (x, y);
    case GRID_ITEMS: return item_layer->yield_model (x, y);
    case GRID_STONES: return stone_layer->yield_model (x, y);
    case GRID_COUNT: return 0;
    }
    return 0;
}


RubberHandle
CommonDisplay::add_line (V2 p1, V2 p2)
{
    return line_layer->add_line (p1, p2);
}

SpriteHandle
CommonDisplay::add_effect (const V2& pos, Model *m)
{
    Sprite *spr = new Sprite (pos, SPRITE_EFFECT, m);
    return SpriteHandle (effects_layer, effects_layer->add_sprite(spr));
}

SpriteHandle
CommonDisplay::add_sprite (const V2 &pos, Model *m)
{
    Sprite *spr = new Sprite (pos, SPRITE_ACTOR, m);
    return SpriteHandle (sprite_layer, sprite_layer->add_sprite(spr));
}

void
CommonDisplay::new_world (int w, int h)
{
    get_engine()->new_world (w, h);
}

void
CommonDisplay::redraw()
{
    get_engine()->update_screen();
}


void
CommonDisplay::set_floor (int x, int y, Model *m)
{
    floor_layer->set_model (x, y, m);
}

void
CommonDisplay::set_item (int x, int y, Model *m)
{
    item_layer->set_model (x,y , m);
}

void
CommonDisplay::set_stone (int x, int y, Model *m)
{
    stone_layer->set_model (x,y , m);
}




//----------------------------------------------------------------------
// Game Display Engine
//----------------------------------------------------------------------

GameDisplay::GameDisplay(const ScreenArea &a)
: CommonDisplay(a),
  last_frame_time (0),
  redraw_everything(false),
  m_reference_point (),
  m_follower (0),
  inventoryarea(0, 480-64, 640, 64)
{
    status_bar = new StatusBarImpl (inventoryarea);
}

GameDisplay::~GameDisplay()
{
    delete m_follower;
    delete status_bar;
}

void
GameDisplay::tick(double dtime)
{
    get_engine()->tick (dtime);
    status_bar->tick (dtime);

    if (m_follower)
        m_follower->tick (dtime, m_reference_point);
}

void
GameDisplay::new_world (int w, int h)
{
    CommonDisplay::new_world (w, h);
    status_bar->new_world();
    resize_game_area (20,13);
    set_follow_mode (FOLLOW_SCREEN);

//     shadow_layer->new_world(w,h);
}

StatusBar *
GameDisplay::get_status_bar() const
{
    return status_bar;
}

//----------------------------------------
// Scrolling
//----------------------------------------

void
GameDisplay::set_follow_mode (FollowMode m)
{
    switch (m) {
    case FOLLOW_NONE: set_follower(0); break;
    case FOLLOW_SCROLLING:
	set_follower (new Follower_Scrolling(get_engine())); break;
    case FOLLOW_SCREEN:
	set_follower (new Follower_Screen(get_engine())); break;
    };
}

void
GameDisplay::set_follower (Follower *f)
{
    delete m_follower;
    if ((m_follower = f))
        follow_center();
}

void
GameDisplay::follow_center()
{
    if (m_follower)
        m_follower->center (m_reference_point);
}

void
GameDisplay::set_reference_point (const V2 &point)
{
    m_reference_point = point;
}

void
GameDisplay::get_reference_point_coordinates(int *x, int *y) {
    get_engine()->world_to_screen(m_reference_point, x, y);
}

//----------------------------------------
// Screen updates
//----------------------------------------

void
GameDisplay::redraw_all (Screen *scr)
{
    get_engine()->mark_redraw_screen();
    redraw_everything = true;
    scr->update_all();
    redraw (scr);
}

void
GameDisplay::redraw(px::Screen *screen)
{
    GC gc(screen->get_surface());
    CommonDisplay::redraw();
    if (status_bar->has_changed() || redraw_everything) {
        status_bar->redraw (gc, inventoryarea);
        screen->update_rect(inventoryarea);
    }
    if (options::ShowFPS) {
        char fps[20];
        sprintf (fps,"fps: %d\n", int(1000.0/(SDL_GetTicks()-last_frame_time)));
        last_frame_time = SDL_GetTicks();
        Font *f = enigma::GetFont("levelmenu");

        clip(gc);
        Rect area (0,0,80,20);
        set_color (gc, 0,0,0);
        box (gc, area);
        f->render (gc, 0,0, fps);

        screen->update_rect(area);
    }
    if (redraw_everything)
        draw_borders(gc);
    screen->flush_updates();
    redraw_everything = false;
}

void
GameDisplay::draw_all (GC &gc)
{
    get_engine()->draw_all(gc);
    status_bar->redraw (gc, inventoryarea);
    draw_borders(gc);
}

void
GameDisplay::draw_borders (GC &gc)
{
    RectList rl;
    rl.push_back (gc.drawable->size());
    rl.sub (get_engine()->get_area());
    rl.sub (inventoryarea);
    clip(gc);
    set_color (gc, 0, 0, 0);
    for (RectList::iterator i=rl.begin(); i!=rl.end(); ++i) {
        box (gc, *i);
    }
}

void GameDisplay::resize_game_area (int w, int h)
{
    DisplayEngine *e = get_engine();
    int neww = w * e->get_tilew();
    int newh = h * e->get_tileh();
    int screenw = 640;
    int screenh = 480-64;       // - height of status bar
    if (neww > screenw || newh > screenh)
    {
        enigma::Log << "Illegal screen size ("<< neww << "," << newh
                    << "): larger than physical display\n";
        return;
    }
    Rect r ((screenw-neww)/2, (screenh-newh)/2, neww, newh);
    e->set_screen_area (r);
    follow_center();
}



//----------------------------------------------------------------------
// GLOBAL FUNCTIONS
//----------------------------------------------------------------------

void
display::Init()
{
    InitModels();

    ScreenArea gamearea(0,0,640, 480-64);
    gamedpy = new GameDisplay (gamearea);
}

void
display::Shutdown()
{
    delete gamedpy;
    ShutdownModels();
}

void display::Tick (double dtime) {
    gamedpy->tick(dtime);
}

StatusBar * display::GetStatusBar() {
    return gamedpy->get_status_bar();
}

void display::NewWorld(int w, int h) {
    gamedpy->new_world (w, h);
}

void display::FocusReferencePoint() {
    gamedpy->follow_center();
}

void display::SetReferencePoint (const px::V2 &point) {
    gamedpy->set_reference_point (point);
}

void display::SetFollowMode(FollowMode m) {
    gamedpy->set_follow_mode(m);
}

void display::GetReferencePointCoordinates(int *x, int *y) {
    gamedpy->get_reference_point_coordinates(x, y);
}

Model *display::SetModel (const GridLoc &l, Model *m) {
    return gamedpy->set_model (l, m);
}

Model *display::SetModel (const GridLoc &l, const string &modelname) {
    return SetModel(l, MakeModel(modelname));
}

void display::KillModel(const GridLoc & l) {
    delete YieldModel(l);
}

Model *display::GetModel(const GridLoc &l) {
    return gamedpy->get_model (l);
}

Model *display::YieldModel(const GridLoc &l) {
    return gamedpy->yield_model (l);
}

void display::AddEffect (const V2& pos, const char *modelname) {
    gamedpy->add_effect (pos, MakeModel(modelname));
}

SpriteHandle
display::AddSprite (const V2& pos, const char *modelname)
{
    Model *m = modelname ? MakeModel(modelname) : 0;
    return gamedpy->add_sprite (pos, m);
}

void display::ToggleFlag(DisplayFlags flag)
{
    toggle_flags (display_flags, flag);
}

void display::DrawAll (GC &gc) {
    gamedpy->draw_all(gc);
}

void display::RedrawAll(Screen *screen) {
    gamedpy->redraw_all(screen);
}

void display::Redraw(Screen *screen) {
    gamedpy->redraw(screen);
}

void display::ResizeGameArea (int w, int h) {
    gamedpy->resize_game_area (w, h);
}
const Rect& display::GetGameArea () {
    return gamedpy->get_engine()->get_area();
}

RubberHandle
display::AddRubber (const V2 &p1, const V2 &p2)
{
    return gamedpy->add_line (p1, p2);
}
