/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: items.cc,v 1.87.2.7 2003/10/11 13:45:24 dheck Exp $
 */

#include "items.hh"
#include "game.hh"
#include "object_mixins.hh"
#include "display.hh"
#include "sound.hh"
#include "player.hh"
#include <string>
#include <vector>
#include <algorithm>
#include <cmath>
#include <cassert>

using namespace items;
using namespace world;
using namespace std;

//----------------------------------------
// Item implementation
//----------------------------------------
Item::Item(const char *name)
: TGridObject<GRID_ITEMS>(name)
{}

string
Item::get_inventory_model()
{
    return get_kind();
}

void Item::stone_change (Stone */*st*/) {}
void Item::on_stonehit (Stone */*st*/) {}
void Item::on_drop (Actor */*a*/) {}
void Item::on_pickup (Actor */*a*/) {}

ItemAction Item::activate(Actor* /*a*/, GridPos /*p*/)
{ return ITEM_DROP; }


px::V2 Item::get_force(Actor */*a*/)
{ return px::V2(); }

bool
Item::actor_hit(Actor *actor)
{
    const double ITEM_RADIUS = 0.3;
    px::V2 item_center(get_pos().x + 0.5, get_pos().y + 0.5);
    double dist = length(actor->get_pos()-item_center);
    return (dist < ITEM_RADIUS);
}

//----------------------------------------
// Various simple items
//----------------------------------------

#define DEF_ITEM(classname, kindname)           \
class classname : public Item {                 \
    CLONEOBJ(classname);                        \
public:                                         \
    classname() : Item(kindname) {}             \
}

#define DEF_FRAGILE_ITEM(classname, kindname)           \
class classname : public Item {                         \
    CLONEOBJ(classname);                                \
public:                                                 \
    classname() : Item(kindname) {}                     \
private:                                                \
    bool on_laserhit(Direction) {                       \
        SetItem(get_pos(), MakeItem("it-explosion1"));  \
        return false;                                   \
    }                                                   \
}

namespace
{
    DEF_ITEM(MagicWand, "it-magicwand");
    DEF_ITEM(Floppy, "it-floppy");
    DEF_ITEM(Odometer, "it-odometer");
    DEF_ITEM(Wrench, "it-wrench");

    DEF_ITEM(Cherry, "it-cherry");

    DEF_FRAGILE_ITEM(Coffee, "it-coffee");

    class Dummyitem : public Item {
        CLONEOBJ(Dummyitem);
    public:
        Dummyitem() : Item("it-dummy") {}

        void on_pickup(Actor *) {
            int code = int_attrib("code");
            fprintf(stderr, "Picked up item 0x%x\n", code);
        }
        void on_drop(Actor *) {
            int code = int_attrib("code");
            fprintf(stderr, "Dropped item 0x%x\n", code);
        }
    };

    class Weight : public Item {
        CLONEOBJ(Weight);
    public:
	Weight() : Item("it-weight") {}
        void on_pickup(Actor *a) {
           ActorInfo *ai = a->get_actorinfo();
           ai->mass += 10.0;
        }
        ItemAction activate(Actor *, GridPos)
        {
            return ITEM_KEEP;
        }
    };

    class Pin : public Item {
        CLONEOBJ(Pin);
    public:
        void on_pickup(Actor *a) {
            a->set_spikes(true);
        }
        void on_drop(Actor *a) {
            a->set_spikes(false);
        }
        Pin() : Item("it-pin") {}
    };

    class Sword : public Item {
        CLONEOBJ(Sword);
        bool on_laserhit(Direction /*d*/) {
            play_sound("st-magic");
            world::SetItem(get_pos(), MakeItem("it-hammer"));
            return false;
        }
    public:
        Sword() : Item("it-sword") {}
    };

    class Hammer : public Item {
        CLONEOBJ(Hammer);
        bool on_laserhit(Direction /*d*/) {
            if (GameCompatibility != GAMET_PEROXYD) {
                play_sound("st-magic");
                world::SetItem(get_pos(), MakeItem("it-sword"));
            }
            return false;
        }
    public:
        Hammer() : Item("it-hammer") {}
    };

    class ExtraLife : public Item {
        CLONEOBJ(ExtraLife);
        std::string get_inventory_model() {
            if (player::CurrentPlayer()==0)
                return "inv-blackball";
            else
                return "inv-whiteball";
        }

        bool on_laserhit(Direction /*d*/) {
            play_sound("st-magic");
            SetItem(get_pos(), MakeItem("it-glasses"));
            return false;
        }

    public:
        ExtraLife() : Item("it-extralife") {}
    };

    class Umbrella : public Item {
        CLONEOBJ(Umbrella);

        bool on_laserhit(Direction /*d*/) {
            SetItem(get_pos(), MakeItem("it-explosion1"));
            return false;
        }

        ItemAction activate(Actor *a, GridPos) {
            SendMessage(a, "shield");
            return ITEM_KILL;
        }


    public:
        Umbrella() : Item ("it-umbrella") {}
    };

    class Key : public Item {
        CLONEOBJ(Key);
    public:
    	Key() : Item("it-key") { set_attrib("keycode", 0.0); }
    };
    class Key_a : public Item {
        CLONEOBJ(Key_a);
    public:
    	Key_a() : Item("it-key_a") { set_attrib("keycode", 1.0); }
    };
    class Key_b : public Item {
        CLONEOBJ(Key_b);
    public:
    	Key_b() : Item("it-key_b") { set_attrib("keycode", 2.0); }
    };
    class Key_c : public Item {
        CLONEOBJ(Key_c);
    public:
    	Key_c() : Item("it-key_c") { set_attrib("keycode", 3.0); }
    };
}


//----------------------------------------
// Brush
//
// Can "paint" some stones and remove ash
//----------------------------------------
namespace
{
    class Brush : public Item {
        CLONEOBJ(Brush);
    public:
        Brush() : Item("it-brush") {}
    private:
        bool on_laserhit(Direction) {
            SetItem(get_pos(), MakeItem("it-explosion1"));
            return false;
        }
        ItemAction activate(Actor *, GridPos p) {
            if (Item *it = GetItem(p))
                SendMessage (it, "brush");
            return ITEM_DROP;
        }
    };
}

//----------------------------------------
// Coins
//
// :value    1,2,4: how many time-units this coin buys
//----------------------------------------
namespace
{
    class Coin : public Item {
        CLONEOBJ(Coin);
    public:
        Coin(int value=1) : Item("it-coin") { set_attrib("value", value); }
    private:
        void init_model() { set_model(get_inventory_model()); }

        bool on_laserhit(Direction /*d*/) {
            play_sound("st-magic");
            switch(get_value()) {
            case 1: SetItem (get_pos(), MakeItem("it-umbrella")); break;
            case 2: SetItem (get_pos(), MakeItem("it-hammer")); break;
            default: SetItem (get_pos(), MakeItem("it-extralife")); break;
            }
            return false;
        }

        void on_stonehit(Stone */*st*/) {
            switch(get_value()) {
            case 1: set_attrib("value", 2.0); init_model(); break;
            case 2: set_attrib("value", 4.0); init_model(); break;
            default: break;
            }
        }

        string get_inventory_model()
        {
            switch(get_value()) {
            case 1: return "it-coin1";
            case 2: return "it-coin2";
            default: return "it-coin4";
            }
        }

        int get_value() {
            int value = int_attrib("value");
//            assert (value==1 || value==2 || value==4);
            return value;
        }
    };
}

//----------------------------------------
// Hollows & Hills
//----------------------------------------

/** \page it-hills Hills and Hollows

Hills and hollows are placed on the floor and can
make the movement difficult.

\subsection hillsm Messages
- \b trigger	will convert a hill to a hollow and vice versa
- \b shovel     decreases the size of the hill/hollow

\image html it-hill.png
*/
namespace
{
    class HillHollow : public Item {
    protected:
        enum Type { HILL, HOLLOW, TINYHILL, TINYHOLLOW };

        HillHollow(const char *name, Type t);

        void transmute(Type newtype);
        V2 vec_to_center (V2 v);
        double get_radius() const { return m_radius[m_type]; }

    private:
        double get_forcefac() const { return m_forcefac[m_type]; }
        void shovel();

        // Item interface
        px::V2 get_force(Actor *a);
        bool actor_hit(Actor */*a*/) { return false; }
        void on_stonehit(Stone *st);

        // Object interface.
        void message(const string &m, const Value &);

        // Variables.
        static double m_radius[4], m_forcefac[4];
        Type m_type;
    };
}

double HillHollow::m_radius[4] = {0.5, 0.5, 0.3, 0.3};
double HillHollow::m_forcefac[4] = {90,-90, 90, -90};


HillHollow::HillHollow(const char *name, Type t)
: Item(name), m_type(t)
{}

void HillHollow::on_stonehit(Stone */*st*/)
{
    shovel();
}

void HillHollow::shovel()
{
    if (m_type==HOLLOW)
        transmute (TINYHOLLOW);
    else if (m_type==HILL)
        transmute (TINYHILL);
    else
        KillItem(get_pos());
}


void HillHollow::message(const string &m, const Value &v)
{
    if (m=="trigger" || m=="signal") {
        Type flippedkind[] = {HOLLOW,HILL, TINYHOLLOW,TINYHILL};
        transmute(flippedkind[m_type]);
    }
    else if (m=="shovel")
        shovel();
}

V2 HillHollow::vec_to_center (V2 v)
{
    return v-get_pos().center();
}

V2 HillHollow::get_force(Actor *a)
{
    V2     v    = vec_to_center(a->get_pos());
    double dist = length(v);

    if (dist > get_radius())
        return px::V2();        // not on the hill

    if (dist <= 0) { // exactly on hill-top
        ActorInfo *ai = a->get_actorinfo();
        if (length(ai->vel) <= 0) { // no velocity
            // we are never "exactly" on the top!
            v = px::V2(DoubleRand(0.01, 0.05), DoubleRand(0.01, 0.05));
        }
    }

    return get_forcefac()*v; // get the force
}

void HillHollow::transmute(Type newtype)
{
    m_type = newtype;
    static const char *modelname[] = { "it-hill", "it-hollow",
                                       "it-tinyhill", "it-tinyhollow" };
    set_model(modelname[m_type]);
}

namespace
{
    class Hill : public HillHollow {
        CLONEOBJ(Hill);
    public:
        Hill() : HillHollow("it-hill", HILL) {}
    };
    class TinyHill : public HillHollow {
        CLONEOBJ(TinyHill);
    public:
        TinyHill() : HillHollow("it-tinyhill", TINYHILL) {}
    };

    class TinyHollow : public HillHollow {
        CLONEOBJ(TinyHollow);
    public:
        TinyHollow() : HillHollow("it-tinyhollow", TINYHOLLOW) {}
    };

    /*
     * Hollows are special in that they can end the current level
     * if they have each a small white marble inside them.
     */
    class Hollow : public HillHollow {
        INSTANCELISTOBJ(Hollow);
    public:
        Hollow();
    private:
        // Item interface.
        bool actor_hit(Actor *a);

        // Functions.
        bool near_center_p (Actor *a);
        void check_if_level_finished();

        // Variables.
        Actor *whiteball;       // The small white ball that is currently being tracked
        Uint32 enter_time;      // ... when did it enter the hollow?
    };
}

Hollow::InstanceList Hollow::instances;

Hollow::Hollow() : HillHollow("it-hollow", HOLLOW), whiteball(0)
{}

bool
Hollow::near_center_p (Actor *a)
{
    return (length(vec_to_center(a->get_pos())) < get_radius()/2);
}

bool
Hollow::actor_hit(Actor *a)
{
    if (whiteball==0 && a->is_kind("ac-whiteball-small") && near_center_p(a)) {
        whiteball  = a;
        enter_time = SDL_GetTicks();
    }
    else if (whiteball==a) {
        if (!near_center_p(a))
            whiteball = 0;
        else
            check_if_level_finished();
    }
    return false;
}

/* If (a) every small white ball is in a hollow and (b) each ball has
   been inside the hollow for at least MINTIME milliseconds, finish
   the level. */
void
Hollow::check_if_level_finished()
{
    const unsigned MINTIME = 1000;

    unsigned size = instances.size();
    unsigned cnt = 0;
    for (unsigned i=0; i<size; ++i) {
        Hollow *h = static_cast<Hollow*> (instances[i]);
        if (!h->whiteball || SDL_GetTicks() - h->enter_time < MINTIME)
            continue;
        else 
            ++cnt;
    }

    if (cnt == player::CountActorsOfKind("ac-whiteball-small"))
        enigma::FinishLevel();
}

//----------------------------------------
// Springs
//----------------------------------------

/** \page it-spring Spring

Activating a spring will make the marble jump.
A jumping marble does not fall into abyss or water.

Springs come in two flavors: it-spring1 stays in the inventory,
whereas it-spring2 drops to the floor when you activate it.

\image html it-spring1.png
*/
namespace
{
    class Spring1 : public Item {
        CLONEOBJ(Spring1);
    public:
        Spring1() : Item("it-spring1") {}
    private:
        ItemAction activate(Actor *a, GridPos)
        {
            SendMessage(a, "jump");
            return ITEM_KEEP;
        }

        const ObjectTraits *get_traits () const {
            static ObjectTraits traits("it-spring1", OBJTYPE_Item);
            return &traits;
        }
    };

    class Spring2 : public Item {
        CLONEOBJ(Spring2);
    public:
        Spring2() : Item("it-spring2") {}
    private:
        ItemAction activate(Actor *a, GridPos)
        {
            SendMessage(a, "jump");
            return ITEM_DROP;
        }
        const ObjectTraits *get_traits () const {
            static ObjectTraits traits("it-spring2", OBJTYPE_Item);
            return &traits;
        }
    };
}

//----------------------------------------
// Springboard
//----------------------------------------
namespace
{
    class Springboard : public Item {
        CLONEOBJ(Springboard);
    public:
        Springboard() : Item("it-springboard") {}

        bool actor_hit(Actor *a) {
            const double ITEM_RADIUS = 0.3;
            px::V2 item_center(get_pos().x + 0.5, get_pos().y + 0.5);
            double dist = length(a->get_pos()-item_center);
            if (dist < ITEM_RADIUS) {
                set_anim("it-springboard_anim");
                SendMessage(a, "jump");
            }
            return false;
        }

        void animcb() {
            set_model("it-springboard");
        }
    };
}


// --------------
//      Brake
// --------------
// Brake is only used to hold st-brake while it is in Actor inventory
namespace
{
    class Brake : public Item {
        CLONEOBJ(Brake);
    public:
        Brake() : Item("it-brake") {}

        void on_creation() {
            GridPos p = get_pos();
            KillItem(p); // kill myself
            SetStone(p, MakeStone("st-brake"));
        }

        string get_inventory_model() {
            return "st-brake";
        }

        ItemAction activate(Actor *, GridPos) {
            return ITEM_DROP_AS_STONE;
        }
    };
}


//----------------------------------------
// Explosion
//----------------------------------------
namespace
{
    class Explosion : public Item {
        CLONEOBJ(Explosion);
    public:
        enum Strength {
            WEAK,               // Explode but do nothing else.
            MEDIUM,             // Explode and leave a hole in the ground.
            STRONG              // Explode and shatter the floor.
        };
        Explosion(Strength st=WEAK);

    private:
        void init_model() {set_anim("expl");}
        bool actor_hit(Actor *actor) {
            SendMessage(actor, "shatter");
            return false;
        }
        void animcb();

        // Variables.
        Strength strength;
    };
}

Explosion::Explosion(Strength st)
: Item("it-explosion"), strength(st)
{}


void Explosion::animcb()
{
    switch (strength) {
    case WEAK:
        world::KillItem(get_pos());
        break;
    case MEDIUM:
	if (Floor *fl = GetFloor(get_pos()))
	    if (fl->is_destroyable())
                SetItem(get_pos(), MakeItem("it-hollow"));
        break;
    case STRONG:
	if (Floor *fl = GetFloor(get_pos()))
	    if (fl->is_destroyable())
		SetItem(get_pos(), MakeItem("it-debris"));
	
        break;
    }
}


//----------------------------------------
// Document.
//----------------------------------------

namespace
{
    class Document : public Item {
    public:
        Document() : Item("it-document") {
            set_attrib("text", "");
        }
    private:
        CLONEOBJ(Document);

        ItemAction activate(Actor */*a*/, GridPos)
        {
            string txt;
            if (string_attrib ("text", &txt))
                display::GetStatusBar()->show_text(txt);
            return ITEM_KILL;	       // remove from inventory
        }
        void message(const string &msg, const Value &/*val*/) {
            bool explode = false;

            if (msg == "ignite") {
                // dynamite does not blow Documents in Oxyd1
                explode = enigma::GameCompatibility != GAMET_OXYD1;
            }
            else if (msg == "expl" || msg == "bombstone")
                SetItem(get_pos(), MakeItem("it-explosion1"));
        }
        bool on_laserhit(Direction) {
            SetItem(get_pos(), MakeItem("it-explosion1"));
            return false;
        }
        const ObjectTraits *get_traits () const {
            static ObjectTraits traits("it-document", OBJTYPE_Item);
            return &traits;
        }
    };
}

//----------------------------------------
// Dynamite.
//----------------------------------------
namespace
{
    class Dynamite : public Item {
        CLONEOBJ(Dynamite);
    public:
        Dynamite() : Item("it-dynamite"), state(IDLE) {}
    private:
        enum State { IDLE, BURNING };
        State state;

        void change_state(State newstate) {
            if (newstate==BURNING && state==IDLE) {
                state = BURNING;
                set_anim("it-dynamite-burning");
            }
        }

        void explode () {
            GridPos p = get_pos();
            SendExplosionEffect(p, DYNAMITE);
            play_sound("explosion2");
            SetItem(p, new Explosion(Explosion::MEDIUM));
        }

        void animcb() { explode(); }
        void message(const string &msg, const Value &/*val*/) {
            if (msg == "ignite" || msg == "expl" || msg == "bombstone")
                change_state(BURNING);
            else if (msg == "explode") // currently unused in c++ code
                explode();
        }
        bool on_laserhit(Direction) {
            change_state(BURNING);
            return false;           // block light
        }
        void on_drop(Actor *) { change_state(BURNING); }
        bool actor_hit(Actor *a) {
            if (state == BURNING)
                return false;   // don't pick up burning dynamite

            return Item::actor_hit(a);
        }
        const ObjectTraits *get_traits () const {
            static ObjectTraits traits("it-dynamite", OBJTYPE_Item);
            return &traits;
        }
    };
}

// ----------------------------
//      BombBase
// ----------------------------
// base class for BlackBomb and WhiteBomb

namespace {
    class BombBase : public Item {
    public:
        BombBase(const char *kind) : Item(kind), m_burning(false) {}

    protected:
        virtual void message(const string &msg, const Value &) {
            if (msg == "ignite" || msg == "expl")
                burn();
            else if (msg == "explode")
                explode();
        }

    private:
        // Variables
        bool m_burning;

        // Private methods
        virtual void explode() = 0;

        void burn() {
            if (!m_burning) {
                m_burning = true;
                set_anim(burn_anim());
            }
        }

        void animcb() { explode (); }

        bool on_laserhit(Direction) {
            explode();
            return false;       // block light
        }
        bool actor_hit(Actor *) { return false; }

        void on_stonehit(Stone *st) {
            switch (enigma::GameCompatibility) {
                case GAMET_OXYD1:
                case GAMET_OXYDMAGNUM:
                    if (!st->is_kind("st-wood?")) // st-wood does not blow bombs
                        explode();
                    break;
                default :
                    explode();
                    break;
            }
        }

        virtual const char *burn_anim() const = 0;
    };
}

//----------------------------------------
// BlackBomb.
//----------------------------------------

/** \page it-blackbomb Black Bomb

When black bombs explode, they destroy the floor tile underneath them.

\image html it-blackbomb.png
*/

namespace
{
    class BlackBomb : public BombBase  {
        CLONEOBJ(BlackBomb);
    public:
        BlackBomb() : BombBase("it-blackbomb") {}
    private:
        const char *burn_anim() const { return "it-blackbomb-burning"; }
        void explode() {
            GridPos p = get_pos();
            SendExplosionEffect(p, BLACKBOMB);
            play_sound("explosion1");
            SetItem(p, MakeItem("it-explosion3"));
        }
        void message(const string &msg, const Value &val) {
            if (msg == "bombstone") {
                KillItem(get_pos());
            }
            BombBase::message(msg, val);
        }
    };
}

//----------------------------------------
// WhiteBomb.
//----------------------------------------

/** \page it-whitebomb White Bomb

When white bombs explode, they destroy the floor tile underneath them and
neighboring floors.

\image html it-whitebomb.png
*/

namespace
{
    class WhiteBomb : public BombBase  {
        CLONEOBJ(WhiteBomb);
    public:
        WhiteBomb() : BombBase("it-whitebomb") {}

    private:
        const char *burn_anim() const { return "it-whitebomb-burning"; }
        void explode() {
            GridPos p = get_pos();
            SendExplosionEffect(p, WHITEBOMB);
            play_sound("explosion1");
            SetItem(p, MakeItem("it-explosion3"));
        }

        // WhiteBomb does not react on message "bombstone" in Oxyd1
    };
}

//----------------------------------------
// Trigger.
//----------------------------------------
namespace
{
    class Trigger : public Item {
        CLONEOBJ(Trigger);
    public:
        Trigger();
    private:
        // Variables
        bool m_pressedp;
        int m_actorcount;

        // Methods
        void update_state();

        // Item interface
        void init_model();
        bool actor_hit (Actor *) { return false; }
        void actor_enter(Actor *) { m_actorcount += 1; update_state(); }
        void actor_leave(Actor *) { m_actorcount -= 1; update_state(); }
        void stone_change(Stone *st) { update_state(); }
    };
}

Trigger::Trigger() : Item("it-trigger")
{
    m_pressedp = false;
    m_actorcount = 0;
    set_attrib("invisible", 0.0);
}

void Trigger::init_model() 
{
    if (int_attrib("invisible"))
        set_model("invisible");
    else if (m_pressedp)
        set_model("it-trigger1");
    else
        set_model("it-trigger");
}

void Trigger::update_state()
{
    Stone *st = GetStone(get_pos());
    bool stone_pressure = st && !st->is_floating();
    bool pressedp = stone_pressure || (m_actorcount > 0);

    if (m_pressedp != pressedp) {
        m_pressedp = pressedp;

        if (m_pressedp) {
            world::PerformAction(this, true);
            play_sound("it-triggerdown");
        } else {
            world::PerformAction(this, false);
            play_sound("it-triggerup");
        }
        init_model();
    }
}


//----------------------------------------
// Seed
//----------------------------------------
namespace
{
    class Seed : public Item {
        CLONEOBJ(Seed);
    public:
        enum Type { WOOD, NOWOOD, VOLCANO };
        Seed( Type inittype=WOOD) : Item ("it-seed"), state(IDLE), type(inittype) {}

    private:
        enum State { IDLE, GROWING } state;
        enum Type type;

        bool actor_hit(Actor *a) {
            if (state==GROWING)
                return false;   // do not pickup growing seed
            return Item::actor_hit(a);
        }
        void on_drop(Actor *) { start_growing(); }
        void on_stonehit(Stone *) { start_growing(); }

        bool on_laserhit(Direction /*d*/) {
            if (state == IDLE)
                start_growing();

            return false;
        }

        void message(const string &msg, const Value &) {
            if (msg == "grow") {
                start_growing();
            }
        }

        void start_growing() {
            state = GROWING;
            set_anim("it-seed-growing");
        }

        void animcb() {
            KillStone(get_pos());
            Stone *st = world::MakeStone( get_stone_model());
            TransferObjectName(this, st);
            world::SetStone(get_pos(), st);
            world::KillItem(get_pos());
        }

        const char* get_stone_model() {
            const char *model = 0;

            switch( type) {
                case WOOD:    model = "st-wood-growing"; break;
                case NOWOOD:  model = "st-greenbrown-growing"; break;
                case VOLCANO: model = "st-volcano-growing"; break;
            }

            return model;
        }
    };
}


//----------------------------------------
// ShogunDot
//----------------------------------------

/** \page it-shogundot Shogun Dot

\subsection shogundota Attributes
- \b size:            1..3  (smallest..largest)
- \b target, \b action:   as usual
*/
namespace
{
    class ShogunDot : public Item {
        CLONEOBJ(ShogunDot);
    public:
        ShogunDot(int size=1);
    private:
        void set_size(int s) { set_attrib("size", s); }
        int get_size() const;

        void init_model();
        void message(const string &str, const Value &v);
        void stone_change(Stone *st);
        bool actor_hit(Actor */*a*/) { return false; }


        // Variables.
        bool activated;
    };
}

ShogunDot::ShogunDot(int size)
: Item("it-shogun"), activated(false)
{
    set_size(size);
}

int
ShogunDot::get_size() const
{
    int size = int_attrib("size");
    assert(1<=size && size<=3);
    return size;
}

void
ShogunDot::init_model()
{
    static const char *models[] = {"it-shogun1", "it-shogun2", "it-shogun3"};
    set_model(models[get_size()-1]);
}

void
ShogunDot::stone_change(Stone *st)
{
    if (activated) {
        if (st == 0) {
            warning("stone_change: Stone disappeared w/o sending me a proper message!");
        }
    }
    else {
        if (st) {               // some Stone on inactive ShogunDot
            // if ShogunDot was set _after_ ShogunStone during level startup,
            // the ShogunDot does not get activated.
            SendMessage(st, "renotify"); // ask stone to notify me (again)
        }
    }
}

void
ShogunDot::message(const string &str, const Value &/*v*/)
{
    int         size = get_size();
    const char *s    = str.c_str();

    if (activated && 0==strcmp(s, "noshogun")) {
        activated = false;
        world::PerformAction(this, false);
    }
    else {
        bool size_matches =
            (strncmp(s, "shogun", 6) == 0)    &&
            ((s[6]-'0')              == size) &&
            (s[7] == 0);

        if (size_matches != activated) {
            activated = size_matches;
            world::PerformAction(this, activated);
        }
    }
}

//----------------------------------------
// Magnet
//----------------------------------------
namespace
{

    class Magnet : public OnOffItem {
        class Magnet_FF : public world::ForceField {
        public:
            Magnet_FF() : m_active(false), strength(30) {}

            void set_pos(GridPos p) { center = p.center(); }

            V2 get_force(Actor */*a*/, V2 x, V2 /*v*/, double /*time*/)
            {
                if (!m_active)
                    return V2();
                V2 dv = center - x;
                double dist = length(dv);
                if (dist < 0.1)
                    return V2();
                else
                    return strength * dv/(dist*dist);
            }

            bool   m_active;
            V2     center;
            double strength;
        };


        CLONEOBJ(Magnet);
    public:
        Magnet() : OnOffItem ("it-magnet") {
            set_attrib("strength",30.0);
        }
    private:
        void init_model() {
            if (is_on())
                set_model("it-magnet-on");
            else
                set_model("it-magnet-off");
        }
        void on_creation() {
            ff.m_active = is_on();
            ff.set_pos(get_pos());
            world::AddForceField(&ff);
            Item::on_creation();
        }
        void on_removal() {
            world::RemoveForceField(&ff);
            Item::on_removal();
        }
        bool actor_hit(Actor */*a*/) { return false; }

        virtual void notify_onoff(bool on) {
            ff.m_active = on;
        }


        Magnet_FF ff;
    };
}

//----------------------------------------
// WormHole
//----------------------------------------

/** \page it-wormhole Worm hole

Worm holes teleport actors to another place.  They have a variable
attracting force field.

\subsection wormholea Attributes
- \b targetx:       X coordinate of the destination
- \b targety:       Y coordinate of the destination
- \b strength:      Strength of the force field (default: 50)
- \b range:         Range of the force field

\subsection wormholee Example
\verbatim
set_item("it-wormhole", 1,1, {targetx=5.5, targety=10.5, strength=50, range=5})
\endverbatim
*/

namespace
{
    class WormHole_FF : public world::ForceField {
    public:
        WormHole_FF() : strength(50), range(1000) {}

        void set_pos(GridPos p) { center = p.center(); }

        V2 get_force(Actor */*a*/, V2 x, V2 /*v*/, double /*time*/)
        {
            V2 b = center - x;
            double bb = length(b);
            if (bb>0 && bb < range) {
                return strength*b/(bb*bb);
            }
            return V2();
        }

        V2     center;          // Center of the force field
        double strength;        // Strength of the force
        double range;           // Range of the force
    };

    class WormHole : public Item, public enigma::TimeHandler
    {
        CLONEOBJ(WormHole);
    public:
        WormHole() : Item("it-wormhole") {
            set_attrib("targetx", 9.5);
            set_attrib("targety", 9.5);
            set_attrib("strength",50.0);
            set_attrib("range", 1000.0);
        }
    private:
        void on_creation() {
            Item::on_creation();
            ff.set_pos(get_pos());
            double_attrib("strength", &ff.strength);
            double_attrib("range", &ff.range);
            world::AddForceField(&ff);
        }
        void on_removal() {
            world::RemoveForceField(&ff);
            Item::on_removal();
        }

        bool actor_hit(Actor *a);

        V2 vec_to_center (V2 v) {return v-get_pos().center();}
        bool near_center_p (Actor *a) {
            return (length(vec_to_center(a->get_pos())) < 0.5/4);
        }

        // Variables.
        WormHole_FF ff;
    };
}

bool
WormHole::actor_hit(Actor *actor)
{
    if (near_center_p(actor)) {
        display::AddEffect (get_pos().center(), "ring-anim");
        double x=0, y=0;
        if (double_attrib("targetx", &x) && double_attrib("targety", &y)) {
            play_sound("warp");
            world::WarpActor(actor, x, y, false);
        }
    }
    return false;
}

//----------------------------------------
//      Vortex
//----------------------------------------

/** \page it-vortex Vortex

Vortexes teleport actors to another place.

They may be opened or closed. Is a vortex is closed, the actor cannot enter.

\subsection vortexa Attributes
- \b targetx:       X coordinate of the destination
- \b targety:       Y coordinate of the destination

\subsection vortexe Example
\verbatim
set_item("it-vortex-open", 1,1, {targetx=5.5, targety=10.5})
set_item("it-vortex-closed", 3,1, {targetx=7.5, targety=10.5})
\endverbatim

\subsection vortexm Messages
- \b open       opens the vortex
- \b close      closes the vortex
- \b signal     signal value: 1 -> "open"; 0 -> "close"
*/

namespace
{
    class Vortex : public Item, public TimeHandler
    {
        CLONEOBJ(Vortex);
    public:
        Vortex(bool opened)
        : Item("it-vortex"), state(opened ? OPEN : CLOSED), 
          close_after_warp(!opened)
        {
        }

    private:

        static const double RANGE = 0.5/2;

        bool actor_hit(Actor*);
        void init_model() {
            switch(state) {
            case OPEN:   set_model("it-vortex-open"  ); break;
            case CLOSED: set_model("it-vortex-closed"); break;
            case OPENING: set_anim("it-vortex-opening"); break;
            case CLOSING: set_anim("it-vortex-closing"); break;
            }
        }
        void animcb() {
            if (state == CLOSING) {
                state = CLOSED;
                init_model();
            }
            else if (state == OPENING) {
                state = OPEN;
                init_model();
            }
        }

        void alarm() {
            warp();
        }

        void message(const string &msg, const Value &/*val*/);

        V2 vec_to_center (V2 v) { return v-get_pos().center(); }
        bool near_center_p (Actor *a) {
            return length(vec_to_center(a->get_pos())) < RANGE;
        }

        V2 target() {
            double tx, ty;
            if (!double_attrib("targetx", &tx) || !double_attrib("targety", &ty)) {
                // no target attributes -> search for signal
                Object *target_obj = FindSignalDestination(this);
                V2 t;

                if (target_obj) {
                    GridObject *gridTarget = dynamic_cast<GridObject*>(target_obj);

                    if (gridTarget)
                        t = gridTarget->get_pos().center();
                    else {
                        target_obj->warning("signal target is not a GridObject");
                        t = get_pos().center(); // take own position
                    }
                }
                else {
                    warning("no target attributes and no signal found");
                    t = get_pos().center(); // take own position
                }
                set_target(t);
                return t;
            }
            return V2(tx, ty);
        }
        void set_target(V2 t) {
            set_attrib("targetx", t[0]);
            set_attrib("targety", t[1]);
        }

        void open() {
            if (state == CLOSED || state == CLOSING) {
                state = OPENING;
                init_model();
            }
        }
        void close() {
            if (state == OPEN || state == OPENING) {
                state = CLOSING;
                init_model();
            }
        }
        void openclose() {
            if (state == OPEN || state == OPENING) close();
            else                                   open();
        }

        void warp();            // warp swallowed actor(s)
        void warp_to(V2 t);     // warp() using alternate target

        // Variables
        enum State { OPEN, CLOSED, OPENING, CLOSING } state;
        bool close_after_warp;
    };
}

bool
Vortex::actor_hit(Actor *actor)
{
    if (state == OPEN && near_center_p(actor)) {
        SendMessage(actor, "fallvortex");
        // Note: actor sends message "warp" to this vortex
        // when fall-anim completed
    }
    return false;
}

void Vortex::warp_to(V2 t) {
//     warning("warp_to(%f, %f) called", t[0], t[1]);
    V2 old_target(target());
    set_target(t);
    warp();
    set_target(old_target);
}

// warps all actors in Range to target position
void Vortex::warp() {
//     warning("warp() called");
    vector<Actor*> actors;
    if (GetActorsInRange(get_pos().center(), RANGE, actors)) {
        Actor *actor = 0;
        if (actors.size() == 1) {
            actor = actors[0];
        }
        else {              // multiple actors in range
            warning("multiple actors in range - random selecting one of them");
            // @@@ FIXME:  search the one actor which completed its anim
            actor = actors[IntegerRand(0, actors.size()-1)]; // until fixed just take one
        }

        assert(actor);

        V2       v_target = target();
        GridPos  p_target(v_target);
        Item    *it       = GetItem(p_target);

        if (it && it->is_kind("it-vortex")) {
            Vortex *v = dynamic_cast<Vortex*>(it); // destination vortex
            assert(v);
            Stone *st = GetStone(p_target);
            if (st) {
//                 warning("Destination is blocked by %s", st->get_kind());
                warp_to(v->target()); // take target of target as destination
            }
            else {
                bool rewarp = true;

                if (v->state == OPEN) { // destination is open
                    play_sound("warp");
                    world::WarpActor(actor, v_target[0], v_target[1], false);
                    SendMessage(actor, "rise");
                    rewarp = false;

                    if (this != v && close_after_warp)
                        close(); // close source vortex
                }
                else if (v->state == CLOSED) { // destination is closed
                    SendMessage(v, "open");
                }

                if (rewarp) {
                    g_timer.set_alarm(this, 0.4, false); // call warp again after some time
                }
            }
        }
        else {
            world::WarpActor(actor, v_target[0], v_target[1], false);
            SendMessage(actor, "appear");
            if (close_after_warp)
                close();
        }
    }
    else {
        warning("message 'warp' received w/o actor in range");
    }
}

void Vortex::message(const string &msg, const Value &val) {
    // warning("received '%s'", msg.c_str());
    if (msg == "signal") {
        if (val.get_type() == Value::DOUBLE) {
            int value = static_cast<int>(val.get_double());
            // close_after_warp = false; (no - see Per.Oxyd #22)
            if (value) open();
            else close();
        }
    }
    else if (msg == "open")
        open();
    else if (msg == "close" || (msg == "arrival" && close_after_warp))
        close();
    else if (msg == "warp")
        warp();
}

//----------------------------------------
// YinYang item
//----------------------------------------
namespace
{
    class YinYang : public Item {
        CLONEOBJ(YinYang);
    public:
        YinYang() : Item("it-yinyang") {}

        string get_inventory_model() {
            if (player::CurrentPlayer()==0)
                return  "it-yinyang";
            else
                return "it-yanying";
        }

        ItemAction activate(Actor */*a*/, GridPos) {
            // Switch to other marble
            player::SwapPlayers();
            sound::PlaySound ("switch");
            return ITEM_KEEP;
        }
    };
}

//----------------------------------------
// Spade
//----------------------------------------
namespace
{
    class Spade : public Item {
        CLONEOBJ(Spade);
    public:
        Spade() : Item("it-spade") {}

        ItemAction activate(Actor *, GridPos p) {
            if (Item *it=GetItem(p)) {
                SendMessage(it, "shovel");
                return ITEM_KEEP;
            }
            return ITEM_DROP;
        }
    };
}

//----------------------------------------
// Pipes
//----------------------------------------
namespace
{
    class Pipe : public Item {
        CLONEOBJ(Pipe);
        ItemID m_id;
    public:
        Pipe(const char *kind, ItemID id) : Item(kind), m_id(id) {}
        int get_id() const { return m_id; }

        void message(const string &msg, const Value &/*val*/) {
            if (msg == "expl")
                SetItem (get_pos(), MakeItem("it-explosion1"));
        }
    };
}

//----------------------------------------
// Pullers
//----------------------------------------
namespace
{
    class Puller : public Item {
    public:
        Puller(Direction dir=SOUTH) : Item ("it-puller"),active(false)
	{ set_orientation(dir); }
    private:
        bool active;
        CLONEOBJ (Puller);

        string get_inventory_model() {
            string mname = get_kind();
            mname += to_suffix(get_orientation());
            return mname;
        }
        void init_model() {
            set_model(get_inventory_model());
        }

        bool actor_hit(Actor *a) {
            if (active)
                return false;
            return Item::actor_hit(a);
        }

        void on_drop(Actor */*a*/) { activate(); }

        void activate() {
            active=true;
            set_anim("it-puller-active");
            play_sound("puller");
        }
        void animcb() {
            Direction dir      = get_orientation();
            GridPos   stonepos = move(get_pos(), reverse(dir));

            send_impulse(stonepos, dir);
            play_sound("explosion2");
            SetItem(get_pos(), new Explosion(Explosion::WEAK));
        }

	Direction get_orientation() const {
	    return Direction(int_attrib("orientation"));
	}
        void set_orientation(Direction dir) {
            set_attrib("orientation", Value(dir));
        }
    };
}

//----------------------------------------
// Cracks
//----------------------------------------
namespace
{
    class Crack : public Item {
        CLONEOBJ(Crack);
    public:
        Crack(int type=0) : Item("it-crack"), state(IDLE), anim_end(false)
	{
	    set_attrib("type", type);
	    set_attrib("fixed", 0.0);
	}
    private:
        enum State { IDLE, CRACKING1, CRACKING2 } state;
        bool anim_end;

        int get_type() const { return int_attrib("type"); }
	bool is_fixed() const { return int_attrib("fixed"); }

        void init_model() {
            if (int t=get_type()) {
                if (t > 3) {
                    state = CRACKING1;
                    set_anim("it-crack_anim1");
                //SetItem(get_pos(), MakeItem("it-debris"));
                }else {
                    char modelname[20];
                    sprintf(modelname, "it-crack%d", t);
                    set_model(modelname);
                }
            }
            else
                set_model("invisible");
        }
        void animcb() {
            if (state == CRACKING2) {
                GridPos p= get_pos();
                SetFloor(p, MakeFloor("fl-abyss"));
                KillItem(p);
            } else {
                state = CRACKING2;
                set_anim("it-crack_anim2");
            }
        }

        void crack(const GridPos &p) {
            if (Floor *fl = GetFloor(p)) {
                string k = fl->get_kind();
                if (fl->is_destroyable()) {
                    if (Item *it = GetItem(p))
                        SendMessage (it, "crack");
                    else if (do_crack())
                        SetItem(p, MakeItem("it-crack"));
                }
            }
        }

        void actor_enter(Actor *a) {
            if (a->is_on_floor()) {
                SendMessage(this, "crack");

                if (get_type() <= 3) {
                    GridPos p = get_pos();
                    crack (move(p, NORTH));
                    crack (move(p, EAST));
                    crack (move(p, SOUTH));
                    crack (move(p, WEST));
                }
            }
        }
        bool actor_hit(Actor *a) {
            if (anim_end)
                SendMessage(a, "fall");
            return false;
        }
        void message(const string &msg, const Value &/*val*/) {
            if (msg == "crack" && state==IDLE && !is_fixed()) {
                int type = get_type();
                if ((type == 0 && do_crack()) || (type > 0)) {
                    set_attrib("type", Value(int_attrib("type") + 1));
                    play_sound("crack");
                    init_model();
                }
            }
        }

        bool do_crack() {
	    if (!is_fixed()) {
		double brittleness = enigma::Brittleness;
		double_attrib ("brittleness", &brittleness);
		double rnd = DoubleRand(0, 1);
		return rnd < brittleness;
	    }
	    else
		return false;
        }
    };

    class Debris : public Item {
        CLONEOBJ(Debris);
        bool actor_hit(Actor *a) { SendMessage(a, "fall"); return false; }
        void init_model() {set_anim("it-debris");}
        void animcb() {
            GridPos p = get_pos();
            SetFloor(p, MakeFloor("fl-abyss"));
            KillItem(p);
        }
    public:
        Debris():Item("it-debris") {}
    };
}

//----------------------------------------
// Burning floors
//
// This items can burn. The fire spreads and destroys items and actors.
//----------------------------------------
namespace
{
    class Burnable : public Item {
	CLONEOBJ(Burnable);
    public:
	enum State { IDLE, IGNITE, BURNING, FIREPROOF, ASH };
	Burnable( State initstate=IDLE): Item("it-burnable") {
	    state = initstate;
	}
    private:
	State state;

	void message(const string &msg, const Value &) {
	    if ((msg == "trigger" || msg == "ignite") && state==IDLE) {
		state = IGNITE; // start burning
		init_model();
	    } else if (msg == "extinguish") {   // stop / never start burning
		state = FIREPROOF;
		init_model();
	    } else if (msg == "brush" && (state == ASH || state == FIREPROOF)) {
		KillItem(get_pos());    // The brush will clean the floor
	    }
	}

	void animcb() {
	    GridPos p = get_pos();
	    if (state == IGNITE || state == BURNING) {
		bool spread = true;
		if( Stone *st = GetStone( p)) {
		    if( ! st->is_floating())
			spread = false; // only hollow stones allow the fire to spread

		    string model = st->get_kind();
		    if( model == "st-wood1" || model == "st-wood2") {
			KillStone( p); // The fire has burnt away the wooden stone
			spread = true;
		    }
		}

		// spread to neighbouring tiles
		if( spread) {
		    if( DoubleRand(0, 1) > 0.3) ignite (move(p, NORTH));
		    if( DoubleRand(0, 1) > 0.3) ignite (move(p, EAST));
		    if( DoubleRand(0, 1) > 0.3) ignite (move(p, SOUTH));
		    if( DoubleRand(0, 1) > 0.3) ignite (move(p, WEST));
		}
	    }
	    if (state == IGNITE) {
		if( Floor *fl = GetFloor( p)) { // The fire has burnt away the wooden floor
		    string model = fl->get_kind();
		    if( model == "fl-wood" || model == "fl-stwood")
			SetFloor( p, MakeFloor("fl-abyss"));
		}
		state = BURNING;
		init_model();
	    } else if (state == BURNING) {  // stop burning after some random time
		if( DoubleRand(0, 1) > 0.7)
		    state = ASH;
		else
		    state = BURNING;

		init_model();
	    }
	}
	void init_model() {
	    switch (state) {
	    case IDLE:      set_model("invisible"); break;
	    case IGNITE:    set_anim("it-burnable_ignite"); break;
	    case BURNING:   set_anim("it-burnable_burning"); break;
	    case FIREPROOF: set_model("it-burnable_fireproof"); break;
	    case ASH:       set_model("it-burnable_ash"); break;
	    }
	}

	bool actor_hit(Actor *a) {
	    if (state == IGNITE || state == BURNING)
		SendMessage(a, "shatter");
	    return false;
	}

	void ignite( GridPos p) {
	    bool only_ignite = false;

	    if( Stone *st = GetStone( p)) {
		if( ! st->is_floating() && ! st->is_movable())
		    return; // Stone does not allow to ignite.
		if( st->is_movable())
		    only_ignite = true; // only ignit burnable items
	    }

	    if (Item *it = GetItem(p)) {    // spread to other items
		string model = it->get_kind();
		// ignite burnable items
		if( model=="it-burnable" || model=="it-dynamite"
		    || model=="it-blackbomb" || model=="it-whitebomb")
		    SendMessage (it, "ignite");
		// cracks are not strong enought
		else if( model=="it-crack")
		    SetItem( p, MakeItem("it-debris"));
		// all other items except some fire-proof ones will burn
		else if (model != "it-extinguisher" && model != "it-hill" && model != "it-hollow"
			 && model != "it-tinyhill" && model != "it-tinyhollow") {
		    SetItem( p, MakeItem("it-burnable-ignited"));
		    //SendMessage (it, "ignite");
		}
	    } else if (!only_ignite) {    // spread on the same floor (if stone allows)
		if( Floor *fl1 = GetFloor( get_pos())) {
		    string model1 = fl1->get_kind();
		    if( Floor *fl2 = GetFloor( p)) {
			string model2 = fl2->get_kind();
			if( model1 == model2)
			    SetItem( p, MakeItem("it-burnable"));
		    }
		}
	    }
	}
    };
}

//----------------------------------------
// Fire Extinguisher
//
// This items can extinguish burning floor.
//----------------------------------------
namespace
{
    class Extinguisher : public Item {
        CLONEOBJ(Extinguisher);
    public:
        Extinguisher( int load=2): Item("it-extinguisher") {
    	    set_attrib("load", load);
        }
    private:
        int get_load() const { return int_attrib("load"); }
        void set_load( int load) { set_attrib("load", load); }

        void extinguish( GridPos p) {
            if (Item *it = GetItem(p)) {
                SendMessage (it, "extinguish");
            } else {
                SetItem( p, MakeItem("it-burnable-fireproof"));
            }
        }

        //
        // Item interface
        //

        ItemAction activate(Actor *a, GridPos p);

        void init_model() {
            set_model (get_inventory_model());
        }

        string get_inventory_model() {
            switch (get_load()) {
            case 0:  return("it-extinguisher_empty"); break;
            case 1:  return("it-extinguisher_medium"); break;
            default: return("it-extinguisher_full"); break;
            }
        }
    };
}

ItemAction 
Extinguisher::activate(Actor *a, GridPos p) 
{
    if (get_load() > 0) {
        extinguish (p);
        extinguish (move(p, NORTH));
        extinguish (move(p, SOUTH));
        extinguish (move(p, EAST));
        extinguish (move(p, WEST));
        if (get_load() > 1) {
            // full extinguisher has a larger range
            extinguish (move(move(p, NORTH),NORTH));
            extinguish (move(move(p, NORTH),EAST));
            extinguish (move(move(p, SOUTH),SOUTH));
            extinguish (move(move(p, SOUTH),WEST));
            extinguish (move(move(p, EAST),EAST));
            extinguish (move(move(p, EAST),SOUTH));
            extinguish (move(move(p, WEST),WEST));
            extinguish (move(move(p, WEST),NORTH));
        }
        set_load( get_load() - 1);

        // ### HACK ###
        // update the player's inventory
        if (player::Inventory *inv = player::GetInventory(a)) {
            inv->redraw();
        }
    }
    return ITEM_DROP;
}



//----------------------------------------
// Flags
//
// Flags can be used to set a new respawn point for the black or white
// marble.
//----------------------------------------
namespace
{
    class FlagBlack : public Item {
        CLONEOBJ(FlagBlack);
        void on_drop(Actor *) {
            player::SetRespawnPositions(get_pos(), true);
        }
        void on_pickup(Actor *) {
            player::RemoveRespawnPositions(true);
        }

    public:
        FlagBlack(): Item("it-flagblack") {}
    };

    class FlagWhite : public Item {
        CLONEOBJ(FlagWhite);
        void on_drop(Actor *) {
            player::SetRespawnPositions(get_pos(), false);
        }
        void on_pickup(Actor *) {
            player::RemoveRespawnPositions(false);
        }

    public:
        FlagWhite(): Item("it-flagwhite") {}
    };
}

//----------------------------------------
// Blocker
//
// If a 'BolderStone' moves over a 'Blocker' the 'Blocker' starts growing
// and replaces itself by a BlockerStone.
//
//----------------------------------------

namespace
{
    static const char * const stateName[] = { "IDLE", "SHRINKED", "BOLDERED", "COVERED" };

    class Blocker : public Item, public TimeHandler {
        CLONEOBJ(Blocker);
    public:
        Blocker(bool shrinked_recently)
        : Item("it-blocker"),
          state(shrinked_recently ? SHRINKED : IDLE)
        {}

    private:
        enum State {
            IDLE,               // nothing special
            SHRINKED,           // recently shrinked by a BolderStone (which has not arrived yet)
            BOLDERED,           // BolderStone has arrived (the one that made me shrink)
            COVERED             // BolderStone will make me grow (but is on top of me)
            // Note: BOLDERED and COVERED are used for all kinds of stones
            // (e.g. when a stone is on top and the Blocker receives a "close" message)
        } state;

        void change_state(State new_state) {
            if (state != new_state) {
                if (state == SHRINKED) {
                    g_timer.remove_alarm(this);
                }
                else if (new_state == SHRINKED) {
                    g_timer.set_alarm(this, 0.5, false);
                }

                state = new_state;
            }
        }

        void on_creation() {
            if (state == SHRINKED) {
                g_timer.set_alarm(this, 0.5, false);
            }
            Item::on_creation();
        }
        void on_removal() {
            change_state(IDLE);
            Item::on_removal();
        }

        void message(const string &msg, const Value &val);

        void stone_change(Stone *st);

        void grow() {
            Stone *st = world::MakeStone("st-blocker-growing");
            world::SetStone(get_pos(), st);
            TransferObjectName(this, st);
            world::KillItem(get_pos());
        }

        void alarm() {
            if (state == SHRINKED) { // BolderStone did not arrive in time
                change_state(IDLE);
            }
        }

        bool actor_hit(Actor */*a*/) { return false; }

    };

};

void 
Blocker::message(const string &msg, const Value &val) 
{
    if (msg == "trigger" || msg == "openclose") {
        switch (state) {
        case IDLE:
        case SHRINKED:
            grow(); // if no stone on top -> grow
            break;

            // if stone on top -> toggle state (has no effect until stone leaves)
        case BOLDERED:
            change_state(COVERED);
            break;
        case COVERED:
            change_state(BOLDERED);
            break;
        }
    }
    else {
        int open = -1;

        if (msg == "signal") {
            if (val.get_type() == Value::DOUBLE) {
                // val: 1 means "shrink", 0 means "grow"
                open = static_cast<int>(val.get_double());
                warning("received signal %i", open);
            }
            else {
                assert(0);
            }
        }
        else if (msg == "open")
            open = 1;
        else if (msg == "close")
            open = 0;

        if (open == 1)  { // shrink
            if (state == COVERED)
                change_state(BOLDERED);
        }
        else { // grow
            if (state == BOLDERED)
                change_state(COVERED);
            else if (state == SHRINKED)
                change_state(IDLE); // remove alarm

            if (state == IDLE) {
                if (Stone *st = GetStone(get_pos())) {
                    if (st->is_kind("st-bolder"))
                        change_state(BOLDERED); // occurs in Per.Oxyd #84
                    else
                        change_state(COVERED);
                }
                else {
                    grow();
                }
            }
        }
    }
}

void Blocker::stone_change(Stone *st) 
{
    if (st) {
        if (st->is_kind("st-bolder")) { // bolder arrived
            switch (state) {
            case IDLE:
                change_state(COVERED);
                break;
            case SHRINKED:
                change_state(BOLDERED);
                break;
            case COVERED:
            case BOLDERED:
                // two BolderStones running directly next to each other
                // let second pass as well (correct? siegfried says yes)
                break;
            }
        }
        else { // any other stone
            change_state(BOLDERED);
        }
    }
    else {              // stone disappeared
        switch (state) {
        case BOLDERED:
            change_state(IDLE);
            break;
        case COVERED:
            grow();
            break;
        case IDLE:
        case SHRINKED:
            // no action
            break;
        }
    }
}



//----------------------------------------
// Soother
//----------------------------------------

namespace
{
    class Soother : public Item {
    	CLONEOBJ(Soother);
    public:
    	Soother() : Item ("it-soother") {}

        ItemAction activate(Actor *a, GridPos) {
            play_sound("warp");
            world::RespawnActor(a);
            return ITEM_KILL;
        }
    };

}

//----------------------------------------
// Bridge item (for Oxyd compatibility)
//
// Floor tiles seem to be static in Oxyd and cannot change dynamically
// or be animated.  For this reason, every bridge floor in Oxyd has to
// be combined with a bridge "item" that receives the signals, shows
// the animation and sets or removes the floor.
//----------------------------------------
namespace
{
    class OxydBridge : public Item {
        CLONEOBJ(OxydBridge);
    public:
        OxydBridge() : Item ("it-bridge-oxyd") {}

        void message(const string& msg, const Value &val) {
            if (msg == "signal") {
                int ival = to_int (val);
                printf ("value %d\n", ival);
                Floor *floor = GetFloor (get_pos());
                if (ival > 0)
                    SendMessage (floor, "close");
                else
                    SendMessage (floor, "open");
            }
        }
        bool actor_hit(Actor */*a*/) { return false; }
    };
}

//----------------------------------------
// Sensors
//----------------------------------------
namespace
{
    class Sensor : public Item {
        CLONEOBJ(Sensor);
    public:
        Sensor() : Item ("it-sensor") {}

        bool actor_hit (Actor *) {
            PerformAction (this, true);
            return false;
        }
    };

    class InverseSensor : public Item {
        CLONEOBJ(InverseSensor);
    public:
        InverseSensor() : Item ("it-inversesensor") {}

        bool actor_hit (Actor *) {
            PerformAction (this, false);
            return false;
        }
    };
}

// ----------------------
//      SignalFilters
// ----------------------
namespace {
    class SignalFilterItem : public Item {
        CLONEOBJ(SignalFilterItem);
    public:
        SignalFilterItem(const char *kind, int type_) : Item(kind), type(type_) {
            assert(type >= 0 && type <= 1);
        }
    private:

        void init_model() { set_model("invisible"); }

        void message(const string& m, const Value& val) {
            if (m == "signal") {
                int value = to_int(val);
//                 warning("received signal with value %i", value);
                if (value)
                    PerformAction(this, type);
            }
        }

        // type of signal filter
        // 0 : receive 1 -> send 0
        // 1 : receive 1 -> send 1
        int type;
    };
}

// ---------------------
//      EasyModeItem
// ---------------------
/** \page it-easymode Easy-Mode Item

In easy game mode this item kills the stone at it's position.
Then in both modes it kills itself.

E.g. it can be used to hide walls in easy game mode.

\subsection easye Example
\verbatim
set_item("it-easymode", 10,10)
\endverbatim

\ref st-easymode
*/


namespace
{
    class EasyModeItem : public Item {
        CLONEOBJ(EasyModeItem);
    public:
        EasyModeItem() : Item("it-easymode") {}

        void message(const string& m, const Value& ) {
            if (m == "init") {
                // does not work in on_creation() because items are created
                // before stones are created.
                if (options::Difficulty == DIFFICULTY_EASY) {
                    KillStone(get_pos());
                }
                KillItem(get_pos());
            }
        }
    };
}

// ----------------
//      Glasses
// ----------------

namespace
{
    class Glasses : public Item {
        CLONEOBJ(Glasses);
    public:
        Glasses() : Item("it-glasses") {}
    private:
        static bool wears_glasses(Actor *a) {
            return player::GetInventory(a)->find("it-glasses") != -1;
        }

        void on_drop(Actor *a) {
            if (!wears_glasses(a)) // 'this' was the only it-glasses
                BroadcastMessage("glasses", 0.0, GRID_STONES_BIT);
        }
        void on_pickup(Actor *a) {
            if (!wears_glasses(a)) // no glasses before
                BroadcastMessage("glasses", 1.0, GRID_STONES_BIT);
        }
        void on_stonehit(Stone */*st*/) {
            play_sound("shatter");
            SetItem(get_pos(), MakeItem("it-glasses-broken"));
        }
    };

    class BrokenGlasses : public Item {
        CLONEOBJ(BrokenGlasses);
    public:
        BrokenGlasses() : Item("it-glasses-broken") {}
    private:
    };
}

//----------------------------------------
// Invisible Abyss
//----------------------------------------
namespace
{
    class InvisibleAbyss : public Item {
        CLONEOBJ(InvisibleAbyss);
    public:
        InvisibleAbyss() : Item("it-abyss") {}

    private:
        bool actor_hit(Actor *a) {
            SendMessage(a, "fall");
            return false;
        }

        void init_model() { set_model("invisible"); }

    };
}

//----------------------------------------
// Landmine
//----------------------------------------
namespace
{
    class Landmine : public Item {
        CLONEOBJ(Landmine);
    public:
        Landmine() : Item("it-landmine") {}

    private:

        bool actor_hit(Actor *a) {
            const double ITEM_RADIUS = 0.3;
            px::V2 item_center(get_pos().x + 0.5, get_pos().y + 0.5);
            double dist = length(a->get_pos()-item_center);
            if (dist < ITEM_RADIUS) {
                SetItem(get_pos(), MakeItem("it-explosion2"));
            }
            return false;
        }

        void on_stonehit(Stone */*st*/) {
            SetItem(get_pos(), MakeItem("it-explosion2"));
        }
    };
}

//----------------------------------------
// Remaining items (still need to be implemented)
//----------------------------------------
namespace
{
    class Rubberband : public Item {
    	SINGLETONOBJ(Rubberband);
    public:
    	Rubberband() : Item ("it-rubberband") {}
    };

    class HStrip : public Item {
        CLONEOBJ(HStrip);
    public:
        HStrip() : Item ("it-hstrip") {
        }
        bool actor_hit(Actor *a) {
            double ycenter = get_pos().y + 0.5;
            const double MAXDIST = 6.0/32;
            if (fabs(a->get_pos()[1] - ycenter) > MAXDIST) {
                if (Floor *fl = GetFloor(get_pos()))
                    fl->actor_contact(a);
            }
            return false;
        }

        bool covers_floor() const { return true; }
    };

    class VStrip : public Item {
        CLONEOBJ(VStrip);
    public:
        VStrip() : Item ("it-vstrip") {
        }
        bool actor_hit(Actor *a) {
            double xcenter = get_pos().x + 0.5;
            const double MAXDIST = 5.0/32;
            if (fabs(a->get_pos()[0] - xcenter) > MAXDIST) {
                if (Floor *fl = GetFloor(get_pos()))
                    fl->actor_contact(a);
            }
            return false;
        }

        bool covers_floor() const { return true; }
    };

}


void items::Init()
{
    using world::Register;

    Register(new BlackBomb);
    Register("it-blocker-new", new Blocker(true));
    Register(new Blocker(false));
    Register(new Brake);
    Register(new Brush);
    Register(new Burnable);
    Register("it-burnable-ignited", new Burnable(Burnable::IGNITE));
    Register("it-burnable-fireproof", new Burnable(Burnable::FIREPROOF));
    Register("it-burnable-ash", new Burnable(Burnable::ASH));
    Register(new Cherry);
    Register(new Coffee);
    Register(new Coin);
    Register("it-coin1", new Coin);
    Register("it-coin2", new Coin(2));
    Register("it-coin4", new Coin(4));
    Register(new Crack);
    Register("it-crack0", new Crack(0));
    Register("it-crack1", new Crack(1));
    Register("it-crack2", new Crack(2));
    Register("it-crack3", new Crack(3));
    Register(new Debris);
    Register(new Document);
    Register(new Dummyitem);
    Register(new Dynamite);
    Register("it-explosion1", new Explosion(Explosion::WEAK));
    Register("it-explosion2", new Explosion(Explosion::MEDIUM));
    Register("it-explosion3", new Explosion(Explosion::STRONG));
    Register(new Extinguisher);
    Register("it-extinguisher-full", new Extinguisher(2));
    Register("it-extinguisher-empty", new Extinguisher(0));
    Register(new ExtraLife);
    Register(new EasyModeItem);
    Register(new FlagBlack);
    Register(new FlagWhite);
    Register(new Floppy);
    Register(new Glasses);
    Register(new BrokenGlasses);
    Register(new Hammer);
    Register(new Hill);
    Register(new Hollow);
    Register(new HStrip);
    Register(new InverseSensor);
    Register(new InvisibleAbyss);
    Register(new Key);
    Register(new Key_a);
    Register(new Key_b);
    Register(new Key_c);
    Register(new Landmine);
    Register(new MagicWand);
    Register(new Magnet);
    Register(new Odometer);
    Register(new OxydBridge);
    Register(new Pin);

    Register(new Pipe ("it-pipe-e", IT_PIPE_E));
    Register(new Pipe ("it-pipe-w", IT_PIPE_W));
    Register(new Pipe ("it-pipe-s", IT_PIPE_S));
    Register(new Pipe ("it-pipe-n", IT_PIPE_N));
    Register(new Pipe ("it-pipe-es", IT_PIPE_ES));
    Register(new Pipe ("it-pipe-ne", IT_PIPE_NE));
    Register(new Pipe ("it-pipe-sw", IT_PIPE_SW));
    Register(new Pipe ("it-pipe-wn", IT_PIPE_WN));
    Register(new Pipe ("it-pipe-h", IT_PIPE_H));
    Register(new Pipe ("it-pipe-v", IT_PIPE_V));

    Register(new Puller);
    Register("it-puller-n", new Puller(NORTH));
    Register("it-puller-e", new Puller(EAST));
    Register("it-puller-s", new Puller(SOUTH));
    Register("it-puller-w", new Puller(WEST));
    Register(new Seed);
    Register("it-seed_wood", new Seed(Seed::WOOD));
    Register("it-seed_nowood", new Seed(Seed::NOWOOD));
    Register("it-seed_volcano", new Seed(Seed::VOLCANO));
    Register(new Sensor);
    Register(new ShogunDot);
    Register ("it-shogun-s", new ShogunDot(1));
    Register ("it-shogun-m", new ShogunDot(2));
    Register ("it-shogun-l", new ShogunDot(3));
    Register(new SignalFilterItem("it-signal-filter0", 0));
    Register(new SignalFilterItem("it-signal-filter1", 1));
    Register(new Soother);
    Register(new Spade);
    Register(new Spring1);
    Register(new Spring2);
    Register(new Springboard);
    Register(new Sword);
    Register(new TinyHill);
    Register(new TinyHollow);
    Register(new Trigger);
    Register(new Umbrella);
    Register ("it-vortex-closed", new Vortex(false));
    Register ("it-vortex-open", new Vortex(true));
    Register(new VStrip);
    Register(new Weight);
    Register(new WhiteBomb);
    Register(new Wrench);
    Register(new WormHole);
    Register(new YinYang);
}
