/*
 * Copyright (C) 2002,2003 Daniel Heck
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: player.hh,v 1.15 2003/08/16 23:56:57 reallysoft Exp $
 */
#ifndef PLAYER_HH
#define PLAYER_HH

/*
 * Player management.  Deals mostly with inventory management,
 * switching between players, respawning, etc.  Players are numbered
 * from 0..N-1, where N is the number of players in the current game
 * (currently always 1 or 2).
 */

#include "fwd.hh"
#include "world.hh"
#include <vector>

namespace player
{
    using world::Item;
    using world::Actor;

    class Inventory {
    public:
        Inventory();
        ~Inventory();
        void add_item(Item *i);
        int size() { return items.size(); } // number of items
        void clear();

        bool is_full() const { return items.size() == max_items; }

        void  rotate(int dir=1);
        void  activate_first();
        Item *get_item(int idx);
        Item *yield_item(int idx);
        Item *yield_first();

        int find(const std::string& kind, int start_idx = 0);

        void redraw();

    private:
        // Private methods.
        Item *remove_item(Item *it);

        // Private variables.
        static const unsigned max_items;
        std::vector<Item*> items;
    };

    /* Start a new game for N players.  Every player begins with two
       extra lifes. if 'add_yinyang' is true every player additionally
       gets one it-yinyang. */
    void NewGame(int nplayers, bool add_yinyang);

    /* This is called whenever a new level is reached in a running
       game.  The inventories of all players are cleaned up, i.e., all
       items except for extra lifes are removed. */
    void NewWorld();

    /* Call this function as soon as the current level is finished; it
       will remove the actors of all players from the level. */
    void LevelFinished();

    /* This function is only used by the YinYang items to exchange the
       two players. */
    void SwapPlayers();

    // set/remove respawn positions for all black or all white actors
    // (used when it-flagwhite/black is dropped)
    void SetRespawnPositions(enigma::GridPos pos, bool black);
    void RemoveRespawnPositions(bool black);

    int      CurrentPlayer();
    void     SetCurrentPlayer(unsigned iplayer);
    unsigned NumberOfRealPlayers();
    bool     IsCurrentPlayer(Actor *a);

    Inventory *GetInventory(int iplayer);
    Inventory *GetInventory(Actor *a);
    Item      *wielded_item (Actor *a);
    bool       wielded_item_is(Actor *a, const std::string &kind);

    void Suicide();

    void AddActor(int iplayer, Actor *a);

    bool     AllActorsDead();
    unsigned CountActorsOfKind(const char *kind);

    void InhibitPickup (bool yesno);
    void PickupItem(Actor *a, enigma::GridPos p);
    void PickupStoneAsItem(Actor *a, enigma::GridPos p);
    void RotateInventory(int dir=1);
    void ActivateItem();

    void Tick (double dtime);

    // move counter
    void InitMoveCounter();
    int  IncMoveCounter(int increment = 1);
    int  GetMoveCounter();
}
#endif
