/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libmac.so
 *
 *   File: disks.c
 */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "mac_plugin.h"

/*
 *  Function:  create_mac_disk_private_data
 *
 *  We keep private data about every logical disk object
 *  that has mac metadata ... by hanging our private
 *  data off the logical disk storage object ... that
 *  doesn't belong to us!
 *
 *  This routine is called to create private data for the
 *  specified logical disk.
 */
int create_mac_disk_private_data( LOGICALDISK *ld )
{
        int rc=0;
        disk_private_data_t *disk_pdata;


        LOG_ENTRY();

        REQUIRE(ld != NULL);

        if ( get_mac_disk_private_data(ld ) == NULL ) {

                disk_pdata = (disk_private_data_t *) calloc(1, sizeof(disk_private_data_t) );

                if ( disk_pdata ) {

                        disk_pdata->signature          = MAC_DISK_PDATA_SIGNATURE;
                        disk_pdata->vsectors_per_block = ld->geometry.bytes_per_sector >> EVMS_VSECTOR_SIZE_SHIFT;
                        ld->consuming_private_data = disk_pdata;

                }
                else {
                        rc = ENOMEM;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  delete_mac_disk_private_data
 *
 *  This routine is called to delete our private data for the
 *  specified logical disk.
 */
int delete_mac_disk_private_data( LOGICALDISK *ld )
{
        int rc=EINVAL;
        disk_private_data_t *disk_pdata=NULL;


        LOG_ENTRY();

        REQUIRE(ld != NULL);

        disk_pdata = (disk_private_data_t *) ld->consuming_private_data;

        if ( disk_pdata ) {

                if (disk_pdata->signature == MAC_DISK_PDATA_SIGNATURE) {
                        free(disk_pdata);
                        ld->consuming_private_data=NULL;
                        rc = 0;
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function:  get_mac_disk_private_data
 *
 *  This routine is called to obtain a ptr to the private
 *  data we keep for the specified logical disk.
 */
disk_private_data_t * get_mac_disk_private_data( LOGICALDISK *ld )
{
        disk_private_data_t  *disk_pdata = NULL;

        if (ld) {

                disk_pdata = (disk_private_data_t *) ld->consuming_private_data;

                if ( disk_pdata ) {

                        if (disk_pdata->signature != MAC_DISK_PDATA_SIGNATURE) {
                                disk_pdata = NULL;
                        }

                }

        }

        return disk_pdata;
}


/*
 *  Function:  delete_all_mac_disk_private_data
 *
 *  This routine is called to delete -ALL- copies of disk
 *  private data we may be keeping.  It is only called by our
 *  plug-in cleanup api.
 */
void  delete_all_mac_disk_private_data( void )
{
        int rc;
        list_anchor_t disk_list;
        disk_private_data_t *disk_pdata; 
        LOGICALDISK *ld=NULL;
        list_element_t iter;

	disk_list = EngFncs->allocate_list();
        if (disk_list) {

                rc = EngFncs->get_object_list( DISK,
                                               DATA_TYPE,
                                               NULL,
                                               NULL,
                                               NOT_MUST_BE_TOP,
                                               &disk_list );
                if (!rc) {

                        LIST_FOR_EACH( disk_list, iter, ld ) {
                                disk_pdata = get_mac_disk_private_data(ld);
                                if (disk_pdata) {
                                        delete_mac_disk_private_data(ld);                                
                                }
                        }

                }

        }

}


