/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

typedef int (*dialog_event_handler)(void *dialog, int key);
typedef void (*dialog_show_func)(void *dialog);
typedef void (*dialog_delete_func)(void *dialog);
typedef void (*dialog_delete_cb)(void *dialog);

/**
 *	struct dialog_list - maintains info on a chain/sequence of dialog windows
 *	@list: the chain of dialog windows in the order they are to be presented
 *	@current: corresponds to the current dialog being processed
 *
 *	This structure maintains information necessary to navigate through a
 *	series of multiple dialog windows. The current link element corresponds
 *	to the current window being displayed. The process_dialog_list_events()
 *	is responsible for processing the events for dialogs in the list. When
 *	current gets set to NULL, this indicates to the event handler to stop
 *	processing all windows.
 */
struct dialog_list {
	GList *list;
	GList *current;
};

/**
 *	struct dialog_window - the base class for any dialog window
 *	@panel: the panel id for the dialog window
 *	@win: the window id of the dialog
 *	@status: the dialog status, see DLG_STATUS_ possibilities below
 *	@window_list: address of struct containing info on the chain of dialogs we are in
 *	@menu: the button menu at the bottom of the dialog window
 *	@user_data: misc stuff the caller wants to keep with window
 *	@handler: the handler responsible for processing events for this dialog
 *	@show_func: the function responsible for drawing additional window elements
 *	@delete_func: the function responsible for cleaning up after a dialog
 *	@delete_cb: the user supplied event handler to call prior to calling delete_func
 *	@help_button: the menu item for the help button
 *	@next_button: the menu item for the next button
 *	@prev_button: the menu item for the previous button
 *	@cancel_button: the menu item for the cancel button
 *
 *	This structure maintains information necessary for the generic dialog
 *	handler to allow processing events for a sequence of dialog windows.
 *	All standard dialog windows are popup windows with a title along with
 *	a horizontal menu at the bottom with Help, Cancel, Previous, and Next
 *	button.
 */
struct dialog_window {
	PANEL *panel;
	WINDOW *win;
	int status;
	struct dialog_list *list;
	struct horizontal_menu *menu;
	void *user_data;
	dialog_event_handler handler;
	dialog_show_func show_func;
	dialog_delete_func delete_func;
	dialog_delete_cb delete_cb;
	struct menu_item *help_button;
	struct menu_item *next_button;
	struct menu_item *prev_button;
	struct menu_item *cancel_button;
};

#define DLG_STATUS_ACTIVE	1
#define DLG_STATUS_CLOSING	2

int ok_button_activated(struct menu_item *item);
void show_message_dialog(char *title, char *fmt, ...);
inline void set_dialog_delete_cb(struct dialog_window *dialog, dialog_delete_cb delete_cb);
int close_window_button_activated(struct menu_item *item);
int cancel_button_activated(struct menu_item *item);
void make_next_button_sensitive(struct dialog_window *dialog);
void make_next_button_insensitive(struct dialog_window *dialog);
void init_dialog_window(struct dialog_window *dialog,
			const char *title,
			const char *help_text,
			dialog_event_handler handler,
			dialog_show_func show_func,
			dialog_delete_func delete_func,
			dialog_delete_cb delete_cb,
			char *next_button_text,
			menuitem_activate_cb next_button_callback,
			char *prev_button_text,
			menuitem_activate_cb prev_button_callback,
			void *user_data);
struct dialog_window *create_dialog_window(
			const char *title,
			const char *help_text,
			dialog_event_handler handler,
			dialog_show_func show_func,
			dialog_delete_func delete_func,
			dialog_delete_cb delete_cb,
			char *next_button_text,
			menuitem_activate_cb next_button_callback,
			char *prev_button_text,
			menuitem_activate_cb prev_button_callback,
			void *user_data);
void show_dialog_window(struct dialog_window *dialog);
void delete_dialog_window(struct dialog_window *dialog, void *not_used);
int process_dialog_window_events(struct dialog_window *dialog, int key);
void init_dialog_list(struct dialog_list *list);
void append_dialog_to_list(struct dialog_window *dialog, struct dialog_list *list);
inline GList *get_next_dialog(struct dialog_window *dialog);
inline GList *get_previous_dialog(struct dialog_window *dialog);
void process_dialog_list_events(struct dialog_list *list);
void delete_dialog_list(GList *list);
void process_modal_dialog(struct dialog_window *dialog);
