/* giFTui
 * Copyright (C) 2003 the giFTui team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include "main.h"

#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>

#include "ui_util.h"

/* str <=> GdkColor */

gboolean
str_to_color (GdkColor *color, const gchar *str)
{
	guint16 *c, *end;
	gchar *s, *n, *cstr;
	
	g_return_val_if_fail (color != NULL, FALSE);
	g_return_val_if_fail (str != NULL, FALSE);
	
	c = &color->red;
	end = &color->red + 3;
	
       	cstr = g_strdup (str);
	s = cstr;
	n = s;
	
	while (c < end && n)
	{
		if ((n = strchr (s, '.')))
			*(n++) = '\0';
		*(c++) = (guint16) atoi (s);
		s = n;
	}
	g_free (cstr);
	
	return TRUE;
}

gchar *
color_to_str (const GdkColor *color)
{
	g_return_val_if_fail (color != NULL, NULL);
	
	return g_strdup_printf ("%hu.%hu.%hu\n", color->red, color->green, color->blue);
}

/* All */

void
widget_set_sensitivity_invert (GtkWidget *widget)
{
	if (GTK_WIDGET_SENSITIVE (widget))
		gtk_widget_set_sensitive (widget, FALSE);
	else
		gtk_widget_set_sensitive (widget, TRUE);
	
	return;
}

/* Separators */

void
widget_vhseparator_add (GtkBox *box, gint orientation)
{
	GtkWidget *sep;
	
	g_return_if_fail (GTK_IS_BOX (box));
	
	if (orientation == HORIZONTAL)
		sep = gtk_hseparator_new ();
	else
		sep = gtk_vseparator_new ();
	gtk_box_pack_start (GTK_BOX (box), sep, FALSE, FALSE, 0);
	gtk_widget_show (sep);
	
	return;
}

/* Menus */

GtkWidget *
widget_menu_add (GtkMenuItem *item)
{
	GtkWidget *menu;
	
	menu = gtk_menu_new ();
	if (item != NULL)
		gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), menu);
	gtk_widget_show (menu);
	
	return menu;
}

GtkWidget *
widget_menu_item_add (GtkMenu *menu, const gchar *title)
{
	GtkWidget *item;
	
	g_return_val_if_fail (GTK_IS_MENU (menu), NULL);
	
	if (title != NULL)
		item = gtk_menu_item_new_with_label (title);
	else
	{
		item = gtk_menu_item_new ();
		gtk_widget_set_sensitive (item, FALSE);
	}
	gtk_menu_append (menu, item);
	gtk_widget_show (item);
	
	return item;
}

GtkWidget *
widget_menu_image_add (GtkMenu *menu, const gchar *title, GtkWidget *image)
{
	GtkWidget *item;
	
	g_return_val_if_fail (GTK_IS_MENU (menu), NULL);
	
	if (!title)
		item = gtk_image_menu_item_new ();
	else
		item = gtk_image_menu_item_new_with_mnemonic (title);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (item), image);
	gtk_menu_append (menu, item);
	gtk_widget_show (item);
	
	return item;
}

/* Checks */

GtkWidget *
widget_check_add (GtkBox *box, const gchar *title)
{
	GtkWidget *hbox, *label, *check;
	
	g_return_val_if_fail (GTK_IS_BOX (box), NULL);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
	
	if (title)
	{
		label = gtk_label_new (title);
		gtk_box_pack_start_defaults (GTK_BOX (hbox), label);
		gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	}
	
	check = gtk_check_button_new ();
	gtk_box_pack_start_defaults (GTK_BOX (hbox), check);
	
	gtk_widget_show_all (hbox);
	
	return check;
}


/* Entry */

GtkWidget *
widget_entry_add_chars (GtkBox *box, const gchar *title,
			const gchar *value, gint max_char)
{
	GtkWidget *entry;
	GtkWidget *hbox;
	GtkWidget *label;
	
	g_return_val_if_fail (GTK_IS_BOX (box), NULL);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
	
	if (title)
	{
		label = gtk_label_new (title);
		gtk_box_pack_start_defaults (GTK_BOX (hbox), label);
		gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	}
	
	entry = gtk_entry_new_with_max_length (max_char);
	if (value)
		gtk_entry_set_text (GTK_ENTRY (entry), value);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), entry);
	
	gtk_widget_show_all (hbox);
	
	return entry;
}

/*gchar *
widget_entry_get_chars (GtkEntry *entry)
{
	g_return_val_if_fail (GTK_IS_ENTRY (entry), NULL);
	
	return g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
}*/

void
widget_entry_set_chars (GtkEntry *entry, const gchar *text)
{
	g_return_if_fail (GTK_IS_ENTRY (entry));
	
	gtk_entry_set_text (GTK_ENTRY (entry), text);
	
	return;
}

/* Spin */

GtkWidget *
widget_spin_add_float (GtkBox *box, const gchar *title,
		       gfloat value, gfloat max, gfloat min,
		       gfloat step)
{
	GtkWidget *hbox;
	GtkWidget *spin;
	GtkWidget *label;
	GtkObject *adj;
	
	g_return_val_if_fail (GTK_IS_BOX (box), NULL);
	
	adj = gtk_adjustment_new (value, min, max, step, step, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
		
	label = gtk_label_new (title);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), label);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0.2, 3);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), spin);
	
	gtk_widget_show_all (hbox);
	
	return spin;
}

/*gfloat
widget_spin_get_float (GtkSpinButton *spin)
{
	g_return_val_if_fail (GTK_IS_SPIN_BUTTON (spin), 0);
	
	return gtk_spin_button_get_value_as_float (spin);
}*/

void
widget_spin_set_float (GtkSpinButton *spin, gfloat value)
{
	g_return_if_fail (GTK_IS_SPIN_BUTTON (spin));
	
	gtk_spin_button_set_value (spin, (gdouble) value);
	
	return;
}

/**/

GtkWidget *
widget_spin_add_int (GtkBox *box, const gchar *title,
		     gint value, gint max, gint min,
		     gint step)
{
	GtkWidget *hbox;
	GtkWidget *spin;
	GtkWidget *label;
	GtkObject *adj;
	
	g_return_val_if_fail (GTK_IS_BOX (box), NULL);
	
	adj = gtk_adjustment_new ((gfloat) value, (gfloat) min, (gfloat) max,
				  (gfloat) step, (gfloat) step, 0);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
		
	label = gtk_label_new (title);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), label);
	gtk_label_set_justify (GTK_LABEL (label), GTK_JUSTIFY_LEFT);
	
	spin = gtk_spin_button_new (GTK_ADJUSTMENT (adj), 0.2, 0);
	gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), spin);
		
	gtk_widget_show_all (hbox);
	
	return spin;
}

/*gint
widget_spin_get_int (GtkSpinButton *spin)
{
	g_return_val_if_fail (GTK_IS_SPIN_BUTTON (spin), 0);
	
	return gtk_spin_button_get_value_as_int (spin);
}*/

void
widget_spin_set_int (GtkSpinButton *spin, gint value)
{
	g_return_if_fail (GTK_IS_SPIN_BUTTON (spin));
	
	gtk_spin_button_set_value (spin, (gdouble) value);
	
	return;
}

/* Color selection */

static void
_color_selected (GtkWidget *button, GtkColorSelectionDialog *sel)
{
	GdkColor *c;
	GtkWidget *col;
	
	col = GTK_WIDGET (g_object_get_data (G_OBJECT (sel), "button"));
	c = g_object_get_data (G_OBJECT (col), "color");
	
	if (c)
		gtk_color_selection_get_current_color (GTK_COLOR_SELECTION (sel->colorsel), c);
	
	gtk_widget_destroy (GTK_WIDGET (sel));
	
	return;
}

static void
_color_select (GtkWidget *button, gpointer data)
{
	GdkColor *c;
        GtkColorSelectionDialog *sel;
	
	sel = GTK_COLOR_SELECTION_DIALOG (gtk_color_selection_dialog_new (_("Select the color")));
	gtk_color_selection_set_has_opacity_control (GTK_COLOR_SELECTION (sel->colorsel), FALSE);
	
	c = g_object_get_data (G_OBJECT (button), "color");
	if (c)
		gtk_color_selection_set_current_color (GTK_COLOR_SELECTION (sel->colorsel), c);
	
	
	g_object_set_data (G_OBJECT (sel), "button", button);
	g_signal_connect (sel->ok_button, "clicked",
			  G_CALLBACK (_color_selected),
			  sel);
	g_signal_connect_swapped (sel->cancel_button, "clicked",
				  G_CALLBACK (gtk_widget_destroy),
				  sel);
	
	gtk_widget_show (GTK_WIDGET (sel));
	
	return;
}

GtkWidget *
widget_button_color_sel_add_from_str (GtkBox *box, const gchar *text)
{
	GtkWidget *hbox;
	GtkWidget *label;
	GtkWidget *button;
	
	g_return_val_if_fail (GTK_IS_BOX (box), NULL);
	
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (box), hbox, FALSE, FALSE, 0);
	
	if (text)
	{
		label = gtk_label_new (text);
		gtk_box_pack_start_defaults (GTK_BOX (hbox), label);
	}
	
	button = gtk_button_new_from_stock (GTK_STOCK_SELECT_COLOR);
	gtk_box_pack_start_defaults (GTK_BOX (hbox), button);
	g_signal_connect (button, "clicked",
			  G_CALLBACK (_color_select), NULL);
	
	gtk_widget_show_all (hbox);
	
	return button;
}

/* Tooltips */
void
widget_tooltips_set_tip (GtkTooltips *tips, GtkWidget *widget, const gchar *text)
{
	g_return_if_fail (GTK_IS_TOOLTIPS (tips));
	g_return_if_fail (GTK_IS_WIDGET (widget));
	
	gtk_tooltips_set_tip (tips, widget, text, NULL);
	
	return;
}

/**/

/* Menu from GtkItemFactoryEntry */

/* If window != NULL we use accels. */
GtkWidget *
gtk_menu_from_factoryentry (GtkItemFactoryEntry *entrys, GType type, gint nb,
			    GtkWidget *window, gpointer data)
{
	GtkWidget *menu;
	GtkItemFactory *item_factory;
	GtkAccelGroup *accel_group = NULL;
	
	g_return_val_if_fail (entrys != NULL, NULL);
	g_return_val_if_fail (nb > 0, NULL);
	
	if (window)
		accel_group = gtk_accel_group_new ();
	item_factory = gtk_item_factory_new (type, "<main>", accel_group);
	gtk_item_factory_create_items (item_factory, nb, entrys, data);
	if (window)
		gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);
	
	menu = gtk_item_factory_get_widget (item_factory, "<main>");
	
	return menu;
}

/* Combo */

#define COMBO_MAX_LENGHT 12

GtkWidget *
gtk_combo_add_with_title (GtkBox *box, const gchar *title, gint max_char)
{
	GtkWidget *label;
	GtkWidget *combo;
	
	g_return_val_if_fail (GTK_IS_BOX (box), NULL);
	
	if (title)
	{
		label = gtk_label_new (title);
		gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 5);
	}
	
	combo = gtk_combo_new ();
	gtk_combo_set_case_sensitive (GTK_COMBO (combo), FALSE);
	gtk_entry_set_max_length (GTK_ENTRY (GTK_COMBO (combo)->entry), max_char);
	gtk_combo_disable_activate (GTK_COMBO (combo));
	gtk_box_pack_start (GTK_BOX (box), combo, TRUE, TRUE, 5);
	
	return combo;
}

static GList *
_find_str_in_combo_list (GList *list, const gchar *text)
{
	while (list)
	{
		if (!g_utf8_collate ((gchar *) list->data, text))
			return list;
		list = list->next;
	}
	
	return NULL;
}

void
gtk_combo_entry_add_text (GtkCombo *combo, const gchar *text)
{
	GList *list, *old;
	
	g_return_if_fail (GTK_IS_COMBO (combo));
	
	if (!text)
		return;
	
	list = (GList *) g_object_get_data (G_OBJECT (combo), "widget-list");
	
	if (list)
	{
		if (!(old = _find_str_in_combo_list (list, text)))
		{
			if ((g_list_length (list) >= COMBO_MAX_LENGHT))
			{
				GList *last;
				
				last = list;
				while (last->next)
					last = last->next;
				
				g_free (last->data);
				list = g_list_delete_link (list, last);
			}
			list = g_list_prepend (list, g_strdup (text));
		}
		else if (list != old)
		{
			list = g_list_remove_link (list, old);
			
			list->prev = old;
			old->next = list;
			
			list = old;
		}
	}
	else
		list = g_list_append (list, g_strdup (text));
	
	g_object_set_data (G_OBJECT (combo), "widget-list", list);
	
	gtk_combo_set_popdown_strings (combo, list);
	
	return;
}

/* GtkTreeStore */

void
gtk_tree_store_remove_children (GtkTreeStore *store, GtkTreeIter *parent)
{
       	GtkTreeIter child;
	
	g_return_if_fail (GTK_IS_TREE_STORE (store));
	g_return_if_fail (parent != NULL);
	
	while (gtk_tree_model_iter_n_children (GTK_TREE_MODEL (store), parent))
	{
		gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (store), &child, parent, 0);
		gtk_tree_store_remove (store, &child);
	}
	
	return;
}

/* GtkTreeSelection */

/* Allow us to remove lines from a GtkTreeStore without lose others
 * selected lines. */
void
gtk_tree_selection_selected_foreach_rm (GtkTreeSelection *selection,
					GtkTreeSelectionForeachFunc func,
					gpointer data)
{
	GList *list, *l;
	GtkTreeView *view;
	GtkTreeModel *model;
	GtkTreeIter iter;
	
	g_return_if_fail (GTK_IS_TREE_SELECTION (selection));
	g_return_if_fail (func != NULL);
	
	view = GTK_TREE_VIEW (gtk_tree_selection_get_tree_view (selection));
	model = gtk_tree_view_get_model (view);
	
	while ((list = gtk_tree_selection_get_selected_rows (selection, NULL)))
	{
		l = list;
		while (l)
		{
			if (gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) list->data))
			{
				gtk_tree_selection_unselect_iter (selection, &iter);
				func (model, (GtkTreePath *) list->data, &iter, data);
				
				break;
			}
			l = l->next;
		}
		g_list_foreach (list, (GFunc) gtk_tree_path_free, NULL);
		g_list_free (list);
	}
	
	return;
}
