# GNU Enterprise Forms - GTK UI Driver - User Interface
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: UIdriver.py 7008 2005-02-11 17:03:06Z reinhard $

import sys
import string
import types
import math

from gnue.common.apps import i18n
from gnue.forms.uidrivers._base import Exceptions

try:
  import pygtk
  pygtk.require ('2.0')

  import gtk
  import pango

  ## check if we really imported gtk 2.0
  if not hasattr (gtk, "keysyms") or gtk.pygtk_version [1] < 2:
    print u_("Import error: You need pygtk 2.2 to use the gtk2 user "
             "interface driver.")
    print u_("You are actually using gtk %s and pygtk %s") \
             % ("%s.%s.%s" % gtk.gtk_version,
                "%s.%s.%s" % gtk.pygtk_version)

    raise ImportError

except AssertionError, ImportError:
  raise Exceptions.DriverNotSupported, \
      _("The GNUe-Forms GTK driver requires PyGTK and GTK 2.x.")

from gnue.common import events
from gnue.common.apps import GConfig


from gnue.forms.GFForm import *

from gnue.forms.uidrivers._base import UIdriver as BaseDriver
from gnue.forms.uidrivers._commonGuiToolkit import UIdriver as commonToolkit

from gnue.forms.uidrivers.gtk2.GFApp import *
from gnue.forms.uidrivers.gtk2.SplashScreen import *
from gnue.forms.uidrivers.gtk2.widgets._base  import *
from gnue.forms.uidrivers.gtk2 import about

# =============================================================================
# This class implements a User Interface for GTK2
# =============================================================================

class GFUserInterface (commonToolkit.GFUserInterface):

  _WidgetToGFObj = {}
  _WidgetToUIObj = {}

  _MBOX_KIND = {'info'    : {'type'   : gtk.MESSAGE_INFO,
                             'buttons': gtk.BUTTONS_CLOSE},
                'warning' : {'type'   : gtk.MESSAGE_WARNING,
                             'buttons': gtk.BUTTONS_CLOSE},
                'question': {'type'   : gtk.MESSAGE_QUESTION,
                             'buttons': gtk.BUTTONS_YES_NO},
                'error'   : {'type'   : gtk.MESSAGE_ERROR,
                             'buttons': gtk.BUTTONS_CLOSE}}

  _RESPONSE = {gtk.RESPONSE_OK    : True,
               gtk.RESPONSE_CLOSE : True,
               gtk.RESPONSE_YES   : True,
               gtk.RESPONSE_NO    : False,
               gtk.RESPONSE_CANCEL: None }


  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------
  
  def __init__ (self, *args, **params):
    commonToolkit.GFUserInterface.__init__ (self, *args, **params)

    self.name = "GTK2"

    self._display   = gtk.gdk.display_manager_get ().get_default_display ()
    self._clipboard = gtk.Clipboard(self._display, "CLIPBOARD")

    

  # ---------------------------------------------------------------------------
  # Initialize user interface
  # ---------------------------------------------------------------------------

  def _initialize (self):

    self.app = getApp ()

    if not self._disableSplash:
      self.splash = UISplashScreen ()
      self.splash.Show ()
      gtk.timeout_add (1500, lambda splash: splash.destroy (), self.splash)

    self.font       = APPFONT

    sr = (CHAR_WIDTH, CHAR_HEIGHT)
    for widgetModule in self._supportedWidgets.values ():
      if hasattr (widgetModule, 'size_request'):
        need = widgetModule.size_request ()
        sr = (max (sr [0], need [0]), max (sr [1], need [1]))

    (self.textWidth, self.textHeight) = sr

    gDebug (6, "Metrics %s x %s" % (self.textWidth, self.textHeight))

    self.widgetWidth  = self.textWidth
    self.widgetHeight = self.textHeight




  #############################################################################
  #
  # Private UIBase support functions
  #
  # Called in UIbase functions to perform UI interaction with this specific
  # widget set.
  #

  # ---------------------------------------------------------------------------
  # Tells the application to close it's main window
  # ---------------------------------------------------------------------------

  def _exit (self, formName):

    _form = None

    exitApp = True
    for child in self._children:
      if child._form.name == formName:
        child.mainWindow.hide ()
        child._visible = False
        _form = child._form

      exitApp = exitApp and not child._visible

    if exitApp:
      for child in self._children:
        if not isinstance (child.mainWindow, gtk.Dialog):
          child.mainWindow.destroy ()
        else:
          child.mainWindow.response (gtk.RESPONSE_CLOSE)

    # if the main form is a dialog we do *not* quit the main loop manually
    if formName == '__main__' and _form.style != 'dialog':
      self.app.quit ()


  def _beep (self):
    pass

  #############################################################################
  #
  # Incoming Event Processors
  #
  # Processes the incoming events from other objects
  # From here down should be nothing but eventListeners listed

  #
  # mainLoop
  #
  # The primary loop of the user interface.  Called once the UI is
  # fully activated (multiple loops are simulated by GFApp)
  #
  def mainLoop(self):
    self.app.mainLoop()


  #
  # formAlert
  #
  # Rings a bell and alters the statusbar to display
  # a line of text
  #
  def formAlert(self, event):
    ui = self._gfObjToUIWidget[event._form]
    if ui.statusBar is not None:
      ui.statusBar1.push (ui.statusBar1.get_context_id ("tip"), \
                          unicode (event.data))


  # ---------------------------------------------------------------------------
  # Change the cursor into the watch
  # ---------------------------------------------------------------------------

  def beginWait (self, event):
    ui = self._gfObjToUIWidget [event._form]
    if ui.mainWindow.window is not None:
      ui.mainWindow.window.set_cursor (gtk.gdk.Cursor (gtk.gdk.WATCH))

    self.processMessages ()



  # ---------------------------------------------------------------------------
  # Called whenever forms leaves a "wait" state
  # ---------------------------------------------------------------------------

  def endWait (self, event):
    ui = self._gfObjToUIWidget [event._form]
    if ui.mainWindow.window is not None:
      ui.mainWindow.window.set_cursor (gtk.gdk.Cursor (gtk.gdk.LEFT_PTR))

    self.processMessages ()


  # ---------------------------------------------------------------------------
  # Get some text from the clipboard
  # ---------------------------------------------------------------------------

  def getClipboardContents (self, event):

    gDebug (6, "Retrieving clipboard contents")

    widget = self._UIform.mainWindow.get_focus ()
    if isinstance (widget, gtk.TextView):
      gDebug (6, "Clipboard is %s" % self._clipboard)

      tBuffer = widget.get_buffer ()
      pos = tBuffer.get_iter_at_mark (tBuffer.get_insert ())
      tBuffer.paste_clipboard (self._clipboard, pos, widget.get_editable())

    elif hasattr (widget, 'paste_clipboard'):
      gDebug (6, "calling %s.paste_clipboard" % widget)
      widget.paste_clipboard ()

    event.__result__ = None


  # ---------------------------------------------------------------------------
  # Set some text into the clipboard
  # ---------------------------------------------------------------------------

  def setClipboardContents (self, event):

    gDebug (6, "Setting clipboard contents")

    widget = self._UIform.mainWindow.get_focus ()
    if isinstance (widget, gtk.TextView):
      tBuffer = widget.get_buffer ()
      tBuffer.copy_clipboard (self._clipboard)

    if hasattr (widget, 'copy_clipboard'):
      gDebug (6, "calling %s.copy_clipboard ()" % widget)
      widget.copy_clipboard ()


  # ---------------------------------------------------------------------------
  # Set the forms title
  # ---------------------------------------------------------------------------

  def setTitle (self, event):
    ui = self._gfObjToUIWidget [event._form]

    try:
      ui.mainWindow.set_title (event.title)
    except AttributeError:
      pass

  # ---------------------------------------------------------------------------
  # create a modal message box
  # ---------------------------------------------------------------------------

  def _showMessage (self, message, kind = 'Info', title = None, cancel = False):
    """
    This function creates a message box of a given kind and returns True, False
    or None depending on the button pressed.
    @param message: the text of the messagebox
    @param kind: type of the message box. Valid types are 'Info', 'Warning',
        'Question', 'Error'
    @param title: title of the message box
    @param cancel: If True a cancel button will be added to the dialog
    @return: True if the Ok-, Close-, or Yes-button was pressed, False if the
        No-button was pressed or None if the Cancel-button was pressed.
    """
    mbRec  = self._MBOX_KIND.get (kind.lower ())
    dialog = gtk.MessageDialog (parent = None, flags = gtk.DIALOG_MODAL,
        type = mbRec ['type'], buttons = mbRec ['buttons'],
        message_format = self.__makeSafe (message))

    if title is not None and len (title):
      dialog.set_title (self.__makeSafe (title))

    cButtons = [gtk.BUTTONS_CANCEL, gtk.BUTTONS_OK_CANCEL]
    if cancel and not mbRec ['buttons'] in cButtons:
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)

    res = dialog.run ()
    dialog.destroy ()

    return self._RESPONSE [res]


  # ---------------------------------------------------------------------------
  # Display an about box
  # ---------------------------------------------------------------------------

  def aboutBox (self, params, form):
    dialog = about.AboutBox (params, form)
    dialog.run ()
    dialog.destroy ()


  # ---------------------------------------------------------------------------
  # Make sure strings get converted to unicode
  # ---------------------------------------------------------------------------

  def __makeSafe (self, aValue):
    if isinstance (aValue, types.StringType):
      return unicode (aValue, i18n.encoding)
    else:
      return aValue


  # ---------------------------------------------------------------------------
  # Show an exception dialog
  # ---------------------------------------------------------------------------

  def _showException (self, group, name, message, detail):
    dialog = ExceptionDisplay (group, name, message, detail)
    try:
      dialog.run ()
    finally:
      dialog.destroy ()


  # ---------------------------------------------------------------------------
  # Process all pending events, but do not block if there are no events
  # ---------------------------------------------------------------------------

  def processMessages (self):
    while gtk.events_pending ():
      gtk.main_iteration (False)



# -----------------------------------------------------------------------------
# Get font metrics and font description for a given font 
# -----------------------------------------------------------------------------

def _getFontMetrics (fontName = None):
  """
  """
  label = gtk.Label ()

  try:
    if fontName is not None:
      label.modify_font (pango.FontDescription (fontName))

    context  = label.get_pango_context ()
    fontDesc = context.get_font_description ()
    metrics  = context.get_metrics (fontDesc)

    width  = pango.PIXELS (metrics.get_approximate_char_width ())
    height = pango.PIXELS (metrics.get_ascent () + metrics.get_descent ())

    # add ten percent to the average character width
    width  = int (math.ceil (width * 1.1))

  finally:
    label.destroy ()

  return (width, height, fontDesc)



# =============================================================================
# This class implements a dialog for displaying exceptions
# =============================================================================

class ExceptionDisplay (gtk.Dialog):

  _TITLE = {'system'     : _("GNUe Internal System Error"),
            'admin'      : _("GNUe Unexpected Error"),
            'application': _("GNUe Application Error")}

  _FORMAT = {
     'system': u_("An unexpected internal error has occured:\n%s.\n"
                  "This means you have found a bug in GNU Enterprise. "
                  "Please report it to gnue-dev@gnu.org"),
     'admin': u_("An unexpected error has occured:\n%s.\n"
                 "Please contact your system administrator."),
     'application': u_("An unexpected error has occured:\n%s.\n"
                       "Please contact your system administrator.")}

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, group, name, message, detail):

    gtk.Dialog.__init__ (self, self._TITLE.get (group, _('Error')), None,
                       gtk.DIALOG_MODAL, (gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE))

    self.set_border_width (5)

    button = gtk.Button (u_('>> Detail'))
    button.connect ('clicked', self.__toggleDetail)
    self.action_area.pack_start (button)
    button.show ()

    hbox = gtk.HBox (spacing = 16)

    image = gtk.Image ()
    image.set_from_stock (gtk.STOCK_DIALOG_ERROR, gtk.ICON_SIZE_DIALOG)
    image.set_alignment (0, 0)
    hbox.pack_start (image, expand = False)
    image.show ()

    vbox = gtk.VBox (spacing = 16)

    label = gtk.Label (self._FORMAT.get (group, "%s") % message)
    label.set_line_wrap (True)
    label.set_alignment (0, 0)
    vbox.pack_start (label, expand = False)
    label.show ()

    textView = gtk.TextView ()
    textView.set_pixels_above_lines (2)
    textView.set_left_margin (2)
    textView.set_wrap_mode (gtk.WRAP_NONE)
    textView.set_cursor_visible (True)
    # Just set the textview to 'readonly' so using clipboard still works
    textView.set_editable (False)
    textView.modify_font (pango.FontDescription ('monospace'))
    textView.get_buffer ().set_text (detail)

    self.detailView = gtk.ScrolledWindow ()
    self.detailView.add_with_viewport (textView)
    self.detailView.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    (w, h) = self.__getDetailSize (detail)
    self.detailView.set_size_request (w, h)
    textView.show ()
    vbox.pack_start (self.detailView)

    hbox.pack_start (vbox)
    vbox.show ()

    self.vbox.pack_start (hbox, padding = 8)
    hbox.show ()

    self._showsDetail  = False
    self._originalSize = self.get_size ()
    self.set_position (gtk.WIN_POS_CENTER_ALWAYS)


  # ---------------------------------------------------------------------------
  # toggle visibility of the detail display
  # ---------------------------------------------------------------------------

  def __toggleDetail (self, button):
    if self._showsDetail:
      self.detailView.hide_all ()
      button.set_label (u_(">> Details"))
      self.resize (*self._originalSize)
    else:
      self.detailView.show_all ()
      button.set_label (u_("<< Details"))
  
    self._showsDetail = not self._showsDetail


  # ---------------------------------------------------------------------------
  # Try to figure out a width and heigth for the traceback-text
  # ---------------------------------------------------------------------------

  def __getDetailSize (self, detail):

    lines = detail.splitlines ()
    maxH = len (lines) + 3
    maxW = 0

    for line in lines:
      maxW = max (maxW, len (line))

    maxW += 5
    (w, h) = _getFontMetrics ('monospace') [:2]

    return (min (w * maxW, gtk.gdk.screen_width ()),
            min (h * maxH, gtk.gdk.screen_height () - 100))

# =============================================================================
# Get the current default application font and it's metrics
# =============================================================================

(CHAR_WIDTH, CHAR_HEIGHT, APPFONT) = _getFontMetrics ()
