/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  iconbar.c: Vertical iconbar for gpredict.

  Copyright (C)  2001-2003  Alexandru Csete.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>


  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/


#include <gnome.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gnome-canvas-pixbuf.h>
#include <gal/shortcut-bar/e-shortcut-bar.h>
#include "satlog.h"
#include "satlist.h"
#include "satmap.h"
#include "callbacks.h"
#include "iconbar-popup-menu.h"
#include "iconbar.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif


/* The items have to be added in this order */
enum groups {
	SHORTCUT_GROUP_NEWMOD = 0,
	SHORTCUT_GROUP_MODULES,
	SHORTCUT_GROUP_COMM,
	SHORTCUT_GROUP_PRED,
	SHORTCUT_GROUP_MAINTEN
};

enum newmod_items {
	SHORTCUT_ITEM_LIST = 0,
	SHORTCUT_ITEM_MAP,
	SHORTCUT_ITEM_SKY,
	SHORTCUT_ITEM_GLOBE,
	SHORTCUT_ITEM_CRASH
};



extern GtkWidget *app; /* main.c */
extern const gchar *modtypes[]; /* satmodules.c */

EShortcutBar *iconbar;


/* privatye function prototypes */
static void iconbar_click_cb (EShortcutBar *,GdkEvent *,gint,gint);



GtkWidget *iconbar_create ()
{
	/* This function creates and fills the shortcut bar on
	   the left of the main window.
	*/
	GtkWidget *iconbar_w,*frame;
	EShortcutModel *shortcut_model;
	GdkPixbuf *pbuf;
	gchar *path,*uri,**namev;
	gint group_num, item_num;
	GList *list,*node;
	GnomeVFSResult result;
	GnomeVFSDirectoryFilter *filter;
	GnomeVFSFileInfo *fileinfo;

	/* create widget */
	shortcut_model = e_shortcut_model_new ();

	iconbar_w = e_shortcut_bar_new ();
	e_shortcut_bar_set_model (E_SHORTCUT_BAR (iconbar_w),
				  shortcut_model);

	gtk_widget_set (iconbar_w, "width", 110, NULL);

	gtk_signal_connect (GTK_OBJECT (iconbar_w), "item_selected",
			    GTK_SIGNAL_FUNC (iconbar_click_cb), NULL);

	iconbar = E_SHORTCUT_BAR (iconbar_w);

	/**** Create New Module ****/
	group_num = e_shortcut_model_add_group (iconbar->model, -1, _("New"));
	/* Satlist icon */
	path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/satlist.png", NULL);
	pbuf = gdk_pixbuf_new_from_file (path);
	item_num = e_shortcut_model_add_item (iconbar->model, group_num, -1,
					      modtypes[SHORTCUT_ITEM_LIST],
					      "List", pbuf);
	gdk_pixbuf_unref (pbuf);
	g_free (path);
	/* Map icon */
	path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/map.png", NULL);
	pbuf = gdk_pixbuf_new_from_file (path);
	item_num = e_shortcut_model_add_item (iconbar->model, group_num, -1,
					      modtypes[SHORTCUT_ITEM_MAP],
					      "Map", pbuf);
	gdk_pixbuf_unref (pbuf);
	g_free (path);
	/* Skyview icon */
	path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/skyview.png", NULL);
	pbuf = gdk_pixbuf_new_from_file (path);
	item_num = e_shortcut_model_add_item (iconbar->model, group_num, -1,
					      modtypes[SHORTCUT_ITEM_SKY],
					      "Sky", pbuf);
	gdk_pixbuf_unref (pbuf);
	g_free (path);
	/* Satview icon */
	path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/gnome-globe.png", NULL);
	pbuf = gdk_pixbuf_new_from_file (path);
	item_num = e_shortcut_model_add_item (iconbar->model, group_num, -1,
					      modtypes[SHORTCUT_ITEM_GLOBE],
					      "Globe", pbuf);
	gdk_pixbuf_unref (pbuf);
	g_free (path);
	
	/**** show existing modules ***/
	/* create group */
	group_num = e_shortcut_model_add_group (iconbar->model, -1, _("Modules"));

	/* create gnome_vfs_uri */
	path = g_strconcat (g_get_home_dir (), "/.gpredict", NULL);
	uri = gnome_vfs_get_uri_from_local_path (path);
	g_free (path);

	/* lists */
	filter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
						 GNOME_VFS_DIRECTORY_FILTER_NODIRS |
						 GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
						 GNOME_VFS_DIRECTORY_FILTER_EXTENDEDREGEXP |
						 GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
						 GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
						 GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
						 "*.satlist");
	result = gnome_vfs_directory_list_load (&list, uri,
						GNOME_VFS_FILE_INFO_DEFAULT,
						filter);
	gnome_vfs_directory_filter_destroy (filter);
	if (result == GNOME_VFS_OK) {
		/* loop throough the file list */
		for (node = list; node != NULL; node = node->next) {
			fileinfo = node->data;
			namev = g_strsplit (fileinfo->name, ".satlist", 2);
			/* create pixbuf */
			path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/satlist.png", NULL);
			pbuf = gdk_pixbuf_new_from_file (path);
			item_num = e_shortcut_model_add_item (iconbar->model, group_num,
							      -1, modtypes[SHORTCUT_ITEM_LIST],
							      namev[0], pbuf);
			gdk_pixbuf_unref (pbuf);
			g_free (path);
			g_strfreev (namev);
		}
		/* free memory taken up by file list */
		gnome_vfs_file_info_list_free (list);
	}
	else {
		path = g_strdup_printf (_("iconbar_create: gnome_vfs_directory_list_load "\
					  "returned: %s"),
					gnome_vfs_result_to_string (result));
		satlog_log (SAT_LOG_CRITICAL, path);
		g_free (path);
	}
	/* maps */
	filter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
						 GNOME_VFS_DIRECTORY_FILTER_NODIRS |
						 GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
						 GNOME_VFS_DIRECTORY_FILTER_EXTENDEDREGEXP |
						 GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
						 GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
						 GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
						 "*.satmap");
	result = gnome_vfs_directory_list_load (&list, uri,
						GNOME_VFS_FILE_INFO_DEFAULT,
						filter);
	gnome_vfs_directory_filter_destroy (filter);
	if (result == GNOME_VFS_OK) {
		/* loop throough the file list */
		for (node = list; node != NULL; node = node->next) {
			fileinfo = node->data;
			namev = g_strsplit (fileinfo->name, ".satmap", 2);
			/* create pixbuf */
			path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/map.png", NULL);
			pbuf = gdk_pixbuf_new_from_file (path);
			item_num = e_shortcut_model_add_item (iconbar->model, group_num,
							      -1, modtypes[SHORTCUT_ITEM_MAP],
							      namev[0], pbuf);
			gdk_pixbuf_unref (pbuf);
			g_free (path);
			g_strfreev (namev);
		}
		/* free memory taken up by file list */
		gnome_vfs_file_info_list_free (list);
	}
	else {
		path = g_strdup_printf (_("iconbar_create: gnome_vfs_directory_list_load "\
					  "returned: %s"),
					gnome_vfs_result_to_string (result));
		satlog_log (SAT_LOG_CRITICAL, path);
		g_free (path);
	}
	/* sky view */
	filter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
						 GNOME_VFS_DIRECTORY_FILTER_NODIRS |
						 GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
						 GNOME_VFS_DIRECTORY_FILTER_EXTENDEDREGEXP |
						 GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
						 GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
						 GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
						 "*.satsky");
	result = gnome_vfs_directory_list_load (&list, uri,
						GNOME_VFS_FILE_INFO_DEFAULT,
						filter);
	gnome_vfs_directory_filter_destroy (filter);
	if (result == GNOME_VFS_OK) {
		/* loop throough the file list */
		for (node = list; node != NULL; node = node->next) {
			fileinfo = node->data;
			namev = g_strsplit (fileinfo->name, ".satsky", 2);
			/* create pixbuf */
			path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/skyview.png", NULL);
			pbuf = gdk_pixbuf_new_from_file (path);
			item_num = e_shortcut_model_add_item (iconbar->model, group_num,
							      -1, modtypes[SHORTCUT_ITEM_SKY],
							      namev[0], pbuf);
			gdk_pixbuf_unref (pbuf);
			g_free (path);
			g_strfreev (namev);
		}
		/* free memory taken up by file list */
		gnome_vfs_file_info_list_free (list);
	}
	else {
		path = g_strdup_printf (_("iconbar_create: gnome_vfs_directory_list_load "\
					  "returned: %s"),
					gnome_vfs_result_to_string (result));
		satlog_log (SAT_LOG_CRITICAL, path);
		g_free (path);
	}
	/* globe/satview */
	filter = gnome_vfs_directory_filter_new (GNOME_VFS_DIRECTORY_FILTER_SHELLPATTERN,
						 GNOME_VFS_DIRECTORY_FILTER_NODIRS |
						 GNOME_VFS_DIRECTORY_FILTER_NODOTFILES |
						 GNOME_VFS_DIRECTORY_FILTER_EXTENDEDREGEXP |
						 GNOME_VFS_DIRECTORY_FILTER_NOSELFDIR |
						 GNOME_VFS_DIRECTORY_FILTER_NOPARENTDIR |
						 GNOME_VFS_DIRECTORY_FILTER_NOBACKUPFILES,
						 "*.satview");
	result = gnome_vfs_directory_list_load (&list, uri,
						GNOME_VFS_FILE_INFO_DEFAULT,
						filter);
	gnome_vfs_directory_filter_destroy (filter);
	if (result == GNOME_VFS_OK) {
		/* loop throough the file list */
		for (node = list; node != NULL; node = node->next) {
			fileinfo = node->data;
			namev = g_strsplit (fileinfo->name, ".satview", 2);
			/* create pixbuf */
			path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/globe.png", NULL);
			pbuf = gdk_pixbuf_new_from_file (path);
			item_num = e_shortcut_model_add_item (iconbar->model, group_num,
							      -1, modtypes[SHORTCUT_ITEM_GLOBE],
							      namev[0], pbuf);
			gdk_pixbuf_unref (pbuf);
			g_free (path);
			g_strfreev (namev);
		}
		/* free memory taken up by file list */
		gnome_vfs_file_info_list_free (list);
	}
	else {
		path = g_strdup_printf (_("iconbar_create: gnome_vfs_directory_list_load "\
					  "returned: %s"),
					gnome_vfs_result_to_string (result));
		satlog_log (SAT_LOG_CRITICAL, path);
		g_free (path);
	}
	g_free (uri);

	/**** Communication shortcuts ****/
	group_num = e_shortcut_model_add_group (iconbar->model, -1, _("Communication"));
	/* Radio icon */
	path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/radio.png", NULL);
	pbuf = gdk_pixbuf_new_from_file (path);
	item_num = e_shortcut_model_add_item (iconbar->model, group_num, -1, "radio:",
					      "Radio", pbuf);
	gdk_pixbuf_unref (pbuf);
	g_free (path);
	/* Antenna icon */
	path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/antenna.png", NULL);
	pbuf = gdk_pixbuf_new_from_file (path);
	item_num = e_shortcut_model_add_item (iconbar->model, group_num, -1, "antenna:",
					      "Antenna", pbuf);
	gdk_pixbuf_unref (pbuf);
	g_free (path);

	/**** Predict shortcuts ****/
	group_num = e_shortcut_model_add_group (iconbar->model, -1, _("Predict"));
	/* Calendar icon */
	path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/calendar.png", NULL);
	pbuf = gdk_pixbuf_new_from_file (path);
	item_num = e_shortcut_model_add_item (iconbar->model, group_num, -1, "pass:",
					      "Passes", pbuf);
	gdk_pixbuf_unref (pbuf);
	g_free (path);

	/**** Maintenance shortcuts ****/
/*  	group_num = e_shortcut_model_add_group (iconbar->model, -1, "Maintenance"); */

	/* create frame around shortcut bar */
	frame = gtk_frame_new (NULL);
	gtk_container_add (GTK_CONTAINER (frame), GTK_WIDGET (iconbar));

	return frame;
}


static void
iconbar_click_cb (EShortcutBar *shortcutbar, GdkEvent *event, gint group, gint item)
{
	/* This function is called when the user clicks on an icon on
	   the shortcut bar. The caller catches ALL mouse clicks regardless
	   of button type or item type.
	*/
	gchar *url=NULL,*name=NULL;
	guint modtype=0;
	GdkPixbuf *pbuf=NULL;

	switch (event->button.button) {
	case 1:
		switch (group) {
		case SHORTCUT_GROUP_NEWMOD:
			switch (item) {
			case -1:
				break;
  			case SHORTCUT_ITEM_LIST:
				satlist_open (NULL, TRUE);
  				break;
			case SHORTCUT_ITEM_MAP:
				satmap_open (NULL, TRUE);
				break;
/*  			case SHORTCUT_ITEM_SKY: */
/*  				break; */
/*  			case SHORTCUT_ITEM_GLOBE: */
/*  				break; */
			default:
				not_finished (NULL, NULL);
				break;
			}
			break;
		case SHORTCUT_GROUP_MODULES:
			switch (item) {
			case -1:
				break;
			default:
				e_shortcut_model_get_item_info (shortcutbar->model,
							       group, item,
							       &url, &name, &pbuf);
				/* what kind of module? */
				if (!g_strcasecmp (url, "satlist"))
					modtype = SHORTCUT_ITEM_LIST;
				else if (!g_strcasecmp (url, "satmap"))
					modtype = SHORTCUT_ITEM_MAP;
				else if (!g_strcasecmp (url, "satsky"))
					modtype = SHORTCUT_ITEM_SKY;
				else if (!g_strcasecmp (url, "satglobe"))
					modtype = SHORTCUT_ITEM_GLOBE;
				switch (modtype) {
				case SHORTCUT_ITEM_LIST:
					satlist_open (name, TRUE);
					break;
				case SHORTCUT_ITEM_MAP:
					satmap_open (name, TRUE);
					break;
				case SHORTCUT_ITEM_SKY:
					break;
				case SHORTCUT_ITEM_GLOBE:
					break;
				default:
					break;
				}
				/* free mem */
  				g_free (url);
				gdk_pixbuf_unref (pbuf);
				break;
			}
			break;
		case SHORTCUT_GROUP_COMM:
			switch (item) {
			case -1:
				break;
			default:
				not_finished (NULL, NULL);
				break;
			}
			break;
		case SHORTCUT_GROUP_PRED:
			switch (item) {
			case -1:
				break;
			default:
				not_finished (NULL, NULL);
				break;
			}
			break;
		case SHORTCUT_GROUP_MAINTEN:
			switch (item) {
			case -1:
				break;
			default:
				not_finished (NULL, NULL);
				break;
			}
			break;
		default:
			break;
		}
		break;
	case 3:
		switch (group) {
		case SHORTCUT_GROUP_MODULES:
			switch (item) {
			case -1:
				break;
			default:
				/* get item info */
				e_shortcut_model_get_item_info (shortcutbar->model,
								group, item,
								&url, &name, &pbuf);
				/* what kind of module? */
				if (!g_strcasecmp (url, "satlist"))
					modtype = SHORTCUT_ITEM_LIST;
				else if (!g_strcasecmp (url, "satmap"))
					modtype = SHORTCUT_ITEM_MAP;
				else if (!g_strcasecmp (url, "satsky"))
					modtype = SHORTCUT_ITEM_SKY;
				else if (!g_strcasecmp (url, "satglobe"))
					modtype = SHORTCUT_ITEM_GLOBE;
				
				/* open popup menu */
				iconbar_popup_item_menu (name, modtype, &event->button);
				/* free mem */
				g_free (url);
				gdk_pixbuf_unref (pbuf);
				
				break;
			}
		default:
			break;
		}
		break;
	default:
		break;
	}


}


void
iconbar_remove_module (gchar *name, guint type)
{
	/* This function should be called when an existing  module
	   is opened or deleted. It will remove that module entry from
	   the iconbar.
	   URL is a string representation of the module type.
	   For opened modules that are closed, it could be called from
	   the satmodule_remove_module function.
	*/
	gchar *modurl=NULL,*modname=NULL;
	GdkPixbuf *pbuf=NULL;
	gint item,n;

	n = e_shortcut_model_get_num_items (iconbar->model, SHORTCUT_GROUP_MODULES);
	for (item=0; item<n; item++) {
		e_shortcut_model_get_item_info (iconbar->model,
						SHORTCUT_GROUP_MODULES, item,
						&modurl, &modname, &pbuf);
		/* what kind of module? */
		if ( !g_strcasecmp (modtypes[type], modurl) && !g_strcasecmp (name, modname) ) {
			e_shortcut_model_remove_item (iconbar->model,
						      SHORTCUT_GROUP_MODULES, 
						      item);
			g_free (modurl);
			g_free (modname);
			gdk_pixbuf_unref (pbuf);
			return;
		}
		g_free (modurl);
		g_free (modname);
		gdk_pixbuf_unref (pbuf);
	}

}
	
		


void
iconbar_add_module (gchar *name, guint type)
{
	/* This function adds a new module entry to the iconbar.
	   It could/should be called when an open module is
	   closed.
	*/
	gchar *path;
	GdkPixbuf *pbuf;
	gint item_num;

	switch (type) {
	case SHORTCUT_ITEM_LIST:
		path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/satlist.png", NULL);
		pbuf = gdk_pixbuf_new_from_file (path);
		item_num = e_shortcut_model_add_item (iconbar->model,
						      SHORTCUT_GROUP_MODULES, -1,
						      modtypes[SHORTCUT_ITEM_LIST],
						      name, pbuf);
		gdk_pixbuf_unref (pbuf);
		g_free (path);
		break;
	case SHORTCUT_ITEM_MAP:
		path = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/map.png", NULL);
		pbuf = gdk_pixbuf_new_from_file (path);
		item_num = e_shortcut_model_add_item (iconbar->model,
						      SHORTCUT_GROUP_MODULES, -1,
						      modtypes[SHORTCUT_ITEM_MAP],
						      name, pbuf);
		gdk_pixbuf_unref (pbuf);
		g_free (path);
		break;
	default:
		break;
	}
}
