/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satmap.c: Graphical visualization module for Gnome Predict showing
            selected satellites on a map.


  Copyright (C)  2001-2003  Alexandru Csete.

  Authors:   Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/


#include <gnome.h>
#include <libgnomeui/gnome-window-icon.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gdk-pixbuf/gnome-canvas-pixbuf.h>
#include <gconf/gconf-client.h>
#include <math.h>
#include <string.h>
#include "satlog.h"
#include "satdata.h"
#include "sataos.h"
#include "fileio.h"
#include "util.h"
#include "qth.h"
#include "satmodules.h"
#include "iconbar.h"
#include "sattabs.h"
#include "extra-widgets.h"
#include "satmap-druid.h"
#include "satmap-print.h"
#include "satmap.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif

#ifndef PI
#  define PI 3.141592653589793
#endif

#ifndef TWOPI
#  define TWOPI 2.0*PI
#endif

#ifndef HALFPI
#  define HALFPI PI/2.0
#endif

#ifndef R0
#  define R0 6378.135
#endif


enum satmap_mode_e {
	SATMAP_MODE_NONE = 0,  /* actually not a valid mode... */
	SATMAP_MODE_TAB,       /* widget is in the notebook */
	SATMAP_MODE_WIN,       /* widget is in a separate window */
	SATMAP_MODE_FULLSCR    /* wisget is blowing out screen! */
};


/* Private function prototypes */
static void satmap_switch_mode_cb     (GtkWidget *, GtkWidget *);
static void satmap_save_cb            (GtkWidget *, GtkWidget *);
static void satmap_store_filename     (GtkWidget *, GtkWidget *);
static void satmap_filesel_destroy_cb (GtkWidget *, GtkWidget *);
static void satmap_fullscreen_cb      (GtkWidget *, GtkWidget *);
static void satmap_configure_cb       (GtkWidget *, gpointer);
static void satmap_close_cb           (GtkWidget *, GtkWidget *);
static void satmap_close_cleanup      (GtkWidget *);
static gint satmap_timer_cb           (gpointer);
static gdouble arccos                 (gdouble, gdouble);
static void satmap_prefs_changed      (GConfClient *, guint, GConfEntry *, gpointer);


gint satwin_delete_cb (GtkWidget *, GdkEvent *, GtkWidget *);
void satwin_close_cb  (GtkWidget *, GtkWidget *);


extern qth_struc qth;       /* qth.c  */
extern GtkWidget *app;      /* main.c */
extern GConfClient *client; /* main.c */
extern GtkWidget *app;      /* main.c */

void satmap_open_cb (GtkWidget *widget, gpointer data)
{
	/* This is just a gtk-callback function that calls
	   satlist_open.
	*/
	satmap_open (NULL, TRUE);
}


gint satmap_open (gchar *name, gboolean addtolist)
{
	/* This function creates the widget that shows selected
	   satellites on a map. The configuration file is expected
	   to be located at $HOME/.gpredict/name.satmap. If name is
	   NULL the druid will be invoked.
	   If addtolist is TRUE, the module will be added to the 
	   list of open modules (this should not be the case when
	   we start the program and open the modules from the
	   existing list).
	*/

	/* widgets */
	GtkWidget *hbox,*vbox,*pixmap;
	GtkWidget *close,*save,*print,*config,*tabwin,*fullscr;
	GtkWidget *satmap;
	GnomeCanvasItem *mapitem,*qthmark,*qthlabel,*mapgroup,*bgd;
	GdkPixbuf *map;
	GtkTooltips *kooltip;
	/* sat related */
	sat_t *sat;             /* pointer to a single sat struct */
	satmap_sat_t *smsat = NULL;    /* pointer to a single satmap sat struct */
	GList *slist = NULL;    /* list of satmap sat structs */
	/* range circle */
	gdouble ssplat,ssplong,beta,azimuth,rangelat,rangelong,num,dem,sx,sy,sx2;
	gdouble deg2rad=1.74532925199e-2;
	guint azi;
	/* other vars */
	gchar *configfile,*menulabel,*pixmapfile;
	gchar *tmp,**argv = NULL;
	gint argc,i,timer,delay;
	guint gconf_id;
	gint satindex;
	gchar *logmsg;

	/**** FIXME: If there is an open module with the specified name, the new
	      module will not be created, but the contents of the config file
	      will be overwritten! See bug #521816 in bug tracker.
	*/
	if (!name)
		name = g_strdup (satmap_druid_run (NULL));

	/* Try again */
	if (!name)
		return -1;

	/* Test whether module exists or not */
	if (!satmodules_test_module (name, SATMOD_TYPE_MAP)) {
		/* delete from list, just in case ... */
		satmodules_remove_module (name, SATMOD_TYPE_MAP);
		return -1;
	}

	if (addtolist) {
		/* is there a module with the specified name already open? */
		if (satmodules_test_open_modules (name, SATMOD_TYPE_MAP)) {
			gnome_app_error (GNOME_APP (app),
					 _("There is already a MAP with the selected name open.\n"\
					   "Please close that MAP or specify another name."));
			return -1;
		}
		iconbar_remove_module (name, SATMOD_TYPE_MAP);
		satmodules_add_module (name, SATMOD_TYPE_MAP);
	}
	/* Create config file path and get sats */
	configfile = g_strconcat ("=", g_get_home_dir (), "/.gpredict/",
				  name, ".satmap=", SATMAP_SAT_LIST_PATH, NULL);
	gnome_config_get_vector (configfile, &argc, &argv);
	g_free (configfile);

	/* Empty list? */
	if (!g_strcasecmp (argv[0], "")) {
		/**********  FIXME: fix up error dialog */
		satmodules_remove_module (name, SATMOD_TYPE_MAP);
		return -1;
	}

	/* get some default values */
	/**** FIXME: Add more */
	delay = gconf_client_get_int (client, SATMAP_TIMER_PATH, NULL);
	if (!delay)
		delay = SATMAP_DEF_TIMER;

	/****  buttons  ****/
	/* close button */
	pixmapfile = g_strconcat ( PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				   G_DIR_SEPARATOR_S, "cancel.png", NULL);
	pixmap = gnome_pixmap_new_from_file (pixmapfile);
	close = gnome_pixmap_button (pixmap, _("  Close"));
	g_free (pixmapfile);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, close, _("Close this map"), NULL);
	/* mode */
	pixmapfile = g_strconcat ( PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				   G_DIR_SEPARATOR_S, "windows_small.png", NULL);
	pixmap = gnome_pixmap_new_from_file (pixmapfile);
	tabwin = gnome_pixmap_button (pixmap, _("  Mode"));
	g_free (pixmapfile);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, tabwin, _("Switch betwen notebook and window mode"), NULL);

	/* config */
/*  	pixmapfile = g_strconcat ( PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons", */
/*  				   G_DIR_SEPARATOR_S, "prefs_20.png", NULL); */
/*  	pixmap = gnome_pixmap_new_from_file (pixmapfile); */
/*  	config = gnome_pixmap_button (pixmap, _("Settings")); */
/*  	g_free (pixmapfile); */
/*  	kooltip = gtk_tooltips_new (); */
/*  	gtk_tooltips_set_tip (kooltip, config, _("Configure this map"), NULL); */

	/* save */
	pixmapfile = g_strconcat ( PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				   G_DIR_SEPARATOR_S, "save_as_20.png", NULL);
	pixmap = gnome_pixmap_new_from_file (pixmapfile);
	save = gnome_pixmap_button (pixmap, _("  Save"));
	g_free (pixmapfile);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, save, _("Save map to file"), NULL);
	/* print */
	pixmapfile = g_strconcat ( PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				   G_DIR_SEPARATOR_S, "print_20.png", NULL);
	pixmap = gnome_pixmap_new_from_file (pixmapfile);
	print = gnome_pixmap_button (pixmap, _("  Print"));
	g_free (pixmapfile);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, print, _("Print map"), NULL);
	/* full screen */
/*  	fullscr = gtk_button_new_with_label (_("Full Screen")); */

	hbox = gtk_hbutton_box_new ();
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (hbox), 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (hbox), GTK_BUTTONBOX_START);
	gtk_box_pack_start (GTK_BOX (hbox), close, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), tabwin, TRUE, TRUE, 0);
/*  	gtk_box_pack_start (GTK_BOX (hbox), config, TRUE, TRUE, 0); */
	gtk_box_pack_start (GTK_BOX (hbox), save, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), print, TRUE, TRUE, 0);
/*  	gtk_box_pack_start (GTK_BOX (hbox), fullscr, TRUE, TRUE, 0); */


	/* Gnome Canvas */
	gtk_widget_push_visual (gdk_rgb_get_visual ());
	gtk_widget_push_colormap (gdk_rgb_get_cmap ());
	satmap = gnome_canvas_new ();
	gtk_widget_pop_visual ();
	gtk_widget_pop_colormap ();
	gtk_widget_set_usize (satmap, SATMAP_CANVAS_WIDTH, SATMAP_CANVAS_HEIGHT);
	gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (satmap), 1);
	gnome_canvas_set_scroll_region (GNOME_CANVAS (satmap),
					0.0, 0.0,
					SATMAP_CANVAS_WIDTH,
					SATMAP_CANVAS_HEIGHT);

	/* create black border */
	bgd = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (satmap)),
				     gnome_canvas_rect_get_type (),
				     "x1", (gdouble) 0.0,
				     "y1", (gdouble) 0.0,
				     "x2", (gdouble) SATMAP_CANVAS_WIDTH,
				     "y2", (gdouble) SATMAP_CANVAS_HEIGHT,
				     "fill_color", "black",
				     NULL);

	/* Create a main group which corresponds to the position and
	   size of the map. This is to allow a black border around the
	   world map. Thus all canvas items that should be displayed
	   on the map should use this group instead of the canvas group
	   (gnome_canvas_root).
	*/
	mapgroup = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (satmap)),
					  gnome_canvas_group_get_type (),
					  "x", (SATMAP_CANVAS_WIDTH-SATMAP_MAP_WIDTH)/2.0,
					  "y", (SATMAP_CANVAS_HEIGHT-SATMAP_MAP_HEIGHT)/2.0,
					  NULL);

	/* Load Earth map */
	configfile = g_strconcat ("=", g_get_home_dir (), "/.gpredict/",
				  name, ".satmap=", SATMAP_SAT_MAP_PATH, NULL);
	tmp = gnome_config_get_string (configfile);
	g_free (configfile);

	if (!tmp) {
		/* try default map */
		tmp = gconf_client_get_string (client, SATMAP_DEFMAP_PATH, NULL);
		if (!tmp) {
			/* try hardcoded version */
			tmp = g_strconcat (PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "maps",
					   G_DIR_SEPARATOR_S, "earth_800.jpg", NULL);
		}
	}
	map = gdk_pixbuf_new_from_file (tmp);
	g_free (tmp);
	
	/* create a new canvas item and scale it if it has a different size */
	mapitem = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
					 gnome_canvas_pixbuf_get_type (),
					 "pixbuf", map,
					 "width_set", TRUE,
					 "height_set", TRUE,
					 "width", SATMAP_MAP_WIDTH,
					 "height", SATMAP_MAP_HEIGHT,
					 NULL );
//	gnome_canvas_item_lower_to_bottom (mapitem);
	gdk_pixbuf_unref (map);

	/* QTH */
	qthmark = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
					 gnome_canvas_rect_get_type (),
					 "x1", CX(qth.lon)-1.0, "y1", CY(qth.lat)-1.0,
					 "x2", CX(qth.lon)+1.0, "y2", CY(qth.lat)+1.0,
					 "fill_color", "cyan",
					 NULL);
	qthlabel = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
					  gnome_canvas_text_get_type (),
					  "text", qth.name,
					  "x", CX(qth.lon)+3.0,
					  "y", CY(qth.lat),
					  "anchor", GTK_ANCHOR_W,
					  "font", "-*-clean-medium-r-*-*-12-*-*-*-*-*-*-*",
					  "fill_color", "cyan",
					  NULL);

	/* create list of satellites */
	for (i=0; i<argc; i++) {
		/* check whether we really have this satellite in the database */
		satindex = satdata_get_sat_index (0, argv[i]);
		if (satindex!= -1) {
			smsat = SATMAP_SAT (g_malloc (sizeof (satmap_sat_t)));
			smsat->name = g_strdup (argv[i]);
			smsat->index = satindex;
			sat = satdata_get_sat (smsat->index);
			/* canvas group for this satellite (reference coords) */
			smsat->group = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
							      gnome_canvas_group_get_type (),
							      "x", CX (sat->lon), "y", CY (sat->lat), NULL);
			smsat->mark = gnome_canvas_item_new (GNOME_CANVAS_GROUP (smsat->group),
							     gnome_canvas_rect_get_type (),
							     "x1", -1.0, "y1", -1.0,
							     "x2", 1.0, "y2", 1.0,
							     "fill_color", "yellow",
							     NULL);
			smsat->text = gnome_canvas_item_new (GNOME_CANVAS_GROUP (smsat->group),
							     gnome_canvas_text_get_type (),
							     "text", smsat->name,
							     "x", 0.0, "y", 1.0,
							     "anchor", GTK_ANCHOR_N,
							     "font", "-*-clean-medium-r-*-*-12-*-*-*-*-*-*-*",
							     "fill_color", "yellow",
							     NULL);
			
			/* Range Circle Calculations
			   Borrowed from gsat 0.9.0 by Xavier Crehueras, EB3CZS
			   who borrowed from John Magliacane, KD2BD.
			*/
			ssplat = sat->lat*deg2rad;
			ssplong = sat->lon*deg2rad;
			beta = (0.5*sat->fp)/R0;
			
			for (azi=0; azi<SATMAP_RANGE_CIRCLE_POINTS; azi++)
			{
				azimuth = deg2rad*(double)azi;
				rangelat = asin(sin(ssplat)*cos(beta)+cos(azimuth)*sin(beta)*cos(ssplat));
				num = cos(beta)-(sin(ssplat)*sin(rangelat));
				dem = cos(ssplat)*cos(rangelat);
				
				if (azi==0 && (beta > HALFPI-ssplat))
					rangelong = ssplong+PI;
				
				else if (azi==180 && (beta > HALFPI+ssplat))
					rangelong = ssplong+PI;
				
				else if (fabs(num/dem) > 1.0)
					rangelong = ssplong;
				
				else {
					if ((180-azi) >= 0)
						rangelong = ssplong-arccos(num,dem);
					else
						rangelong = ssplong+arccos(num,dem);
				}
				
				while (rangelong < 0.0)
					rangelong += TWOPI;
				
				while (rangelong > (2.0*PI))
					rangelong -= TWOPI;
				
				rangelat = rangelat/deg2rad;
				rangelong = rangelong/deg2rad;
				
				/* Convert range circle data to map-based
				   coordinates and draw on map */
				
				if (rangelong > 180.0)
				{
					rangelong = rangelong-180.0;
					sx = (gint)(800-(rangelong*800/360.0));
				}
				else
					sx = (gint)((800/2.0)-(rangelong*800/360.0));
				
				sy = (gint)((400/2.0)-(rangelat*400/180.0));
				
				/* store position for Plot Range Circle*/
				/**** FIXME: VERY SLOW!!!!! */
				smsat->range[azi] = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
									   gnome_canvas_rect_get_type (),
									   "x1", sx, "x2", sx,
									   "y1", sy, "y2", sy,
									   "fill_color", "yellow",
									   "outline_color", "yellow",
									   "width_pixels", 1,
									   NULL);
				/* fixes bug #537372 */
				if (sx < CX (sat->lon))
					sx += SATMAP_MAP_WIDTH;
				/* Mirror the half circle to the left */
				sx2 = sx-2*(sx-CX(sat->lon));
				if (sx2 < 0.0) /* outside the map */
					sx2 = SATMAP_MAP_WIDTH+sx2; 
				smsat->range[azi+180] = gnome_canvas_item_new (GNOME_CANVAS_GROUP (mapgroup),
									       gnome_canvas_rect_get_type (),
									       "x1", sx2, "x2", sx2,
									       "y1", sy, "y2", sy,
									       "fill_color", "yellow",
									       "outline_color", "yellow",
									       "width_pixels", 1,
									       NULL);
				
			}
			slist = g_list_append (slist, smsat);
		}
		else {
			logmsg = g_strdup_printf (_("%s: Satellite %s not in database!"),
						  __FUNCTION__, argv[i]);
			satlog_log (SAT_LOG_WARNING, logmsg);
			g_free (logmsg);
		}
	}
	g_strfreev (argv);

	/* Create main vertical box */
	vbox = gtk_vbox_new (FALSE, 0);

	/* start timer */
	timer = gtk_timeout_add (delay, satmap_timer_cb, vbox);

	/* attach some data to the main container widget */
	gtk_object_set_data (GTK_OBJECT (vbox), "timer", GINT_TO_POINTER (timer));
	gtk_object_set_data (GTK_OBJECT (vbox), "name", name);
	gtk_object_set_data (GTK_OBJECT (vbox), "list", slist);
	gtk_object_set_data (GTK_OBJECT (vbox), "mode", GUINT_TO_POINTER (SATMAP_MODE_TAB));
	gtk_object_set_data (GTK_OBJECT (vbox), "qthm", qthmark);
	gtk_object_set_data (GTK_OBJECT (vbox), "qthl", qthlabel);

	gtk_box_pack_start (GTK_BOX (vbox), satmap, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

	/* add gconf notifier (module reloads prefs autiomatically) */
	gconf_id = gconf_client_notify_add (client, SATMAP_ROOT_PATH,
					    satmap_prefs_changed,
					    vbox, NULL, NULL);
	gtk_object_set_data (GTK_OBJECT (vbox), "gconfid", GUINT_TO_POINTER (gconf_id));

	/* Connect button signals or disable */
/*  	gtk_widget_set_state (config, GTK_STATE_INSENSITIVE); */
	gtk_signal_connect (GTK_OBJECT (save), "clicked",
			    GTK_SIGNAL_FUNC (satmap_save_cb), satmap);
	gtk_signal_connect (GTK_OBJECT (print), "clicked",
			    GTK_SIGNAL_FUNC (satmap_print_cb), satmap);
/*  	gtk_widget_set_state (fullscr, GTK_STATE_INSENSITIVE); */
	gtk_signal_connect (GTK_OBJECT (tabwin), "clicked",
			    GTK_SIGNAL_FUNC (satmap_switch_mode_cb), vbox);
	gtk_signal_connect (GTK_OBJECT (close), "clicked",
			    GTK_SIGNAL_FUNC (satmap_close_cb), vbox);

	gtk_widget_show_all (vbox);
	menulabel = g_strconcat (name, _(" (map)"), NULL);
	sattabs_add_tab_menu (vbox, name, menulabel);
	g_free (menulabel);
// you can open in window using this...
//	satmap_switch_mode_cb (tabwin, vbox);

	return 0;
}


static void
satmap_switch_mode_cb (GtkWidget *button, GtkWidget *vbox)
{
	/* This function is called when the user clicks on the
	   Tab/Window button.
	*/
	guint mode;
	gchar *name,*menulabel,*buff;
	gint page;
	GtkWidget *satwin;

	mode = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (vbox), "mode"));
	name = gtk_object_get_data (GTK_OBJECT (vbox), "name");
	switch (mode) {
	case SATMAP_MODE_TAB:
		page = sattabs_lookup_tab (vbox);
		if (page != -1) {
			/* delete from notebook */
			gtk_widget_ref (vbox);
			sattabs_remove_tab (page);
			/* update state */
			gtk_object_set_data (GTK_OBJECT (vbox), "mode",
					     GUINT_TO_POINTER (SATMAP_MODE_WIN));
			/* create window */
			satwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
			gtk_object_set_data (GTK_OBJECT (vbox), "window", satwin);
			gtk_window_set_title (GTK_WINDOW (satwin), name);
			/* window icon */
			buff = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/map.png", NULL);
			gnome_window_icon_set_from_file (GTK_WINDOW (satwin), buff);
			g_free (buff);
			/* connect window signals */
			gtk_signal_connect (GTK_OBJECT (satwin), "delete-event",
					    GTK_SIGNAL_FUNC (satwin_delete_cb), vbox);
			gtk_signal_connect (GTK_OBJECT (satwin), "destroy",
					    GTK_SIGNAL_FUNC (satwin_close_cb), vbox);
			gtk_container_add (GTK_CONTAINER (satwin), vbox);
			gtk_widget_show_all (satwin);
		}
		break;
	case SATMAP_MODE_WIN:
		/* get and destroy the window */
		satwin = gtk_object_get_data (GTK_OBJECT (vbox), "window");
		gtk_object_remove_data (GTK_OBJECT (vbox), "window");
		gtk_container_remove (GTK_CONTAINER (satwin), vbox);
		gtk_widget_destroy (satwin);
		/* reconfigure module */
		gtk_object_set_data (GTK_OBJECT (vbox), "mode",
				     GUINT_TO_POINTER (SATMAP_MODE_TAB));
		menulabel = g_strconcat (name, _(" (map)"), NULL);
		sattabs_add_tab_menu (vbox, name, menulabel);
		g_free (menulabel);
		break;
	case SATMAP_MODE_FULLSCR:
		break;
	default:
		satlog_log (SAT_LOG_CRITICAL, _("SATMAP: Invalid mode (BUG!)"));
	}
}

static void
satmap_save_cb  (GtkWidget *widget, GtkWidget *canvas)
{
	/* This function is called when the user clicks on the
	   Save button. It pops up a file selector dialog.
	*/
	GdkImlibImage *tmp = 0;
	GtkWidget *filesel,*filter,*hbox,*label;

	/* get image */
	tmp = gdk_imlib_create_image_from_drawable (canvas->window, NULL,
						    0, 0,
						    SATMAP_CANVAS_WIDTH,
						    SATMAP_CANVAS_HEIGHT);

	/* create fileselector widget and attach image */
	filesel = gtk_file_selection_new (_("Save Map As..."));
	gtk_object_set_data (GTK_OBJECT (filesel), "image", (gpointer) tmp);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC (satmap_store_filename),
			    (gpointer) filesel);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC (satmap_filesel_destroy_cb),
			    (gpointer) filesel);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION(filesel)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC (satmap_filesel_destroy_cb),
			    (gpointer) filesel);

	/* Create filter widget */
	label = gtk_label_new (_("File type:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	filter = filesel_graphic_filter ();
	gtk_object_set_data (GTK_OBJECT (filesel), "filter", filter);

	hbox = gtk_hbox_new (TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 5);
	gtk_box_pack_start (GTK_BOX (hbox), filter, TRUE, TRUE, 5);
	gtk_container_add (GTK_CONTAINER (GTK_FILE_SELECTION (filesel)->main_vbox),
			   hbox);

	gtk_widget_show_all (filesel);
}


/* in extra-widgets.c */
extern const gchar *gfx_filter_desc[];
extern const gchar *gfx_filter_ext[];


static void
satmap_store_filename (GtkWidget *button, GtkWidget *filesel)
{
	/* This function is called when the user clicks on the
	   OK-button of the file selector dialog.
	*/
	gchar *fname=NULL,*tmp=NULL;
	gint i=0;
	guint filti;
	GtkWidget *menu,*item;

	menu = gtk_option_menu_get_menu (GTK_OPTION_MENU (gtk_object_get_data (GTK_OBJECT (filesel), "filter")));
	item = gtk_menu_get_active (GTK_MENU (menu));
	filti = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (item), "filter"));
	if (filti >= GFX_FILTER_NUM) {
		satlog_log (SAT_LOG_CRITICAL, _("Selected file type out of range (bug)"));
		gnome_app_error (GNOME_APP (app), _("Selected file type out of range (bug)"));
		return;
	}
	if (filti) {
		/* User selected type */
		tmp = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));
		if (tmp) {
			fname = g_strconcat (tmp, ".", gfx_filter_ext[filti], NULL);
			i = gdk_imlib_save_image ((GdkImlibImage *) gtk_object_get_data (GTK_OBJECT (filesel), "image"),
						  fname, 0);
		}
	}
	else {
		/* Automatic file type from extension */
		fname = gtk_file_selection_get_filename (GTK_FILE_SELECTION (filesel));
		if (fname) {
			i = gdk_imlib_save_image((GdkImlibImage *) gtk_object_get_data (GTK_OBJECT (filesel),
											"image"),
						 fname, 0);
		}
	}
/*  	if (tmp) */
/*  		g_free (tmp); */
/*  	if (fname) */
/*  		g_free (fname); */
	if (i)
		satlog_log (SAT_LOG_INFO, _("SATMAP: gdk_imlib_save() returned non-zero value."));
}


static void
satmap_filesel_destroy_cb (GtkWidget *button, GtkWidget *filesel)
{
	gdk_imlib_destroy_image ((GdkImlibImage *) gtk_object_get_data (GTK_OBJECT (filesel), "image"));
	gtk_widget_destroy (GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (filesel), "filter")));
	gtk_widget_destroy (filesel);
}

static void
satmap_fullscreen_cb  (GtkWidget *widget, GtkWidget *vbox)
{
}


static void
satmap_configure_cb (GtkWidget *widget, gpointer data)
{
}


static void
satmap_close_cb (GtkWidget *button, GtkWidget *vbox)
{
	/* This function is called when the user clicks on
	   the "Close" button. It checks whether the Satmap
	   module is in a notebook tab or in a window, and
	   removes the widget.
	*/
	gint tabpos;
	GtkWidget *window;
	gchar *module;

	/* stop timer */
	gtk_timeout_remove (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (vbox), "timer")));
	gtk_object_remove_data (GTK_OBJECT (vbox), "timer");
	/* remove GConf notifier */
	gconf_client_notify_remove (client, GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (vbox), "gconfid")));

	module = g_strdup (gtk_object_get_data (GTK_OBJECT (vbox), "name"));
	gtk_object_remove_data (GTK_OBJECT (vbox), "name");
	gtk_object_remove_data (GTK_OBJECT (vbox), "mode");
	/* re-add module to iconbar */
	iconbar_add_module (module, SATMOD_TYPE_MAP);
	/* remove module from list */
	satmodules_remove_module (module, SATMOD_TYPE_MAP );
	g_free (module);

	tabpos = sattabs_lookup_tab (vbox);
	if (tabpos != -1) {
		sattabs_remove_tab (tabpos);
		satmap_close_cleanup (vbox);
		gtk_widget_destroy (vbox);
	}
	else {
		window = gtk_object_get_data (GTK_OBJECT (vbox), "window");
		gtk_object_remove_data (GTK_OBJECT (vbox), "window");
		satmap_close_cleanup (vbox);
		gtk_widget_destroy (vbox);
		if (window)
			gtk_widget_destroy (window);
	}
}


static void
satmap_close_cleanup (GtkWidget *vbox)
{
	/* This function cleans up some memory that has been
	   allocated to a satmap. The data is attached to the
	   "vbox".
	*/
	GList *slist;

        slist = (GList *) gtk_object_get_data (GTK_OBJECT (vbox), "list");
	gtk_object_remove_data (GTK_OBJECT (vbox), "list");
	while (slist) {
		g_free (SATMAP_SAT (slist->data)->name);
		g_free (slist->data);
		slist = g_list_next (slist);
	}
	slist = g_list_first (slist);
	g_list_free (slist);
}


static gint
satmap_timer_cb (gpointer data)
{
	/* This function does the update on the satmap.
	   It is called automatically at regular intervals.
	*/
	GList *list;
	satmap_sat_t *ssat;
	sat_t *sat;
	/* range circle */
	gdouble ssplat,ssplong,beta,azimuth,rangelat,rangelong,num,dem,sx,sy,sx2;
	gdouble deg2rad=1.74532925199e-2;
	gchar *logmsg;
	guint azi;
	GnomeCanvasItem *item;

	/* update QTH mark */
	item = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (data), "qthm"));
	gnome_canvas_item_set (item, "x1", CX(qth.lon)-1.0, "y1", CY(qth.lat)-1.0,
			       "x2", CX(qth.lon)+1.0, "y2", CY(qth.lat)+1.0, NULL);
	/* update QTH label */
	item = GNOME_CANVAS_ITEM (gtk_object_get_data (GTK_OBJECT (data), "qthl"));
	gnome_canvas_item_set (item, "text", qth.name, "x", CX(qth.lon)+3.0, "y", CY(qth.lat), NULL);

	/* get the list of satellites */
	list = (GList *) gtk_object_get_data (GTK_OBJECT (data), "list");

	while (list) {
		ssat = (satmap_sat_t *)list->data;
		sat = satdata_get_sat (ssat->index);
		if (sat) {

			gnome_canvas_item_set (ssat->group,
					       "x", CX (sat->lon),
					       "y", CY (sat->lat),
					       NULL);
			/* we need to do this, otherwise canvas won't get updated */
			gnome_canvas_item_set (ssat->mark,
					       "x1", -1.0,
					       NULL);
			gnome_canvas_item_set (ssat->text,
					       "x", 0.0, "y", 1.0,
					       NULL);
			/* Range Circle Calculations
			   Borrowed from gsat 0.9.0 by Xavier Crehueras, EB3CZS
			   who borrowed from John Magliacane, KD2BD.
			*/
			/* as of CVS rev. 1.36 some optimizations have been done by alexc */
			ssplat = sat->lat*deg2rad;
			ssplong = sat->lon*deg2rad;
			beta = (0.5*sat->fp)/R0;
			for (azi=0; azi<SATMAP_RANGE_CIRCLE_POINTS; azi++)
			{
				azimuth = deg2rad*(double)azi;
				rangelat = asin(sin(ssplat)*cos(beta)+cos(azimuth)*sin(beta)*cos(ssplat));
				num = cos(beta)-(sin(ssplat)*sin(rangelat));
				dem = cos(ssplat)*cos(rangelat);
				
				if (azi==0 && (beta > HALFPI-ssplat))
					rangelong = ssplong+PI;
				
				else if (azi==180 && (beta > HALFPI+ssplat))
					rangelong = ssplong+PI;
				
				else if (fabs(num/dem) > 1.0)
					rangelong = ssplong;
				
				else {
					if ((180) >= azi)
						rangelong = ssplong-arccos(num,dem);
					else
						rangelong = ssplong+arccos(num,dem);
				}
				
				while (rangelong < 0.0)
					rangelong += TWOPI;
				
				while (rangelong > (TWOPI))
					rangelong -= TWOPI;
				
				rangelat = rangelat/deg2rad;
				rangelong = rangelong/deg2rad;
				
				/* Convert range circle data to map-based
				   coordinates and draw on map */
				if (rangelong > 180.0)
				{
					rangelong = rangelong-180.0;
					sx = (gint)(SATMAP_MAP_WIDTH-(rangelong*SATMAP_MAP_WIDTH/360.0));
				}
				else
					sx = (gint)((SATMAP_MAP_WIDTH/2.0)-(rangelong*SATMAP_MAP_WIDTH/360.0));
				
				sy = (gint)((SATMAP_MAP_HEIGHT/2.0)-(rangelat*SATMAP_MAP_HEIGHT/180.0));
				
				/* store position for Plot Range Circle*/
				/**** FIXME: VERY SLOW!!!!! */
/*  			gnome_canvas_item_set (ssat->range[azi], "x1", sx, "x2", sx, */
/*  					       "y1", sy, "y2", sy, NULL); */
				/* Let's try to access rect->re coordinates directly ... uuuhh... */
				GNOME_CANVAS_RECT (ssat->range[azi])->re.x1 = sx;
				GNOME_CANVAS_RECT (ssat->range[azi])->re.x2 = sx;
				GNOME_CANVAS_RECT (ssat->range[azi])->re.y1 = sy;
				GNOME_CANVAS_RECT (ssat->range[azi])->re.y2 = sy;
				gnome_canvas_item_request_update (ssat->range[azi]);
				/* fixes bug #537372 */
				if (sx < CX (sat->lon))
					sx += SATMAP_MAP_WIDTH;
				/* Mirror the half circle to the left */
				sx2 = sx-2*(sx-CX(sat->lon));
				if (sx2 < 0.0) /* outside the map */
					sx2 = SATMAP_MAP_WIDTH+sx2;
				/* TO SLOOOOW... */
/*  			gnome_canvas_item_set (ssat->range[azi+180], "x1", sx2, "x2", sx2, */
/*  					       "y1", sy, "y2", sy, NULL); */
				GNOME_CANVAS_RECT (ssat->range[azi+180])->re.x1 = sx2;
				GNOME_CANVAS_RECT (ssat->range[azi+180])->re.x2 = sx2;
				GNOME_CANVAS_RECT (ssat->range[azi+180])->re.y1 = sy;
				GNOME_CANVAS_RECT (ssat->range[azi+180])->re.y2 = sy;
				gnome_canvas_item_request_update (ssat->range[azi+180]);
			}
		}
		else {
			logmsg = g_strdup_printf (_("%s: Missing satellite detected. This satellite is now a zombie."),
						    __FUNCTION__);
			satlog_log (SAT_LOG_WARNING, logmsg);
			g_free (logmsg);
		}
		list = g_list_next (list);
	}

	return TRUE;
}




gint satwin_delete_cb (GtkWidget *widget, GdkEvent *event, GtkWidget *vbox)
{
	/* This function is connected to the "delete" event signal
	   of the satmap window. It returns false, so that the window
	   will be destroyed, ie. the next function will be called.
	*/
	satmap_close_cb (NULL, vbox);
	return FALSE;
}


void satwin_close_cb (GtkWidget *widget, GtkWidget *vbox)
{
	/****** FIXME: should clean up some memory */

	/* This function is connected to the "destroy" signal
	   of the satmap window. It calls the satmap_close
	   function, which in turn will clean up the memory
	   allocated for one module.
	*/



}



static gdouble arccos (gdouble x, gdouble y)
{
	/* This function implements the arccosine function,
	   returning a value between 0 and two pi.
	   Borrowed from gsat 0.9 by Xavier Crehueras, EB3CZS
	*/

	/* somewhat optimized version by alexc (as of CVS rev. 1.36 */

	if (x && y) {
		if (y > 0.0)
			return acos (x/y);
		else if (y < 0.0)
			return PI + acos (x/y);
	}
	return 0.0;
}


static void
satmap_prefs_changed (GConfClient *client, guint cid, GConfEntry *entry, gpointer vbox)
{
	/* This function reloads the preferences for the satellite map.
	   It is called automatically by GConf when a value under
	   /apps/gpredict/satmap is changed.
	*/

	gint timer,delay;

	gtk_timeout_remove (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (vbox), "timer")));
	delay = gconf_client_get_int (client, SATMAP_TIMER_PATH, NULL);
	if (!delay)
		delay = SATMAP_DEF_TIMER;
	timer = gtk_timeout_add (delay, satmap_timer_cb, vbox);
	gtk_object_set_data (GTK_OBJECT (vbox), "timer", GINT_TO_POINTER (timer));
} 
