##
# /toggle whitelist_notify [default ON]
# Print a message in the status window if someone not on the whitelist messages us
#
# /toggle whitelist_log_ignored_msgs [default ON]
# if this is on, ignored messages will be logged to ~/.irssi/whitelist.log
#
# /set whitelist_nicks phyber etc
# nicks that are allowed to msg us (whitelist checks for a valid nick before a valid host)
#
# /toggle whitelist_nicks_case_sensitive [default OFF]
# do we care which case nicknames are in?
#
# /set whitelist_hosts *!*@*isp.com *!ident@somewhere.org
# Hosts that are allowed to message us, space delimited
#
# Thanks to Geert for help/suggestions on this script
##

use strict;
use Irssi;
use Irssi::Irc;

use vars qw($VERSION %IRSSI);
$VERSION = "0.9a";
%IRSSI = (
	authors		=> "David O\'Rourke",
	contact		=> "irssi \[at\] null-routed.com",
	name		=> "whitelist",
	description	=> "Whitelist specific nicks or hosts and ignore messages from anyone else.",
	licence		=> "GPLv2",
	changed		=> "10.7.2003 20:48"
);

my $whitenick;
my $whitehost;
my $tstamp;

my %htr = ( );
foreach my $i (0..255) {
	my $ch = chr($i);
	$htr{$ch} = "\Q$ch\E";
}
$htr{'?'} = '.';
$htr{'*'} = '.*';

sub host_to_regexp($) {
	my ($mask) = @_;
	$mask = lc_host($mask);
	$mask =~ s/(.)/$htr{$1}/g;
	return $mask;
}

sub lc_host($) {
	my ($host) = @_;
	$host =~ s/(.+)\@(.+)/sprintf("%s@%s", $1, lc($2));/eg;
	return $host;
}

sub timestamp {
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
	my @timestamp = ($year+1900,$mon+1,$mday,$hour,$min,$sec);
	$tstamp = "$timestamp[0]-$timestamp[1]-$timestamp[2] $timestamp[3]:$timestamp[4]:$timestamp[5] :";
}

sub whitelist_check {
	my ($server, $msg, $nick, $address) = @_;
	my @nicks = Irssi::settings_get_str('whitelist_nicks');
	my @hosts = Irssi::settings_get_str('whitelist_hosts');
	my $warning = Irssi::settings_get_bool('whitelist_notify');
	my $casesensitive = Irssi::settings_get_bool('whitelist_nicks_case_sensitive');
	my $logging = Irssi::settings_get_bool('whitelist_log_ignored_msgs');
	my $logfile = Irssi::get_irssi_dir."/whitelist.log";
	my $hostmask = "$nick!$address";
	my $tag = $server->{tag};
	my $valid = 0;
	
	foreach my $whitenick (split(/ /, "@nicks")) {
		if ($casesensitive == 0) {
			$nick = lc($nick);
			$whitenick = lc($whitenick);
		}
		if ($nick eq $whitenick) {
			$valid = 1;
		}
	last if ($valid == 1);
	}
	
	return if ($valid == 1);
	
	foreach my $whitehost (split(/ /, "@hosts")) {
		$whitehost = host_to_regexp($whitehost);
		if ($hostmask =~ /$whitehost/) {
			$valid = 1;
		}
	last if ($valid == 1);
	}
	
	return if ($valid == 1);
	
	# stop if the message isn't from a whitelisted address
	if ($valid == 0) {
		# print a notice if that setting is enabled
		# this could flood your status window if someone is flooding you with messages
		if ($warning == 1) {
			Irssi::print "\[$tag\] $nick \[$address\] attempted to send private message.";
		}
		
		if ($logging == 1) {
			timestamp();
			open(WLLOG, ">>$logfile") or return;
			print WLLOG "$tstamp"." \[$tag\] $nick \[$address\]: $msg\n";
			close(WLLOG);
		}
		Irssi::signal_stop();
		return;
	}
}

Irssi::settings_add_bool('whitelist', 'whitelist_notify' => 1);
Irssi::settings_add_bool('whitelist', 'whitelist_log_ignored_msgs' => 1);
Irssi::settings_add_str('whitelist', 'whitelist_nicks', '');
Irssi::settings_add_bool('whitelist', 'whitelist_nicks_case_sensitive' => 0);
Irssi::settings_add_str('whitelist', 'whitelist_hosts', '');

Irssi::signal_add_first('message private', 'whitelist_check');
