#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <pthread.h>

#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include "gnet_lib.h"
#include "gnet_engine.h"
#include "gnet_search.h"
#include "gnet_channel.h"
#include "gnet_msg.h"
#include "gnet_proto.h"

struct query*
gnet_find_query(struct gnet *gnet, char *id){
    struct list_head *p, *tmp;
    struct query *q;

    list_for_each_safe(p, tmp, &gnet->g_queries){
	q = list_entry(p, struct query, list);

	if(time(NULL) - q->stamp > q->time_out){
	    TRACE("query expired");

	    list_del(&q->list);
	    free(q);
	    continue;
	}

	if(!strncmp(q->guid, id, 16)){
	    TRACE("query found");
	    return q;
	}
    }

    TRACE("query not found");

    return NULL;
}

void
gnet_stop_search(struct gnet *gnet, unsigned long id){
    struct list_head *p;
    struct query *q;

    pthread_mutex_lock(&gnet->g_channels_lock);

    list_for_each(p, &gnet->g_queries){
	q = list_entry(p, struct query, list);
	if(q->id == id){
	    TRACE("query found");
	    TRACE("search stopped");
	    list_del(&q->list);
	    free(q);
	    break;
	}

    }

    pthread_mutex_unlock(&gnet->g_channels_lock);
    
}

int
gnet_start_search(struct gnet *gnet, char *txt, void(*cb)(void*, struct gnet_locator*, unsigned long), void *ctx, unsigned to, unsigned long *srch_id){
    struct message *msg;
    struct query *query;
    char *data;

    TRACE("starting search for %s...", txt);

    if(!(query = malloc(sizeof(struct query))))
	return -1;

    if(!(msg = gnet_create_message(NULL, GNET_MSG_QUERY, gnet->g_cfg->query_ttl, 0, strlen(txt) + 3))){
	WARN("could not create message: %s", strerror(errno));
	free(query);
	return -1;
    }

    data = msg->m_data + GNET_HDR_SIZE;
    *(uint16_t*)data = HTOGS(gnet->g_cfg->min_speed);
    strcpy(data + 2, txt);

    memcpy(query->guid, msg->m_data, 16);
    query->call_back = cb;
    query->ctx = ctx;
    query->stamp = time(NULL);
    query->id = gnet->g_search_seq++;

    if(srch_id)
	*srch_id = query->id;

    if(to)
	query->time_out = to;
    else
	query->time_out = gnet->g_cfg->query_wait;

    pthread_mutex_lock(&gnet->g_channels_lock);

    list_add_tail(&query->list, &gnet->g_queries);
    gnet_deliver_message_all(gnet, NULL, msg);

    pthread_mutex_unlock(&gnet->g_channels_lock);

    gnet_engine_signal(gnet, 0);

    return 0;
}

int
gnet_send_push(struct gnet *gnet, struct gnet_locator *loc, unsigned short port){
    struct list_head *p;
    struct channel *chan;
    struct message *msg;
    struct sockaddr_in addr;
    socklen_t len;
    char *data;
    int found = 0;

    TRACE("sending a push for %s", loc->name);

    if(!(msg = gnet_create_message(NULL, GNET_MSG_PUSH, gnet->g_cfg->query_ttl, 0, 26))){
	WARN("could not create message: %s", strerror(errno));
	return -1;
    }

    data = msg->m_data + GNET_HDR_SIZE;

    memcpy(data, loc->guid, 16);
    *(uint32_t*)(data + 16) = HTOGL(loc->index);
    *(uint16_t*)(data + 24) = HTOGS(port);

    pthread_mutex_lock(&gnet->g_channels_lock);

    list_for_each_prev(p, &gnet->g_channels){
	chan = list_entry(p, struct channel, c_list);

	if(chan->c_state == CHANNEL_STATE_CONNECTED){
	    len = sizeof(struct sockaddr_in);
	    
	    if(getsockname(chan->c_fd, (struct sockaddr*)&addr, &len) < 0){
		TRACE("getsockname call failed: %s", strerror(errno));	
	    }else{
		TRACE("local interface: %s", inet_ntoa(*(struct in_addr*)&addr.sin_addr.s_addr));

		*(uint32_t*)(data + 20) = addr.sin_addr.s_addr;		
		found = 1;

		break;
	    }
	}	
    }

    if(!found){
	WARN("could not determine local interface ip!");
	pthread_mutex_unlock(&gnet->g_channels_lock);
	return -1;
    }

    gnet_deliver_message_all(gnet, NULL, msg);
    pthread_mutex_unlock(&gnet->g_channels_lock);

    gnet_engine_signal(gnet, 0);

    return 0;
}
