#include <fcntl.h>
#include <unistd.h>

#include <vector>
#include <string>

#include <lufs/proto.h>
#include <lufs/fs.h>

using namespace std;

#include "tools.h"
#include "file_handle.h"
#include "fs_modules.h"

/********************************************************************************
				Helper Classes
********************************************************************************/

FileHandle* FileHandleArray::Find( const char* name )
{
	FileHandle *p;
    TRACE("looking for " << name << ", size=" << size());
	
    for(iterator i = begin(); i != end(); i++ )
	{
		p = *i;
		if(p && p->name == name)
		{
			TRACE("handle found");
			return p;
		}
    }
    return NULL;
}

int FileHandleArray::Delete( const char* name )
{
	FileHandle *p;
	int r;
    TRACE("closing for " << name << ", size=" << size());
    for(iterator i = begin(); i != end(); i++ )
	{
		p = *i;
		if(p && p->name == name)
		{
			TRACE("handle found");
			erase(i);
			r = p->Close();
			delete p;
			return r;
		}
    }
	return 0;
}

void FileHandleArray::Insert( FileHandle* handle )
{
	TRACE("Inserting : "<<handle->name);
	FileHandle *p;
	int r;
    for(iterator i = begin(); i != end(); i++ )
	{
		p = *i;
		if(p && p->name == handle->name)
		{
			TRACE("handle found");
			if( p !=handle )
			{
				erase(i);
				r = p->Close();
				delete p;
				push_back(handle);
			}
			return;
		}
    }
	
	push_back( handle );
}

void FileHandleArray::Cleanup()
{
	TRACE("Cleanup. ");
	time_t now = time(0);
	FileHandle *p;
	bool more = false;
	do
	{
		more = false;
		for(iterator i = begin(); i != end(); i++ )
		{
			p = *i;
			if(p && p->IsTimedOut(now))
			{
				fprintf(stderr,"deleting %s",p->name.c_str());
				//TRACE("deleting "<<handle->name.c_str());
				erase(i);
				delete p;
				more = true;
				break;	// stupid STL iterators break on erase()
			}
		}
	} while( more );
}

/********* FileHandle methods **************/

FileHandle::FileHandle(const char* _name, const char* _real_name )
{
	name = _name;
	real_name = _real_name;
	KeepAlive();
}

FileHandle::~FileHandle()
{
}

void FileHandle::KeepAlive()
{
	timeout = time(0) + file_handle_timeout;
}

bool FileHandle::IsTimedOut(time_t t)
{
	return t>timeout;
}

/********* FileHandle_Default methods **************/

FileHandle_Default::FileHandle_Default(const char* name, const char* real_name) : FileHandle(name,real_name)
{
	handle = -1;
}

FileHandle_Default::~FileHandle_Default()
{
	Close();
}

int FileHandle_Default::Open()
{
	TRACE(Identity()<<" tries to open "<<name);
	if( handle != -1 )
		return handle;
	else
		return handle = open(name.c_str(),O_RDONLY);
}

int FileHandle_Default::Close()
{
	if( handle != -1 )
		return close(handle);
	else
		return 0;
}

int FileHandle_Default::Stat(struct lufs_fattr *fattr)
{
    struct stat stat;

    if(fstat(handle, &stat) < 0)
		return -1;

    fattr->f_mode = stat.st_mode;
    fattr->f_nlink = stat.st_nlink;
    fattr->f_uid = (getuid() == stat.st_uid) ? 1 : 0;
    fattr->f_gid = (getgid() == stat.st_gid) ? 1 : 0;
    fattr->f_size = stat.st_size;
    fattr->f_atime = stat.st_atime;
    fattr->f_mtime = stat.st_mtime;
    fattr->f_ctime = stat.st_ctime;

    return 0;
}

int FileHandle_Default::Read(long long pos, unsigned long length, char* buf)
{
	if(Open()<0)
		return -1;

    if(lseek(handle, pos, SEEK_SET) < 0)
	{
		TRACE("seek failed");
		return -1;
    }

    int res = read(handle, buf, length);
    TRACE("read: "<< res);
    return res;
}


/********* FileHandle_XWav methods **************/

FileHandle_XWav::FileHandle_XWav(const char* name, const char* real_name, const char* cmd) :
	SeekablePipe(name, real_name, cmd)
{
}

FileHandle_XWav::~FileHandle_XWav()
{
	Close();
}

typedef struct
{
	char riff[4];
	unsigned long file_length;
	char wave[4];
} wav_header_t;

// stats original file instead.
int FileHandle_XWav::Stat(struct lufs_fattr *fattr)
{
    struct stat st;

	TRACE("Statting : " << real_name);
	
    if(stat(real_name.c_str(), &st) < 0)
		return -1;

	wav_header_t header;
	if( Read(0,sizeof(header),(char*)&header) != sizeof(header) )
		return -1;
	
	TRACE(header.riff);
	TRACE(header.wave);
	
	if(memcmp(header.riff,"RIFF",4))
	{
		ERROR("No RIFF!");
		return -1;
	}
	if(memcmp(header.wave,"WAVE",4))
	{
		ERROR("No WAVE!");
		return -1;
	}
	
    fattr->f_mode = st.st_mode;
    fattr->f_nlink = st.st_nlink;
    fattr->f_uid = (getuid() == st.st_uid) ? 1 : 0;
    fattr->f_gid = (getgid() == st.st_gid) ? 1 : 0;
    fattr->f_size = header.file_length+8;
    fattr->f_atime = st.st_atime;
    fattr->f_mtime = st.st_mtime;
    fattr->f_ctime = st.st_ctime;

	TRACE("Stat : Success ! : "<<real_name);
    return 0;
}

/********* FileCache **************/

SeekablePipe::SeekablePipe(const char* name, const char* real_name, const char* cmd) : FileHandle(name,real_name)
{
	already_read = 0;
	eof = true;
	pipe = 0;
	tmp_file = 0;
	command = cmd;
}

SeekablePipe::~SeekablePipe()
{
	Close();
}

int SeekablePipe::Open()
{
	if(!pipe)
	{
		TRACE("virtual file name : "<<name);
		TRACE("   real file name : "<<real_name);
		TRACE("          command : "<<command);
		
		// open pipe from process
		pipe = popen(command.c_str(), "r");
		if( !pipe )
			goto error;
		TRACE("Opened.");
		eof = false;
		already_read = 0;
	}

	if(!tmp_file)
	{
		// create temporary filename
		tmp_file = tmpfile();
		TRACE("Temp file created.");
		if( !tmp_file )
			goto error;
	}
	
	return 0;
	
error:
	TRACE("Can(t open.");
	Close();
	return -1;
}

int SeekablePipe::Stat(struct lufs_fattr *fattr)
{
	return -1;	// can't stat a pipe !
}

int SeekablePipe::Read(long long pos, unsigned long length, char* buf)
{
	if(Open()<0)
		return -1;

	Cache(pos+length);

    if(fseek(tmp_file, pos, SEEK_SET))
	{
		TRACE("seek failed");
		return -1;
    }

    int res = fread(buf, 1, length, tmp_file);
    return res;
}

int SeekablePipe::Cache( long long upto )
{
	char block[32768];
	if(eof) return -1;
		
	while(already_read < upto)
	{
		int nr = fread( block, 1, sizeof(block), pipe );
		eof = feof(pipe);
		if(!nr) break;
	    if(fseek(tmp_file, 0, SEEK_END))
		{
			TRACE("seek failed");
			return -1;
	    }
		fwrite(block,nr,1,tmp_file);
		already_read += nr;
	}
	return 0;
}

int SeekablePipe::Close()
{
	int x=0;
	if(pipe)
	{
		x=fclose(pipe);
		pipe=0;
	}
	if(tmp_file)
	{
		x|=fclose(tmp_file);
		tmp_file=0;
	}
	return x;
}


/********* FileHandle_Flac methods **************/

FileHandle_Flac::FileHandle_Flac(const char* name, const char* real_name) :
	FileHandle_XWav(name, real_name, (string("flac -c -d \"")+real_name+"\"").c_str())
{
}

/*
FileHandle_Ogg::FileHandle_Ogg(const char* name, const char* real_name) :
	FileHandle_XWav(name, real_name, (string("oggdec -o - \"")+real_name+"\"").c_str())
{
}

FileHandle_MP3::FileHandle_MP3(const char* name, const char* real_name) :
	FileHandle_XWav(name, real_name, (string("mpg123 -w - \"")+real_name+"\"").c_str())
{
}
*/
