/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 asyncsocket.cpp  -  Core asynchronous socket wrapper

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/
 
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>

#ifdef HAVE_SYS_POLL_H
#include <sys/poll.h>
#else
#include "fakepoll.h"
#endif

#include <unistd.h>
#include <fcntl.h>
#include <typeinfo>

#include "mutella.h"
#include "asyncsocket.h"
#include "common.h"
#include "asyncdns.h"

bool is_fd_ok(SOCKET fd);
static HANDLE s_PollThread = (unsigned long)-1;

MAsyncSocket::MAsyncSocket()
{
	m_hSocket = INVALID_SOCKET;
	m_bConnecting = false;
	m_bListening = false;
	m_bNewSocket = true;
	m_nSelectFlags = 0;
	m_bSelectActive = false;
	m_dwActivity = 0;
	m_nLastError = errno;
	m_nAsyncDnsReqID = -1;
}

MAsyncSocket::~MAsyncSocket()
{
	//ASSERT(s_PollThread == (unsigned long)-1 || s_PollThread == MThread::currentThread());
	if (m_hSocket != INVALID_SOCKET)
		Close();
}

BOOL MAsyncSocket::Create(UINT nSocketPort /*= 0*/, int nSocketType /*=SOCK_STREAM*/,
		long lEvent /*= FD_READ | FD_WRITE | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE*/,
		LPCTSTR lpszSocketAddress /*=NULL*/)
{
//#ifdef _DEBUG
//	if (lEvent & FD_WRITE)
//		TRACE2("WARNING: creating a socket with FD_WRITE on for ", typeid(*this).name());
//#endif
	if (Socket(nSocketType, lEvent))
	{
		if (Bind(nSocketPort,lpszSocketAddress))
			return TRUE;
		int nResult = GetLastError();
		Close();
		SetLastError(nResult);
	}
	return FALSE;
}

BOOL MAsyncSocket::Attach(SOCKET hSocket,
		long lEvent /*= FD_READ | FD_WRITE | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE*/)
{
//#ifdef _DEBUG
//	if (lEvent & FD_WRITE)
//		TRACE2("WARNING: attaching a socket with FD_WRITE on for ", typeid(*this).name());
//#endif
	ASSERT(hSocket != INVALID_SOCKET);

	m_hSocket = hSocket;
	MAsyncSocket::AttachHandle(hSocket, this);
	
	m_bSelectActive = true;
	// notify Heartbeat() that something has changed
	SetDirty();
	return AsyncSelect(lEvent);
}
	
SOCKET MAsyncSocket::Detach()
{
	SOCKET hSocket = m_hSocket;
	m_bConnecting = false;
	m_bListening = false;
	if (AsyncSelect(0))
	{
		MAsyncSocket::KillSocket(hSocket, this);
		m_hSocket = INVALID_SOCKET;
		// notify Heartbeat() that something has changed
		SetDirty();
		return hSocket;
	}
	// notify Heartbeat() that something has changed
	SetDirty();
	return INVALID_SOCKET;
}

BOOL MAsyncSocket::GetPeerName(CString& rPeerAddress, UINT& rPeerPort)
{
	SOCKADDR_IN sockAddr;
	memset(&sockAddr, 0, sizeof(sockAddr));
	MLock lock(g_LibcMutex);

	socklen_t nSockAddrLen = sizeof(sockAddr);
	BOOL bResult = GetPeerName((SOCKADDR*)&sockAddr, &nSockAddrLen);
	if (bResult)
	{
		rPeerPort = ntohs(sockAddr.sin_port);
		rPeerAddress = ::inet_ntoa(sockAddr.sin_addr);
		m_nLastError = errno;
	}
	return bResult;
}

BOOL MAsyncSocket::GetPeerName(SOCKADDR* lpSockAddr, socklen_t* lpSockAddrLen)
{
	BOOL bRes = ( 0 == ::getpeername(m_hSocket, lpSockAddr, lpSockAddrLen) );
	m_nLastError = errno;
	return bRes;
}

BOOL MAsyncSocket::GetSockName(CString& rSocketAddress, UINT& rSocketPort)
{
	SOCKADDR_IN sockAddr;
	memset(&sockAddr, 0, sizeof(sockAddr));
	MLock lock(g_LibcMutex);

	socklen_t nSockAddrLen = sizeof(sockAddr);
	BOOL bResult = GetSockName((SOCKADDR*)&sockAddr, &nSockAddrLen);
	if (bResult)
	{
		rSocketPort = ntohs(sockAddr.sin_port);
		rSocketAddress = ::inet_ntoa(sockAddr.sin_addr);
		m_nLastError = errno;
	}
	return bResult;
}

BOOL MAsyncSocket::GetSockName(SOCKADDR* lpSockAddr, socklen_t* lpSockAddrLen)
{
	BOOL bRes = ( 0 == ::getsockname(m_hSocket, lpSockAddr, lpSockAddrLen) );
	m_nLastError = errno;
	return bRes;
}

BOOL MAsyncSocket::Accept(MAsyncSocket& rConnectedSocket,
		SOCKADDR* lpSockAddr /*=NULL*/, socklen_t* lpSockAddrLen /*=NULL*/)
{
	ASSERT(rConnectedSocket.m_hSocket == INVALID_SOCKET);
	//ASSERT(MAsyncSocket::FromHandle(INVALID_SOCKET) == NULL);

	MAsyncSocket::AttachHandle(INVALID_SOCKET, &rConnectedSocket);

	SOCKET hTemp = ::accept(m_hSocket, lpSockAddr, lpSockAddrLen);
	m_nLastError = errno;

	if (hTemp == INVALID_SOCKET)
	{
		DWORD dwProblem = GetLastError();
		MAsyncSocket::DetachHandle(rConnectedSocket.m_hSocket);
		rConnectedSocket.m_hSocket = INVALID_SOCKET;
		SetLastError(dwProblem);
	}
	else /*if (MAsyncSocket::FromHandle(INVALID_SOCKET) != NULL)*/
	{
		rConnectedSocket.m_hSocket = hTemp;
		MAsyncSocket::DetachHandle(INVALID_SOCKET);
		MAsyncSocket::AttachHandle(hTemp, &rConnectedSocket);
		rConnectedSocket.m_bSelectActive = true;
		
		int val = 1;
		::setsockopt(hTemp,SOL_SOCKET,SO_REUSEADDR,(char*)&val,sizeof(val));
		::fcntl(hTemp,F_SETFL,O_NONBLOCK);
		m_nLastError = errno;
	}

	return (hTemp != INVALID_SOCKET);
}

BOOL MAsyncSocket::Bind(UINT nSocketPort, LPCTSTR lpszSocketAddress /*=NULL*/)
{
	SOCKADDR_IN sockAddr;
	memset(&sockAddr,0,sizeof(sockAddr));

	LPSTR lpszAscii = T2A((LPTSTR)lpszSocketAddress);
	sockAddr.sin_family = AF_INET;

	if (lpszAscii == NULL)
		sockAddr.sin_addr.s_addr = htonl(INADDR_ANY);
	else
	{
		MLock lock(g_LibcMutex);
		DWORD lResult = ::inet_addr(lpszAscii);
		m_nLastError = errno;
		if (lResult == INADDR_NONE)
		{
			SetLastError(EINVAL);
			return FALSE;
		}
		sockAddr.sin_addr.s_addr = lResult;
	}

	sockAddr.sin_port = htons((u_short)nSocketPort);

	return Bind((SOCKADDR*)&sockAddr, sizeof(sockAddr));
}

BOOL MAsyncSocket::Bind(const SOCKADDR* lpSockAddr, socklen_t nSockAddrLen)
{
	BOOL bRes = ( 0 == bind(m_hSocket, lpSockAddr, nSockAddrLen) );
	m_nLastError = errno;
	return bRes;
}

void MAsyncSocket::Close()
{
	m_bConnecting = false;
	m_bListening = false;
	if (m_hSocket != INVALID_SOCKET)
	{
		if (0 != ::close(m_hSocket))
		{
			TRACE2("WARNING: closing closed socket ", m_hSocket);
		}
		m_nLastError = errno;
		MAsyncSocket::KillSocket(m_hSocket, this);
		m_hSocket = INVALID_SOCKET;
	}
	// notify Heartbeat() that something has changed
	SetDirty();
}

/*BOOL MAsyncSocket::Connect(LPCTSTR lpszHostAddress, UINT nHostPort)
{
	ASSERT(lpszHostAddress != NULL);
	
	MLock lock(g_LibcMutex);

	SOCKADDR_IN sockAddr;
	memset(&sockAddr,0,sizeof(sockAddr));

	LPSTR lpszAscii = T2A((LPTSTR)lpszHostAddress);
	sockAddr.sin_family = AF_INET;
	sockAddr.sin_addr.s_addr = ::inet_addr(lpszAscii);
	m_nLastError = errno;

	if (sockAddr.sin_addr.s_addr == INADDR_NONE)
	{
		LPHOSTENT lphost;
		lphost = ::GetHostByName(lpszAscii);
		if (lphost != NULL)
			sockAddr.sin_addr.s_addr = ((LPIN_ADDR)lphost->h_addr)->s_addr;
		else
		{
			SetLastError(EINVAL);
			return FALSE;
		}
	}

	sockAddr.sin_port = htons((u_short)nHostPort);

	return Connect((SOCKADDR*)&sockAddr, sizeof(sockAddr));
}*/

class MAsyncConnectByName : public MAsyncDns {
public:
	MAsyncConnectByName(const CString& sHost, UINT nPort, SOCKET hSocket) :
		MAsyncDns(sHost),
		m_hSocket(hSocket),
		m_nPort(nPort)
	{
	}
	void OnErrorDelayed() {
		MAsyncSocket* pS = MAsyncSocket::FromHandle(m_hSocket);
		if (!pS || pS->m_nAsyncDnsReqID != m_nRequestID)
		{
			if (pS)
			{
				TRACE3("OnResolveDelayed() : socket object for fd ", m_hSocket, " is around, but ID is wrong");
				TRACE4("OnResolveDelayed() : (pS->m_nAsyncDnsReqID != m_nRequestID) : ", pS->m_nAsyncDnsReqID, " != ", m_nRequestID);
			}
			else
			{
				TRACE3("OnResolveDelayed() : socket object for fd ", m_hSocket, " has gone");
			}
			return;
		}
		//TRACE3("OnResolveDelayed() : socket object for fd ", m_hSocket, " is still around");
		pS->Close();
		pS->SetLastError(GetLastError());
		pS->OnClose(GetLastError());
	}
	void OnResolveDelayed() {
		MAsyncSocket* pS = MAsyncSocket::FromHandle(m_hSocket);
		if (!pS || pS->m_nAsyncDnsReqID != m_nRequestID)
		{
			if (pS)
			{
				TRACE3("OnResolveDelayed() : socket object for fd ", m_hSocket, " is around, but ID is wrong");
				TRACE4("OnResolveDelayed() : (pS->m_nAsyncDnsReqID != m_nRequestID) : ", pS->m_nAsyncDnsReqID, " != ", m_nRequestID);
			}
			else
			{
				TRACE3("OnResolveDelayed() : socket object for fd ", m_hSocket, " has gone");
			}
			return;
		}
		//TRACE3("OnResolveDelayed() : socket object for fd ", m_hSocket, " is still around");
		SOCKADDR_IN sockAddr;
		memset(&sockAddr,0,sizeof(sockAddr));
		sockAddr.sin_family = AF_INET;
		sockAddr.sin_addr.s_addr = m_nHostS_addr;
		sockAddr.sin_port = htons((u_short)m_nPort);
		//
		if ( (!pS->Connect((SOCKADDR*)&sockAddr, sizeof(sockAddr))) && pS->GetLastError()!=EINPROGRESS )
		{
			int nErrno = pS->GetLastError();
			pS->Close();
			pS->SetLastError(nErrno);
			pS->OnClose(nErrno);
		}
	}
protected:
	SOCKET m_hSocket;
	UINT   m_nPort;
};

BOOL MAsyncSocket::Connect(LPCTSTR lpszHostAddress, UINT nHostPort)
{
	ASSERT(lpszHostAddress != NULL);

	MLock lock(g_LibcMutex);

	SOCKADDR_IN sockAddr;
	memset(&sockAddr,0,sizeof(sockAddr));

	LPSTR lpszAscii = T2A((LPTSTR)lpszHostAddress);
	sockAddr.sin_family = AF_INET;
	sockAddr.sin_addr.s_addr = ::inet_addr(lpszAscii);
	m_nLastError = errno;

	if (sockAddr.sin_addr.s_addr == INADDR_NONE)
	{
		lock.unlock();
		SetLastError(EINPROGRESS);
		MAsyncConnectByName* pACN = new MAsyncConnectByName(lpszHostAddress, nHostPort, m_hSocket);
		m_nAsyncDnsReqID = pACN->GetRequestID();
		pACN->Release();
		return FALSE;
	}

	sockAddr.sin_port = htons((u_short)nHostPort);
	return Connect((SOCKADDR*)&sockAddr, sizeof(sockAddr));
}

BOOL MAsyncSocket::Connect(const SOCKADDR* lpSockAddr, socklen_t nSockAddrLen)
{
    SOCKADDR sockAddr;
	memcpy(&sockAddr,lpSockAddr,nSockAddrLen);
	int retval;
    bool bRes = (0 == (retval = ::connect(m_hSocket, &sockAddr, nSockAddrLen)));
    m_nLastError = errno;
	m_bConnecting = bRes ? true : (m_nLastError == EINPROGRESS);
	m_bSelectActive = m_bConnecting;
	// notify Heartbeat() that something has changed
	SetDirty();
	return bRes;
}

//BOOL MAsyncSocket::IOCtl(long lCommand, DWORD* lpArgument)
//{
//	return (SOCKET_ERROR != ioctl(m_hSocket, lCommand, lpArgument));
//}

BOOL MAsyncSocket::Listen(int nConnectionBacklog /*=5*/)
{
	m_bListening = (0 == ::listen(m_hSocket, nConnectionBacklog));
	m_nLastError = errno;
	m_bSelectActive = m_bListening;
	// notify Heartbeat() that something has changed
	SetDirty();
	return m_bListening;
}

int MAsyncSocket::Receive(void* lpBuf, int nBufLen, int nFlags /*=0*/)
{
	int nRec = ::recv(m_hSocket, (LPSTR)lpBuf, nBufLen, nFlags);
	m_nLastError = errno;
	return nRec;
}

BOOL MAsyncSocket::ShutDown(int nHow /*=sends*/)
{
	// notify Heartbeat() that something has changed
	SetDirty();
	BOOL bRes = ( 0 == ::shutdown(m_hSocket,nHow) );
	m_nLastError = errno;
	return bRes;
}

int MAsyncSocket::Send(const void* lpBuf, int nBufLen, int nFlags /*=0*/)
{
	if (is_fd_ok(m_hSocket))
	{
		int nSent = ::send(m_hSocket, (LPSTR)lpBuf, nBufLen, nFlags);
		m_nLastError = errno;
		return nSent;
	}
	return 0;
}

BOOL MAsyncSocket::AsyncSelect(long lEvent /*= FD_READ | FD_WRITE | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE*/)
{
	long nOldFlags = m_nSelectFlags;
	m_nSelectFlags = lEvent;
	// notify Heartbeat() that something has changed
	if (nOldFlags != m_nSelectFlags)
		SetDirty();
	return true;
}

BOOL MAsyncSocket::ModifySelectFlags(long lEventAdd, long lEventRemove)
{
	long nOldFlags = m_nSelectFlags;
	m_nSelectFlags &= ~lEventRemove;
	m_nSelectFlags |= lEventAdd;
	// notify Heartbeat() that something has changed
	if (nOldFlags != m_nSelectFlags)
		SetDirty();
	return true;
}

void MAsyncSocket::OnReceive(int nErrorCode)
{
}

void MAsyncSocket::OnSend(int nErrorCode)
{
}

void MAsyncSocket::OnOutOfBandData(int nErrorCode)
{
}

void MAsyncSocket::OnAccept(int nErrorCode)
{
}

void MAsyncSocket::OnConnect(int nErrorCode)
{
}

void MAsyncSocket::OnClose(int nErrorCode)
{
}

BOOL MAsyncSocket::Socket(int nSocketType /*=SOCK_STREAM*/,
		long lEvent /*= FD_READ | FD_WRITE | FD_OOB | FD_ACCEPT | FD_CONNECT | FD_CLOSE*/,
		int nProtocolType /*=0*/,
		int nAddressFormat /*=PF_INET*/)
{
//#ifdef _DEBUG
//	if (lEvent & FD_WRITE)
//		TRACE2("WARNING: calling 'socket' with FD_WRITE on for ",typeid(*this).name());
//#endif
	ASSERT(m_hSocket == INVALID_SOCKET);

	m_hSocket = socket(nAddressFormat,nSocketType,nProtocolType);
	if (m_hSocket != INVALID_SOCKET)
	{
		MAsyncSocket::AttachHandle(m_hSocket, this);
		
		int val = 1;
		setsockopt(m_hSocket,SOL_SOCKET,SO_REUSEADDR,(char*)&val,sizeof(val));
		fcntl(m_hSocket,F_SETFL,O_NONBLOCK);

		// notify Heartbeat() that something has changed
		SetDirty();
		return AsyncSelect(lEvent);
	}
	return FALSE;

}

// statics
int MAsyncSocket::GetLastError()
{
	return m_nLastError;
}

void MAsyncSocket::SetLastError(int nError)
{
	m_nLastError = nError;
}


//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//
// static socket maps and message scheduling
//

typedef map<SOCKET,MAsyncSocket*> MapSocket_MAS;

static MapSocket_MAS  s_mapSockets;
static MMutex         s_mutexMap;

// lazy poll vector filling
static MMutex          s_mutexPollVec;
static bool            s_bPollVecDirty = true;
static struct pollfd * s_aPollFDs = NULL;
static long            s_nPollFDs = 0;
static long            s_nPollVecSize = 0;
//static ulong           s_dwHeartbeatCounter = 0;

// implementation

void MAsyncSocket::SetDirty()
{
	s_mutexPollVec.lock();
	s_bPollVecDirty = true;
	s_mutexPollVec.unlock();
}

MAsyncSocket* MAsyncSocket::FromHandle(SOCKET hSocket)
{
	return MAsyncSocket::LookupHandle(hSocket);
}

void MAsyncSocket::KillSocket(SOCKET hSocket, MAsyncSocket* pSocket)
{
	//MAsyncSocket::AsyncSelect(hSocket, 0); //deselect
	ASSERT(hSocket == INVALID_SOCKET || MAsyncSocket::LookupHandle(hSocket) == pSocket);
	MAsyncSocket::DetachHandle(hSocket);
}

void MAsyncSocket::AttachHandle(SOCKET hSocket, MAsyncSocket* pSocket)
{
	s_mutexMap.lock();
	s_mapSockets[hSocket] = pSocket;
	// notify Heartbeat()
	MAsyncSocket::SetDirty();
	s_mutexMap.unlock();
	ASSERT(MAsyncSocket::LookupHandle(hSocket) == pSocket);
}

void MAsyncSocket::DetachHandle(SOCKET hSocket)
{
	s_mutexMap.lock();
	// notify Heartbeat()
	MAsyncSocket::SetDirty();
	s_mapSockets.erase(hSocket);
	s_mutexMap.unlock();
}

MAsyncSocket* MAsyncSocket::LookupHandle(SOCKET hSocket)
{
	MLock lock(s_mutexMap);
	MapSocket_MAS::iterator it = s_mapSockets.find(hSocket);
	if (it != s_mapSockets.end())
		return (*it).second;
	return NULL;
}

#ifndef _DEBUG

#define CALL_WITH_BLOCKING_CHECK(_object, _function)\
	_object->_function;

#else //_DEBUG

#define CALL_WITH_BLOCKING_CHECK(_object, _function)\
{\
	clock_t _TimeBefore = clock();\
	int _nTimeBefore = xtime();\
	CString _sClsName = typeid(*_object).name();\
	_object->_function;\
	double _dt = ((double)(clock() - _TimeBefore))/CLOCKS_PER_SEC;\
	if (_dt>0.2)\
		printf("blocking in %s::%s detected (%g ms)\n", _sClsName.c_str(), #_function, _dt*1000);\
	else if (_nTimeBefore + 1 < xtime())\
		printf("blocking in %s::%s detected with xtime\n", _sClsName.c_str(), #_function);\
}

#endif //_DEBUG

#define ACTIVITY_ITEM_SLOW 0x10
#define ACTIVITY_ITEM_FAST 0x04
struct SPollSock {
	int   nFD;
	short nEvents;
	unsigned long dwActivity;
};

inline bool byActivity (const SPollSock& s1, const SPollSock& s2)
{
	// sort decending by activity
	return (s1.dwActivity>s2.dwActivity);
}

void MAsyncSocket::Heartbeat(int nSleep)
{
	s_PollThread = MThread::currentThread();
	// called by the "main event loop" thread
	int retval;
	int nLastError;
	int nSockFlags;
	int nFDs;
	long lEvent;
	short nRevents;
	MAsyncSocket* pSocket;
	MapSocket_MAS::iterator ih;
	//struct pollfd * aPollFDs;
	// vector 'iterator'
	struct pollfd * pPFD;
	//
	s_mutexMap.lock();
	// first fill or update pollfds vector
	nFDs = s_mapSockets.size();
	if (!nFDs)
	{
		// the map is empty
		s_mutexMap.unlock();
		{
			// Do a sleep.  This will wait the *full* usec time.
			// Note nSleep is in *milliseconds*, not nanos.
			int nsresult = -1;
			struct timespec rqtp, rmtp;
			rqtp.tv_nsec = nSleep * 1000L;
			rqtp.tv_sec = rmtp.tv_sec = rmtp.tv_nsec = 0;

			while (nsresult!=0)
			{
				nsresult = safe_nanosleep(&rqtp, &rmtp);
				if (nsresult!=0)
				{
					if (errno==EINTR)
					{
						// Reset our timespecs and rerun
						// the remaining time.
						rqtp.tv_sec = rmtp.tv_sec;
						rqtp.tv_nsec = rmtp.tv_nsec;
						rmtp.tv_sec = rmtp.tv_nsec = 0;
					}
					else nsresult = 0; // got something else.
				}
			}
		}
		return;
	}
	// ok, we have fds in the map
	s_mutexPollVec.lock();
	if (s_bPollVecDirty || !s_aPollFDs)
	{
		// first select the sockets which have something to say and sort them by activity
		vector<SPollSock> vecSocks;
		vecSocks.reserve(nFDs); // this will make push_back quicker
		SPollSock* pBack;
		for (ih = s_mapSockets.begin(); ih!=s_mapSockets.end(); ++ih )
		{
			pSocket = (*ih).second;
			ASSERT(pSocket);
			if (pSocket->m_hSocket != INVALID_SOCKET && pSocket->m_bSelectActive && pSocket->m_nSelectFlags)
			{
				vecSocks.resize(vecSocks.size()+1);
				pBack = &vecSocks.back();
				pBack->nFD = (*ih).first;
				pBack->nEvents = 0;
				pBack->dwActivity = pSocket->m_dwActivity;
				// this will cause kind of decay
				pSocket->m_dwActivity *= 0x0f;
				pSocket->m_dwActivity >>= 4;
				//

				ASSERT(pSocket->m_hSocket == pBack->nFD);
				lEvent = pSocket->m_nSelectFlags;
				if ( lEvent & (FD_READ) )
					pBack->nEvents |= POLLIN;
				if ( lEvent & (FD_OOB) )
					pBack->nEvents |= POLLPRI;
				if ( lEvent & (FD_WRITE) )
					pBack->nEvents |= POLLOUT;
				// connecting
				if ( pSocket->m_bConnecting && (lEvent & (FD_CONNECT)) )
					pBack->nEvents |= POLLOUT;
				// listening
				if ( pSocket->m_bListening && (lEvent & (FD_ACCEPT)) )
					pBack->nEvents |= POLLIN;
			}
		}
		s_nPollFDs = vecSocks.size();
		// TODO: s_nPollFDs can be 0 here
		// sort the vector
		sort(vecSocks.begin(), vecSocks.end(), byActivity);
		// now copy this vector into the actual poll vector
		if (s_nPollFDs > s_nPollVecSize || s_nPollFDs*10 < s_nPollVecSize)
		{
			if (s_aPollFDs)
				delete [] s_aPollFDs;
			s_nPollVecSize = s_nPollFDs;
			s_aPollFDs = new pollfd[s_nPollVecSize];
			ASSERT(s_aPollFDs);
		}
		// vector 'iterator'
		pPFD = s_aPollFDs;
		//TRACE("updated poll array:");
		for (vector<SPollSock>::iterator iv = vecSocks.begin(); iv != vecSocks.end(); ++iv, ++pPFD)
		{
			pPFD->fd = iv->nFD;
			pPFD->events = iv->nEvents;
			pPFD->revents = 0;
			//TRACE4("socket = ", iv->nFD,"  activity = ", iv->dwActivity);
		}
		s_bPollVecDirty = false;
	}
	s_mutexPollVec.unlock();
	s_mutexMap.unlock();

	ASSERT(s_aPollFDs);
	ASSERT(s_nPollVecSize >= s_nPollFDs);
	// poll full array only 1/4 times
	// nope, this seems to cause lags
	//s_dwHeartbeatCounter = (s_dwHeartbeatCounter+1) % 4;
	//if (s_dwHeartbeatCounter)
	//	retval = poll(s_aPollFDs, s_nPollFDs/2, nSleep);
	//else
	retval = poll(s_aPollFDs, s_nPollFDs, nSleep);
	//
    nLastError = errno;
    if (retval>0)
    {
    	nFDs = s_nPollFDs;
    	for ( pPFD = s_aPollFDs; retval>0 && nFDs>0; ++pPFD, --nFDs)
    	{
    		if (nRevents=pPFD->revents)
    		{
    			--retval; // to save looping
    			pPFD->revents = 0; // we are going to use the same array next time
    			pSocket = MAsyncSocket::FromHandle(pPFD->fd);
    			if (pSocket)
    			{
    				ASSERT(pSocket->m_hSocket==pPFD->fd);
    				// ensure that the socket is non-blocking
    				if (pSocket->m_bNewSocket)
    				{
    					nSockFlags = fcntl(pPFD->fd,F_GETFL,0);
    					fcntl(pPFD->fd,F_SETFL,nSockFlags|O_NONBLOCK);	
    					pSocket->m_bNewSocket = false;
    				}
    				// acknowledge the socket's activity
    				//pSocket->m_dwActivity += s_dwHeartbeatCounter ? ACTIVITY_ITEM_FAST : ACTIVITY_ITEM_SLOW;
    				pSocket->m_dwActivity += ACTIVITY_ITEM_FAST;
    				
    				// call appropriate notifiers
    				if ( nRevents & (POLLERR|POLLHUP|POLLNVAL) )
    				{
    					// error condition
    					if ( (pSocket->m_nSelectFlags & FD_CLOSE) && (nRevents & POLLHUP) )
    					{
    						//printf("poll-hangup\n");
    						//pSocket->OnClose(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnClose(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    					pSocket->Close();
    					break;
    				}
    				if ( nRevents & POLLOUT )
    				{
						//TRACE2(">>> POLLOUT event for ", typeid(*pSocket).name());
    					if (pSocket->m_bConnecting)
    					{
    						pSocket->m_bConnecting = false;
							SetDirty(); // we'll have to rebuild the array
    						//pSocket->OnConnect(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnConnect(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    					else
    					{
							//TRACE2(">>> Calling OnSend() for ", typeid(*pSocket).name());
    						//pSocket->OnSend(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnSend(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    				}
    				if ( nRevents & POLLPRI )
    				{
    					//pSocket->OnOutOfBandData(nLastError);
    					CALL_WITH_BLOCKING_CHECK(pSocket,OnOutOfBandData(nLastError))
    					TRACE("OnOutOfBandData occured");
    					if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    						continue;
    				}
    				if ( nRevents & POLLIN )
    				{
    					if (pSocket->m_bListening)
    					{
    						//pSocket->OnAccept(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnAccept(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    					else
    					{
    						//pSocket->OnReceive(nLastError);
    						CALL_WITH_BLOCKING_CHECK(pSocket,OnReceive(nLastError))
    						if ( pSocket != MAsyncSocket::FromHandle(pPFD->fd) )
    							continue;
    					}
    				}
    			}
    			else
    			{
    				// TODO: proper analysis why it happens, close FD if required, etc
    				TRACE("AsyncSocket dissapeared while in poll");
    				::close(pPFD->fd);
    			}
    		}
    	}
    }
#ifdef _DEBUG
    else if (retval<0)
    {
		//printf("poll error: retwal=%d, errno=%d serving %d sockets\n", retval, errno, nFDs);
    }
#endif //_DEBUG
}

bool is_fd_ok(SOCKET fd)
{
	int nError = 0;
	socklen_t nSize = sizeof(nError);
	int nReturn = getsockopt(fd, SOL_SOCKET,SO_ERROR,(char*)&nError,&nSize);
	ASSERT(nReturn || nSize == sizeof(int));
	if (nError && nError!=EWOULDBLOCK && nError!=EAGAIN && nError!=EINPROGRESS)
	{
		//printf("possibly bad socket %d, error= %d\n",fd,nError);
		return false;
	}
	return 0 == nReturn;
}

