/****************************************************************************
** $Id: mthread_p.h,v 1.10 2002/07/07 13:37:28 maksik Exp $
**
** MThread class for Unix
**
** Created : 20001309
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of the kernel module of the Qt GUI Toolkit.
**
** This file may be distributed under the terms of the Q Public License
** as defined by Trolltech AS of Norway and appearing in the file
** LICENSE.QPL included in the packaging of this file.
**
** This file may be distributed and/or modified under the terms of the
** GNU General Public License version 2 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.
**
** Licensees holding valid Qt Enterprise Edition or Qt Professional Edition
** licenses may use this file in accordance with the Qt Commercial License
** Agreement provided with the Software.
**
** This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
** WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
**
** See http://www.trolltech.com/pricing.html or email sales@trolltech.com for
**   information about Qt Commercial License Agreements.
** See http://www.trolltech.com/qpl/ for QPL licensing information.
** See http://www.trolltech.com/gpl/ for GPL licensing information.
**
** Contact info@trolltech.com if any conditions of this licensing are
** not clear to you.
**
**********************************************************************/

#ifndef MTHREAD_P_H
#define MTHREAD_P_H

#include <errno.h>
#include <unistd.h>

#ifndef _USE_SGI_STL
#include <map>
#else
#include <sgi_stl/map>
#endif //_USE_SGI_STL

// Thread definitions for UNIX platforms

#if defined(_OS_LINUX_)
#  ifndef _XOPEN_SOURCE
#    define _XOPEN_SOURCE 500
#  endif //_XOPEN_SOURCE
// Linux: couldn't we query macros from <unistd.h> instead of library versions?
#  if (__GLIBC__ == 2) && (__GLIBC_MINOR__ == 0)
// Linux with glibc 2.0.x - POSIX 1003.4a thread implementation
#    define M_HAS_RECURSIVE_MUTEX
#    define M_USE_PTHREAD_MUTEX_SETKIND
#    define M_NORMAL_MUTEX_TYPE PTHREAD_MUTEX_ERRORCHECK_NP
#    define M_RECURSIVE_MUTEX_TYPE PTHREAD_MUTEX_RECURSIVE_NP
#  else
// Linux with glibc 2.1.x - POSIX 1003.1c thread implementation
#    define M_HAS_RECURSIVE_MUTEX
#    undef  M_USE_PTHREAD_MUTEX_SETKIND
#    define M_NORMAL_MUTEX_TYPE PTHREAD_MUTEX_DEFAULT
#    define M_RECURSIVE_MUTEX_TYPE PTHREAD_MUTEX_RECURSIVE
#  endif
#elif defined (_OS_CYGWIN_)
// CygWin
#  define M_HAS_RECURSIVE_MUTEX
#  undef  M_USE_PTHREAD_MUTEX_SETKIND
#  define M_NORMAL_MUTEX_TYPE PTHREAD_MUTEX_DEFAULT
#  define M_RECURSIVE_MUTEX_TYPE PTHREAD_MUTEX_RECURSIVE
#  define M_MANUAL_LOCK_TRACKING
#elif defined(_OS_OSF_)
// Digital UNIX - 4.0 and later has a POSIX 1003.1c implementation
//   - should we assume > 4.0?
#  define M_HAS_RECURSIVE_MUTEX
#  undef  M_USE_PTHREAD_MUTEX_SETKIND
#  define M_NORMAL_MUTEX_TYPE PTHREAD_MUTEX_ERRORCHECK
#  define M_RECURSIVE_MUTEX_TYPE PTHREAD_MUTEX_RECURSIVE
#elif defined(_OS_AIX_)
// AIX 4.3.x
#  define M_HAS_RECURSIVE_MUTEX
#  undef  M_USE_PTHREAD_MUTEX_SETKIND
#  define M_NORMAL_MUTEX_TYPE PTHREAD_MUTEX_ERRORCHECK
#  define M_RECURSIVE_MUTEX_TYPE PTHREAD_MUTEX_RECURSIVE
#elif defined(_OS_HPUX_)
// We only support HP/UX 11.x
#  define M_HAS_RECURSIVE_MUTEX
#  undef  M_USE_PTHREAD_MUTEX_SETKIND
#  define M_NORMAL_MUTEX_TYPE PTHREAD_MUTEX_ERRORCHECK
#  define M_RECURSIVE_MUTEX_TYPE PTHREAD_MUTEX_RECURSIVE
#elif defined (_OS_FREEBSD_) || defined(_OS_OPENBSD_)
// FreeBSD and OpenBSD use the same user-space thread implementation
#  define M_HAS_RECURSIVE_MUTEX
#  undef  M_USE_PTHREAD_MUTEX_SETKIND
#  define M_NORMAL_MUTEX_TYPE PTHREAD_MUTEX_ERRORCHECK
#  define M_RECURSIVE_MUTEX_TYPE PTHREAD_MUTEX_RECURSIVE
#elif defined(_OS_DARWIN_)
// Darwin in its current rev (xnu-201) has a pretty limited POSIX implementation
#  undef  M_HAS_RECURSIVE_MUTEX
#  undef  M_USE_PTHREAD_MUTEX_SETKIND
#  undef  M_USE_PTHREAD_MUTEX_TYPE
#  undef  M_RECURSIVE_MUTEX_TYPE
#elif defined(_OS_SOLARIS_)
// Solaris 2.7 and later - we use the native Solaris threads implementation
#  undef  M_HAS_RECURSIVE_MUTEX
#  undef  M_USE_PTHREAD_MUTEX_SETKIND
#  undef  M_NORMAL_MUTEX_TYPE
#  undef  M_RECURSIVE_MUTEX_TYPE
#elif defined(_OS_IRIX_)
#  define M_HAS_RECURSIVE_MUTEX
#  undef  M_USE_PTHREAD_MUTEX_SETKIND
#  define M_NORMAL_MUTEX_TYPE PTHREAD_MUTEX_ERRORCHECK
#  define M_RECURSIVE_MUTEX_TYPE PTHREAD_MUTEX_RECURSIVE
#else
// Fall through for systems we don't know about
#  warning "Assuming non-POSIX 1003.1c thread implementation. Might fail."
#  undef  M_HAS_RECURSIVE_MUTEX
#  undef  M_USE_PTHREAD_MUTEX_SETKIND
#  undef  M_NORMAL_MUTEX_TYPE
#  undef  M_RECURSIVE_MUTEX_TYPE
#endif

typedef std::map<HANDLE,MThread*> MapHndlTrd;

static MMutex *dictMutex = 0;
static MapHndlTrd *thrDict = 0;


extern "C" { static void * start_thread(void * t); }

#if defined(_OS_SOLARIS_)

#include <thread.h>
// Function usleep() is in C library but not in header files on Solaris 2.5.1.
// Not really a surprise, usleep() is specified by XPG4v2 and XPG4v2 is only
// supported by Solaris 2.6 and better.
// So we are trying to detect Solaris 2.5.1 using macro _XOPEN_UNIX which is
// defined by <unistd.h> when XPG4v2 is supported.
//#if !defined(_XOPEN_UNIX)
//typedef unsigned int useconds_t;
//extern "C" int usleep(useconds_t);
//#endif


class MMutexPrivate {
public:
    mutex_t mutex;

    MMutexPrivate(bool recursive = FALSE)
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    mutex_init( &mutex, NULL, NULL );

#ifdef CHECK_RANGE
	if( ret )
	    fprintf(stderr, "MMutex::MMutex: init failure: %s\n", strerror( ret ) );
#endif
    }

    virtual ~MMutexPrivate()
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    mutex_destroy( &mutex );

#ifdef CHECK_RANGE
	if ( ret )
	    fprintf(stderr, "MMutex::~MMutex: destroy failure: %s\n", strerror( ret ) );
#endif
    }

    virtual void lock()
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    mutex_lock(&mutex);

#ifdef CHECK_RANGE
	if (ret)
	    fprintf(stderr,"MMutex::lock: mutex lock failure: %s\n", strerror(ret));
#endif
    }

    virtual void unlock()
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    mutex_unlock(&mutex);

#ifdef CHECK_RANGE
	if (ret)
	    fprintf(stderr,"MMutex::unlock: mutex unlock failure: %s\n", strerror(ret));
#endif
    }

    virtual bool locked()
    {
	int ret = mutex_trylock(&mutex);

	if (ret == EBUSY) {
	    return TRUE;
	} else if (ret) {
#ifdef CHECK_RANGE
	    fprintf(stderr,"MMutex::locked: try lock failed: %s\n", strerror(ret));
#endif
	} else {
	    mutex_unlock(&mutex);
	}

	return FALSE;
    }

#if defined(CHECK_RANGE) || !defined(M_HAS_RECURSIVE_MUTEX)
    virtual int type() const { return M_MUTEX_NORMAL; }
#endif
};


class MRMutexPrivate : public MMutexPrivate
{
public:
    int count;
    HANDLE owner;
    mutex_t mutex2;

    MRMutexPrivate()
	: MMutexPrivate(TRUE)
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    mutex_init( &mutex2, NULL, NULL );


#ifdef CHECK_RANGE
	if( ret )
	    fprintf(stderr, "MMutex::MMutex: init failure: %s\n", strerror( ret ) );
#endif

	count = 0;
    }

    ~MRMutexPrivate()
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    mutex_destroy(&mutex2);

#ifdef CHECK_RANGE
	if( ret )
	    fprintf(stderr, "MMutex::MMutex: destroy failure: %s\n", strerror( ret ) );
#endif
    }

    void lock()
    {
	mutex_lock(&mutex2);

	if(count > 0 && owner == MThread::currentThread()) {
	    count++;
	} else {
	    mutex_unlock(&mutex2);
	    MMutexPrivate::lock();
	    mutex_lock(&mutex2);

	    count = 1;
	    owner = MThread::currentThread();
	}

	mutex_unlock(&mutex2);
    }

    void unlock()
    {
	mutex_lock(&mutex2);

	if (owner != MThread::currentThread()) {
#ifdef CHECK_RANGE
	    fprintf(stderr,"MMutex::unlock: unlock from different thread than locker\n");
	    fprintf(stderr,"                was locked by %d, unlock attempt from %d\n",
		     owner, MThread::currentThread());
	    mutex_unlock(&mutex2);
#endif

	    return;
	}

	// do nothing if the count is already 0... to reflect the behaviour described
	// in the docs
	if (count && (--count) < 1) {
	    MMutexPrivate::unlock();
	    count=0;
	}

	mutex_unlock(&mutex2);
    }

    bool locked()
    {
	mutex_lock(&mutex2);
	bool ret = MMutexPrivate::locked();
	mutex_unlock(&mutex2);

	return ret;
    }

#if defined(CHECK_RANGE) || !defined(M_HAS_RECURSIVE_MUTEX)
    int type() const { return M_MUTEX_RECURSIVE; }
#endif
};


class MThreadPrivate {
public:
    thread_t thread_id;
    MWaitCondition* thread_done;
    bool finished, running, autodelete;

    MThreadPrivate()
	: thread_id(0), finished(FALSE), running(FALSE)
    {
	thread_done = NULL;
	autodelete = false;
    // this assumes that the first thread is created from
    // yet single-threaded application
	if (! dictMutex)
	{
	    dictMutex = new MMutex;
	    ASSERT(dictMutex);
	}
	if (! thrDict)
	{
		thrDict = new MapHndlTrd;
		ASSERT(thrDict);
	}
    }

    ~MThreadPrivate()
    {
    	erase();
    }

    void erase(){
    	dictMutex->lock();
		if (thread_id)
	    	thrDict->erase((HANDLE) thread_id);
		dictMutex->unlock();

		thread_id = 0;
    }

    bool thread_wait(MMutex *mtx, unsigned long time){
    	if (thread_done)
    		return thread_done->wait(mtx, time);
    	return true;
    }

    void init(MThread *that)
    {
	that->d->running = TRUE;
	that->d->finished = FALSE;

	int ret = thr_create( NULL, NULL, start_thread, that, THR_DETACHED,
			      &thread_id );

#ifdef CHECK_RANGE
	if (ret)
	    fprintf(stderr,"MThread::start: thread creation error: %s\n", strerror(ret));
#endif
    }

/*    static void internalRun(MThread *that)
    {
	dictMutex->lock();
	thrDict->insert(MapHndlTrd::value_type(MThread::currentThread(), that));
	dictMutex->unlock();

	that->d->running = TRUE;
	that->d->finished = FALSE;

	that->run();

	dictMutex->lock();

    MapHndlTrd::iterator it = thrDict->find(MThread::currentThread());
	if ( it != thrDict->end() ) {
        MThread *there = (*it).second;
        ASSERT(there);
	    there->d->running = FALSE;
	    there->d->finished = TRUE;

	    there->d->thread_done.wakeAll();
	}

	thrDict->erase(MThread::currentThread());
	dictMutex->unlock();
	}
};*/

	MapHndlTrd::iterator it = thrDict->find(MThread::currentThread());
	if ( it != thrDict->end() ) {
		MThread *there = (*it).second;
		ASSERT(there);
		there->d->running = FALSE;
		there->d->finished = TRUE;
		MWaitCondition* pWait = there->d->thread_done;
		there->d->thread_done = NULL;
		
		if (there->d->autodelete)
		{
			thrDict->erase(it);
			dictMutex->unlock();
			// delete the object
			delete there;
			pWait->wakeAll();
			delete pWait;
			return;
		}
		pWait->wakeAll();
		delete pWait;
	}

	thrDict->erase(MThread::currentThread());
	dictMutex->unlock();
    }
};


class MWaitConditionPrivate {
public:
    cond_t cond;
    MMutex mutex;
    int nrefs;

    MWaitConditionPrivate()
    {
    nrefs=1;
#ifdef CHECK_RANGE
	int ret =
#endif
	    cond_init(&cond, NULL, NULL );

#ifdef CHECK_RANGE
	if( ret )
	    fprintf(stderr, "MWaitCondition::MWaitCondition: event init failure %s\n", strerror( ret ) );
#endif
    }
private:
    ~MWaitConditionPrivate()
    {
	int ret = cond_destroy(&cond);
	if( ret ) {
#ifdef CHECK_RANGE
	    fprintf(stderr, "MWaitCondition::~MWaitCondition: event destroy failure %s\n", strerror( ret ) );
#endif

	    // seems we have threads waiting on us, lets wake them up
	    cond_broadcast(&cond);
	}
    }
public:
	// references
	inline void addref(){
	mutex.lock();
	++nrefs;
	mutex.unlock();
	}
	inline void addref_lock(){
	mutex.lock();
	++nrefs;
	}
	inline void release(){
	mutex.lock();
	ASSERT(nrefs>0);
	if (--nrefs<=0)
	{
		mutex.unlock();
		delete this;
		return;
	}
	mutex.unlock();
	}
	inline void release_unlock(){
	ASSERT(nrefs>0);
	if (--nrefs<=0)
	{
		mutex.unlock();
		delete this;
		return;
	}
	mutex.unlock();
	}
	// functionality
    void wakeOne()
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    cond_signal(&(cond));

#ifdef CHECK_RANGE
	if ( ret ) fprintf(stderr,"MWaitCondition::wakeOne: wake error: %s\n",strerror(ret));
#endif
    }

    void wakeAll()
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    cond_broadcast(& (cond) );

#ifdef CHECK_RANGE
	if( ret ) fprintf(stderr,"MWaitCondition::wakeAll: wake error: %s\n",strerror(ret));
#endif
    }

    bool wait(unsigned long time)
    {
	addref_lock();

	int ret;
	if (time != ULONG_MAX) {
	    timespec ti;
	    ti.tv_sec = (time / 1000);
	    ti.tv_nsec = (time % 1000) * 1000000;

	    ret = cond_timedwait(&(cond), &(mutex.d->mutex), &ti);
	} else {
	    ret = cond_wait ( &(cond), &(mutex.d->mutex) );
	}

	release_unlock();

#ifdef CHECK_RANGE
	if( ret ) fprintf(stderr,"MWaitCondition::wait: wait error:%s\n",strerror(ret));
#endif

	return (ret == 0);
    }

    bool wait(MMutex *mtx, unsigned long time)
    {
	if (! mtx) return FALSE;

#ifdef CHECK_RANGE
	if (mtx->d->type() == M_MUTEX_RECURSIVE)
	    fprintf(stderr,"MWaitCondition::wait: warning - using recursive mutexes with\n"
				       "                      conditions is undefined!\n");
#endif

#ifndef M_HAS_RECURSIVE_MUTEX
	int c = 0;
	HANDLE id = 0;

	if (mtx->d->type() == M_MUTEX_RECURSIVE) {
	    MRMutexPrivate *rmp = (MRMutexPrivate *) mtx->d;
	    mutex_lock(&(rmp->mutex2));

	    if (! rmp->count) {
#  ifdef CHECK_RANGE
		fprintf(stderr,"MWaitCondition::unlockAndWait: recursive mutex not locked!\n");
#  endif

		return FALSE;
	    }

	    c = rmp->count;
	    id = rmp->owner;

	    rmp->count = 0;
	    rmp->owner = 0;

	    mutex_unlock(&(rmp->mutex2));
	}
#endif

	adderf();

	int ret;
	if (time != ULONG_MAX) {
	    timespec ti;
	    ti.tv_sec = (time / 1000);
	    ti.tv_nsec = (time % 1000) * 1000000;

	    ret = cond_timedwait(&(cond), &(mtx->d->mutex), &ti);
	} else {
	    ret = cond_wait ( &(cond), &(mtx->d->mutex) );
	}
	
	release();

#ifndef M_HAS_RECURSIVE_MUTEX
	if (mtx->d->type() == M_MUTEX_RECURSIVE) {
	    MRMutexPrivate *rmp = (MRMutexPrivate *) mtx->d;
	    mutex_lock(&(rmp->mutex2));
	    rmp->count = c;
	    rmp->owner = id;
	    mutex_unlock(&(rmp->mutex2));
	}
#endif

#ifdef CHECK_RANGE
	if ( ret ) fprintf(stderr,"MWaitCondition::wait: wait error:%s\n",strerror(ret));
#endif

	return (ret == 0);
    }
};


#else // ! defined(_OS_SOLARIS_)


#include <pthread.h>


class MMutexPrivate {
public:
    pthread_mutex_t mutex;
#if defined(M_MANUAL_LOCK_TRACKING)
	int nLockCount;
#endif

    MMutexPrivate(bool recursive = FALSE)
    {
	pthread_mutexattr_t attr;
	pthread_mutexattr_init(&attr);

#if defined(M_MANUAL_LOCK_TRACKING)
	nLockCount = 0;
#endif

#if defined(M_HAS_RECURSIVE_MUTEX)
	if (recursive) {

#  if defined(M_RECURSIVE_MUTEX_TYPE)
#    if defined(M_USE_PTHREAD_MUTEX_SETKIND)
	    pthread_mutexattr_setkind_np(&attr, M_RECURSIVE_MUTEX_TYPE);
#    else
	    pthread_mutexattr_settype(&attr, M_RECURSIVE_MUTEX_TYPE);
#    endif
#  endif

	} else {
#endif

#if defined(M_NORMAL_MUTEX_TYPE)
#  if defined(M_USE_PTHREAD_MUTEX_SETKIND)
	    pthread_mutexattr_setkind_np(&attr, M_NORMAL_MUTEX_TYPE);
#  else
	    pthread_mutexattr_settype(&attr, M_NORMAL_MUTEX_TYPE);
#  endif
#endif

#if defined(M_HAS_RECURSIVE_MUTEX)
	}
#endif

#ifdef CHECK_RANGE
	int ret =
#endif
	    pthread_mutex_init( &mutex, &attr );

	pthread_mutexattr_destroy(&attr);

#ifdef CHECK_RANGE
	if( ret )
	    fprintf(stderr, "MMutex::MMutex: init failure: %s\n", strerror( ret ) );
#endif
    }

    virtual ~MMutexPrivate()
    {
#ifdef CHECK_RANGE
#if defined(M_MANUAL_LOCK_TRACKING)
	if (nLockCount)
		fprintf(stderr, "MMutex::~MMutex: destroing locked mutex");
#endif //M_MANUAL_LOCK_TRACKING
	int ret =
#endif
	    pthread_mutex_destroy( &mutex );

#ifdef CHECK_RANGE
	if ( ret )
	    fprintf(stderr, "MMutex::~MMutex: destroy failure: %s\n", strerror( ret ) );
#endif
    }

    virtual void lock()
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    pthread_mutex_lock(&mutex);

#if defined(M_MANUAL_LOCK_TRACKING)
	++nLockCount;
#endif //M_MANUAL_LOCK_TRACKING

#ifdef CHECK_RANGE
	if (ret)
	    fprintf(stderr,"MMutex::lock: mutex lock failure: %s\n", strerror(ret));
#endif
    }

    virtual void unlock()
    {
#if defined(M_MANUAL_LOCK_TRACKING)
	--nLockCount;
#endif //M_MANUAL_LOCK_TRACKING


#ifdef CHECK_RANGE
	int ret =
#endif
	    pthread_mutex_unlock(&mutex);

#ifdef CHECK_RANGE
	if (ret)
	    fprintf(stderr,"MMutex::unlock: mutex unlock failure: %s\n", strerror(ret));
#endif
    }

    virtual bool locked()
    {
	int ret = pthread_mutex_trylock(&mutex);

	if (ret == EBUSY)
	{
		return TRUE;
	} else if (ret) {
#ifdef CHECK_RANGE
	    fprintf(stderr,"MMutex::locked: try lock failed: %s\n", strerror(ret));
#endif
	} else {
	    pthread_mutex_unlock(&mutex);
	}

#if defined(M_MANUAL_LOCK_TRACKING)
	return nLockCount;
#else
	return FALSE;
#endif //M_MANUAL_LOCK_TRACKING
    }

#if defined(CHECK_RANGE) || !defined(M_HAS_RECURSIVE_MUTEX)
    virtual int type() const { return M_MUTEX_NORMAL; }
#endif
};


class MRMutexPrivate : public MMutexPrivate
{
public:
#ifndef M_HAS_RECURSIVE_MUTEX
    int count;
    HANDLE owner;
    pthread_mutex_t mutex2;

    ~MRMutexPrivate()
    {
#  ifdef CHECK_RANGE
	int ret =
#  endif
	    pthread_mutex_destroy(&mutex2);

#  ifdef CHECK_RANGE
	if( ret )
	    fprintf(stderr, "MMutex::MMutex: destroy failure: %s\n", strerror( ret ) );
#  endif

    }

    void lock()
    {
	pthread_mutex_lock(&mutex2);

	if(count > 0 && owner == MThread::currentThread()) {
	    count++;
	} else {
	    pthread_mutex_unlock(&mutex2);
	    MMutexPrivate::lock();
	    pthread_mutex_lock(&mutex2);

	    count = 1;
	    owner = MThread::currentThread();
	}

	pthread_mutex_unlock(&mutex2);
    }

    void unlock()
    {
	pthread_mutex_lock(&mutex2);

	if (owner != MThread::currentThread()) {
#ifdef CHECK_RANGE
	    fprintf(stderr,"MMutex::unlock: unlock from different thread than locker\n");
	    fprintf(stderr,"                was locked by %d, unlock attempt from %d\n",
		     owner, MThread::currentThread());
	    pthread_mutex_unlock(&mutex2);
#endif

	    return;
	}

	// do nothing if the count is already 0... to reflect the behaviour described
	// in the docs
	if (count && (--count) < 1) {
	    MMutexPrivate::unlock();
	    count=0;
	}

	pthread_mutex_unlock(&mutex2);
    }

    bool locked()
    {
	pthread_mutex_lock(&mutex2);
	bool ret = MMutexPrivate::locked();
	pthread_mutex_unlock(&mutex2);

	return ret;
    }
#endif

    MRMutexPrivate()
	: MMutexPrivate(TRUE)
    {
#ifndef M_HAS_RECURSIVE_MUTEX
	pthread_mutexattr_t attr;
	pthread_mutexattr_init(&attr);

#  ifdef CHECK_RANGE
	int ret =
#  endif
	    pthread_mutex_init( &mutex2, &attr );

	pthread_mutexattr_destroy(&attr);

#  ifdef CHECK_RANGE
	if( ret )
	    fprintf(stderr, "MMutex::MMutex: init failure: %s\n", strerror( ret ) );
#  endif

	count = 0;
#endif
    }

#if defined(CHECK_RANGE) || !defined(M_HAS_RECURSIVE_MUTEX)
    int type() const { return M_MUTEX_RECURSIVE; }
#endif
};


class MThreadPrivate {
public:
    pthread_t thread_id;
    MWaitCondition* thread_done;      // Used for MThread::wait()
    bool finished, running, autodelete;

    MThreadPrivate()
	: thread_id(0), finished(FALSE), running(FALSE)
    {
	autodelete = false;
	thread_done = NULL;
    // this assumes that the first thread is created from
    // yet single-threaded application
	if (! dictMutex)
	{
		dictMutex = new MMutex;
		ASSERT(dictMutex);
	}
	if (! thrDict)
	{
		thrDict = new MapHndlTrd;
		ASSERT(thrDict);
	}
    }

    ~MThreadPrivate()
    {
    	erase();
    }

    void erase(){
    	dictMutex->lock();
		if (thread_id)
	    	thrDict->erase((HANDLE) thread_id);
		dictMutex->unlock();

		thread_id = 0;
    }

    bool thread_wait(MMutex *mtx, unsigned long time){
    	if (thread_done)
    		return thread_done->wait(mtx, time);
    	return true;
    }

    void init(MThread *that)
    {
    //cout << "I1" << flush;
	that->d->running = TRUE;
	that->d->finished = FALSE;

	pthread_attr_t attr;
	pthread_attr_init(&attr);
	pthread_attr_setinheritsched(&attr, PTHREAD_INHERIT_SCHED);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);

	//cout << "I2" << flush;
	int ret = pthread_create( &thread_id, &attr, start_thread, (void *) that );
    //cout << "I3" << flush;
	pthread_attr_destroy(&attr);
    //cout << "I4" << flush;
	if ( ret ) {
#ifdef CHECK_RANGE
	    fprintf(stderr,"MThread::start: thread creation error: %s\n", strerror(ret));
#endif
		//cout << "I!" << flush;
	    return;
	}
	//cout << "I5" << flush;
    }

    static void internalRun(MThread *that)
    {
    //cout << "R" << flush;
	dictMutex->lock();
	thrDict->insert(MapHndlTrd::value_type(MThread::currentThread(), that));
	// create "wait condition" object
	ASSERT(that->d->thread_done == NULL);
	that->d->thread_done = new MWaitCondition();
	ASSERT(that->d->thread_done);
	// if the thread calls "delete this" juct created MWaitCondition will join
	// the army of memory leaks. Thread should call setSutoDelete() instead
	// whereas I dont understand why cant I just remember this pinter and delete it
	// in the end
	dictMutex->unlock();
	
	that->run();

	dictMutex->lock();
	
	MapHndlTrd::iterator it = thrDict->find(MThread::currentThread());
	if ( it != thrDict->end() ) {
		MThread *there = (*it).second;
		ASSERT(there);
		there->d->running = FALSE;
		there->d->finished = TRUE;
		MWaitCondition* pWait = there->d->thread_done;
		there->d->thread_done = NULL;
		
		if (there->d->autodelete)
		{
			thrDict->erase(it);
			dictMutex->unlock();
			// delete the object
			delete there;
			pWait->wakeAll();
			delete pWait;
			return;
		}
		pWait->wakeAll();
		delete pWait;
	}

	thrDict->erase(MThread::currentThread());
	dictMutex->unlock();
    }
};


class MWaitConditionPrivate {
public:
    pthread_cond_t cond;
    MMutex         mutex;
    u_long         nrefs;

    MWaitConditionPrivate()
    {
	nrefs = 1;
	pthread_condattr_t cattr;
#ifdef HAVE_PTHREAD_CONDATTR_INIT
	pthread_condattr_init(&cattr);
#else
	memset(&cattr, 0, sizeof(cattr));
#endif

#ifdef CHECK_RANGE
	int ret =
#endif
	    pthread_cond_init(&cond, &cattr);

#ifdef HAVE_PTHREAD_CONDATTR_DESTROY
	pthread_condattr_destroy(&cattr);
#endif

#ifdef CHECK_RANGE
	if( ret )
	    fprintf(stderr, "MWaitCondition::MWaitCondition: event init failure %s\n", strerror( ret ) );
#endif
    }

private:
    ~MWaitConditionPrivate()
    {
	int ret = pthread_cond_destroy(&cond);
	if( ret ) {
#ifdef CHECK_RANGE
	    fprintf(stderr, "MWaitCondition::~MWaitCondition: event destroy failure %s\n", strerror( ret ) );
#endif

	    // seems we have threads waiting on us, lets wake them up
	    pthread_cond_broadcast(&cond);
	}
    }
public:
	// references
	inline void addref(){
	mutex.lock();
	++nrefs;
	mutex.unlock();
	}
	inline void addref_lock(){
	mutex.lock();
	++nrefs;
	}
	inline void release(){
	mutex.lock();
	ASSERT(nrefs>0);
	if (--nrefs<=0)
	{
		mutex.unlock();
		delete this;
		return;
	}
	mutex.unlock();
	}
	inline void release_unlock(){
	ASSERT(nrefs>0);
	if (--nrefs<=0)
	{
		mutex.unlock();
		delete this;
		return;
	}
	mutex.unlock();
	}
	// functionality
    void wakeOne()
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    pthread_cond_signal( &(cond) );

#ifdef CHECK_RANGE
	if ( ret ) fprintf(stderr,"MWaitCondition::wakeOne: wake error: %s\n",strerror(ret));
#endif
    }

    void wakeAll()
    {
#ifdef CHECK_RANGE
	int ret =
#endif
	    pthread_cond_broadcast(&(cond));

#ifdef CHECK_RANGE
	if( ret ) fprintf(stderr,"MWaitCondition::wakeAll: wake error: %s\n",strerror(ret));
#endif
    }

    bool wait(unsigned long time)
    {
	addref_lock();
	
#if defined(M_MANUAL_LOCK_TRACKING)
		// wait will unlock the mutex -- decrese the counter
		--mutex.d->nLockCount;
#endif
	
	int ret;
	if (time != ULONG_MAX) {
	    timespec ti;
	    ti.tv_sec = (time / 1000);
	    ti.tv_nsec = (time % 1000) * 1000000;

	    ret = pthread_cond_timedwait(&(cond), &(mutex.d->mutex), &ti);
	} else {
	    ret = pthread_cond_wait ( &(cond), &(mutex.d->mutex) );
	}

#if defined(M_MANUAL_LOCK_TRACKING)
		++mutex.d->nLockCount;
#endif

	release_unlock();

#ifdef CHECK_RANGE
	if( ret ) fprintf(stderr,"MWaitCondition::wait: wait error:%s\n",strerror(ret));
#endif

	return (ret == 0);
    }

    bool wait(MMutex *mtx, unsigned long time)
    {
	if (! mtx) return FALSE;

#ifdef CHECK_RANGE
	if (mtx->d->type() == M_MUTEX_RECURSIVE)
	    fprintf(stderr,"MWaitCondition::unlockAndWait: warning - using recursive mutexes with\n"
				       "                               conditions is undefined!\n");
#endif

#ifndef M_HAS_RECURSIVE_MUTEX
	int c = 0;
	HANDLE id = 0;

	if (mtx->d->type() == M_MUTEX_RECURSIVE) {
	    MRMutexPrivate *rmp = (MRMutexPrivate *) mtx->d;
	    pthread_mutex_lock(&(rmp->mutex2));

	    if (! rmp->count) {
#  ifdef CHECK_RANGE
		fprintf(stderr,"MWaitCondition::unlockAndWait: recursive mutex not locked!\n");
#  endif

		return FALSE;
	    }

	    c = rmp->count;
	    id = rmp->owner;

	    rmp->count = 0;
	    rmp->owner = 0;

	    pthread_mutex_unlock(&(rmp->mutex2));
	}
#endif

	addref();

#if defined(M_MANUAL_LOCK_TRACKING)
	// wait will unlock the mutex -- decrese the counter
	--mtx->d->nLockCount;
#endif

	int ret;
	if (time != ULONG_MAX) {
	    timespec ti;
	    ti.tv_sec = (time / 1000);
	    ti.tv_nsec = (time % 1000) * 1000000;

	    ret = pthread_cond_timedwait(&(cond), &(mtx->d->mutex), &ti);
	} else {
	    ret = pthread_cond_wait ( &(cond), &(mtx->d->mutex) );
	}

#if defined(M_MANUAL_LOCK_TRACKING)
	// wait will unlock the mutex -- decrese the counter
	++mtx->d->nLockCount;
#endif

	release();

#ifndef M_HAS_RECURSIVE_MUTEX
	if (mtx->d->type() == M_MUTEX_RECURSIVE) {
	    MRMutexPrivate *rmp = (MRMutexPrivate *) mtx->d;
	    pthread_mutex_lock(&(rmp->mutex2));
	    rmp->count = c;
	    rmp->owner = id;
	    pthread_mutex_unlock(&(rmp->mutex2));
	}
#endif

#ifdef CHECK_RANGE
	if ( ret ) fprintf(stderr,"MWaitCondition::wait: wait error:%s\n",strerror(ret));
#endif

	return (ret == 0);
    }
};


#endif // defined(_OS_SOLARIS_)


extern "C" {
    static void *start_thread(void *t)
    {
	MThreadPrivate::internalRun( (MThread *) t );
	return 0;
    }
}


#endif // MTHREAD_P_H
