/* $Id: nws_db.c,v 1.7 2004/03/03 05:48:36 graziano Exp $ */

#include "config_nws.h"

#include <stdio.h>
#include <stdlib.h>

#include "diagnostic.h"
#include "osutil.h"
#include "strutil.h"
#include "nws_db.h"

/* we use the DB backend only if we picked it up from autoconf */
#ifndef NWS_WITH_DB
int
ConnectToNwsDB( const char *name) {
	ERROR("CloseNwsDB: no postgresql installation found!\n");
	return 0;
}
int
WriteNwsDB(     const char *name,
		double timeout,
		double seq,
		const char *data,
		unsigned int dataLen) {
	ERROR("WriteNwsDB: no postgresql installation found!\n");
	return 0;
}
int 
ReadNwsDB(	const char *name,
		char *where,
		int wanted,
		int whereSize,
		double seq,
		double *timeout,
		double *seqOut,
		int *howMany,
		int *recSize) {
	ERROR("ReadNwsDB: no postgresql installation found!\n");
	return 0;
}
char *
GetTables() {
	ERROR("GetTables: no postgresql installation found!\n");
	return NULL;
}
int
CheckTable(const char *name) {
	ERROR("CheckTable: no postgresql installation found!\n");
	return 0;
}

#else


#ifdef HAVE_LIBPQ_FE_H
#	include <libpq-fe.h>
#elif HAVE_POSTGRESQL_LIBPQ_FE_H
#	include <postgresql/libpq-fe.h>
#endif


/* global variables */
static PGconn *db = NULL;		/* connection to the database */

/* close connection to the database */
void
CloseNwsDB() {
	if (db != NULL) {
		LOG1("CloseNwsDB: closing connection with %s\n", PQdb(db));
		PQfinish(db);
	}
}


/* create a typical NWS table named #name#. The fields are already known.
 * Returns 1 if the succesful 0 otherwise. 
 *
 * The table has the following fields:
 * 	timeout			timeout from the memory
 * 	seq			sequential number
 * 	data			the data as coming from the client
 */
static int
CreateNwsTable(	const char *name) {
	int ret;
	char *query;
	PGresult *res;

	/* sanity check */
	if (name == NULL) {
		ERROR("CreateNwsTable: NULL parameter\n");
		return 0;
	}
	if (db == NULL) {
		ERROR("CreateNwsTable: database not initialized\n");
		return 0;
	}

	/* let's create the query */
	query = (char *)MALLOC(80 + strlen(name) + 1);
	if (query == NULL) {
		ABORT("CreateNwsTable: out of memory\n");
	}
	sprintf(query, "create table \"%s\" ( timeout NUMERIC(10), seq NUMERIC(10), data varchar(%d) )", name, MAX_RECORD_SIZE);

	/* let's do it */
	res = PQexec(db, query);
	if (res == NULL) {
		ABORT("CreateNwsTable: out of memory\n");
	}
	if (PQresultStatus(res) != PGRES_COMMAND_OK) {
		ERROR1("CreateNwsTable: %s", PQerrorMessage(db));
		ret = 0;
	} else {
		LOG1("Created table %s\n", name);
		ret = 1;
	}
	PQclear(res);
	FREE(query);

	return ret;
}

/* initialize the connection to the database: returns 1 if succesful, 0
 * otherwise */
int
ConnectToNwsDB(	const char *dbname) {
	char *query;
	int ret;

	/* sanity check */
	if (dbname == NULL) {
		ERROR("ConnectToNwsDB: NULL name!\n");
		return 0;
	}

	/* if we have an old connection, close it */
	if (db != NULL) {
		CloseNwsDB();
	}

	/* create the query string */
	query = (char *)MALLOC(20 + strlen(dbname) + 2);
	if (query == NULL) {
		ABORT("ConnectNwsToDB: out of memory\n");
	}
	sprintf(query, "dbname = %s", dbname);

	/* now open the new connection */
	db = PQconnectdb(query);
	if (db == NULL) {
		ABORT("ConnectToNwsDB: out of memory\n");
	}
	ret = 1;
	if (PQstatus(db) != CONNECTION_OK) {
		ERROR1("ConnectToNwsDB: %s\n", PQerrorMessage(db));
		ret = 0;
		CloseNwsDB();
	}
	FREE(query);
	LOG1("ConnectToDB: connecte to %s\n", dbname);
	
	return ret;
}


/* store the data into the specified table. #name# is the table name,
 * #timeout# is the memory timeout, #seq# the sequence number,
 * #data# a pointer to the experimental data and #dataLen# is how many char
 * can I take for this write. Return 1 on success, 0 on failure.
 */
int
WriteNwsDB(	const char *name,
		double timeout,
		double seq,
		const char *data,
		unsigned int dataLen) {
	PGresult *res;
	int done;
	char *query,		/* holds the sql command */
		*tmp;		/* holds the user's value */

	/* sanity check */
	if (name == NULL || data == NULL) {
		ERROR("WriteNwsDB: NULL parameter\n");
		return 0;
	}
	if (db == NULL) {
		ERROR("WriteNwsDB: database not initialized\n");
		return 0;
	}

	/* let's generate the query */
	query = (char *)MALLOC(40 + strlen(name) + dataLen + 22);
	tmp = (char *)MALLOC(dataLen + 1);
	if (query == NULL || tmp == NULL) {
		ABORT("WriteNwsDB: out of memory\n");
	}
	memcpy(tmp, data, dataLen);
	tmp[dataLen] = '\0';
	sprintf(query, "insert into \"%s\" values ( %10.0f, %10.0f, '%s' )", name, timeout, seq, tmp);
	FREE(tmp);

	/* We don't know if we already have the table, so we try to write
	 * it, if we get a failure we try to create the table and if
	 * succesful we write again. */
	for (done = -2; done < 0; ) {
		res = PQexec(db, query);
		if (res == NULL) {
			ABORT("WriteNwsDB: out of memory\n");
		}
		if (PQresultStatus(res) != PGRES_COMMAND_OK) {
			/* we failed: try to create the table */
			if (done == -2 && CreateNwsTable(name)) {
				/* we have another chance */
				done++;
				continue;
			}
			ERROR1("WriteNwsDB: %s", PQerrorMessage(db));
			done = 0;
		} else {
			/* success */
			done = 1;
		}
		break;
	}
	PQclear(res);
	FREE(query);

	return done;
}

/* read from a table */
int
ReadNwsDB(	const char *name,
		char *where,
		int wanted,
		int whereSize,
		double seq,
		double *timeout,
		double *seqOut,
		int *howMany,
		int *recSize) {
	PGresult *res;
	int done, size, tmp;
	char *query;		/* holds the sql command */

	/* sanity check */
	if (name == NULL || where == NULL || timeout == NULL || howMany == NULL || recSize == NULL) {
		ERROR("ReadNwsDB: NULL parameter\n");
		return 0;
	}
	if (db == NULL) {
		ERROR("ReadNwsDB: database not initialized\n");
		return 0;
	}

	/* let's generate the query */
	query = (char *)MALLOC(80 + strlen(name) + 11);
	if (query == NULL) {
		ABORT("ReadNwsDB: out of memory\n");
	}
	sprintf(query, "select seq, timeout, data from \"%s\" where seq > %10.0f order by seq DESC", name, seq);

	res = PQexec(db, query);
	if (res == NULL) {
		ABORT("ReadNwsDB: out of memory\n");
	}
	FREE(query);
	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
		ERROR1("ReadNwsDB: %s", PQerrorMessage(db));
		done = 0;
	} else {
		/* let's pack the result into where */
		if (PQnfields(res) < 3) {
			ERROR("ReadNedDB: possibly inconsistent table\n");
			done = 0;
		} else {
			tmp = size = 0;
			*howMany = *recSize = *seqOut = 0;
			for (done = 0; done < PQntuples(res)  && done < wanted; done++) {
				/* and this is what came from the client */
				query = PQgetvalue(res, done, 2);
				/* let's copy in the right place */
				if (*recSize == 0) {
					*recSize = strlen(query);
				} else if (*recSize != strlen(query)) {
					INFO("ReadNwsDB: different record lenghts!\n");
				}
				if (size + strlen(query) > whereSize) {
					INFO("ReadNwsDB: not enouogh space, doprring extra records\n");
					break;
				}
				memcpy(where + size, query, strlen(query));
				size += strlen(query);
				where[size] = ' ';
				where[++size] = '\0';
				tmp++;

				/* set the first sequence */
				if (*seqOut == 0) {
					/* get the sequence number */
					*seqOut = strtod(PQgetvalue(res, done, 2), NULL);
				}
			}
			/* set the record count */
			*howMany = tmp;

			/* set the timeout */
			*timeout = strtod(PQgetvalue(res, done - 1, 1), NULL);
		}

		/* success */
		done = 1;
	}
	PQclear(res);

	return done;


}

/* returns a \n separated list of tables names. It can returns NULL */
char *
GetTables() {
	PGresult *res;
	char *tmp, *ret;
	int i, size;

	/* sanity check */
	if (db == NULL) {
		ERROR("GetTables: database not initialized\n");
		return NULL;
	}
	 ret = NULL;

	/* let's query for the tables name */
	res = PQexec(db, "select tablename from pg_tables where schemaname = 'public'");
	if (res == NULL) {
		ABORT("GetTables: out of memory\n");
	}
	if (PQresultStatus(res) != PGRES_TUPLES_OK) {
		ERROR1("GetTables: %s", PQerrorMessage(db));
		return NULL;
	}

	/* let's construct the list of names */
	size = 0;
	for (i = 0; i < PQntuples(res); i++) {
		tmp = PQgetvalue(res, i, 0);
		ret = (char *)REALLOC(ret, size + strlen(tmp) + 2);
		if (ret == NULL) {
			ABORT("GetTables: out of memory\n");
		}
		memcpy(ret + size, tmp, strlen(tmp));
		size += strlen(tmp);
		ret[size] = '\n';
		ret[++size] = '\0';
	}

	return ret;
}

/* check if table is a correct one. Returns 1 on success. */
int
CheckTable(const char *name) {
	char *query;
	PGresult *res;
	int ret = 0;

	/* sanity check */
	if (name == NULL) {
		ERROR("ChecKTable: NULL parameter\n");
		return 0;
	}
	if (db == NULL) {
		ERROR("CheckTable: database not initialized\n");
		return 0;
	}

	/* create the query */
	query = (char *)MALLOC(50 + strlen(name));
	if (query == NULL) {
		ABORT("CheckTable: out of memory\n");
	}
	sprintf(query, "select seq, timeout, data from \"%s\"", name);
	res = PQexec(db, query);
	if (res == NULL) {
		ABORT("CheckTable: out of memory\n");
	}
	FREE(query);
	if (PQresultStatus(res) == PGRES_TUPLES_OK) {
		ret = 1;
	} 
	PQclear(res);

	return ret;
}

#endif	/* NWS_WITH_DB */
