/*
 * rtp-switcher.cc --
 *
 *      Keeps a table of sources.  When it receives a packet, it changes
 *      the packets header info according to the table and passes it to the
 *      target.  This is used when broadcasting packets received from the
 *      studio session.
 *      FIXME: This file needs some major work.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "tclcl.h"
#include "module.h"
#include "pktbuf.h"
#include "rtp.h"
#include "media-timer.h"


class CSwitcherFrameInfo
{
    /* these are always stored in host format */
public:
    u_int32_t its_;
    u_int32_t ots_;
    u_int16_t iseqno_;
    u_int16_t oseqno_;

    CSwitcherFrameInfo () {
	its_ = ots_ = iseqno_ = oseqno_ = 0;
    }
};


class CSwitcherInfo
{
public:
    CSwitcherInfo();
    CSwitcherInfo(unsigned int iInSourceId, unsigned int iOutSourceId);
    CSwitcherInfo *next;

    u_int32_t isrcid_;
    u_int32_t osrcid_;

    u_int32_t its_;
    u_int32_t ots_;

    u_int16_t iseqno_;
    u_int16_t oseqno_;

    u_int32_t tsoff_; // Timestamp offset
    int fmt_;         // Current format of this source

    CSwitcherFrameInfo finfo[2];
};

CSwitcherInfo::CSwitcherInfo()
{
   isrcid_ = 0;
   osrcid_ = 0;

   its_ = 0;
   ots_ = 0;

   iseqno_ = 0;
   oseqno_ = 0;

   tsoff_ = 0;
   fmt_ = 0;
}

CSwitcherInfo::CSwitcherInfo(unsigned int iInSourceId, unsigned int iOutSourceId)
{
   isrcid_ = iInSourceId;
   osrcid_ = iOutSourceId;

   its_ = 0;
   ots_ = 0;

   iseqno_ = 0;
   oseqno_ = 0;

   tsoff_ = 0;
   fmt_ = 0;
}

class CRTPSwitcher : public PacketModule, public MediaTimer
{
  public:
  CRTPSwitcher();
  virtual void recv(pktbuf *);
  virtual int command(int argc, const char* const *argv);

  void MapSource(unsigned int iInSourceId, unsigned int iOutSourceId);
  void UnmapSource(unsigned int iInSourceId, unsigned int iOutSourceId);

  protected:
  int m_bDebug;

  CSwitcherInfo* m_pcSwitcherInfo;
};

static
class CRTPSwitcherClass : public TclClass {
  public:
  CRTPSwitcherClass() : TclClass("Module/RTPSwitcher") {}
  TclObject* create(int /* argc */, const char*const* /* argv */) {
    return (new CRTPSwitcher);
  }
} dm_rtp_switcher;


CRTPSwitcher::CRTPSwitcher() : PacketModule(), MediaTimer()
{
  m_bDebug = 0;
  m_pcSwitcherInfo = NULL;
}

int
CRTPSwitcher::command(int argc, const char*const* argv)
{
  unsigned int uiInSourceId;
  unsigned int uiOutSourceId;

  if (argc == 4) {
    if (strcmp(argv[1], "MapSource") == 0) {
      uiInSourceId = strtoul(argv[2], (char**) NULL, 0);
      uiOutSourceId = strtoul(argv[3], (char**) NULL, 0);
      MapSource(uiInSourceId, uiOutSourceId);
      return (TCL_OK);
    }

    if (strcmp(argv[1], "UnmapSource") == 0) {
      uiInSourceId = strtoul(argv[2], (char**) NULL, 0);
      uiOutSourceId = strtoul(argv[3], (char**) NULL, 0);
      UnmapSource(uiInSourceId, uiOutSourceId);
      return (TCL_OK);
    }
  }

  return(PacketModule::command(argc, argv));
}

/*
 * To map input source id to output src id, just lookup 
 * the switcherInfo data structure.
 *
 * To map input timestamp to output timestamp, we do the following:
 * - keep around a two mappings of input ts -> output ts (M1 & M2)
 * - look at the ts of curr packet.
 * - if it matches M1 or M2 just set the ts and we are done
 * - if it does belongs to the mapping, then it is either an old
 *   frame or a new frame.  If old, discard packet, if new, discard
 *   M1, copy M2 to M1, generate and new mapping M2.
 * 
 * To map input seq no to output seq no, we keep a reference input and output 
 * seqno for each frame.  We set the output seqno  to:
 *  curr input seqno - ref input seqno + ref output seqno
 * Determining the reference output seqno is done using information 
 * from jpeghdr. 
 */
void
CRTPSwitcher::recv(pktbuf *pkt)
{
    CSwitcherInfo* srcinfo;
    rtphdr* rtpHdr = (rtphdr *)pkt->dp;
    unsigned int uiInSourceId;

    // check that there's a decoder attached
    if (target_ == NULL) {
	pkt->release();
	return;
    }

    uiInSourceId = (u_int32_t) ntohl(rtpHdr->rh_ssrc);

    // find the switch info entry that is associated with this pkt
    for (srcinfo = m_pcSwitcherInfo; srcinfo != NULL; srcinfo = srcinfo->next) {
	if (srcinfo->isrcid_ == uiInSourceId) {
	    break;
	}
    }

    // we couldn't find the entry
    if (srcinfo == NULL) {
	pkt->release();
	return;
    }

    /* Rewrite the src id. */
    rtpHdr->rh_ssrc = htonl(srcinfo->osrcid_);

    u_int32_t currts = ntohl(rtpHdr->rh_ts);
    u_int16_t seqno;
    int fmt = ntohs(rtpHdr->rh_flags) & 0x7F;
    if (fmt != srcinfo->fmt_) {
	/* We have switch format. Restart. */
	/* FIXME: We should do something more intelligent 
	 * here, to fix the sequence number */
	srcinfo->finfo[0].its_ = 0;
	srcinfo->finfo[0].ots_ = 0;
	srcinfo->finfo[0].iseqno_ = 0;
	srcinfo->finfo[0].oseqno_ = 0;

	srcinfo->finfo[1].its_ = 0;
	srcinfo->finfo[1].ots_ = 0;
	srcinfo->finfo[1].iseqno_ = 0;
	srcinfo->finfo[1].oseqno_ = 0;

	srcinfo->fmt_ = fmt;
    }

    /*
    * If input rtp ts is something we have seen recently, 
    * update the output timestamp based on previously 
    * generated info.
    */
    if (srcinfo->finfo[0].its_ == currts) {
      rtpHdr->rh_ts = htonl(srcinfo->finfo[0].ots_);
      seqno = ntohs(rtpHdr->rh_seqno) - srcinfo->finfo[0].iseqno_ +
	    srcinfo->finfo[0].oseqno_;
      rtpHdr->rh_seqno = htons(seqno);

    } else if (srcinfo->finfo[1].its_ == currts) {
      rtpHdr->rh_ts = htonl(srcinfo->finfo[1].ots_);
      seqno = ntohs(rtpHdr->rh_seqno) - srcinfo->finfo[1].iseqno_ +
	    srcinfo->finfo[1].oseqno_;
      rtpHdr->rh_seqno = htons(seqno);
    } 
    /*
    * If input rtp ts is something really old, throw it away.
    * If we receive frame 2, and then frame 4, and then frame 3,
    * frame 3 will be thrown away!  
    */
    else if (currts < srcinfo->finfo[0].its_ || currts < srcinfo->finfo[1].its_) {
	pkt->release();
	return;
    }
    /*
    * If it is more recent than everything  we have seen so far, 
    * generate new timestamp.
    */
    else {
	srcinfo->finfo[0].its_ = srcinfo->finfo[1].its_;
	srcinfo->finfo[0].ots_ = srcinfo->finfo[1].ots_;
	srcinfo->finfo[0].iseqno_ = srcinfo->finfo[1].iseqno_;
	srcinfo->finfo[0].oseqno_ = srcinfo->finfo[1].oseqno_;

	srcinfo->finfo[1].its_ = currts;
	srcinfo->finfo[1].ots_ = media_ts();
	srcinfo->finfo[1].iseqno_ = ntohs(rtpHdr->rh_seqno);
	srcinfo->finfo[1].oseqno_ = 
	    srcinfo->finfo[1].iseqno_ -
	    srcinfo->finfo[0].iseqno_ +
	    srcinfo->finfo[0].oseqno_;

	rtpHdr->rh_ts = htonl(srcinfo->finfo[1].ots_);
	rtpHdr->rh_seqno = htons(srcinfo->finfo[1].oseqno_);
    }

    target_->recv(pkt);

  if (m_bDebug) {
    //    printf("ssrcid = %u\n", (u_int32_t) ntohl(((rtphdr *)(pkt->dp))->rh_ssrc));
    printf("Isrcid=%u, OsrcId=%u, InTs#=%lu, OutTs#=%u\n",
	   srcinfo->isrcid_,
	   srcinfo->osrcid_,
	   (unsigned long) ntohl(srcinfo->its_),
	   srcinfo->ots_);
  }
}

void
CRTPSwitcher::MapSource(unsigned int uiInSourceId, unsigned int uiOutSourceId)
{
  CSwitcherInfo* srcinfo;

  // check that we don't have this entry already
  for (srcinfo = m_pcSwitcherInfo;
       srcinfo != NULL;
       srcinfo = srcinfo->next) {

    // enough to check that all outgoing src id's are different
    if (srcinfo->osrcid_ == uiOutSourceId)
      return;
  }

  // otherwise we need to create a new entry and add it
  srcinfo = new CSwitcherInfo(uiInSourceId, uiOutSourceId);

  // insert the new switcher info
  srcinfo->next = m_pcSwitcherInfo;
  m_pcSwitcherInfo = srcinfo;
}

void
CRTPSwitcher::UnmapSource(unsigned int uiInSourceId,
			  unsigned int uiOutSourceId)
{
  CSwitcherInfo* pcSwitcherInfo;
  CSwitcherInfo* pcSwitcherInfoTemp;


  pcSwitcherInfoTemp = NULL;

  // iterate though the list and find the info entry
  for (pcSwitcherInfo = m_pcSwitcherInfo;
       pcSwitcherInfo != NULL;
       pcSwitcherInfo = pcSwitcherInfo->next) {

    // match the entry info object
    if (pcSwitcherInfo->isrcid_ == uiInSourceId &&
	pcSwitcherInfo->osrcid_ == uiOutSourceId ) {

      // if the trailing pointer is null
      if (pcSwitcherInfoTemp == NULL) {
	m_pcSwitcherInfo = pcSwitcherInfo->next;
      } else {
	// else it's either at the end or middle
	pcSwitcherInfoTemp->next = pcSwitcherInfo->next;
      }

      delete pcSwitcherInfo;
      return;
    }

    pcSwitcherInfoTemp = pcSwitcherInfo;
  }

  // didn't find a match so print out an error
  printf("ERROR: CRTPSwitcher::UnmapSource");
}
