/*
 * effect-module.cc --
 *
 *      EffectModule definitions . An EffectModule is a Module that 
 *      gets frames from one or several Modules, performs an effect 
 *      on them, and returns the resulting frame to the target.
 *
 * Copyright (c) 1993-2001 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/* @(#) $Header: /usr/mash/src/repository/mash/mash-1/fx/effects/effect-module.cc,v 1.1 2002/02/07 04:18:19 chema Exp $ */


#include "effect-module.h"

EffectModule::EffectModule(int inputs, int csss)
	: Module(), output_(0), csss_(csss), inputs_(inputs)
{
	// check the csss is correct
	if ((csss_ != 444) && (csss_ != 422) && 
			(csss_ != 420) && (csss_ != 411)) {
		fprintf (stdout, "EffectModule: color subsampling scheme unknown (%d)\n", 
				csss_);
		abort();
	}

	// initialize all the expected input-frame sizes to 0x0
	for (int i = 0; i < EFFECT_MAX_INPUTS; i++) {
		in_size(i, 0, 0);
	}

	// create the buffer frames
	for (int i = 0; i < (inputs-1); i++) {
		buffer_[i] = new Uncompressed;
	}

	// create the output frame
	output_ = new Uncompressed;

	return;
}


void EffectModule::recv(Uncompressed* input, int id)
{
	// check the input dimensions
	if (!in_samesize(id, input)) {
		in_size(id, input->w_, input->h_);
	}

	// check the output dimensions coincide with the input ones
	if (!output_->samesize(input)) {
		output_->init (input, csss_, VIDREP_ALLOC_MEMORY);
	}

	// if this is a non triggering frame, just copy it and return
	if (id != 0) {
		// TBD: I'm not very happy with not copying the non-triggering
		//  buffers. This works but...

		// TBD: I don't think double buffering is needed. The reason is that
		//  this runs on a tcl interpreter, which is composed of only one 
		//  process and doesn't preempt scripts until they return the control.
		//  We should think carefully about this, anyway.

		if (!buffer_[id-1]->samesize(input)) {
			buffer_[id-1]->init (input, csss_, VIDREP_OVERLOAD_MEMORY);
		}
		return;
	}

	// a triggering frame: check the chromas and trigger the effect
	if (csss_ != input->get_csss()) {
		input->add_chroma(csss_);
	}
	input->check_chroma();
	for (int i = 0; i < (inputs_ - 1); i++) {
		buffer_[i]->check_chroma();
	}

	// copy the input timestamp
	output_->ts_ = input->ts_;

	// trigger the effect
	trigger_effect (input);

	return;
}


int EffectModule::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();

	// number of inputs
	if (argc == 2 && strcmp(argv[1], "get-inputs") == 0) {
		char result[1024];
		sprintf (result, "%d", inputs_);
		tcl.result(result);
		return (TCL_OK);
	}

//	if (argc == 3 && strcmp(argv[1], "set-inputs") == 0) {
//		inputs_ = atoi(argv[2]);
//		return (TCL_OK);
//	}

//	if ((argc == 2) && (strcmp(argv[1], "target") == 0)) {
//		if (target_ != 0) {
//			tcl.result(target_->name());
//		}
//		return (TCL_OK);
//	}

//	if ((argc == 3) && (strcmp(argv[1], "target") == 0)) {
//		target_ = (Module*)TclObject::lookup(argv[2]);
//		return (TCL_OK);
//	}

	return (Module::command(argc, argv));
}

