/*
 * mb-items.h --
 *
 *      MediaBoard Canvas items
 *
 * Copyright (c) 1996-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Header: /usr/mash/src/repository/mash/mash-1/mb/mb-items.h,v 1.11 2002/02/03 03:16:30 lim Exp $
 */

#ifndef __MB_ITEMS_H__
#define __MB_ITEMS_H__

#include "mb/mb.h"
#include "mb/mb-nethost.h"
#include "mb/mb-pkt.h"
#include "compat/tkcompat.h"

typedef enum PageItemType_enum {
	PgItemNone=0,			/* in case we need a no-types object */
	PgItemLine,			/* 1 */
	PgItemRect,			/* 2 */
	PgItemOval,			/* 3 */
	PgItemMLine,			/* 4 */
	PgItemText,			/* 5 */
	PgItemImage,			/* 6 */
	PgItemPS,			/* 7 */
	PgItemInvalid			/* This value i	s really num types + 1 */
} PageItemType;

// gives the name in string as part of commands to MBSender
extern const char *caPageItemTypeNames[];

// gives the type name in tkCanvas for each item type
extern const char *caPgItemTyp2CanvItemTyp[];

// gives the number of points to be stored in each item
extern const int caPIT2nPts[];

/* forward decl */
class MBCanvas;

/*
 * A PageItem is an item that can be drawn onto media board
 * The current implementation corresponds closely with the tkCanvas items
 */
class PageItem {
public:
	PageItem(PageItemType type) : type_(type), numPoints_(0),
		aPoints_(NULL) {}
	virtual ~PageItem() { delete[] aPoints_; }

	inline static PageItemType strName2Type(const char *szName) {
		int pi;
		for (pi = PgItemNone+1; pi<PgItemInvalid; pi++)
			if (strcmp(szName, caPageItemTypeNames[pi]) == 0)
				break;
		return ((PageItemType)pi);
	}
	inline static int type2NumPts(PageItemType type) {
		assert( ((int)type) > 0 && ((int)type) < ((int)PgItemInvalid)
			&& "alert: inva	lid types used in Type2NumPts()");
		return caPIT2nPts[type];
	}
	static PageItem* createItem(PageItemType type);
	static PageItem* create(PageItemType type, Bool inclCoords, Byte *pb);

	virtual void getCopy(PageItem** ppItem) const {
		(*ppItem)->type_=type_;
		(*ppItem)->numPoints_=numPoints_;
		(*ppItem)->aPoints_= new Point[numPoints_];
		memcpy((char*)( (*ppItem)->aPoints_),(char*)aPoints_,
		       sizeof(Point)*numPoints_);
	}

	static char* getProp(const char *szTypeName, const char *szPrefix);

	virtual Bool configure(int argc, const char *const*argv);
	char * getPropsAsStr(MBCanvas* pCanv);

	char *formatConfigValue(MBCanvas *pCanv, Tk_ConfigSpec* pSpec,
				Bool* pfree);

	// called by create on all derived items
	virtual void  extract(Byte* pb)=0;
	virtual Byte* packetize(Bool inclCoords, Byte *pb) ;
	virtual u_int getPacketLen(Bool inclCoords);

	virtual Bool mustFrag() { return FALSE; }

	// sets/gets the points
	void setPoints(Point* aPoints, int numPoints) {
		numPoints_ = numPoints;
		// release the memory if it is NOT pointing to the same
		// location
		if (aPoints!=aPoints_) delete[] aPoints_;
		aPoints_ = aPoints;
	}
	int getNumPoints() const { return numPoints_; }
	const Point& getPointByIdx(int index) const {
		assert(index<=numPoints_ &&
		       "GetPointByIdx: index out of range!");
		return aPoints_[index];
	}
	Point* getPoints() const { return aPoints_; }
	PageItemType getType() const { return type_; }

private:
	PageItemType type_;
	int numPoints_;
	Point* aPoints_;
};


typedef enum ArrowType_enum {
	ArrowNone=0,		/* no arrows */
	ArrowFirst,		/* 1 */
	ArrowLast,		/* 2 */
	ArrowBoth,		/* 3 */
	ArrowInvalid		/* This value is really num types+1 */
} ArrowType;


extern char* aArrowTypeNames[];


class LineItem : public PageItem {
public:
	LineItem(PageItemType type): PageItem(type),
		width_(1),arrow_(ArrowNone),fg_(NULL) {}
	virtual ~LineItem(){
		if (fg_) MB_FreeColor(fg_);
	}

	double getWidth() { return width_; } // added by Yatin for TGMB

	static int parseArrowType( ClientData clientData, Tcl_Interp* interp,
				   Tk_Window tkwin, char *value,
				   char *recordPtr, int offset );
	static char* printArrowType(ClientData clientData,
				    Tk_Window tkwin,
				    char *recordPtr, int offset,
				    Tcl_FreeProc **freeProcPtr);

	// virtual functions from PageItem
	virtual void getCopy(PageItem** ppResultItem) const;
	virtual void extract(Byte* pb);
	virtual Byte* packetize(Bool inclCoords, Byte *pb);

	virtual u_int getPacketLen(Bool inclCoords) {
		return sizeof(Pkt_LineItem) +
			PageItem::getPacketLen(inclCoords);
	}

public:
	/* DATA  *********/
	int width_;			/* Width of line. */
	ArrowType arrow_;		/* Indicates arrowhead types: */
				/* "none", "first", "last", or "both". */
	XColor *fg_;			/* Foreground color for line. */
};


class PolyItem : public PageItem {
public:
	PolyItem(PageItemType type): PageItem(type),
		width_(1), fillC_(NULL), outlineC_(NULL) {}

	virtual ~PolyItem() {
		if (fillC_) MB_FreeColor(fillC_);
		if (outlineC_) MB_FreeColor(outlineC_);
	}

	/* 2 static functions for custom parsing */
	static int parseArrowType( ClientData clientData, Tcl_Interp* interp,
				   Tk_Window tkwin, char *value,
				   char *recordPtr, int offset );
	static char* printArrowType(ClientData clientData, Tk_Window tkwin,
				    char *recordPtr, int offset,
				    Tcl_FreeProc **freeProcPtr);

	// virtual functions from PageItem
	virtual void getCopy(PageItem** ppItem) const;
	virtual void extract(Byte* pb);
	virtual Byte* packetize(Bool inclCoords, Byte *pb);
	virtual u_int getPacketLen(Bool inclCoords) {
		return (sizeof(Pkt_PolyItem)+PageItem::getPacketLen(inclCoords));
	}

	/* DATA *********/
	int width_;			/* Width of outline. */
	XColor *fillC_;		/* Color of fillings, null if don't fill */
	XColor *outlineC_;		/* Foreground color for outline. */
};


class TextItem : public PageItem {
public:
	TextItem(): PageItem(PgItemText), fillC_(NULL), font_(NULL),
		szText_(NULL) {}
	virtual ~TextItem() {
		if (fillC_) MB_FreeColor(fillC_);
		if (font_) MB_FreeFont(font_);
		delete[] szText_;
	}

	// virtual functions from PageItem
	virtual void getCopy(PageItem** ppItem) const;
	virtual void extract(Byte* pb);
	virtual Byte* packetize(Bool inclCoords, Byte *pb);
	virtual u_int getPacketLen(Bool inclCoords);

	// note: szText will be deleted by destructor
	void setText(char* szText) { szText_ = szText; }

	/* DATA *********/
	XColor *fillC_;		/* Color of text */
	Tk_Font font_;		/* font to use */
	char *szText_;
};


class ImageItem : public PageItem {
public:
	ImageItem(): PageItem(PgItemImage), szImageName_(NULL) {}
	~ImageItem() { delete[] szImageName_; }

	virtual Bool configure(int /*argc*/, const char *const* /*argv*/)
		{ return TRUE; }

	virtual void getCopy(PageItem** ppItem) const {
		*ppItem = new ImageItem();
		PageItem::getCopy(ppItem);
		ImageItem* pImgItem = (ImageItem*) *ppItem;
		::AllocNCopy( &(pImgItem->szImageName_), szImageName_);
	}
	virtual void  extract(Byte* /*pb*/) {}

	const char* getImageName() { return szImageName_; }

	void setImageName(const char* szImageName) {
		assert(!szImageName_ && "should not set this twice or more");
		::AllocNCopy(&szImageName_, szImageName);
	}
	virtual Bool mustFrag() { return TRUE; }

	/* DATA *********/
	char* szImageName_;
};


class PSItem : public PageItem {
public:
	PSItem(): PageItem(PgItemPS), szFileName_(NULL) {}
	~PSItem() { delete[] szFileName_; }

	virtual void getCopy(PageItem ** ppItem) const {
		*ppItem = new PSItem();
		PageItem::getCopy(ppItem);
		PSItem *pPSItem = (PSItem*) *ppItem;
		::AllocNCopy( &(pPSItem->szFileName_), szFileName_);
	}

	virtual void extract(Byte* /*pb*/) {}

	const char* getFileName() { return szFileName_; }
	void setFileName(const char* szFileName) {
		assert(!szFileName_ &&
		       "should not set file name twice or more");
		::AllocNCopy(&szFileName_, szFileName);
	}
	virtual Bool mustFrag() { return TRUE; }

	/* DATA *********/
	char* szFileName_;
};


#endif /* #ifdef __MB_ITEMS_H__ */
