/*
 * session-srmv2.cc --
 *
 *      FIXME: This file needs a description here.
 *
 * Copyright (c) 1998-2002 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * A. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * B. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * C. Neither the names of the copyright holders nor the names of its
 *    contributors may be used to endorse or promote products derived from this
 *    software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
 * IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <tcl.h>
#include <math.h>
#include <limits.h>
#include <fcntl.h>

#include "manager-srmv2.h"
#include "session-srmv2.h"
#include "ntp-time.h"
#include "all-types.h"
#include "tclcl.h"
#include "ns-map.h"
#include "ns-srmv2.h"
#include "timer.h"
#include "random.h"

static class SRMv2_SessionClass : public TclClass {
public:
	SRMv2_SessionClass() : TclClass("Session/SRMv2") { }
	TclObject* create(int, const char*const*) {
		return new SRMv2_Session;
	}
} srmv2_session_class;


SRMv2_Session::SRMv2_Session() :
	Transmitter(), SessionHandler(),
	badproto_(0), simcount_(0),
	c1_(1.0), c2_(1.0),
	pb_(0), lsrc_(0), manager_(0)
{
	bind("lossProb_", &lossProb_);
	dh_.manager(this);
 	ch_.manager(this);

	tokenBucket_ = new SRMv2_TokenBucket(this);
	srcs_ = new Tcl_HashTable;
	Tcl_InitHashTable(srcs_, 4);

	delayinfo_ = new Tcl_HashTable;
	Tcl_InitHashTable(delayinfo_, TCL_ONE_WORD_KEYS);
}

SRMv2_Source *
SRMv2_Session::source(int *srcID, unsigned int addr, int *created)
{
	Tcl_HashEntry *eptr = Tcl_FindHashEntry(srcs_, (char *)srcID);

	if (!eptr) {
		SRMv2_Source *src = new SRMv2_Source(srcID);
		eptr = Tcl_CreateHashEntry(srcs_, (char *)srcID, created);
		Tcl_SetHashValue(eptr, src);
	}
	SRMv2_Source *src = (SRMv2_Source *) Tcl_GetHashValue(eptr);
	/*
	 * Since the IPv4 address may change during the
	 * lifetime of a session, this information must
	 * be updated each time.
	 */
	src->ipaddr(addr);
	return src;
}


SRMv2_Source *
SRMv2_Session::source(SRMv2_Source *src)
{
	int *srcID = src->source_id();
	int created = 0;

	Tcl_HashEntry *eptr = Tcl_FindHashEntry(srcs_, (char *)srcID);

	if (!eptr) {
		eptr = Tcl_CreateHashEntry(srcs_, (char *)srcID, &created);
		Tcl_SetHashValue(eptr, src);
	}
	return (SRMv2_Source *) Tcl_GetHashValue(eptr);
}

int
SRMv2_Session::command(int argc, const char*const* argv)
{
	Tcl& tcl = Tcl::instance();
	char* cp = tcl.buffer();

	if (argc == 2) {
		if (strcmp(argv[1], "exit") == 0) {
			exit(0);
		}
		if (strcmp(argv[1], "data-net") == 0) {
			sprintf(cp, "%s", dh_.net()->name());
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ctrl-net") == 0) {
			sprintf(cp, "%s", ch_.DataHandler::net()->name());
			tcl.result(cp);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "local-source") == 0) {
			sprintf(cp, "%s", lsrc_->name());
			tcl.result(cp);
			return (TCL_OK);
		}
		/* To not turn on announcements while debugging */
		if (strcmp(argv[1], "start-announce") == 0) {
			start_announce();
			return (TCL_OK);
		}
	} else if (argc == 3) {
		if (strcmp(argv[1], "c1_") == 0) {
			c1_ = atof(argv[2]);
//			printf("Setting c_1 to %f\n", c1_);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "c2_") == 0) {
			c2_ = atof(argv[2]);
//			printf("Setting c_2 to %f\n", c2_);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "peak-rate") == 0) {
			/* in Kb/s */
			tokenBucket_->rate(atof(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "buffer-pool") == 0) {
			pool_ = (SRMv2_BufferPool*)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "set-manager") == 0) {
			manager_ = (SRMv2_Manager *)TclObject::lookup(argv[2]);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "local-source") == 0) {
			lsrc_ = (SRMv2_Source*)TclObject::lookup(argv[2]);
			/* Enter into src hash table */
			source(lsrc_);
			return (TCL_OK);
		}
		if (strcmp(argv[1], "log-file") == 0) {
			logfile_ = fopen(argv[2], "w+");
//			logfile_ = stdout;
			return (TCL_OK);
		}
	} else if (argc == 4) {
		if (strcmp(argv[1], "data-net") == 0) {
			dh_.net((Network*)TclObject::lookup(argv[2]));
			return (TCL_OK);
		}
		if (strcmp(argv[1], "ctrl-net") == 0) {
			ch_.net((Network*)TclObject::lookup(argv[2]));
			return (TCL_OK);
		}
	} else if (argc == 6) {
		if (strcmp(argv[1], "req-send") == 0) {
			if (tokenBucket_) {
				unsigned int a[4];
				for (int i = 0; i < 4; i ++) {
					a[i] = (unsigned int) atoi(argv[i+2]);
				}
				tokenBucket_->request_to_send(a[0], a[1], a[2], a[3]);
			} else {
				printf("warning : SRMv2_Session has no tokenBucket_ at RTS\n");
			}
			return (TCL_OK);
		}
	} else if (argc == 7) {
		if (strcmp(argv[1], "transmit-data") == 0) {
			SRMv2_Source *ss = (SRMv2_Source*)TclObject::lookup(argv[2]);
			send_adu(ss, (char *)argv[3], atoi(argv[4]), 0,
				 atoi(argv[5]), atoi(argv[5]), atoi(argv[6]), SRMv2_DATA);
			return (TCL_OK);
		}
	}
	return (Transmitter::command(argc, argv));
}

SRMv2_TokenBucket::SRMv2_TokenBucket(SRMv2_Session *session) : session_(session)
{
	avbl_ = 1024;
	rate_ = 16 * 1024; /* 128 Kb/s in Bytes/sec */
	wait_ = 0.0;
	sampleTime_ = ntptime();
}

void
SRMv2_TokenBucket::request_to_send(unsigned int cid, unsigned int seqno,
				   unsigned int sbytes, unsigned int ebytes)
{
	double prevTime = sampleTime_;
	sampleTime_ = ntptime();
	avbl_ += int(ntp2msec((int) (sampleTime_ - prevTime)) * rate_/1000);
	int excess_ = ebytes - sbytes + 1 - avbl_;
	if (excess_ < 0) excess_ = 0;
	double wait_ = excess_ * 1.0/rate_;
	seqno_  = seqno; cid_ = cid;
	sbytes_ = sbytes; ebytes_ = ebytes;
	msched((int) wait_ * 1000);
}

void
SRMv2_TokenBucket::timeout()
{
	double prevTime = sampleTime_;
	sampleTime_ = ntptime();
	avbl_ += int(ntp2msec((int) (sampleTime_ - prevTime)) * rate_/1000);
	session_->clear_to_send(cid_, seqno_, sbytes_, ebytes_, avbl_);
}

void
SRMv2_TokenBucket::update_avbl(unsigned int xmitted_bytes)
{
	/*
	 * If there was a small error in taking the timeout,
	 * we don't allow avbl_ to go below zero.
	 */
	double prevTime = sampleTime_;
	sampleTime_ = ntptime();
	avbl_ += int(ntp2msec((int) (sampleTime_ - prevTime)) * rate_/1000 - xmitted_bytes);
	avbl_ = MAX(avbl_, 0);
}

void
SRMv2_Session::clear_to_send(unsigned int cid, unsigned int seqno, unsigned int sbytes,
			     unsigned int ebytes, int avbl)
{
	Tcl& tcl = Tcl::instance();
	char *data, *aname;
	NameMap *map = lsrc_->namemap();
	if (map)
		aname = map->getname(cid);
	if (aname) {
		if (strcmp(aname, "/map") == 0) {
			data = map->getname(seqno);
		} else {
			tcl.evalf("%s reada %s %d", manager_->name(), aname, seqno);
			data = tcl.result();
		}
	}
	if (data != 0 && strlen(data) > 0) {
		unsigned int len = (unsigned) strlen(data);
		unsigned char frag;
		if (sbytes != 0 || ebytes + 1 < len) {
			frag = 1;
//			printf("Setting frag [%d -- %d]\n", sbytes, ebytes);
		}
		send_adu(lsrc_, data, seqno, sbytes, ebytes, len - 1, cid, SRMv2_DATA, frag);
	}
}

/*
 * recover:
 * Explicitly query the application to check if "aname"
 * ADU's from 'start' to 'end' need to be repaired.
 * Note that this is a little inefficient if the
 * application does not require the whole range to
 * be repaired. We postpone further optimizations to
 * repair the exact sub-ranges of the loss range.
 * Recall that repair requests are ranges.
 */

int
SRMv2_Session::recover(char *aduname, unsigned int start, unsigned int end)
{
	Tcl& tcl = Tcl::instance();

	tcl.evalf("%s recover %s %d %d", manager_->name(), aduname, start, end);
	int flag = atoi(tcl.result());
	return flag;
}


/*
 * Protocol currently supports only 64 KB long ADUs.
 * For ADUs larger than 64 KB< we need a library that
 * handles fragmentation into pieces <= 64 KB
 *
 * NTP time should be given by the application!
 */
void
SRMv2_Session::send_adu(SRMv2_Source *src, char *data, unsigned int seqno,
		      unsigned int sbytes, unsigned int ebytes, unsigned int adulbyte,
		      unsigned int cid, unsigned char type, unsigned char f)
{
	unsigned int mss;
	unsigned char more = 0;
	unsigned char frag = 0;
	unsigned int sentbytes = sbytes;
	unsigned short int incr;
	char *dp = data;
	dp += sbytes;

	if (!pool_) {
		printf("send failed. No buffer pool.\n");
		abort();
	}
//	printf("S: (%d:%d) \n", cid, seqno);

	/*
	 * Check for fragmentation
	 * Fragement header consumes  ~4B
	 */
	mss = pool_->ip_mtu();
	if (ebytes - sbytes + 1 > mss) {
		mss -= sizeof(srmv2_fraghdr);
		frag = 1;
	}
	frag |= f;
	while (sentbytes < ebytes) {
		if (!pb_)
			pb_ = pool_->alloc(type);
		pool_->build_conhdr(pb_, src->source_id(), cid);
		incr = MIN(ebytes - sentbytes + 1, mss);
		pool_->build_aduhdr(pb_, 0, incr, seqno, ntp64time(), frag);
//		printf("Fragment = %d (orig = %d)\n", frag, f);
		if (frag) {
			more = ((sentbytes + incr - 1) < ebytes) ? 1 : 0;
			more |= (adulbyte != ebytes);
//			printf("Building FRAGMENT header (%d)\n", sentbytes);
			pool_->build_fraghdr(pb_, more, sentbytes);
		}
//		printf("S %d %d.%d %d.%d M=%d ", cid, seqno, sentbytes, seqno,
//		       sentbytes + incr - 1, more);
		memcpy(pb_->dp, dp, incr);
		pb_->len += incr;
		pb_->dp   = pb_->data;
		simloss(incr);
		sentbytes += incr;
		dp += incr;
		tokenBucket_->update_avbl(incr);
	}
	if (lsrc_ == src) {
//		printf("Registering adu %d\n", cid);
		src->namemap()->regadu(cid, seqno, ebytes);
		src->namemap()->update_prio(cid);
	}
}

void
SRMv2_Session::simloss(int len)
{
	if (len > 100 && (double)random()/INT_MAX <= lossProb_) {
		pb_->release();
		pb_ = 0;
//		printf(" (D)\n");
		return;
	}
//	printf(" \n");
	transmit(pb_);
}



void
SRMv2_Session::transmit(pktbuf* pb)
{
	Network* n = 0;
	n = dh_.net();
	if (n != 0) {
		n->send(pb);
		if (!pool_) {
			printf("pool_ is null\n");
			abort();
		}
		pool_->release(pb);
		pb_ = 0;
	}
}

void
SRMv2_Session::recv(DataHandler* dh)
{
	pktbuf* pb = pool_->alloc(0);
	unsigned int addr;
	int port;

	memset(pb->data, 0, PKTBUF_SIZE);
	int cc = dh->recv(pb->data, sizeof(pb->data), addr, port);
	if (cc <= 0)
		return;
	if (addr == LookupLocalAddr()) {
		pb->release();
		return;
	}
//	printf("R %d\n", cc);
	pb->len = cc;
	parse_header(pb, addr);
}

/*
 * We deal with stacking later.
 */
void
SRMv2_Session::parse_header(pktbuf *pb, unsigned int addr)
{
	/* Parse common header */
	srmv2_cmnhdr *sh           = (srmv2_cmnhdr *)pb->data;
	unsigned char type       = sh->type;
	unsigned char version    = sh->flags >> 6;
	unsigned int ipaddr      = htonl(sh->ipaddr);

	if (version != SRMv2_VERSION) {
		printf("Bad version %d\n", version);
		badproto_ ++;
		return;
	}
	pb->dp = pb->data + sizeof(srmv2_cmnhdr);


	/* Parse container header */
	int srcID[4];
	srmv2_conhdr *chdr = (srmv2_conhdr *)pb->dp;
	for (int i = 0; i < 4; i ++)
		srcID[i] = ntohl(chdr->srcid[i]);
	unsigned int cid = chdr->cid;
	pb->dp       += sizeof(srmv2_conhdr);

	int created;
        SRMv2_Source *src = source(&srcID[0], ipaddr, &created);
	srmv2_aduhdr *sah;
	srmv2_rreqhdr *rh;
	srmv2_announcehdr *salist = 0;
	srmv2_delayhdr    *dvec = 0;
	unsigned int tsend;
	int na, more = 0;
	NameMap *map;

	unsigned int ss, es, sbytes = 0, ebytes = 0, offset = 0;
//	if (cid > 0)
//		printf("R: %d ", cid);

	switch (type) {
	case SRMv2_REXMIT :
	case SRMv2_DATA :
		sah  = (srmv2_aduhdr *)pb->dp;
		pb->dp += sizeof(srmv2_aduhdr);
		if (sah->flags >> 7) {
			offset = ntohl(*((srmv2_fraghdr *)pb->dp));
			more   = offset >> 31;
			offset &= 0x7fffffff;
			pb->dp += sizeof(srmv2_fraghdr);
		}
		pb->layer = offset;
		pb->len = ntohs(sah->alen);
//		if (type == SRMv2_REXMIT) {
//			printf("R rxmit %d %d [%d:%d]\n", cid, ntohl(sah->seqno), offset,
//			       offset + pb->len - 1);
//			if (cid == 1)
//				printf("Packet contains %s\n", pb->dp);
//		}
		manager_->handle_data(src, pb, cid, ntohl(sah->seqno),
				      type, offset, more);
		break;

	case SRMv2_RREQ :
		if (manager_) {
			rh = (srmv2_rreqhdr *)pb->dp;
			ss = ntohl(rh->ss);
			sbytes = ntohl(rh->sbytes);
			es = ntohl(rh->es);
			ebytes = ntohl(rh->ebytes);
			pb->dp += sizeof(srmv2_rreqhdr);
//			printf("R rreq 0x%x %d [%d.%d %d.%d]\n", src->source_id(0),
//			       cid, ss, sbytes, es, ebytes);
			manager_->handle_rreq(src, ss, sbytes, es,
					      ebytes, cid);
		}
		pb->release();
		break;

	case SRMv2_ANNOUNCE :
		na = (pb->len - sizeof(srmv2_conhdr) -
			  sizeof(srmv2_cmnhdr))/sizeof(srmv2_announcehdr);
		salist = (srmv2_announcehdr *)pb->dp;
		map = src->namemap();
		map->get_namespace()->compute_signature();
//		map->get_namespace()->root()->display(1);

		int k;
		for (k = 0; k < na; k++) {
//			printf(" sa(%d) %d.%d \n", ntohl(salist[k].cid),
//			       ntohl(salist[k].sign), ntohl(salist[k].ebytes));
			manager_->handle_announce(src, ntohl(salist[k].cid),
						  ntohl(salist[k].sign),
						  ntohl(salist[k].ebytes));
		}
		pb->release();
		break;

	case SRMv2_DELAYS :
		na = (pb->len - sizeof(unsigned int) - sizeof(srmv2_cmnhdr) -
		      sizeof(srmv2_conhdr))/sizeof(srmv2_delayhdr);
		tsend = ntohl(*(unsigned int*)(pb->dp));
		pb->dp += sizeof(unsigned int);

		dvec = (srmv2_delayhdr *)pb->dp;
//		printf("IPv4 address %s %u\n", intoa(ipaddr), tsend);
		update_delay(ipaddr, tsend, na, dvec);
		pb->release();
		break;

	case SRMv2_NSREPAIR :
		na = (pb->len - sizeof(srmv2_conhdr) -
			     sizeof(srmv2_cmnhdr))/sizeof(srmv2_announcehdr);
		salist = (srmv2_announcehdr *)pb->dp;
		map = src->namemap();
		map->get_namespace()->compute_signature();
		manager_->handle_nsrepair(src, salist, na);
		pb->release();
		break;
	}
}


void
SRMv2_Session::send_rreq(SRMv2_Source *src, unsigned int ss, unsigned int sbytes,
			 unsigned int es, unsigned int ebytes, unsigned int cid)
{
	if (!pool_) {
		printf("send failed. No buffer pool.\n");
		abort();
	}
	if (!pb_) pb_ = pool_->alloc(SRMv2_RREQ);

	pool_->build_conhdr(pb_, src->source_id(), cid);
	pool_->build_rreqhdr(pb_, ss, sbytes, es, ebytes);
//	printf("S REQ %s %d %d.%d %d.%d\n", intoa(src->ipaddr()),
//	       cid, ss, sbytes, es, ebytes);
//	fflush(logfile_);
	pb_->dp  = pb_->data;
	transmit(pb_);
	return;
}


void
SRMv2_Session::send_announce(SRMv2_Source *src, srmv2_announcehdr* sa_list,
			     int count, unsigned char type)
{
	if (!pool_) {
		printf("send failed. No buffer pool.\n");
		abort();
	}
	if (!pb_)
		pb_ = pool_->alloc(type);
	/* FIXME This should be build_srchdr. Wasting some space. */
	pool_->build_conhdr(pb_, src->source_id(), SRMv2_ERROR);
	pool_->build_announcehdr(pb_, sa_list, count);
	pb_->dp  = pb_->data;
	transmit(pb_);
	return;
}

/*
 * Multicast a delay information vector.
 */
void
SRMv2_Session::send_delayinfo()
{
	if (!pool_) {
		printf("send failed. No buffer pool.\n");
		abort();
	}
	if (!pb_)
		pb_ = pool_->alloc(SRMv2_DELAYS);
	pool_->build_conhdr(pb_, lsrc_->source_id(), 0);
	fill_delays(pb_);
	pb_->dp  = pb_->data;
	transmit(pb_);
	return;
}

void
SRMv2_Session::start_announce()
{
	atimer_ = 20000;
	msched(atimer_);
}


void
SRMv2_Session::timeout()
{
	NameMap* map;
	srmv2_announcehdr *sahdr;

	if (lsrc_) {
		map = lsrc_->namemap();
//		printf("Timeout -- Sending new announcement ...\n");
		map->get_namespace()->compute_signature();
//		map->get_namespace()->root()->display(1);
		sahdr = map->hiprio(30);
		if (sahdr) {
			send_announce(lsrc_, sahdr, 30);
			delete [] sahdr;
		}
	}
	send_delayinfo();
	msched(atimer_);
}


float
SRMv2_Session::delay(unsigned int address)
{
	float delay=0.0;

	Tcl_HashEntry *eptr = Tcl_FindHashEntry(delayinfo_, (const char*) address);
	if (eptr) {
		srmv2_delays *record = (srmv2_delays *) Tcl_GetHashValue(eptr);
		delay = ntp2msec((int) (0.5 * record->rtt_est));
	}
	if (delay > 0.0) {
		return delay;
	} else {
		return MIN_BACKOFF_DELAY;
	}
}


/*
 * Compute backoff after reading the estimated delay.  We should be
 * using more conservative constants d1_ and d2_ for repairs,
 * but for now, use the same constants regardless of packet type.
 */
double
SRMv2_Session::backoff(SRMv2_Source *src, unsigned int level)
{
	double d = (double) delay(src->ipaddr());
	double r = (double)random()/double(INT_MAX);
//	printf("Level %u\n", level);
	double p = pow(2, level);
	double b = p * (c1_ + c2_ * r) * d;
//	printf("d = %f, backoff = %f, Level = %u\n",
//	       d, b, level);
	/*
	 * This is an arbitrary rule to avoid retransmitting too early!
	 * We should wait at least one RTT.
	 */
	return b;
}

/*
 * SRMv2_DELAYS :
 *                      (32)
 * +--------+---------+---------------------------+
 * |              tsend (NTP timestamp)           |
 * +--------+---------+---------------------------+
 * |                 IPv4 addr           (0)      |
 * +--------+---------+---------------------------+
 * |               one-way difference    (0)      |
 * +--------+---------+---------------------------+
 * |                 IPv4 addr           (1)      |
 * +--------+---------+---------------------------+
 * |               one-way difference    (1)      |
 * +--------+---------+---------------------------+
                      .....
 * +--------+---------+---------------------------+
 * |                 IPv4 addr           (n-1)    |
 * +--------+---------+---------------------------+
 * |               one-way difference    (n-1)    |
 * +--------+---------+---------------------------+
 */

void
SRMv2_Session::fill_delays(pktbuf *pb)
{

	unsigned int tsend = ntptime();
	*(unsigned int *) pb->dp = htonl(tsend);
	pb->dp  += 4;
	pb->len += 4;
	srmv2_delayhdr* di = (srmv2_delayhdr *)pb->dp;

	Tcl_HashSearch hs;
	unsigned int addr;
	int j = 0;
	srmv2_delays *pDI;

	for (Tcl_HashEntry* pEntry = Tcl_FirstHashEntry(delayinfo_, &hs);
	     pEntry; pEntry = Tcl_NextHashEntry(&hs))
	{
		addr = (unsigned int)Tcl_GetHashKey(delayinfo_, pEntry);
		pDI  = (srmv2_delays *)Tcl_GetHashValue(pEntry);

                /* Ignore estimates that don't provide any information. */
		if (pDI->ow_diff == 0)
			continue;
		di[j].ip_addr = htonl(addr);
		di[j].ow_diff = htonl(pDI->ow_diff);
//		printf("DI: %x %d\n", addr, pDI->ow_diff);
		j++;
	}
	pb->dp  += j * sizeof(srmv2_delayhdr);
	pb->len += j * sizeof(srmv2_delayhdr);
}


/*
 * Update our delay with respect to 'ipaddr'.
 * tsend is in ntp format, IPaddr is a 32-bit IPv4 address.
 *
 * t1
 *   \
 *   t2
 *    |
 *   t3
 *   /
 * t4
 *
 */

void
SRMv2_Session::update_delay(unsigned int addr, unsigned int tsend,
			    unsigned int n, srmv2_delayhdr* dvec)
{
	unsigned int local_ipaddr = LookupLocalAddr();
	if (local_ipaddr == addr) return; /* Discard our own packet. */
	unsigned int now = ntptime();
	/* Update the one-way delay estimate (owd_est) */

	int created;
	Tcl_HashEntry* pEntry = Tcl_CreateHashEntry(delayinfo_, (const char*) addr,
						    &created);
	srmv2_delays* pDI = 0;
	if (created) {
		pDI = new srmv2_delays;
		pDI->rtt_est = 0;
		pDI->ow_diff = 0;
		Tcl_SetHashValue(pEntry, pDI);
	} else {
		pDI = (srmv2_delays *) Tcl_GetHashValue(pEntry);
	}

	/*
	 * For the incoming announcement, t2 = now, t1 = tsend, so
	 * ow_diff = now - tsend.
	 */
	pDI->ow_diff = now - tsend;
//	printf("tsend= %u ; now= %u ; ow_diff= %d ms\n",
//	       tsend, now, pDI->ow_diff);

	/*
	 * t4 = now
	 * td = t1 + delta = t1 + t3 - t2 = t3 - (t2 - t1)
	 * rtt_est = (t4 - t1 - delta) = (now - td)
	 */
	for (unsigned int i=0; i < n; i++) {
		if (ntohl(dvec[i].ip_addr) == local_ipaddr) {
			unsigned int new_sample = (now - tsend + ntohl(dvec[i].ow_diff));
			if (new_sample > 0) {
				if (pDI->rtt_est == 0.0) {
					pDI->rtt_est = new_sample;
				} else {
					pDI->rtt_est = (unsigned int)(0.3 * pDI->rtt_est
								      + 0.7 * new_sample);
				}
				printf("RTT %s %f\n", intoa(addr),
				       ntp2msec(pDI->rtt_est));
//				fflush(logfile_);
			}
			break;
		}
	}
}

