# archive-record.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1997-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# @(#) $Header: /usr/mash/src/repository/mash/mash-1/tcl/archive/recorder/archive-record.tcl,v 1.17 2002/02/03 04:25:13 lim Exp $


import Observable

# The ArchiveSession/Record class provides a shell for implementing session
# classes for RTP and SRM recorder session classes.
#
# Note: This is an abstract base class. Do not create objects directly of
# this class
#
# Any observer that is attached to objects of this class should define
# methods corresponding to the following observable events:
# new_stream:
#      $session notify_observers new_stream <stream>
# Invoked when the recording session detects a new source and creates a
# new stream object for it. stream must be an object of a class that is
# subclassed from ArchiveStream.
# stream_done:
#      $session notify_observers stream_done <stream>
# Invoked when the recording session detects an existing source has
# vanished. stream must be an object of a class that is subclassed from
# ArchiveStream.
# Status: Beta
# Author: Yatin Chawathe and Angela Schuett

Class ArchiveSession/Record -superclass Observable
set classes [ArchiveStream info superclass]
set objectIdx [lsearch $classes Observable]
if { $objectIdx == -1 } {
	ArchiveStream superclass [concat Observable $classes]
}

ArchiveSession/Record instproc init { media } {
	$self next
	$self set stream_count_ 0
	$self media $media
}

ArchiveStream/Record public destroy {} {
	$self next
	# close and delete the data and index files from C, so that the end
	# timestamps can be written properly

}

ArchiveSession/Record instproc catalog { args } {
	$self instvar catalog_
	if { [llength $args]==0 } {
		if [info exists catalog_] {
			return $catalog_
		} else {
			return ""
		}
	} else {
		set catalog_ [lindex $args 0]
	}
}

# 1.  set directory [$session save_in] returns the name of the directory
# in which stream data and index files are stored, if the directory has
# previously been set, otherwise it returns an empty string.
# 2. $session save_in $directory sets the directory name in which the
# stream data and index files for this

ArchiveSession/Record instproc save_in { args } {
	switch -exact -- [llength $args] {
		0 {
			if [info exists directory_] {
				return $directory_
			} else {
				return ""
			}
		}
		1 {
			$self set directory_ [lindex $args 0]
			return
		}
		default {
			error "too many arguments"
		}
	}
}

# 1.  set session_id [$session session_id] returns the session-id string
# if it has previously been set, otherwise it returns an empty string.
# 2.  $session session_id $session_id set the session-id associated with
# this session. session_id is simply a string that contains valid
# filename characters (directory separator characters are not
# permitted). The session-id is used as a prefix for data and index
# filenames.

ArchiveSession/Record instproc session_id { args } {
	switch -exact -- [llength $args] {
		0 {
			if [info exists session_id_] {
				return $session_id_
			} else {
				return ""
			}
		}
		1 {
			$self set session_id_ [lindex $args 0]
			return
		}
		default {
			error "too many arguments"
		}
	}
}

#   1.  set m [$session media] returns a string that represents the media
# associated with this session.
#  2.  $session media $m
#
# Set or query the media associated with this session. Please note that
# the media is automatically set by the init method of this class.
# Programmers should not invoke this method directly to set the media.

ArchiveSession/Record instproc media { args } {
	$self instvar media_
	switch -exact -- [llength $args] {
		0 {
			if [info exists media_] {
				return $media_
			} else {
				return ""
			}
		}
		1 {
			$self set media_ [lindex $args 0]
			$class instvar media_count_
			if { ![info exists media_count_($media_)] } {
				set media_count_($media_) 0
			}
			incr media_count_($media_)
			$self set media_count_ $media_count_($media_)
			return
		}
		default {
			error "too many arguments"
		}
	}
}


# Return a filename that is guaranteed to be unique across all streams
# and all sessions that are part of this recorder process. The filename
# does not contain the ".dat" or ".idx" extension. The filename has the
# following format: dir/session_id-mediamedia_count-count
# where dir is the value returned by the save_in method, session_id is
# the value returned by the session_id method (if the session_id field
# is empty, that part of the filename, including the following hyphen,
# is eliminated), media is the value returned by the media method,
# media_count is a media-specific counter, so that two sessions with the
# same media will not collide on their filenames, and count is a
# session-specific counter, so different streams within a session have
# different filenames.

ArchiveSession/Record instproc generate_filename { } {
	$self instvar directory_ session_id_ media_ media_count_ stream_count_
	set name ""
	if { $session_id_!="" } {
		append name "${session_id_}-"
	}
	incr stream_count_
	append name "${media_}${media_count_}-${stream_count_}"
	return [file join $directory_ $name]
}

ArchiveStream/Record instproc bind { session } {
	$self set archive_session_ $session
	if { [catch {
		set filename [$session generate_filename]
		set dataFile  [new ArchiveFile/Data]
		$dataFile open "${filename}.dat" "w"

		set indexFile [new ArchiveFile/Index]
		$indexFile open "${filename}.idx" "w"
		$self write_to_catalog $session $filename
		$session notify_observers new_stream $self
		$self notify_observers filename "${filename}.dat \[.idx\]"
		$self datafile  $dataFile
		$self indexfile $indexFile
	} error] } {
		return $error
	}
	return ""
}

ArchiveStream/Record instproc write_to_catalog { session filename } {
	set catalog [$session catalog]
	if { $catalog=="" } return

	set dirname    [file dirname $filename]
	set catalogdir [file dirname [$catalog filename]]
	if { [string first $catalogdir $dirname]==0 } {
		# the save_in directory is a child of the catalog directory!
		# Let's use relative paths

		set dirname [file split $dirname]
		set ignore [llength [file split $catalogdir]]

		set dirname [lrange $dirname $ignore end]
		if { [llength $dirname]==0 } {
			set filename [file tail $filename]
		} else {
			set filename [eval file join $dirname \
					[list [file tail $filename]]]
		}
	}
	$session instvar media_ media_count_
	$catalog write_stream $self "$media_$media_count_" \
			"${filename}.dat" "${filename}.idx"
}


ArchiveStream/Record instproc session { } {
	return [$self set archive_session_]
}


ArchiveStream/Record instproc media { } {
	return [[$self set archive_session_] media]
}

