# ui-ctrlmenu.tcl --
#
#       FIXME: This file needs a description here.
#
# Copyright (c) 1993-2002 The Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
#
# A. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
# B. Redistributions in binary form must reproduce the above copyright notice,
#    this list of conditions and the following disclaimer in the documentation
#    and/or other materials provided with the distribution.
# C. Neither the names of the copyright holders nor the names of its
#    contributors may be used to endorse or promote products derived from this
#    software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
# ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
# CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
# OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
# $Header: /usr/mash/src/repository/mash/mash-1/tcl/vic/ui-ctrlmenu.tcl,v 1.80 2002/02/03 04:39:54 lim Exp $


import TextEntry ErrorWindow GlobalWindow KeyEditor GammaEntry Configuration

#
# A control menu to control the various elements of
# the video capture and rendering subsystems.
#
Class ControlMenu -superclass TopLevelWindow -configuration {
	recvOnly 0
	framerate 8
	maxbw -1
	bandwidth 128000
	useHardwareDecode false
	stillGrabber false
}

#
ControlMenu proc fork_histtolut { } {
	$self tkvar ditherStyle_
	$self instvar vframe_ asm_ optionsMenu_

	if { $ditherStyle_ == "gray" } {
		new ErrorWindow "cannot optimize grayscale rendering"
		return
	}
	#
	# create a histogram object and collect stats from decoders
	#
	set ch [[$vframe_ set colorModel_] create-hist]
	set active 0

	foreach src [$asm_ active-sources] {
		set d [$src handler]
		if { ![$src mute] && $d != "" } {
			$d histogram $ch
			set active 1
		}
	}
	if !$active {
		new ErrorWindow "no active, unmuted sources"
		delete $ch
		return
	}
	set pid [pid]
	set outfile /tmp/vicLUT.$pid
	set infile /tmp/vicHIST.$pid

	if { [$ch dump $infile] < 0 } {
		new ErrorWindow "couldn't create $infile"
		delete $ch
		return
	}
	delete $ch
	#
	# fork off a histtolut and use tk fileevent to call back
	# finish_histtolut when the child is done
	#
	#FIXME -n
	set eflag ""
	if { $ditherStyle_ == "ed" } {
		set eflag "-e"
	}
	if [catch \
	  "open \"|histtolut $eflag -n 170 -o $outfile $infile\"" pipe] {
		new ErrorWindow "histtolut not installed in your path"
		return
	}
	fileevent $pipe readable "$self finish_histtolut $pipe $infile $outfile"
	# disable button while histtolut is running
	$optionsMenu_ entryconfigure "Optimize Colormap" \
		-state disabled
	$self instvar path_
	$path_ configure -cursor watch
}

#
ControlMenu proc finish_histtolut { pipe infile outfile } {
	$self instvar path_
	$path_ configure -cursor ""
	$self instvar optionsMenu_ vframe_ asm_
	$optionsMenu_ entryconfigure "Optimize Colormap" -state normal
	set cm [$vframe_ set colorModel_]
	$cm free-colors
	$cm lut $outfile
	if ![$cm alloc-colors] {
		#FIXME unset lut
		$vframe_ revert_to_gray
	}
	#FIXME make this a proc
	foreach src [$asm_ active-sources] {
		set d [$src handler]
		if { $d != "" } {
			$d redraw
		}
	}
	fileevent $pipe readable ""
	close $pipe
}

#
ControlMenu instproc have_transmit_permission {} {
	$self instvar vpipe_
	if { [$vpipe_ input_devices] != "" } {
		return ![$self yesno recvOnly]
	}
	return 0
}

#
# Creates a toplevel window at ".menu".  Send it a ui (<i>mainUI</i>), a
# VideoAgent (<i>agent</i>), a VideoPipeline (<i>vpipe</i>), a
# VisualFrame (<i>vframe</i>), an ActiveSourceManager (<i>asm</i>), and
# a UISrcListWindow (<i>uiSrcListWin</i>).  The <i>ui</i> object must
# implement the accessor methods, <i>use_hw_decode</i> and
# <i>mute_new_sources</i>.
#
ControlMenu instproc init { mainUI agent vpipe vframe asm uiSrcListWin} {
	$self next .menu
	$self instvar ui_ videoAgent_ qval_ lastFmt_ path_ vpipe_ ui_srclist_ vframe_ asm_
	set ui_ $mainUI
	set ui_srclist_ $uiSrcListWin
	set videoAgent_ $agent
	set vpipe_ $vpipe
	set vframe_ $vframe
	set asm_ $asm
	set lastFmt_ ""

	set qval_(h261) 68
	#H.263 change
	set qval_(h263+) 68
	set qval_(h263) 68
	set qval_(nv) 80
	set qval_(nvdct) 80
	set qval_(pvh) 60
	set qval_(jpeg) 29

	$self tkvar useHardwareDecode_ ditherStyle_
	set ditherStyle_ [$vframe_ set dither_]
	#FIXME
	set useHardwareDecode_ [$self yesno useHardwareDecode]

	$self tkvar muteNewSources
	set muteNewSources [$self yesno muteNewSources]
}

#
# Build the menu panel.  Called from toggle_window,
# the first time the Menu button is hit.
#
ControlMenu instproc build w {
	$self create-window $w "Video Settings"
	wm withdraw $w
	catch "wm resizable $w false false"

#	$w configure -background yellow

	frame $w.session
	frame $w.cb
	$self build.xmit $w.cb
	if { [$self yesno useScuba] && [$self get_option megaVideoSession] != ""} {
		frame $w.scuba
		$self build.scuba $w.scuba
	}
	frame $w.encoder
	$self build.encoder $w.encoder
	frame $w.decoder
	$self build.decoder $w.decoder
	$self instvar videoAgent_
	#FIXME should abstract out session descriptor
	$self build.session $w.session \
		[$videoAgent_ session-addr] \
		[$videoAgent_ session-sport]:[$videoAgent_ session-rport] \
		[$videoAgent_ get_local_srcid] \
		[$videoAgent_ session-ttl] \
		[$videoAgent_ local-name]

	button $w.dismiss -text Dismiss -borderwidth 2 -width 8 \
		-relief raised -anchor c \
		-command "$self toggle" -font [$self get_option medfont]

	pack $w.cb -padx 6 -fill x -expand 1
	if { [$self yesno useScuba] && [$self get_option megaVideoSession] != "" } {
		pack $w.scuba -padx 6 -fill x -expand 1
	}
	pack $w.encoder $w.decoder $w.session -padx 6 -fill x -expand 1
	pack $w.dismiss -anchor c -pady 4

	if [$self have_transmit_permission] {
		$self selectInitialDevice
	}

        wm protocol $w WM_DELETE_WINDOW "$self toggle"
}

#
ControlMenu instproc selectInitialDevice {} {
	$self instvar vpipe_ device_ transmitButton_
	$self tkvar transmitButtonState_
	set L [$vpipe_ input_devices]
	set d [$self get_option defaultDevice]
	set selected 0
	foreach v $L {
		if { [$v nickname] == "$d" && \
				[$v attributes] != "disabled" } {
#			puts "selectInitialDevice setting device_"
			set device_ $v
			$self select_device $v
			set selected 1
			break
		}
	}
	if !$selected {
		foreach v $L {
			if { "[$v attributes]" != "disabled" && \
					"[$v nickname]" != "still" } {
#				puts "selectInitialDevice2 setting device_"
				set device_ $v
				$self select_device $v
				set selected 1
				break
			}
		}
	}
	if { $selected && [$self get_option xmitVideoOnStartup]!={} && \
			[$transmitButton_ cget -state] != "disabled" && \
			!$transmitButtonState_} {
		set transmitButtonState_ 1
		$self transmit
	}
}

#
ControlMenu instproc create_global_window {} {
	$self instvar src_ global_win_
	if [info exists global_win_] {
		$self delete_global_window
	} else {
		set global_win_ [new GlobalWindow .globalStats "Session Stats" "$self stats" "$self delete_global_window"]
	}
}

#
ControlMenu instproc delete_global_window {} {
	$self instvar global_win_
	delete $global_win_
	unset global_win_
}

#
ControlMenu instproc stats {} {
	return [[$self set videoAgent_] stats]
}

#
ControlMenu instproc new_hostspec {} {
	$self instvar videoAgent_ addrspec_ namespec_
	if ![info exists addrspec_] {
		return
	}
	set dst [$videoAgent_ session-addr]
	set port [$videoAgent_ session-sport]:[$videoAgent_ session-rport]
	set ttl [$videoAgent_ session-ttl]
	set srcid [$videoAgent_ get_local_srcid]
	$addrspec_ configure -text \
			"Dest: $dst   Port: $port  ID: $srcid  TTL: $ttl"

	set name [$videoAgent_ local-name]
	$namespec_.entry delete 0 end
	$namespec_.entry insert 0 $name

	$self instvar transmitButton_
	$transmitButton_ configure -state normal
}

#
# switches the VideoAgent object. 
# This method should be called always from VicUI::switch-agent
#
ControlMenu instproc switch-agent {new_agent} {
	$self instvar videoAgent_
	$self instvar addrspec_ namespec_


	# set the new VideoAgent
	set videoAgent_ $new_agent


	# recreate the UI
	if ![info exists addrspec_] {
		return
	}
	set dst [$videoAgent_ session-addr]
	set port [$videoAgent_ session-sport]:[$videoAgent_ session-rport]
	set ttl [$videoAgent_ session-ttl]
	set srcid [$videoAgent_ get_local_srcid]
	$addrspec_ configure -text \
			"Dest: $dst   Port: $port  ID: $srcid  TTL: $ttl"

	set name [$videoAgent_ local-name]
	$namespec_.entry delete 0 end
	$namespec_.entry insert 0 $name

	$self instvar transmitButton_
	$transmitButton_ configure -state normal


	# not sure if we should set the session bw without the 
	# user changing it
	# $seld set_sessionbw window value 
}


#
ControlMenu instproc build.session { w dst port srcid ttl name } {
	set f [$self get_option smallfont]

	label $w.title -text Session
	pack $w.title -fill x

	frame $w.nb -relief sunken -borderwidth 2
	pack $w.nb -fill x

	frame $w.nb.frame
	pack append $w.nb \
			$w.nb.frame { top fillx }

	$self instvar addrspec_ sessionspec_ namespec_
	label $w.nb.frame.info -font $f -anchor w \
			-text "Dest: $dst   Port: $port  ID: $srcid  TTL: $ttl"
	set addrspec_ $w.nb.frame.info

	# the "Switch session: ..." line
	frame $w.nb.frame.session
	label $w.nb.frame.session.label -text "Switch session: " -font $f\
			-anchor e -width 20
	new TextEntry "$self update_session" $w.nb.frame.session.entry ""
	pack $w.nb.frame.session.label -side left
	pack $w.nb.frame.session.entry -side left -expand 0 -fill x -pady 2
	set sessionspec_ $w.nb.frame.session

	# the "Name: ..." line
	frame $w.nb.frame.name
	label $w.nb.frame.name.label -text "Name: " -font $f -anchor e -width 7
	new TextEntry "$self update_name" $w.nb.frame.name.entry $name
	pack $w.nb.frame.name.label -side left
	pack $w.nb.frame.name.entry -side left -expand 1 -fill x -pady 2
	set namespec_ $w.nb.frame.name

	# the "Note: ..." line
	frame $w.nb.frame.msg
	label $w.nb.frame.msg.label -text "Note: " -font $f -anchor e -width 7
	new TextEntry "$self update_note" $w.nb.frame.msg.entry ""
	pack $w.nb.frame.msg.label -side left
	pack $w.nb.frame.msg.entry -side left -expand 1 -fill x -pady 2

	#
	# Create the entry widget for editing the session's
	# encryption key.  The audio agent exports methods
	# (invoked by the KeyEditor) to enable/disable encryption.
	#
	$self instvar videoAgent_
	new KeyEditor $w.nb.frame $videoAgent_

	frame $w.nb.frame.b

	# the "Global Stats" button
	button $w.nb.frame.b.stats -text "Global Stats" -borderwidth 2 \
			-anchor c -font $f -command create_global_window
	# for now...
	$w.nb.frame.b.stats configure -state disabled

	# the "Members" button
	$self instvar ui_srclist_
	button $w.nb.frame.b.members -text Members -borderwidth 2 \
			-anchor c -font $f -command "$ui_srclist_ toggle"

	# the "Switch Session" button
#	$self instvar ui_
#	button $w.nb.frame.b.switch -text "Switch Session" -borderwidth 2 \
#			-anchor c -font $f -command "$ui_ switch-agent 224.1.1.1/11111"


	# pack all the widgets
	pack $w.nb.frame.b.stats $w.nb.frame.b.members \
			-side left -padx 4 -pady 2 -anchor c

	pack $w.nb.frame.info $w.nb.frame.session $w.nb.frame.name\
			$w.nb.frame.msg $w.nb.frame.key -fill x -padx 2 -expand 1
	pack $w.nb.frame.b -pady 2 -anchor c
}

#
ControlMenu instproc setFillRate {} {
	$self instvar vpipe_
	global sendingSlides
	$self tkvar transmitButtonState_
	if $transmitButtonState_ {
		if $sendingSlides {
			$vpipe_ fillrate 16
		} else {
			$vpipe_ fillrate 2
		}
	}
}

#
ControlMenu instproc update_session spec {
	$self instvar sessionspec_
	$self instvar ui_

	# delete the text entry
	$sessionspec_.entry delete 0 end

	# check if the new session spec is correct
	set err "You must type a correct specification"
	if {$spec != ""} {
		# we need to create the AddressBlock with a valid spec or it will 
		# exit mash by calling "$self fatal", which cannot be catched 
		# because it ends up calling "exit 1" at Log::fatal{} 
		# (mash/tcl/common/log.tcl).
		set ab [new AddressBlock 224.1.1.1/11111]
		set err [$ab parse $spec]
		delete $ab
	}
	if {$err != ""} {
		Log warn $err
		return 0
	}

	# the spec is valid
	$ui_ switch-agent $spec
	return 0
}

#
ControlMenu instproc update_name name {
	if { $name != ""} {
		$self instvar videoAgent_
		$videoAgent_ set_local_sdes name $name
		return 0
	}
	return -1
}

#
ControlMenu instproc update_note note {
	$self instvar videoAgent_
	$videoAgent_ set_local_sdes note $note
	return 0
}

#
ControlMenu instproc transmit { } {
	$self instvar vpipe_ device_
	$self tkvar transmitButtonState_
	global videoFormat

	if $transmitButtonState_ {
#		puts "controlmenu::transmit calling select on pipeline"
		set err [$vpipe_ select $device_ $videoFormat]
		if { $err != "" } {
			set transmitButtonState_ 0
			new ErrorWindow $err
			$self select_device $device_
			return
		}
		set err [$vpipe_ start]
		if { $err != "" } {
			set transmitButtonState_ 0
			new ErrorWindow $err
			$self select_device $device_
			return
		}
		#FIXME
		$self tx-init
	} else {
		$vpipe_ stop
	}
}

#
ControlMenu instproc release {} {
	$self tkvar transmitButtonState_
	set transmitButtonState_ 0
	[$self set vpipe_] release_device
}

#
ControlMenu instproc build.buttons w {
	set f [$self get_option smallfont]
	$self instvar transmitButton_
	$self tkvar transmitButtonState_
	set transmitButton_ $w.send
	set transmitButtonState_ 0
	checkbutton $w.send -text "Transmit" \
		-relief raised -command "$self transmit" \
		-anchor w -variable [$self tkvarname transmitButtonState_] \
		-font $f \
		-state disabled -highlightthickness 0
	button $w.release -text "Release" \
		-relief raised -command "$self release" \
		-font $f -highlightthickness 0

	pack $w.send $w.release -fill both
}

#
ControlMenu instproc invoke_transmit {} {
	$self instvar transmitButton_
	$transmitButton_ invoke
}

#
ControlMenu instproc set_sessionbw { w value } {
	$self instvar videoAgent_
	$videoAgent_ sessionbw $value
	$w configure -text [format_bps $value]
	update idletasks
}

#
ControlMenu instproc set_bps { w value } {
	$self instvar vpipe_ videoAgent_
	$vpipe_ set_bps $value
	$videoAgent_ local_bandwidth $value
	$w configure -text [format_bps $value]
	update idletasks
}

#
ControlMenu instproc set_fps { w value } {
	$self instvar vpipe_
	$vpipe_ set_fps $value
	$w configure -text "$value f/s"
	update idletasks
}

#
ControlMenu instproc build.sliders w {
	set f [$self get_option smallfont]

	global V
	global btext ftext
	$self instvar videoAgent_
	set key [$videoAgent_ set session_]
	set ftext($key) "0.0 f/s"
	set btext($key) "0.0 kb/s"

	if [$self yesno useScuba] {
		set rctext "Rate Control (SCUBA)"
		set maxbw [$self get_option maxVideoSessionBW]
	} else {
		set rctext "Rate Control"
		set maxbw [$self get_option maxbw]
	}

	frame $w.info
	label $w.info.label -text $rctext -font $f
	label $w.info.fps -textvariable ftext($key) -width 6 \
		-font $f -pady 0 -borderwidth 0
	label $w.info.bps -textvariable btext($key) -width 8 \
		-font $f -pady 0 -borderwidth 0
	pack $w.info.label -side left
	pack $w.info.bps $w.info.fps -side right

	frame $w.bps
	scale $w.bps.scale -orient horizontal -font $f \
		-showvalue 0 -from 1 -to $maxbw \
		-command "$self set_bps $w.bps.value" -width 12 \
		-relief groove
	label $w.bps.value -font $f -width 10 -anchor w

	frame $w.fps
	scale $w.fps.scale -font $f -orient horizontal \
		-showvalue 0 -from 1 -to 30 \
		-command "$self set_fps $w.fps.value" -width 12 \
		-relief groove
	label $w.fps.value -font $f -width 10 -anchor w

	pack $w.info -fill x
	pack $w.bps $w.fps -fill x
	pack $w.bps.scale -side left -fill x -expand 1
	pack $w.bps.value -side left -anchor w
	pack $w.fps.scale -fill x -side left -expand 1
	pack $w.fps.value -side left -anchor w

	if [$self yesno useScuba] {
		set s [$videoAgent_ set session_]
		$w.bps.scale set [$s data-bandwidth]
		$w.fps.scale set 30
	} else {
		$w.bps.scale set [$self get_option bandwidth]
		$w.fps.scale set [$self get_option framerate]
		$w.bps.scale configure -resolution 1000
		$w.bps.scale configure -from 1000
	}

	global fps_slider bps_slider
	set fps_slider $w.fps.scale
	set bps_slider $w.bps.scale
}

#
# called when selecting a new device: insert a grabber control panel
# if it exists and remove the old one (if one was installed)
#
ControlMenu instproc insert_grabber_panel devname {
	set k [string first - $devname]
	if { $k >= 0 } {
		incr k -1
		set devname [string range $devname 0 $k]
	}
	set k [string first " " $devname]
	if { $k >= 0 } {
		incr k -1
		set devname [string range $devname 0 $k]
	}
	set devname [string tolower $devname]
	if {[string range $devname end end] == ":"} {
		set devname [string range $devname 0 [expr [string length $devname] - 2]]
	}
	$self instvar path_
	set w $path_.$devname
	global grabberPanel
	if [info exists grabberPanel] {
		if { "$grabberPanel" == "$w" } {
			return
		}
		pack forget $grabberPanel
		unset grabberPanel
	}
	if { [$class info instprocs build.$devname] != "" } {
		if ![winfo exists $w] {
			frame $w
			#label $w.label -text "Grabber ($devname)"
			#frame $w.frame -relief sunken -borderwidth 0
			#$self build.$devname $w.frame
			#pack $w.label $w.frame -side top -fill x -expand 1
			$self build.$devname $w
			pack $w -side top -fill x -expand 1
		}
# else {
#  		puts "didn't create menu because it already exists"
#		}
		pack $w -before $path_.encoder -padx 6 -fill x
		set grabberPanel $w
	}
}

#
# Called when user selects a particular device (like videopix or xil)
# (and at startup for default device)
#
ControlMenu instproc select_device device {
	global formatButtons \
		videoFormat defaultFormat lastDevice defaultPort inputPort
	$self instvar videoAgent_ vpipe_ sizeButtons_ portButton_ \
		transmitButton_
	$self tkvar transmitButtonState_

	#
	# Remember settings of various controls for previous device
	# and close down the device if it's already open
	#
	set wasTransmitting $transmitButtonState_
	if [info exists lastDevice] {
		set defaultFormat($lastDevice) $videoFormat
		set defaultPort($lastDevice) $inputPort
	}
	set lastDevice $device
	$vpipe_ release_device
	$self configure_formats $device

	if [$videoAgent_ have_network] {
		$transmitButton_ configure -state normal
	}
	$self configure_sizes $device
	$self configure_port $device
	$self configure_norm $device

#	puts "select_device calling insert_grabber_panel"
	$self insert_grabber_panel [$device nickname]
	set videoFormat $defaultFormat($device)
#	puts "select_device calling select_format"
	$self select_format $videoFormat

#	puts "select_device setting vpipe_ device"
        $vpipe_ set device_ $device
#	puts "select_device setting device_"
        $self set device_ $device

#	puts "select_device calling insert_grabber_panel"
#	$self insert_grabber_panel [$device nickname]

	if $wasTransmitting {
		$vpipe_ start
	}
}

#
ControlMenu instproc configure_port { device } {
	$self instvar portButton_

	if [$device supports port *] {
		$portButton_ configure -state normal
		$self attach_ports $device
	} else {
		$portButton_ configure -state disabled
	}
}

#
ControlMenu instproc configure_norm { device } {
	$self instvar normButton_

	if [$device supports norm *] {
		$normButton_ configure -state normal
		$self attach_norms $device
	} else {
		$normButton_ configure -state disabled
	}
}

#
ControlMenu instproc configure_sizes { device } {
	$self instvar sizeButtons_

	if [$device supports size small] {
		$sizeButtons_.b0 configure -state normal
	} else {
		$sizeButtons_.b0 configure -state disabled
	}
	if [$device supports size large] {
		$sizeButtons_.b2 configure -state normal
	} else {
		$sizeButtons_.b2 configure -state disabled
	}
}

#
ControlMenu instproc configure_formats { device } {
	$self instvar vpipe_
	global formatButtons
	set fmtList [$vpipe_ available_formats $device]
	foreach b $formatButtons {
		set fmt [lindex [$b configure -value] 4]
		if { [inList $fmt $fmtList] } {
			$b configure -state normal
		} else {
			$b configure -state disabled
		}
	}
}

#
ControlMenu instproc build.device w {
	set f [$self get_option smallfont]

	set m $w.menu
	menubutton $w -menu $m -text Device... \
		-relief raised -width 10 -font $f
	#
	# Disabled the device button if we have no devices or
	# if we don't have transmit persmission.
	#
	if ![$self have_transmit_permission] {
		$w configure -state disabled
		return
	}
	menu $m
	$self build.device_menu $m
}

#
ControlMenu instproc build.device_menu m {
	set f [$self get_option smallfont]

	global defaultFormat videoFormat
	set videoFormat [$self get_option videoFormat]
	if { $videoFormat == "h.261" } {
		set videoFormat h261
	}
	if { $videoFormat == "h.263+" } {
		set videoFormat h263+
	}
	if { $videoFormat == "h.263" } {
		set videoFormat h263
	}
	$self instvar vpipe_
	foreach d [$vpipe_ input_devices] {
		if { [$d nickname] == "still" && ![$self yesno stillGrabber] } {
			set defaultFormat($d) $videoFormat
			continue
		}
		# this is fragile
		$m add radiobutton -label [$d nickname] \
			-command "$self select_device $d" \
			-value $d -variable device_ -font $f
		if { "[$d attributes]" == "disabled" } {
			$m entryconfigure [$d nickname] -state disabled
		}
		set fmtList [$vpipe_ available_formats $d]
		if [inList $videoFormat $fmtList] {
			set defaultFormat($d) $videoFormat
		} else {
			set defaultFormat($d) [lindex $fmtList 0]
		}
	}
}

#
ControlMenu instproc format_col { w n0 { n1 {} } {n2 {} }} {
	set f [$self get_option smallfont]
	frame $w
	global formatButtons
	radiobutton $w.b0 -text $n0 -relief flat -font $f -anchor w \
		-variable videoFormat -value $n0 -padx 0 -pady 0 \
		-command "$self select_format $n0"
	pack $w.b0 -fill x
	lappend formatButtons $w.b0
	if { $n1 != "" } {
		radiobutton $w.b1 -text $n1 -relief flat -font $f -anchor w \
			-variable videoFormat -value $n1 -padx 0 -pady 0 \
			-command "$self select_format $n1"
		pack $w.b1 -fill x
		lappend formatButtons $w.b1
	} else {
		label $w.b1 -text "" -padx 0 -pady 0
		pack $w.b1 -fill x
	}
	if { $n2 != "" } {
		radiobutton $w.b2 -text $n2 -relief flat -font $f -anchor w \
			-variable videoFormat -value $n2 -padx 0 -pady 0 \
			-command "$self select_format $n2"
		pack $w.b2 -fill x -anchor n
		lappend formatButtons $w.b2
	} else {
		label $w.b2 -text "" -padx 0 -pady 0
		pack $w.b2 -fill x
	}
}

#
ControlMenu instproc build.format w {
	$self format_col $w.p0 nv nvdct cellb
	$self format_col $w.p1 jpeg pvh
	$self format_col $w.p2 h261
	#$self format_col $w.p2 h261 h263 h263+

	frame $w.glue0
	frame $w.glue1

	pack $w.glue0 -side left -fill x -expand 1
	pack $w.p0 $w.p1 $w.p2 -side left
	pack $w.glue1 -side left -fill x -expand 1
}

#
ControlMenu instproc set-port p {
	$self instvar vpipe_
	$vpipe_ set_port $p
}

#
ControlMenu instproc set-norm n {
	$self instvar vpipe_
	$vpipe_ set_norm $n
}

#
ControlMenu instproc set-decimate p {
	$self instvar vpipe_
	$vpipe_ set_decimate $p
}

#
ControlMenu instproc build.size w {
	set f [$self get_option smallfont]

	set b $w.b
	frame $b
	radiobutton $b.b0 -text "small" -command "$self set-decimate 4" \
		-padx 0 -pady 0 \
		-anchor w -variable inputSize -font $f -relief flat -value 4
	radiobutton $b.b1 -text "normal" -command "$self set-decimate 2" \
		-padx 0 -pady 0 \
		-anchor w -variable inputSize -font $f -relief flat -value 2
	radiobutton $b.b2 -text "large" -command "$self set-decimate 1" \
		-padx 0 -pady 0 \
		-anchor w -variable inputSize -font $f -relief flat -value 1
	pack $b.b0 $b.b1 $b.b2 -fill x
	pack $b -anchor c -side left
	global inputSize
	set inputSize 2
	$self instvar sizeButtons_
	set sizeButtons_ $b
}

#
ControlMenu instproc build.port w {
	set f [$self get_option smallfont]
	# create the menubutton but don't defer the menu creation until later
	menubutton $w -menu $w.menu -text Port... \
		-relief raised -width 10 -font $f -state disabled
	global inputPort
	$self instvar portButton_
	set portButton_ $w
	set inputPort undefined
}

#
ControlMenu instproc attach_ports device {
	$self instvar portButton_
	catch "destroy $portButton_.menu"
	set m $portButton_.menu
	$self build.port_menu $device $m
}

#
ControlMenu instproc build.port_menu { device m } {
	global inputPort defaultPort

	set portnames [$device get_attribute port]
	set f [$self get_option smallfont]
	menu $m
	foreach port $portnames {
		$m add radiobutton -label $port \
			-command "$self set-port $port" \
			-value $port -variable inputPort -font $f
	}

	if ![info exists defaultPort($device)] {
		set nn [$device nickname]
		if [info exists defaultPort($nn)] {
			set defaultPort($device) $defaultPort($nn)
		} else {
			set s [$self get_option defaultPort($nn)]
			if { $s != "" } {
				set defaultPort($device) $s
			} else {
				set defaultPort($device) [lindex $portnames 0]
			}
		}
	}
	set inputPort $defaultPort($device)
}

#
ControlMenu instproc build.norm w {
	set f [$self get_option smallfont]

	# create the menubutton but defer the menu creation until later
	menubutton $w -menu $w.menu -text Signal... \
		-relief raised -width 10 -font $f -state disabled
	global inputNorm
	$self instvar normButton_
	set normButton_ $w
	set inputNorm undefined
}

#
ControlMenu instproc attach_norms device {
	$self instvar normButton_
	catch "destroy $normButton_.menu"
	set m $normButton_.menu
	$self build.norm_menu $device $m
}

#
ControlMenu instproc build.norm_menu { device m } {
	global inputNorm defaultNorm

	set normnames [$device get_attribute norm]

	set f [$self get_option smallfont]
	menu $m

	foreach norm $normnames {
		$m add radiobutton -label $norm -command "$self set-norm $norm" \
			-value $norm -variable inputNorm -font $f
	}

	if ![info exists defaultNorm($device)] {
		set nn [$device nickname]
		if [info exists defaultNorm($nn)] {
			set defaultNorm($device) $defaultNorm($nn)
		} else {
			set s [$self get_option defaultNorm($nn)]
			if { $s != "" } {
				set defaultNorm($device) $s
			} else {
				set defaultNorm($device) [lindex $normnames 0]
			}
		}
	}
	set inputNorm $defaultNorm($device)
}

#
ControlMenu instproc build.encoder_buttons w {
	set f [$self get_option smallfont]
	$self build.encoder_options $w.options
	$self build.device $w.device
	$self build.port $w.port
	$self build.norm $w.norm
	pack $w.device $w.port $w.norm $w.options -fill x
}

#
ControlMenu instproc build.encoder_options w {
	set f [$self get_option smallfont]
	set m $w.menu
	menubutton $w -text Options... -menu $m -relief raised -width 10 \
		-font $f

	$self build.encoder_options_menu $m
}

#
ControlMenu instproc build.encoder_options_menu m {
	set f [$self get_option smallfont]
	$self tkvar useJPEGforH261_
	set useJPEGforH261_ [$self yesno [$self get_option useJPEGforH261]]
	menu $m
	$m add checkbutton -label "Sending Slides" \
		-variable sendingSlides -font $f -command "$self setFillRate"
	$m add checkbutton -label "Use JPEG for H261" \
		-variable useJPEGforH261_ \
		-font $f -command "$self restart"
}

#
ControlMenu instproc build.tile w {
	$self instvar asm_ ui_ layout_menu_
	set f [$self get_option smallfont]
	set layout_menu_ $w.menu
	menubutton $w -text Layout... -menu $layout_menu_ -relief raised \
		-width 10 -font $f
	menu $layout_menu_
	set v [$self tkvarname ncol]

	$layout_menu_ add radiobutton -label Single \
		-command "$asm_ redecorate 1" \
		-value 1 -variable $v -font $f
	$layout_menu_ add radiobutton -label Double \
		-command "$asm_ redecorate 2" \
		-value 2 -variable $v -font $f
	$layout_menu_ add radiobutton -label Triple \
		-command "$asm_ redecorate 3" \
		-value 3 -variable $v -font $f
	$layout_menu_ add radiobutton -label Quadruple \
		-command "$asm_ redecorate 4" \
		-value 4 -variable $v -font $f
	$layout_menu_ add radiobutton -label Quintuple \
		-command "$asm_ redecorate 5" \
		-value 5 -variable $v -font $f
	$layout_menu_ add radiobutton -label Sextuple \
		-command "$asm_ redecorate 6" \
		-value 6 -variable $v -font $f
	$layout_menu_ add radiobutton -label Septuple \
		-command "$asm_ redecorate 7" \
		-value 7 -variable $v -font $f
	$layout_menu_ add radiobutton -label Octuple \
		-command "$asm_ redecorate 8" \
		-value 8 -variable $v -font $f
	$layout_menu_ add separator
	$layout_menu_ add checkbutton -label Scrollbars \
		-command "$ui_ set_scrollbars" \
		-font $f -variable [$ui_ tkvarname useScrollbars_]

	# Display the current number of columns/rows.
	$asm_ instvar ncol_ nrow_ list_direction_
	if {$list_direction_ == "vertical"} {
		$layout_menu_ invoke [expr $ncol_ - 1]
	} else {
		$layout_menu_ invoke [expr $nrow_ - 1]
	}

	# If the scrollbars are on, check the Scrollbars checkbox.
	$ui_ instvar scrollbars_on_
    	if {$scrollbars_on_} {
	    	$layout_menu_ invoke [$layout_menu_ index Scrollbars]
    	}
}

#
# Toggle the Scrollbars checkbutton.
#
ControlMenu instproc toggle_scrollcheck {} {
    $self instvar layout_menu_
    if [info exists layout_menu_] {
	$layout_menu_ invoke [$layout_menu_ index Scrollbars]
    }
}

#
# Update the number of columns/rows value.
#
ControlMenu instproc update_layout {n} {
    $self instvar layout_menu_
    if [info exists layout_menu_] {
	$layout_menu_ invoke [expr $n - 1]
    }
}

#
ControlMenu instproc use-hw {} {
	$self tkvar useHardwareDecode_
	return $useHardwareDecode_
}

#
ControlMenu instproc mute-new-sources {} {
	$self tkvar muteNewSources
	return $muteNewSources
}

#
ControlMenu instproc build.decoder_options w {
	$self instvar asm_
	$asm_ instvar autoplace_

	set f [$self get_option smallfont]
	set m $w.menu
	menubutton $w -text Options... -menu $m -relief raised -width 10 \
		-font $f
	menu $m
	if {[info exists autoplace_]} {
    		$m add checkbutton -label "Auto-Place New Sources" \
			-command "$asm_ set_autoplace" \
    	    		-variable [$asm_ tkvarname autoplaceNewSources] -font $f
	}
    	$m add checkbutton -label "Mute New Sources" \
		-variable [$self tkvarname muteNewSources] -font $f
    	$m add checkbutton -label "Use Hardware Decode" \
		-variable [$self tkvarname useHardwareDecode_] -font $f
	$m add separator
    	$m add command -label "Optimize Colormap" \
		-command "$self fork_histtolut" -font $f

	$self instvar optionsMenu_
	set optionsMenu_ $m

	$self tkvar ditherStyle_
	if { $ditherStyle_ == "" } {
		$m entryconfigure "Optimize Colormap" -state disabled
	}

	# If autoplace is on, check the Auto-Place checkbox.
	if {[info exists autoplace_] && $autoplace_} {
	    $m invoke [$m index {Auto-Place New Sources}]
	}
}

#
# Toggle the Auto-Place checkbox.
#
ControlMenu instproc toggle_autocheck {} {
    $self instvar optionsMenu_
    if [info exists optionsMenu_] {
	$optionsMenu_ invoke [$optionsMenu_ index {Auto-Place New Sources}]
    }
}

#
ControlMenu instproc build.external w {
	set f [$self get_option smallfont]
	set m $w.menu
	global outputDeviceList
	if ![info exists outputDeviceList] {
		set outputDeviceList ""
	}
	if { [llength $outputDeviceList] <= 1 } {
		button $w -text External -relief raised \
			-width 10 -font $f -highlightthickness 0 \
			-command "extout_select $outputDeviceList"
	} else {
		menubutton $w -text External... -menu $m -relief raised \
			-width 10 -font $f
		menu $m
		foreach d $outputDeviceList {
			$m add command -font $f -label [$d nickname] \
				-command "extout_select $d"
		}
	}
	if { $outputDeviceList == "" } {
		$w configure -state disabled
	}
}

#
ControlMenu instproc set-dither d {
	$self instvar vframe_
	$vframe_ set-dither $d
}

#
ControlMenu instproc build.dither w {
	set f [$self get_option smallfont]
	$self tkvar ditherStyle_
	if { $ditherStyle_ != "" } {
		set state normal
	} else {
		set state disabled
	}

	set v $w.h0
	frame $v
	set dvar [$self tkvarname ditherStyle_]
	radiobutton $v.b0 -text "Ordered" -command "$self set-dither Dither" \
		-padx 0 -pady 0 \
		-anchor w -variable $dvar -state $state \
		-font $f -relief flat -value Dither
	radiobutton $v.b1 -text "Error Diff" -command "$self set-dither ED" \
		-padx 0 -pady 0 \
		-anchor w -variable $dvar -state $state \
		-font $f -relief flat -value ED
	set v $w.h1
	frame $v
	radiobutton $v.b2 -text Quantize -command "$self set-dither Quant" \
		-padx 0 -pady 0 \
		-anchor w -variable $dvar -state $state \
		-font $f -relief flat \
		-value Quant
	radiobutton $v.b3 -text Gray -command "$self set-dither Gray" \
		-padx 0 -pady 0 \
		-anchor w -variable $dvar -state $state \
		-font $f -relief flat -value Gray

	pack $w.h0.b0 $w.h0.b1 -anchor w -fill x
	pack $w.h1.b2 $w.h1.b3 -anchor w -fill x
	pack $w.h0 $w.h1 -side left
}

#
Class GammaEntry -superclass Entry

#
# Pass a VisualFrame, <i>vframe</i>.
#
GammaEntry instproc init { w value vframe } {
	$self next $w $value
	$self instvar vframe_
	set vframe_ $vframe
}

#
GammaEntry instproc update { w s } {
	$self instvar vframe_
	return [$vframe_ set-gamma $s]
}

#
ControlMenu instproc build.gamma w {
	$self instvar vframe_
	frame $w
	label $w.label -text "Gamma: " -font [$self get_option smallfont] -anchor e
	new GammaEntry $w.entry [$vframe_ set gamma_] $vframe_
	$w.entry configure -width 6
	$self tkvar ditherStyle_
	if { $ditherStyle_ == "" } {
		$w.entry configure -state disabled -foreground gray60
		$w.label configure -foreground gray60
	}
	pack $w.label -side left
	pack $w.entry -side left -expand 1 -fill x -pady 2
}

#
ControlMenu instproc build.decoder w {
	set f [$self get_option smallfont]

	label $w.title -text Display
	frame $w.f -relief sunken -borderwidth 2

	set v $w.f.h0
	frame $v

	$self build.external $v.ext
	$self build.tile $v.tile
	$self build.decoder_options $v.options

	if [winfo exists $v.options] { pack $v.options -fill x -expand 1 }
	if [winfo exists $v.tile   ] { pack $v.tile    -fill x -expand 1 }
	if [winfo exists $v.ext    ] { pack $v.ext     -fill x -expand 1 }

	set v $w.f.h2
	frame $v
	frame $v.dither -relief groove -borderwidth 2
	$self build.dither $v.dither
	frame $v.bot
	$self build.gamma $v.bot.gamma

	$self instvar ui_srclist_
	if { $ui_srclist_ != {} } {
		set top [$ui_srclist_ widget_path]
	} else { set top $w }
	label $v.bot.mode -text "\[[winfo depth $top]-bit\]" -font $f
	pack $v.bot.gamma $v.bot.mode -side left -padx 4
	pack $v.dither $v.bot -anchor c -pady 2

	pack $w.f.h0 -side left -padx 6 -pady 6
	pack $w.f.h2 -side left -padx 6 -pady 6 -fill x -expand 1

	pack $w.title $w.f -fill x
}

#
ControlMenu instproc build.encoder w {
	label $w.title -text Encoder
	frame $w.f -relief sunken -borderwidth 2

	frame $w.f.h0 -relief flat
	frame $w.f.h1 -relief flat
	frame $w.f.h0.eb -relief flat
	frame $w.f.h0.format -relief groove -borderwidth 2
	frame $w.f.h0.size -relief groove -borderwidth 2
	frame $w.f.h0.gap -relief flat -width 4

	$self build.encoder_buttons $w.f.h0.eb
	$self build.format $w.f.h0.format
	$self build.size $w.f.h0.size

	$self build.q $w.f.h1

	pack $w.f.h0.eb -side left -anchor n -fill y -padx 6 -pady 4
	pack $w.f.h0.format -side left -anchor n -fill both -expand 1
	pack $w.f.h0.size -side left -anchor c -fill both
	pack $w.f.h0.gap -side left -anchor c

	pack $w.f.h0 -fill x -pady 4
	pack $w.f.h1 -fill x -pady 6
	pack $w.title $w.f -fill x
}


#
# If the capture device is open, close it.  If transmission
# was active fire it up again.  Some state can only be set at
# device open time, so some controls must resort to this proc.
#
ControlMenu instproc restart { } {
	$self tkvar transmitButtonState_
	$self tkvar useJPEGforH261_
	$self instvar vpipe_
	if $useJPEGforH261_ {
		$self add_option useJPEGforH261 true
	} else {
		$self add_option useJPEGforH261 false
	}
	if $transmitButtonState_ {
		$vpipe_ stop
		$vpipe_ release_device
		$self tx-init
		$vpipe_ start
	} else {
		$vpipe_ release_device
	}
}

#
ControlMenu instproc disable_large_button { } {
	$self instvar sizeButtons_
	global inputSize
	if { $inputSize == 1 } {
		set inputSize 2
		$self set-decimate 2
	}
	$sizeButtons_.b2 configure -state disabled
}

#
ControlMenu instproc enable_large_button { } {
	$self instvar device_ sizeButtons_
	if { [info exists device_] && \
		[$device_ supports size large] } {
		$sizeButtons_.b2 configure -state normal
	}
}

#
ControlMenu instproc setq value {
	$self instvar vpipe_ qvalue_
	set v [$vpipe_ set_quality $value]
	$qvalue_ configure -text $v
}

#
ControlMenu instproc select_format fmt {
	if { $fmt == "h261" } {
		# H.261 supports only QCIF/CIF
		$self disable_large_button
	} else {
		$self enable_large_button
	}

	$self configure_quality $fmt
}

#
ControlMenu instproc configure_quality fmt {
	global videoFormat
	$self instvar qval_ qscale_ qlabel_ qvalue_ lastFmt_
	$self instvar vpipe_ device_

	# get the quality value for the last codec used
	set qval_($lastFmt_) [$qscale_ get]
	set lastFmt_ $videoFormat

	# set the scale value
	if [info exists qval_($fmt)] {
		$qscale_ set $qval_($fmt)
	}

	# set the VideoPipeline format
	$vpipe_ select $device_ $fmt

	# configure the three UI widgets related to quality setting: label, scale, 
	# and value
	if { ([info exists qval_($fmt)]) && \
			([$vpipe_ set_quality $qval_($fmt)] >= 0) } {
		$qscale_ configure -state normal -command "$self setq"
	}
	$qlabel_ configure -foreground [$self get_option foreground]
}

#
ControlMenu instproc tx-init {} {
	$self instvar qscale_
	if { [lindex [$qscale_ configure -state] 4] == "normal" } {
		set cmd [lindex [$qscale_ configure -command] 4]
		eval $cmd [$qscale_ get]
	}

	$self instvar portButton_
	$self instvar normButton_
	global inputPort inputNorm
	if { [$portButton_ cget -state] == "normal" } {
		$self set-port $inputPort
	}
	if { [$normButton_ cget -state] == "normal" } {
		$self set-norm $inputNorm
	}

	$self setFillRate
	update
}

#
ControlMenu instproc build.q w {
	set f [$self get_option smallfont]
	frame $w.tb
	label $w.title -text "Quality" -font $f -anchor w
	label $w.tb.value -text 0 -font $f -width 3
	scale $w.tb.scale -font $f -orient horizontal \
		-showvalue 0 -from 0 -to 99 \
		-width 12 -relief groove
	$self instvar qscale_ qvalue_ qlabel_
	set qscale_ $w.tb.scale
	set qvalue_ $w.tb.value
	set qlabel_ $w.title

	pack $w.tb.scale -side left -fill x -expand 1
	pack $w.tb.value -side left
	pack $w.title -padx 2 -side left
	pack $w.tb -fill x -padx 6 -side left -expand 1
}

#
ControlMenu instproc build.scuba w {
	set f [$self get_option smallfont]
	label $w.label -text SCUBA
	frame $w.frame -relief sunken -borderwidth 2
	pack $w.label -fill x
	pack $w.frame -fill both -expand 1

	set wf $w.frame

	frame $wf.title
	frame $wf.title.lglue
	frame $wf.title.rglue
	label $wf.title.l -text "Local Bandwidth: " -font $f
	label $wf.title.value -font $f -width 8 -anchor w
	pack $wf.title.lglue -expand 1 -fill x -side left
	pack $wf.title.l $wf.title.value -side left
	pack $wf.title.rglue -expand 1 -fill x -side right
	pack $wf.title -fill x -expand 1

	frame $wf.sessbw
	scale $wf.sessbw.scale -orient horizontal -font $f \
		-showvalue 0 -from 1000 -to \
		[$self get_option maxVideoSessionBW] \
		-command "$self set_sessionbw $wf.title.value" -width 12 \
		-relief groove -resolution 1000

	pack $wf.sessbw -fill x -expand 1
	pack $wf.sessbw.scale -fill x -side left -expand 1

	$self instvar ui_
	set s [$ui_ scuba_session]
	$wf.sessbw.scale set [$s set sessionbw_]
}

#
ControlMenu instproc build.xmit w {
	set f [$self get_option smallfont]
	label $w.label -text Transmission
	frame $w.frame -relief sunken -borderwidth 2
	pack $w.label -fill x
	pack $w.frame -fill both -expand 1

	frame $w.frame.buttons
	$self build.buttons $w.frame.buttons
	frame $w.frame.right
	$self build.sliders $w.frame.right

	pack $w.frame.buttons -side left -padx 6
	pack $w.frame.right -side right -expand 1 -fill x -padx 10 -anchor c
}

