/*
 * pfmon_symbols.c  - management of symbol tables
 *
 * Copyright (C) 2002-2003 Hewlett-Packard Co
 * Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of pfmon, a sample tool to measure performance 
 * of applications on Linux/ia64.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307 USA
 */
#include "pfmon.h"

#include <ctype.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <libelf.h>


#ifndef ELF_ST_TYPE
#define ELF_ST_TYPE(val)         ((val) & 0xF)
#endif

struct sym_hash_table_t;

typedef struct _sym_hash_entry {
	struct _sym_hash_entry 	*next, *prev;
	struct _sym_hash_entry 	*hash_next, *hash_prev;
	uintptr_t		addr;	/* start address */
	uint64_t		access_count;
	unsigned long		bucket;

	uintptr_t		eaddr;
	char			*name;
	char			*module;
} sym_hash_entry_t;

typedef struct {
	sym_hash_entry_t	*entries; 	/* entry storage area */
	sym_hash_entry_t	**table;	/* hash table */
	sym_hash_entry_t	*free_list;
	sym_hash_entry_t	*active_list;

	unsigned long		mask;		/* mask used for hashing (2^hash_size-1) */

	uint64_t		accesses;
	uint64_t		misses;
	uint64_t		collisions;

	unsigned long 		max_entries;
	unsigned long 		used_entries;
} sym_hash_table_t;

typedef struct {
	char		*name;			/* symbol name */
	uintptr_t	value;			/* symbol address */
	unsigned long	size;			/* optional symbol size */
	int		type;			/* type : CODE or DATA */
} symbol_t;

typedef struct _module_symbol_t {
	struct _module_symbol_t *next, *prev;	/* linked list */
	symbol_t		*symbol_tab;	/* the table itself */
	char 			*name_space;	/* where strings are stored (ELF) */
	unsigned long		nsyms;		/* number of symbols */
	char			*name;		/* module name */
} module_symbols_t;

static module_symbols_t	*modules_list;

static int
symcmp(const void *a, const void *b)
{
	symbol_t *ap = (symbol_t *)a;
	symbol_t *bp = (symbol_t *)b;

	return ap->value > bp->value;
}

static int
read_sym64(Elf *elf, const char *filename, module_symbols_t *mod_sym)
{
	Elf_Data *data;
	Elf64_Shdr *symtab_hdr = NULL;
	Elf_Scn *symtab_section;
	Elf_Scn *strsym_section;
	Elf64_Sym *symtab_data;
	symbol_t *symbol_tab;
	char *name_space;
	unsigned long table_size;
	size_t strsym_index;
	int i, j;
	int type;

    
	/* first find the symbol table table */
	symtab_section = NULL;
	while ((symtab_section = elf_nextscn(elf, symtab_section)) != 0) {

		symtab_hdr = elf64_getshdr(symtab_section);
		if (symtab_hdr == NULL) {
			vbprintf("cannot get section header\n");
			return -1;
		}
			
		/* is this the symbol table? no DYNSYMTAB? */
		if (symtab_hdr->sh_type == SHT_SYMTAB) goto found;
	}
	vbprintf("no symbol table found for %s\n", filename);
	return -1;
found:
	/* use elf_rawdata since there is no memory image of this data */
	data = elf_rawdata(symtab_section, NULL); 
	if (data == NULL) {
		vbprintf("can't extract raw elf data for symbol table\n");
		return -1;
	}
	symtab_data = (Elf64_Sym *)data->d_buf;
	table_size  = symtab_hdr->sh_size/symtab_hdr->sh_entsize;
  
	/* get the string table */
	strsym_index   = symtab_hdr->sh_link;
	strsym_section = elf_getscn(elf, strsym_index);
  
	/* use elf_rawdata since there is no memory image of this data */
	data = elf_rawdata(strsym_section, NULL); 
	if (data == NULL) {
		vbprintf("can't extract raw elf data for string section\n");
		return -1;
	}
  
	/* allocate space and copy content */
	mod_sym->name_space = name_space = malloc(data->d_size);
	if (name_space  == NULL) {
		vbprintf("can't allocate space for string table\n"); 
		return -1;
	}
  	memcpy(name_space, data->d_buf, data->d_size);
  
  	/* allocate space for the table and set it up */
	mod_sym->symbol_tab = symbol_tab = malloc(table_size * sizeof(symbol_t));
	if (symbol_tab == NULL) {
    		vbprintf("cannot allocate space for symbol table\n");
		return -1;
	}
  	for (i = 0, j= 0; i < table_size; i++) {
    		symbol_tab[j].name  = name_space + symtab_data[i].st_name;

		if (symbol_tab[j].name == NULL || symbol_tab[j].name[0] == '\0') continue;

		type = ELF_ST_TYPE(symtab_data[i].st_info);
		if (type != STT_FUNC &&  type != STT_OBJECT && type != STT_COMMON) continue;

    		symbol_tab[j].value = symtab_data[i].st_value;
		symbol_tab[j].size  = symtab_data[i].st_size;
		symbol_tab[j].type  = type == STT_FUNC ? PFMON_TEXT_SYMBOL : PFMON_DATA_SYMBOL; 
		j++;
	}
  	mod_sym->nsyms = j;
	qsort(symbol_tab, j, sizeof(symbol_t), symcmp);
	return 0;
}

static int
read_sym32(Elf *elf, const char *filename, module_symbols_t *mod_sym)
{
	Elf_Data *data;
	Elf32_Shdr *symtab_hdr = NULL;
	Elf_Scn *symtab_section;
	Elf_Scn *strsym_section;
	Elf32_Sym *symtab_data;
	symbol_t *symbol_tab;
	char *name_space;
	size_t strsym_index;
	int i, j, table_size;
	int type;

    
	/* first find the symbol table table */
	symtab_section = NULL;
	while ((symtab_section = elf_nextscn(elf, symtab_section)) != 0) {

		symtab_hdr = elf32_getshdr(symtab_section);
		if (symtab_hdr == NULL) {
			vbprintf("cannot get section header\n");
			return -1;
		}
			
		/* is this the symbol table? no DYNSYMTAB? */
		if (symtab_hdr->sh_type == SHT_SYMTAB) goto found;
	}
	vbprintf("no symbol table found for %s\n", filename);
	return -1;
found:
	/* use elf_rawdata since there is no memory image of this data */
	data = elf_rawdata(symtab_section, NULL); 
	if (data == NULL) {
		vbprintf("can't extract raw elf data for symbol table\n");
		return -1;
	}
	symtab_data = (Elf32_Sym *)data->d_buf;
	table_size  = symtab_hdr->sh_size/symtab_hdr->sh_entsize;
  
	/* get the string table */
	strsym_index   = symtab_hdr->sh_link;
	strsym_section = elf_getscn(elf, strsym_index);
  
	/* use elf_rawdata since there is no memory image of this data */
	data = elf_rawdata(strsym_section, NULL); 
	if (data == NULL) {
		vbprintf("can't extract raw elf data for string section\n");
		return -1;
	}
  
	/* allocate space and copy content */
	mod_sym->name_space = name_space = malloc(data->d_size);
	if (name_space  == NULL) {
		vbprintf("can't allocate space for string table\n"); 
		return -1;
	}
  	memcpy(name_space, data->d_buf, data->d_size);
  
  	/* allocate space for the table and set it up */
	mod_sym->symbol_tab = symbol_tab = malloc(table_size * sizeof(symbol_t));
	if (symbol_tab == NULL) {
    		vbprintf("cannot allocate space for symbol table\n");
		return -1;
	}
  	for (i = 0, j= 0; i < table_size; i++) {
    		symbol_tab[j].name  = name_space + symtab_data[i].st_name;

		if (symbol_tab[j].name == NULL || symbol_tab[j].name[0] == '\0') continue;

		type = ELF_ST_TYPE(symtab_data[i].st_info);
		if (type != STT_FUNC &&  type != STT_OBJECT && type != STT_COMMON) continue;

    		symbol_tab[j].value = (uintptr_t)symtab_data[i].st_value;
		symbol_tab[j].size  = symtab_data[i].st_size;
		symbol_tab[j].type  = type == STT_FUNC ? PFMON_TEXT_SYMBOL : PFMON_DATA_SYMBOL; 
		j++;
	}
  	mod_sym->nsyms = j;
	qsort(symbol_tab, j, sizeof(symbol_t), symcmp);
	return 0;
}

static int
load_elf_symbols(const char *filename, module_symbols_t *mod_sym)
{
	Elf *elf;
	char *eident;
	int fd;

	if (filename == NULL) return -1;

	fd = open(filename, O_RDONLY);
	if (fd == -1) {
		vbprintf("symbol file for %s not found\n", filename);
		return -1;
	}

  	/* initial call to set internal version value */
	if (elf_version(EV_CURRENT) == EV_NONE) {
		DPRINT(("ELF library out of date"));
		return -1;
	}

  	/* prepare to read the entire file */
	elf = elf_begin(fd, ELF_C_READ, NULL);
	if (elf == NULL) {
		DPRINT(("can't read %s\n", filename));
		return -1;
	}

	/* error checking */
	if (elf_kind(elf) != ELF_K_ELF) {
		DPRINT(("%s is not an ELF file\n", filename));
		return -1;
	}
  
	eident = elf_getident(elf, NULL);
	if (eident[EI_MAG0] != ELFMAG0
	    || eident[EI_MAG1] != ELFMAG1
	    || eident[EI_MAG2] != ELFMAG2
	    || eident[EI_MAG3] != ELFMAG3) {
		DPRINT(("invalid ELF magic in %s\n", filename));
	}

	switch (eident[EI_CLASS]) {
  		case ELFCLASS32:
			if (read_sym32(elf, filename, mod_sym)) {
				DPRINT(("cannot extract symbols from %s\n", filename));
				return -1;
			}
			break;
		case ELFCLASS64:
			if (read_sym64(elf, filename, mod_sym)) {
				DPRINT(("cannot extract symbols from %s\n", filename));
				return -1;
			}
    			break;
    		default:
    			DPRINT(("unsupported ELF class for %s\n", filename));
			return -1;
	}

	return 0;
}

static char *
place_str(unsigned long length)
{
	static char *current_free, *current_end;
	char *tmp;
#define STR_CHUNK_SIZE	options.page_size
	if (length >= STR_CHUNK_SIZE)
		fatal_error("sysmap load string is too long\n");

	/*
	 * XXX: that's bad, we do not keep track of previously allocated
	 * chunks, so we cannot free!
	 */
	if (current_free == NULL || (current_end-current_free) < length) {
		current_free = (char *)malloc(STR_CHUNK_SIZE);
		if (current_free == NULL) return NULL;
		current_end = current_free + STR_CHUNK_SIZE;
	}
	tmp = current_free;
	current_free += length;
	return tmp;
}

/*
 * load kernel symbols using /proc/kallsyms.
 * This file does not contains kernel data symbols but includes code/data
 * symbols from modules. Code symbol size is not provided.
 */
static int
load_kallsyms_symbols(module_symbols_t *mod_sym)
{
#define PFMON_KALLSYMS		"/proc/kallsyms"
#define PFMON_KALLSYMS_SYMBOLS	20000
#define PFMON_KALLSYMS_MAXLEN	256

	FILE *fp;
	uintptr_t min_addr = 0UL;
	symbol_t *symbol_tab;
	symbol_t *tmp;
	char *s, *str_addr, *sym_start, *mod_start, *endptr;
	unsigned long nsyms = 0, idx = 0;
	unsigned long line = 1UL, current_symbol_count;
	unsigned long symtab_old_size = 0, sym_len, mod_len;
	int need_sorting = 0;
	char line_str[PFMON_KALLSYMS_MAXLEN];
	char addr_str[24]; /* cannot be more than 16+2 (for 0x) */
	int type;


	fp = fopen(PFMON_KALLSYMS, "r");
	if (fp == NULL) {
		DPRINT(("file %s not found\n", PFMON_KALLSYMS));
		return -1;
	}

	/*
	 * allocate a default-sized symbol table 
	 */
	current_symbol_count = PFMON_KALLSYMS_SYMBOLS;
	symbol_tab = (symbol_t *)malloc(PFMON_KALLSYMS_SYMBOLS*sizeof(symbol_t));
	if (symbol_tab == NULL) {
		DPRINT(("cannot allocate sysmap table for %lu symbols\n", nsyms));
		goto load_abort;
	}

	idx = 0;

	while(fgets(line_str, sizeof(line_str), fp)) {

		s = line_str;

		while(*s != ' ' && *s !='\0') s++;

		if (*s == '\0') break;

		if (s-line_str > 16+2) goto invalid_address;

		strncpy(addr_str, line_str, s-line_str);
		addr_str[s-line_str] = '\0';

		/* point to object type */
		s++;
		type = tolower(*s);

		/*
		 * convert static data symbols to data
		 */
		if (type == 's') type = 'd';

		/* 
		 * keep only text and data symbols
		 *
		 * skip uninteresting symbols
		 */
		if (type != 't' && type != 'd') {
			while(*s != '\n' && *s != '\0') s++;

			if (*s == '\0') goto invalid_line;

			line++;

			continue;
		}

		/* look for space separator */
		s++;
		if (*s != ' ') goto invalid_line;

		if (idx == current_symbol_count) {
			symtab_old_size = sizeof(symbol_t)*current_symbol_count;
			current_symbol_count <<=1;
			DPRINT(("extending kallsyms symbol table to %lu entries old_size=%lu\n", current_symbol_count, symtab_old_size));
			tmp = (symbol_t *)realloc(symbol_tab, current_symbol_count*sizeof(symbol_t));
			if (tmp == NULL) {
				DPRINT(("cannot extend kallsyms symbol table to %lu entries\n", current_symbol_count));
				goto load_abort;
			}
			symbol_tab = tmp;
		}

    		symbol_tab[idx].type  = type == 't' ? PFMON_TEXT_SYMBOL : PFMON_DATA_SYMBOL;

		/* compute address */
		endptr = NULL;
    		symbol_tab[idx].value  = (uintptr_t )strtoul(addr_str, &endptr, 16);
		if (*endptr != '\0') goto invalid_address;

		/*
		 * check that file is sorted correctly
		 */
		if (idx == 0) 
			min_addr = symbol_tab[idx].value;
		else if (symbol_tab[idx].value < min_addr) 
			need_sorting = 1;


		/* advance to symbol name */
		sym_start = ++s;

		/* look for end-of-string */
		while(*s != '\n' && *s != '\0' && *s != ' ' && *s != '\t') s++;

		if (*s == '\0') goto invalid_line;

		sym_len = s - sym_start;


		/* check for module */
		while(*s != '\n' && *s != '\0' && *s != '[') s++;

		/* symbol belongs to a kernel module */
		if (*s == '[') {
			mod_start = s++;
			while(*s != '\n' && *s != '\0' && *s != ']') s++;
			if (*s != ']') goto invalid_line;
			mod_len = s - mod_start + 1;
		} else {
			mod_len   = 0;
			mod_start = NULL;
		}

		line++;

		/*
		 * place string in our memory pool
		 * +1 for '\0'
		 */
		str_addr = place_str(mod_len + sym_len + 1);
		if (str_addr == NULL) goto error2;


		strncpy(str_addr, sym_start, sym_len);
		if (mod_len) strncpy(str_addr+sym_len, mod_start, mod_len);
		str_addr[sym_len+mod_len] = '\0';

    		symbol_tab[idx].name  = str_addr;
    		symbol_tab[idx].size  = 0; /* use approximation */

		idx++;
	}
	/* record final number of symbols */
	mod_sym->symbol_tab = symbol_tab;
	mod_sym->nsyms = idx;

	/*
	 * normally a kallsyms is already sorted
	 * so we should not have to do this
	 */
	if (need_sorting) qsort(symbol_tab, idx, sizeof(symbol_t), symcmp);

	fclose(fp);

	return 0;
invalid_line:
	warning("sysmap file %s has invalid format, line %lu\n", PFMON_KALLSYMS, line);
	return -1;
error2:
	DPRINT(("sysmap load file cannot place new string\n"));
load_abort:
	fclose(fp);
	return -1;
invalid_address:
	warning("file %s has an invalid address, line %lu\n", PFMON_KALLSYMS, line);
	return -1;

}

static int
load_sysmap_symbols(module_symbols_t *mod_sym)
{
	int fd;
	struct stat st;
	unsigned long nsyms = 0, idx = 0;
	uintptr_t min_addr = 0UL;
	unsigned long line = 1UL;
	char *filename = options.symbol_file;
	char *p, *s, *end, *str_addr, *base;
	char *endptr;
	int type;
	symbol_t *symbol_tab;
	char b[24]; /* cannot be more than 16+2 (for 0x) */
	int need_sorting = 0;


	fd = open(filename, O_RDONLY);
	if (fd == -1) {
		DPRINT(("sysmap  file %s not found\n", filename));
		return -1;
	}

	if (fstat(fd, &st) == -1) {
		DPRINT(("cannot access sysmap file %s\n", filename));
		return -1;
	}

	p = base = mmap(0, st.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
	if (p == (char *)-1) {
		DPRINT(("cannot map sysmap file %s\n", filename));
		goto load_abort;
	}

	end = base + st.st_size;


	/* find number of symbols */
	while (p < end) {
		if (*p == '\n') nsyms++;
		p++;
	}
	mod_sym->symbol_tab = symbol_tab = (symbol_t *)malloc(nsyms * sizeof(symbol_t));
	if (symbol_tab == NULL) {
		DPRINT(("cannot allocate sysmap table for %lu symbols\n", nsyms));
		goto load_abort;
	}

	idx = 0;

	/* now parse symbols */
	p = base;

	while (p < end) {

		/* find end */
		s = p;
		while(s < end && *s != ' ') s++;
		if (s == end) break;
		if (s-p > 16+2) {
			DPRINT(("invalid address at line %lu in %s\n", line, filename));
			goto load_abort;
		}

		strncpy(b, p, s-p);
		b[s-p] = '\0';

		/* point to object type */
		s++;
		type = tolower(*s);

		/* 
		 * keep only text and data symbols
		 * XXX: oversimplification here!
		 */
		if (type != 't' && type != 'd') {
			while(s < end && *s != '\n') s++;
			if (s == end) goto error;
			line++;
			p = s + 1;
			continue;
		}

		/* look for space separator */
		s++;
		if (*s != ' ') goto error;

    		symbol_tab[idx].type  = type == 't' ? PFMON_TEXT_SYMBOL : PFMON_DATA_SYMBOL;

		/* compute address */
    		symbol_tab[idx].value  = (uintptr_t )strtoul(b, &endptr, 16);
		if (*endptr != '\0') {
			DPRINT(("invalid address at line %lu in %s\n", line, filename));
			goto load_abort;
		}

		/*
		 * check that file is sorted correctly
		 */
		if (idx == 0) 
			min_addr = symbol_tab[idx].value;
		else if (symbol_tab[idx].value < min_addr) 
			need_sorting = 1;


		/* advance to symbol name */
		s++;
		p = s;	

		/* look for end-of-line */
		while(s < end && *s != '\n') s++;
		if (s == end) goto error;
		if (s == p) goto error;
		line++;

		/*
		 * place string in our memory pool
		 */
		str_addr = place_str(s-p+1);
		if (str_addr == NULL) goto error2;

		strncpy(str_addr, p, s-p);
		str_addr[s-p] = '\0';
		p = s +1;	

		/* sanity */
		if (idx == nsyms) fatal_error("too many symbol for sysmap files\n");

    		symbol_tab[idx].name  = str_addr;
    		symbol_tab[idx].size  = 0; /* use approximation */

		idx++;
	}
	/* record final number of symbols */
	mod_sym->nsyms = idx;

	/*
	 * cleanup mappings
	 */
	munmap(base, st.st_size);
	close(fd);

	/*
	 * normally a System.map file is already sort
	 * so we should not have to do this
	 */
	if (need_sorting) qsort(symbol_tab, idx, sizeof(symbol_t), symcmp);

	return 0;
error:
	warning("sysmap file %s has invalid format, line %lu\n", filename, line);
	return -1;
error2:
	DPRINT(("sysmap load file cannot place new string\n"));
load_abort:
	close(fd);
	return -1;

}

int
load_kernel_syms(void)
{
	static int kernel_syms_loaded;
	module_symbols_t *p;
	int ret = -1;

	if (kernel_syms_loaded) return 0;

	p = malloc(sizeof(module_symbols_t));
	if (p == NULL) 
		fatal_error("cannot allocate kernel symbol table module\n");

	p->next       = p->prev = NULL;
	p->symbol_tab = NULL;
	p->name_space = NULL;
	p->nsyms      = 0UL;
	p->name       = "kernel";

	/* 
	 * Despite /proc/kallsyms, System.map is still useful because it includes data symbols
	 * We use System.map if specified, otherwise we default to /proc/kallsyms
	 */
	if (options.opt_sysmap_syms) {
		ret = load_sysmap_symbols(p);
		vbprintf("loaded %lu symbols from %s\n", p->nsyms, options.opt_sysmap_syms);
	} else {
		ret = load_kallsyms_symbols(p);
		vbprintf("loaded %lu symbols from /proc/kallsyms\n", p->nsyms);
	}

	if (ret) return ret;
		
	modules_list = p;

	return 0;
}

int
load_elf_syms(const char *filename)
{
	module_symbols_t *p;

	p = malloc(sizeof(module_symbols_t));
	if (p == NULL) 
		fatal_error("cannot allocate symbol table module for %s\n", filename);

	p->prev = NULL;
	p->symbol_tab = NULL;
	p->name_space = NULL;
	p->nsyms      = 0UL;
	p->name       = "app";

	if (load_elf_symbols(filename, p)) {
		free(p);
		return -1;
	}

	/*
	 * XXX: not thread-safe
	 */
	p->next      = modules_list;
	modules_list = p;

	vbprintf("loaded %lu symbols from ELF file %s\n", p->nsyms, filename);

	return 0;
}

/*
 * mostly for debug
 */
void
print_syms(void)
{
	module_symbols_t *mod;
	symbol_t *symbol_tab;
	char *mod_name;
	unsigned long i, nsyms;

	for (mod = modules_list ; mod ; mod = mod->next) {

		nsyms      = mod->nsyms;
		symbol_tab = mod->symbol_tab;
		mod_name   = mod->name;

		for (i = 0; i < nsyms; i++) {
			printf("%p %c %8lu %s:%s:\n", 
				(void *)symbol_tab[i].value, 
				symbol_tab[i].type == PFMON_TEXT_SYMBOL ? 'T' : 'D',
				symbol_tab[i].size,
				mod_name,
				symbol_tab[i].name);
		}
	}
}

int
find_sym_addr(char *name, int type, uintptr_t *start, uintptr_t *end)
{
	module_symbols_t *mod;
	symbol_t *symbol_tab;
	char *p;
	unsigned long i, nsyms;
	int has_mod_name = 0;
	char mod_name[32];

	if (name == NULL || *name == '\0' || start == NULL) return -1;

	/*
	 * check for module name
	 */
	mod_name[0] = '\0';
	p = strchr(name, ':');
	if (p) {
		strncpy(mod_name, name, p - name); 
		mod_name[p-name] = '\0';
		name = p + 1;
		has_mod_name = 1;
	}

	for(mod = modules_list; mod; mod = mod->next) {
		if (has_mod_name && strcmp(mod_name, mod->name)) continue;
		nsyms      = mod->nsyms;
		symbol_tab = mod->symbol_tab;
		for (i = 0; i < nsyms; i++) {
			if (!strcmp(name, symbol_tab[i].name) && symbol_tab[i].type == type) 
				goto found;
		 }
	}

	return -1;
found:
	*start = symbol_tab[i].value;
	if (end) {
		if (symbol_tab[i].size != 0) {
			*end = *start + symbol_tab[i].size; 
			//vbprintf("symbol %s: [0x%lx-0x%lx)=%ld bytes\n", name, *start, *end, symbol_tab[i].size);
		} else {
			vbprintf("using approximation for size of symbol %s\n", name);

			if (i == (nsyms-1)) {
				warning("cannot find another symbol to approximate size of %s\n", name);
				return -1;
			}

		        /*
		 	 * XXX: Very approximative and maybe false at times
		 	 * Use carefully
		 	 */
			*end = symbol_tab[i+1].value;
		}
		vbprintf("symbol %s (%s): [%p-%p)=%ld bytes\n", 
				name, 
				type == PFMON_TEXT_SYMBOL ? "code" : "data",
				(void *)*start, 
				(void *)*end, 
				*end-*start);
	}
	return 0;
}

int
find_sym_byaddr(uintptr_t addr, int type, char **name, char **module, uintptr_t *start, uintptr_t *end)
{
	module_symbols_t *mod;
	symbol_t *symbol_tab;
	unsigned long i, nsyms;

	/* table is assumed sorted by address */
	for(mod = modules_list ; mod ; mod = mod->next) {
		nsyms      = mod->nsyms;
		symbol_tab = mod->symbol_tab;
		for (i = 0; i < nsyms; i++) {
			if (symbol_tab[i].value > addr && symbol_tab[i].type == type) goto found;
		}
	}
	return -1;
found:
	if (i==0) {
		DPRINT(("invalid address 0x%lx < 0x%lx, first entry\n", symbol_tab[i].value));
		return -1;
	}
	/* i > 0 */
	i--;
	*name  = symbol_tab[i].name;
	if (start) *start   = symbol_tab[i].value;
	if (end)   *end     = symbol_tab[i].size ? symbol_tab[i].value+symbol_tab[i].size : symbol_tab[i+1].value ;
	if (module) *module = mod->name;
	return 0;
}

int
is_exact_sym(uintptr_t addr, int type)
{
	module_symbols_t *mod;
	symbol_t *symbol_tab;
	unsigned long i, nsyms;

	/* table is assumed sorted by address */
	for(mod = modules_list; mod ; mod = mod->next) {
		nsyms      = mod->nsyms;
		symbol_tab = mod->symbol_tab;
		for (i = 0; i < nsyms; i++) {
			if (symbol_tab[i].value == addr && symbol_tab[i].type == type) return 1;
		}
	}
	return 0;
}

static void
init_hash_table_lists(sym_hash_table_t *hash)
{
	sym_hash_entry_t *entries, *prev = NULL;
	unsigned long max;
	unsigned long i;

	/*
	 * initialize entries free list 
	 */
	max     = hash->max_entries - 1;
	entries = hash->entries;

	for(i = 0; i < max; i++) {
		entries[i].next = entries+i+1;
		entries[i].prev = prev;
		prev = entries+i;
	}
	entries[i].next = NULL;
	entries[i].prev = prev;

	hash->free_list   = entries;
	hash->active_list = NULL;


	/*
	 * initialize hash table
	 */
	max = hash->mask+1;
	for(i=0; i < max; i++) {
		hash->table[i] = NULL;
	}
}

static sym_hash_entry_t *
find_victim_sym_hash(sym_hash_table_t *hash)
{
	sym_hash_entry_t *p, *q = NULL;
	unsigned long min = ~0;

	/*
	 * look for a victim across ALL entries
	 */
	for (p = hash->active_list ; p ; p = p->next) {
		if (p->access_count < min) {
			q = p;
			min = p->access_count;
		}
	}
	/*
	 * detach entry from hash list
	 */
	if (q->hash_prev) 
		q->hash_prev->hash_next = q->hash_next;
	else
		hash->table[q->bucket] = q->hash_next;

	if (q->hash_next) 
		q->hash_next->hash_prev = q->hash_prev;

	return q;
}

int
pfmon_syms_hash_find(void *hash_desc, uintptr_t addr, char **name, char **module, uintptr_t *start_addr)
{
	sym_hash_table_t *hash = (sym_hash_table_t *)hash_desc;
	sym_hash_entry_t *hash_list;
	sym_hash_entry_t *p;
	unsigned long bucket;
	unsigned long collisions = 0UL;

	/*
	 * our hash function, we ignore the bottom 8 bits
	 */
	bucket = ((intptr_t)addr>>8) & hash->mask;

	hash_list = hash->table[bucket];

	hash->accesses++;

	for (p = hash_list; p ; p = p->next) {
		if (p->addr <= addr && p->eaddr > addr) goto found;
		collisions++;
	}

	hash->misses++;

	if (hash->free_list) {
		p               = hash->free_list;
		hash->free_list = hash->free_list->next;

		p->next = hash->active_list;
		p->prev = NULL;

		if (hash->active_list) hash->active_list->prev = p;

		hash->active_list = p;

		hash->used_entries++;

	} else {
		p = find_victim_sym_hash(hash);
	}
	/*
	 * when symbol not found, put entry back into free list
	 */
	if (find_sym_byaddr(addr, PFMON_TEXT_SYMBOL, &p->name, &p->module, &p->addr, &p->eaddr)) {

		if (p->prev)
			p->prev->next = p->next;
		else
			hash->active_list = p->next;

		if (p->next) p->next->prev = p->prev;

		p->next = hash->free_list;
		hash->free_list = p;

		return -1;
	}

	p->bucket    = bucket;
	p->hash_prev = NULL;
	p->hash_next = hash_list;
	hash->table[bucket] = p;

found:
	hash->collisions   += collisions;

	*name       = p->name;
	*module     = p->module;
	*start_addr = p->addr;

	if (p->hash_next) p->hash_next->hash_prev = p->hash_prev;
	if (p->hash_prev) p->hash_prev->hash_next = p->hash_next;

	p->hash_prev = NULL;
	p->hash_next = hash->table[p->bucket];
	hash->table[p->bucket] = p;

	p->access_count++;

	return 0;
}

int
pfmon_syms_hash_alloc(unsigned long hash_log_size, unsigned long max_entries, void **hash_desc)
{
	sym_hash_table_t *hash;
	size_t sz;

	sz = sizeof(sym_hash_table_t) 
	   + max_entries*sizeof(sym_hash_entry_t)
	   + (1UL<<hash_log_size)*sizeof(sym_hash_entry_t *);

	hash = malloc(sz);
	if (hash == NULL) {
		warning("could not allocate %lu bytes for symbol hash table\n", sz);
		return -1;
	}
	hash->entries      = (sym_hash_entry_t *)(hash+1);
	hash->table        = (sym_hash_entry_t **)(hash->entries+max_entries);

	hash->mask         = (1UL<<hash_log_size) - 1UL;
	hash->max_entries  = max_entries;
	hash->accesses     = hash->misses = 0ULL;
	hash->used_entries = 0UL;

	init_hash_table_lists(hash);

	*hash_desc = hash;

	return 0;
}

void 
pfmon_syms_hash_free(void *hash_desc)
{
	sym_hash_table_t *hash = (sym_hash_table_t *)hash_desc;
	
	DPRINT(("accesses %"PRIu64", misses %"PRIu64", miss ratio %.2f%%, %.2f collisions/access\n", 
		hash->accesses,
		hash->misses,
		hash->misses*100.0/hash->accesses,
		hash->collisions*1.0/hash->accesses));

	free(hash_desc);
}
