/*
 * CLightSource.cpp
 * $Id: CLightSource.cpp,v 1.8 2001/11/23 02:03:35 mjanich Exp $
 *
 * Copyright (C) 1999, 2000 Michael Meissner, Michael Guthe
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Definition of the CLightSource class
//  Purpose     : Managment of class providing the interface to a linked
//                list of objects.

/** documentation stuff

  @author Michael Guthe, Michael Meissner

  @version 0.0 //see cvs docu

*/


// Own
////////
#include "GeoGeneric.h"
#include "CLightSource.h"

// System
///////////

//Constructors
////////////////

CLightSource::CLightSource(const CLightSource &light)
/**********************************************************************/
{
   setType(light.getType());
   setStatus(light.getStatus());
   setPosition(light.getPosition());
   setDirection(light.getDirection());
   setColor(light.getColor()[0], 
            light.getColor()[1], 
            light.getColor()[2]);
   setIntensity(light.getIntensity());
   setExponent(light.getExponent());
   setAngle(light.getAngle());
   setConstantAttenuation(light.getConstantAttenuation());
   setLinearAttenuation(light.getLinearAttenuation());
   setQuadraticAttenuation(light.getQuadraticAttenuation());
}


CLightSource::CLightSource(LightSourceType nType)
  : m_LightStatus(ON),
    m_Point(CP3D(0,0,0)),
    m_Direction(CV3D(0,0,-1)),
    m_rfIntensity(1.0),
    m_rfAngle(180.0),
    m_rfExponent(0.0),
    m_rfConstantAttenuation(1.0),
    m_rfLinearAttenuation(0.0),
    m_rfQuadraticAttenuation(0.0)
/**********************************************************************/
{
   setType(nType);
 
   m_arfColor[0] = 1.0;
   m_arfColor[1] = 1.0;
   m_arfColor[2] = 1.0;
}


CLightSource::CLightSource(const CV3D &direction, float rfIntensity)
  : m_LightSourceType(DIRECTIONAL),
    m_LightStatus(ON),
    m_rfAngle(180.0),
    m_rfExponent(0.0),
    m_rfConstantAttenuation(1.0),
    m_rfLinearAttenuation(0.0),
    m_rfQuadraticAttenuation(0.0)
/**********************************************************************/
{
   CV3D directionNormal = direction.getNormalized();
   setType(DIRECTIONAL);
   m_Direction   = directionNormal;
   m_Direction.normalize();
   m_Point       = CP3D(0,0,0) + (-100000 * directionNormal);
   m_rfIntensity = rfIntensity;
 
   m_arfColor[0] = 1.0;
   m_arfColor[1] = 1.0;
   m_arfColor[2] = 1.0;
}



CLightSource::CLightSource(const CP3D &point, float rfIntensity)
  : m_LightStatus(ON),
    m_Direction(CV3D(0,0,0)),
    m_rfAngle(180.0),
    m_rfExponent(0.0),
    m_rfConstantAttenuation(1.0),
    m_rfLinearAttenuation(0.0),
    m_rfQuadraticAttenuation(0.0)
/**********************************************************************/
{
   setType(POINT);
   m_Direction.normalize();
   m_Point       = point;
   m_rfIntensity = rfIntensity;

   m_arfColor[0] = 1.0;
   m_arfColor[1] = 1.0;
   m_arfColor[2] = 1.0;
}



CLightSource::CLightSource(const CP3D &point, const CV3D &direction, 
                           float rfAngle, float rfIntensity)
  : m_LightStatus(ON),
    m_rfExponent(0.0),
    m_rfConstantAttenuation(1.0),
    m_rfLinearAttenuation(0.0),
    m_rfQuadraticAttenuation(0.0)
/**********************************************************************/
{
   setType(CONE);
   m_Direction   = direction;
   m_Direction.normalize();
   m_Point       = point;
   m_rfIntensity = rfIntensity;
   m_rfAngle     = rfAngle;

   m_arfColor[0] = 1.0;
   m_arfColor[1] = 1.0;
   m_arfColor[2] = 1.0;
}


CLightSource::CLightSource(const CP3D &point, const CV3D &direction, 
                           float rfAngle, float rfIntensity, float rfExponent)
  : m_LightStatus(ON),
    m_rfConstantAttenuation(1.0),
    m_rfLinearAttenuation(0.0),
    m_rfQuadraticAttenuation(0.0)
/**********************************************************************/
{
   setType(CONE_DIFFUSE);
   m_Direction   = direction;
   m_Direction.normalize();
   m_Point       = point;
   m_rfIntensity = rfIntensity;
   m_rfAngle     = rfAngle;
   m_rfExponent  = rfExponent;
 
   m_arfColor[0] = 1.0;
   m_arfColor[1] = 1.0;
   m_arfColor[2] = 1.0;
}

  


///////////////////////////////
//                           //
// Functions of CLightSource //
//                           //
///////////////////////////////



// Function   : setIntensity
// Parameters : float rfIntensity
// Purpose    :
// Comments   :
int CLightSource::setIntensity(float rfIntensity)
/**********************************************************/
{
   m_rfIntensity = rfIntensity;
   return 1;
}



// Function   : setPosition
// Parameters : CP3D point
// Purpose    :
// Comments   :
int CLightSource::setPosition(const CP3D &point)
/**********************************************************/
{
   m_Point = point;
   return 1;
}


// Function   : setDirection
// Parameters : CV3D direction
// Purpose    :
// Comments   :
int CLightSource::setDirection(const CV3D &direction)
/**********************************************************/
{
   m_Direction = direction;
   m_Direction.normalize();

   if (getType() == POINT) {
      //cout << "CLightSource::setExponent: Notice that a point ";
      //cout << "light source can not have a direction!" << endl;
      return -1;
   }
   return 1;
}



// Function   : setAngle
// Parameters : float rfAngle
// Purpose    :
// Comments   :
int CLightSource::setAngle(float rfAngle)
/**********************************************************/
{
   m_rfAngle = rfAngle;

   if (getType() != CONE && getType() != CONE_DIFFUSE) {
      //cout << "CLightSource::setExponent: Notice that only a ";
      //cout << "cone light source can have an exponent!" << endl;
      return -1;
   }
   return 1;
}



// Function   : setExponent
// Parameters : float rfExponent
// Purpose    :
// Comments   :
int CLightSource::setExponent(float rfExponent)
/**********************************************************/
{
   m_rfExponent = rfExponent;

   if (getType() != CONE || getType() != CONE_DIFFUSE) {
      //cout << "CLightSource::setExponent: Notice that only a ";
      //cout << "cone light source can have an exponent!" << endl;
      return -1;
   }
   return 1;
}



// Function   : setColor
// Parameters : float rfRed, rfGreen, rfBlue
// Purpose    :
// Comments   :
void CLightSource::setColor(float rfRed, float rfGreen, float rfBlue)
/**********************************************************/
{
   m_arfColor[0] = rfRed;
   m_arfColor[1] = rfGreen;
   m_arfColor[2] = rfBlue;
   return;
}



// Function   : getIntensity
// Parameters : CP3D point
// Purpose    :
// Comments   :
float CLightSource::getIntensity(const CP3D &point) const
/**********************************************************/
{
   CV3D dirVec;
   float rfIntensity;
   float rfCosinusAlpha;

   if (getStatus() == OFF)
      return 0.0;

   if (getType() == DIRECTIONAL) 
      return getIntensity();

   dirVec = point - getPosition();
   rfIntensity = getIntensity()/(float)(dirVec.getX()*dirVec.getX()
                                        +dirVec.getY()*dirVec.getY()
		 		        +dirVec.getZ()*dirVec.getZ());

   if ((getType() == CONE) || (getType() == CONE_DIFFUSE)) {
      dirVec.normalize();
      rfCosinusAlpha = (float) (dirVec * m_Direction);
      if (rfCosinusAlpha < cos(getAngle() * M_PI / 180.0))
         return 0.0;
      if (getType() == CONE_DIFFUSE) 
	 rfIntensity *= (float) pow(rfCosinusAlpha, getExponent());

   }
   return rfIntensity;
}



// Function   : getDirection
// Parameters : CP3D point
// Purpose    :
// Comments   :
CV3D CLightSource::getDirection(const CP3D &point) const
/**********************************************************/
{
   if (getStatus() == OFF)
      return CV3D(0.0,0.0,0.0);
      
   if (getType() == DIRECTIONAL) 
      return m_Direction;

   return point - m_Point;
}
