/* qmv.h - Common definitions for the qmv program.
 *
 * Copyright (C) 2001  Oskar Liljeblad
 *
 * This file is part of the file renaming utilities (renameutils).
 *
 * This software is copyrighted work licensed under the terms of the
 * GNU General Public License. Please consult the file `COPYING' for
 * details.
 */

#ifndef __QMV_H__
#define __QMV_H__

#include <stdio.h>			/* for FILE */
#include "common/llist.h"
#include "common/common.h"	/* for bool */

#define PROGRAM "qmv"

typedef struct _EditFormat EditFormat;
typedef struct _FileSpec FileSpec;
typedef struct _RenamePlan RenamePlan;

struct _EditFormat {
	char *name;
	char *short_name;
	bool (*parse_options)(char *);
	char *(*option_generator)(const char *, int);
	void (*output)(FILE *, LList *);
	bool (*input)(FILE *, LList *);
};

typedef enum {
	STATUS_NONE,
	STATUS_OLD_MISSING,		/* (error) old file no longer exists */
	STATUS_DUPLICATE,		/* (error) part of a set with duplicate destinations */
	STATUS_NO_CHANGE,		/* (no_change) old_name eq new_name */
	STATUS_NEW_EXISTS,		/* (error) new file already exists */
	STATUS_RENAME,			/* (ok) everything is ok, rename file */
	STATUS_CIRCULAR,		/* (ok) a temporary rename need to be made first */
	STATUS_COMPLETED,		/* (no_change) previously completed */
} FileSpecStatus;

typedef enum {
	RENAME_UNDONE,			
	RENAME_COMPLETE,
	RENAME_FAILED,
} FileSpecRenamed;

struct _FileSpec {
	char *old_name;
	char *new_name;
	FileSpecStatus status;
	FileSpecRenamed renamed;
	FileSpec *next_spec;		/* next duplicate or circular */
	FileSpec *dependency;		/* this must be completed first */
};

struct _RenamePlan {
	LList *ok;
	LList *error;
	LList *no_change;
};

/* main.c */
extern bool dummy;
//extern bool show_control_chars;
extern LList *renames;
extern char *renames_directory;
extern char *editor_program;
extern char *edit_filename;
extern EditFormat *format;
extern RenamePlan *plan;
/*bool set_show_control_chars(bool value);*/
void display_qmv_version(void);
void display_qmv_help(FILE *out);

/* editing.c */
EditFormat *find_edit_format_by_name(const char *name);
void edit_command(char **args);
bool edit_renames(bool all);
char *edit_format_generator(const char *text, int state);

/* listing.c */
void list_command(char **args);
bool list_files(char **args);
void process_ls_option(int c);
struct option *append_ls_options(const struct option *options);
FileSpec *new_file_spec(void);
void free_file_spec(FileSpec *spec);
void free_renames(LList *renames);
void display_ls_help(FILE *out);
bool cwd_to_renames_directory();
bool cwd_from_renames_directory();

/* quoting.c */
char *quote_output_file(const char *t);
char *dequote_output_file(const char *str);

/* editformats/(misc).c */
extern EditFormat single_column_format;
extern EditFormat dual_column_format;
extern EditFormat destination_only_format;

/* commandmode.c */
void commandmode_mainloop(void);
void display_commandmode_header(void);

/* variables.c */
void set_command(char **args);
void show_command(char **args);
char *qmv_variable_generator(const char *text, int state);

/* renaming.c */
RenamePlan *make_plan(LList *renames);
void display_plan(RenamePlan *plan, bool show_status);
void free_plan(RenamePlan *plan);
void plan_command(char **args);

/* planaction.c */
bool apply_plan(RenamePlan *plan);
bool retry_plan(RenamePlan *plan);
void apply_command(char **args);
void retry_command(char **args);

#endif
