///###////////////////////////////////////////////////////////////////////////
//
// Burton Computer Corporation
// http://www.burton-computer.com
// $Id: MimeLineReader.cc,v 1.11 2004/04/22 19:30:41 bburton Exp $
//
// Copyright (C) 2000 Burton Computer Corporation
// ALL RIGHTS RESERVED
//
// This program is open source software; you can redistribute it
// and/or modify it under the terms of the Q Public License (QPL)
// version 1.0. Use of this software in whole or in part, including
// linking it (modified or unmodified) into other programs is
// subject to the terms of the QPL.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// Q Public License for more details.
//
// You should have received a copy of the Q Public License
// along with this program; see the file LICENSE.txt.  If not, visit
// the Burton Computer Corporation or CoolDevTools web site
// QPL pages at:
//
//    http://www.burton-computer.com/qpl.html
//

#include <stdexcept>
#include "RegularExpression.h"
#include "MimeLineReader.h"

static const string MBX_FILE_FIRST_LINE("*mbx*");
static const int MBX_FILE_HEADER_LENGTH = 2048;
static const unsigned long MBX_DELETED_FLAG = 0x0002;
static const string MBX_RECORD_HEADER_REGEX("^[ 0-9][0-9]+-[A-Za-z]+-[0-9]+ [0-9]+:[0-9]+:[0-9]+ .?[0-9]+,([0-9]+);[0-9a-zA-Z]+([0-9a-zA-Z][0-9a-zA-Z][0-9a-zA-Z][0-9a-zA-Z])-[0-9a-zA-Z]+$");

MimeLineReader::MimeLineReader(istream &in)
  : m_atEnd(false),
    m_in(in),
    m_count(0),
    m_messageLength(0),
    m_recordLength(0),
    m_isMBX(false),
    m_isFirstRecord(true),
    m_canSeek(false)
{
}

MimeLineReader::~MimeLineReader()
{
}

void MimeLineReader::readMBXFileHeader()
{
  assert(m_isFirstRecord);

  if (!m_canSeek) {
    return;
  }

  if (m_in.get() == '*' &&
      m_in.get() == 'm' &&
      m_in.get() == 'b' &&
      m_in.get() == 'x' &&
      m_in.get() == '*' &&
      m_in.get() == '\r' &&
      m_in.get() == '\n') {
    m_isMBX = true;
    m_in.seekg(MBX_FILE_HEADER_LENGTH, ios::beg);
  } else {
    m_isMBX = false;
    m_in.seekg(0, ios::beg);
  }
}

void MimeLineReader::readMBXRecordHeader()
{
  assert(m_isMBX);

  static RegularExpression header_expr(MBX_RECORD_HEADER_REGEX, 3);

  bool again = true;
  while (again) {
    m_recordLength = 0;
    if (!readLine()) {
      return;
    }

    //cerr << "MBX: HDR: " << m_line << endl;
    if (!header_expr.match(m_line)) {
      throw runtime_error(string("IMAP-MBX: regex for record header did not match: ") + m_line);
    }

    string matched;
    m_recordLength = atoi(header_expr.getMatch(1, matched).c_str());
    unsigned long flags = strtoul(header_expr.getMatch(2, matched).c_str(), 0, 16);
    //cerr << "MBX: FLAGS " << hex << flags << " RECLEN " << dec << m_recordLength << endl;
    if (flags & MBX_DELETED_FLAG) {
      // message is marked deleted
      m_in.ignore(m_recordLength);
      //cerr << "MBX: SKIPPED DELETED MESSAGE" << endl;
    } else {
      m_messageLength = 0;
      again = false;
    }
  }
}

void MimeLineReader::gotoNextRecord()
{
  if (m_isFirstRecord) {
    readMBXFileHeader();
    m_messageLength = 0;
    m_recordLength = 0;
    m_isFirstRecord = false;
  }

  if (m_isMBX) {
    if (m_messageLength < m_recordLength) {
      //cerr << "MBX SKIPPING MSGLEN " << m_messageLength << " RECLEN " << m_recordLength << endl;
      m_in.ignore(m_recordLength - m_messageLength);
    }
    readMBXRecordHeader();
  }
}

bool MimeLineReader::readLine()
{
  assert(m_isMBX || (m_recordLength == 0));

  m_line.erase();

  if (m_atEnd) {
    return false;
  }

  if (!m_in.good()) {
    m_atEnd = true;
    return false;
  }

  if (m_recordLength > 0 && m_messageLength >= m_recordLength) {
    //cerr << "MBX: MSGLEN " << m_messageLength << " RECLEN " << m_recordLength << endl;
    return false;
  }

  m_line.erase();

  char ch;
  bool again = true;
  m_in.get(ch);
  while (again && m_in.good()) {
    m_count += 1;
    m_messageLength += 1;

    if (m_recordLength > 0 && m_messageLength >= m_recordLength) {
      //cerr << "MBX: MSGLEN " << m_messageLength << " RECLEN " << m_recordLength << endl;
      break;
    }

    switch (ch) {
    case '\0':
      m_line += ' ';
      m_in.get(ch);
      break;

    case '\n':
      again = false;
      break;

    case '\r':
      if (m_in.peek() == '\n') {
        m_count += 1;
        m_messageLength += 1;
        m_in.get();
      }
      again = false;
      break;

    default:
      m_line += safe_char(ch);
      m_in.get(ch);
      break;
    }
  }

  if (!m_in.good()) {
    m_atEnd = true;
    return m_line.length() > 0;
  }

  return !m_atEnd;
}
