/*  GNU SED, a batch stream editor.
    Copyright (C) 1999
    Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

#include "config.h"

#include <ctype.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifndef NULL
# include <stdio.h>
#endif

#include "basicdefs.h"
#include "utils.h"
#include "sed.h"
#include "regexp.h"
#ifndef NULL
# define NULL CAST(VOID *)0
#endif

#ifdef gettext_noop
# define N_(String) gettext_noop(String)
#else
# define N_(String) (String)
#endif

extern flagT use_extended_syntax_p;

#define TRANSLATE_LENGTH 256 /*XXX shouldn't this be (UCHAR_MAX+1)?*/

static const char NO_REGEX[] = N_("No previous regular expression");
static const char BAD_MODIF[] = N_("Cannot specify modifiers on empty regexp");



regex_t *
compile_regex(b, flags, needed_sub)
  struct buffer *b;
  int flags;
  int needed_sub;
{
  regex_t *new_regex;
  char error[200];
  int errcode;

  char *last_re = NULL;
  size_t last_re_len;

  /* My reading of IEEE Std 1003.2-1992 is that // means the empty RE.
     But historical and common practice is that // "matches the last RE";
     thus this use of POSIXLY_CORRECT. */
  if (size_buffer(b) == 0 && !POSIXLY_CORRECT)
    {
      if (flags > 0)
	bad_prog(_(BAD_MODIF));
      return NULL;
    }

  last_re_len = size_buffer(b);
  last_re = ck_memdup(get_buffer(b), last_re_len);

  new_regex = MALLOC(1, regex_t);

#ifdef REG_PERL
  errcode = regncomp(new_regex, last_re, last_re_len,
		     (needed_sub ? 0 : REG_NOSUB)
		     | flags
		     | extended_regexp_flags);

#else
  {
    int syntax = ((extended_regexp_flags & REG_EXTENDED)
		   ? RE_SYNTAX_POSIX_EXTENDED
                   : RE_SYNTAX_POSIX_BASIC)
		   & ~RE_UNMATCHED_RIGHT_PAREN_ORD;

    /* Try to allocate space for the fastmap.  */
    new_regex->fastmap = ck_malloc (1 << (8 * sizeof(char)));
    if (new_regex->fastmap == NULL)
      {
        errcode = REG_ESPACE;
        goto error;
      }

    syntax |= (flags & REG_ICASE) ? RE_ICASE : 0;

    /* If REG_NEWLINE is set, newlines are treated differently.  */
    if (flags & REG_NEWLINE)
      { /* REG_NEWLINE implies neither . nor [^...] match newline.  */
        syntax &= ~RE_DOT_NEWLINE;
        syntax |= RE_HAT_LISTS_NOT_NEWLINE;
        /* It also changes the matching behavior.  */
        new_regex->newline_anchor = 1;
      }
    else
      new_regex->newline_anchor = 0;

    new_regex->translate = NULL;

    /* Only PCRE processes \t & co. */
    last_re_len = normalize_text(last_re, last_re_len);
    errcode = re_compile_internal (new_regex, last_re, last_re_len, syntax);

    if (errcode == REG_NOERROR)
      {
        if (re_compile_fastmap (new_regex) == -2)
          {
            /* Some error occurred while computing the fastmap, just forget
               about it.  */
            ck_free (new_regex->fastmap);
            new_regex->fastmap = NULL;
          }
      }
  }
#endif

  FREE(last_re);

  if (errcode)
    {
    error:
      regerror(errcode, NULL, error, 200);
      bad_prog(gettext(error));
    }

  /* Just to be sure, I mark this as not POSIXLY_CORRECT behavior */
  if (new_regex->re_nsub < needed_sub && !POSIXLY_CORRECT)
    {
      char buf[200];
      sprintf(buf, _("Invalid reference \\%d on `s' command's RHS"),
	      needed_sub);
      bad_prog(buf);
    }

  return new_regex;
}


int
match_regex(regex, buf, buflen, buf_start_offset, regarray, regsize)
  regex_t *regex;
  char *buf;
  size_t buflen;
  size_t buf_start_offset;
  regmatch_t *regarray;
  int regsize;
{
  static regex_t *regex_last;
  int ret;

  /* Keep track of the last regexp matched. */
  if (!regex)
    {
      regex = regex_last;
      if (!regex_last)
	bad_prog(_(NO_REGEX));
    }
  else
    regex_last = regex;

#ifdef REG_PERL
  ret = regexec2(regex, 
		 buf, CAST(int)buflen, CAST(int)buf_start_offset,
		 regsize, regarray, 0);

#else
  ret = re_search_internal(regex, buf, buflen, buf_start_offset,
		           buflen - buf_start_offset, buflen,
			   regsize, regarray, 
			   RE_NO_POSIX_BACKTRACKING);

#endif
  return (ret == 0);
}


#ifdef DEBUG_LEAKS
void
release_regex(regex)
  regex_t *regex;
{
  regfree(regex);
  FREE(regex);
}
#endif /*DEBUG_LEAKS*/
