/**
 * @copyright
 * ====================================================================
 * Copyright (c) 2000-2004 CollabNet.  All rights reserved.
 *
 * This software is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at http://subversion.tigris.org/license-1.html.
 * If newer versions of this license are posted there, you may use a
 * newer version instead, at your option.
 *
 * This software consists of voluntary contributions made by many
 * individuals.  For exact contribution history, see the revision
 * history and logs, available at http://subversion.tigris.org/.
 * ====================================================================
 * @endcopyright
 *
 * @file svn_ra_svn.h
 * @brief libsvn_ra_svn functions used by the server
 */




#ifndef SVN_RA_SVN_H
#define SVN_RA_SVN_H

#include <apr.h>
#include <apr_pools.h>
#include <apr_network_io.h>
#include <svn_config.h>

#include "svn_delta.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/** The well-known svn port number. */
#define SVN_RA_SVN_PORT 3690

/** Currently-defined capabilities. */
#define SVN_RA_SVN_CAP_EDIT_PIPELINE "edit-pipeline"

/** A specialized form of @c SVN_ERR to deal with errors which occur in an
 * @c svn_ra_svn_command_handler.
 *
 * An error returned with this macro will be passed back to the other side 
 * of the connection.  Use this macro when performing the requested operation; 
 * use the regular @c SVN_ERR when performing I/O with the client.
 */
#define SVN_CMD_ERR(expr)                                     \
  do {                                                        \
    svn_error_t *svn_err__temp = (expr);                      \
    if (svn_err__temp)                                        \
      return svn_error_create(SVN_ERR_RA_SVN_CMD_ERR,         \
                              svn_err__temp, NULL);           \
  } while (0)

/** an ra_svn connection. */
typedef struct svn_ra_svn_conn_st svn_ra_svn_conn_t;

/** Command handler, used by @c svn_ra_svn_handle_commands. */
typedef svn_error_t *(*svn_ra_svn_command_handler)(svn_ra_svn_conn_t *conn,
                                                   apr_pool_t *pool,
                                                   apr_array_header_t *params,
                                                   void *baton);

/** Command table, used by @c svn_ra_svn_handle_commands.
 *
 * If @c terminate is set, command-handling will cease after command is
 * processed.
 */
typedef struct svn_ra_svn_cmd_entry_t 
{
  const char *cmdname;
  svn_ra_svn_command_handler handler;
  svn_boolean_t terminate;
} svn_ra_svn_cmd_entry_t;

/** Memory representation of an on-the-wire data item. */
typedef struct svn_ra_svn_item_t 
{
  enum {
    SVN_RA_SVN_NUMBER,
    SVN_RA_SVN_STRING,
    SVN_RA_SVN_WORD,
    SVN_RA_SVN_LIST
  } kind;
  union {
    apr_uint64_t number;
    svn_string_t *string;
    const char *word;

    /** Contains @c svn_ra_svn_item_t's. */
    apr_array_header_t *list;
  } u;
} svn_ra_svn_item_t;

typedef svn_error_t *(*svn_ra_svn_edit_callback)(void *baton);

/** Initialize a connection structure for the given socket or
 * input/output files.
 *
 * Either @c sock or @c in_file/@c out_file must be set, not both.
 */
svn_ra_svn_conn_t *svn_ra_svn_create_conn(apr_socket_t *sock,
                                          apr_file_t *in_file,
                                          apr_file_t *out_file,
                                          apr_pool_t *pool);

/** Initialize a connection's capabilities to the ones specified in
 * @a list, which contains svn_ra_svn_item_t entries (which should
 * be of type SVN_RA_SVN_WORD; a malformed data error will result if
 * any are not). */
svn_error_t *svn_ra_svn_set_capabilities(svn_ra_svn_conn_t *conn,
                                         apr_array_header_t *list);

/** Return @c TRUE if @a conn has the capability @a capability, or
 * @c FALSE if it does not. */
svn_boolean_t svn_ra_svn_has_capability(svn_ra_svn_conn_t *conn,
                                        const char *capability);

/** Write a number over the net.
 *
 * Writes will be buffered until the next read or flush.
 */
svn_error_t *svn_ra_svn_write_number(svn_ra_svn_conn_t *conn, apr_pool_t *pool,
                                     apr_uint64_t number);

/** Write a string over the net.
 *
 * Writes will be buffered until the next read or flush.
 */
svn_error_t *svn_ra_svn_write_string(svn_ra_svn_conn_t *conn, apr_pool_t *pool,
                                     const svn_string_t *str);

/** Write a cstring over the net.
 *
 * Writes will be buffered until the next read or flush.
 */
svn_error_t *svn_ra_svn_write_cstring(svn_ra_svn_conn_t *conn,
                                      apr_pool_t *pool, const char *s);

/** Write a word over the net.
 *
 * Writes will be buffered until the next read or flush.
 */
svn_error_t *svn_ra_svn_write_word(svn_ra_svn_conn_t *conn, apr_pool_t *pool,
                                   const char *word);

/** Begin a list.  Writes will be buffered until the next read or flush. */
svn_error_t *svn_ra_svn_start_list(svn_ra_svn_conn_t *conn, apr_pool_t *pool);

/** End a list.  Writes will be buffered until the next read or flush. */
svn_error_t *svn_ra_svn_end_list(svn_ra_svn_conn_t *conn, apr_pool_t *pool);

/** Flush the write buffer.
 *
 * Normally this shouldn't be necessary, since the write buffer is flushed
 * when a read is attempted.
 */
svn_error_t *svn_ra_svn_flush(svn_ra_svn_conn_t *conn, apr_pool_t *pool);

/** Write a tuple, using a printf-like interface.
 *
 * The format string @a fmt may contain:
 *
 *<pre>
 *   Spec  Argument type         Item type
 *   ----  --------------------  ---------
 *   n     apr_uint64_t          Number
 *   r     svn_revnum_t          Number
 *   s     const svn_string_t *  String
 *   c     const char *          String
 *   w     const char *          Word
 *   b     svn_boolean_t         Word ("true" or "false")
 *   (                           Begin tuple
 *   )                           End tuple
 *   ?                           Remaining elements optional
 *   ! (at beginning or end)     Suppress opening or closing of tuple
 * </pre>
 *
 * Inside the optional part of a tuple, 'r' values may be @c
 * SVN_INVALID_REVNUM and 's', 'c', and 'w' values may be @c NULL; in
 * these cases no data will be written.  'n', 'b', and '(' may not
 * appear in the optional part of a tuple.  Either all or none of the
 * optional values should be valid.
 *
 * Use the '!' format specifier to write partial tuples when you have
 * to transmit an array or other unusual data.  For example, to write
 * a tuple containing a revision, an array of words, and a boolean:
 *   SVN_ERR(svn_ra_svn_write_tuple(conn, pool, "r(!", rev));
 *   for (i = 0; i < n; i++)
 *     SVN_ERR(svn_ra_svn_write_word(conn, pool, words[i]));
 *   SVN_ERR(svn_ra_svn_write_tuple(conn, pool, "!)b", flag));
 */
svn_error_t *svn_ra_svn_write_tuple(svn_ra_svn_conn_t *conn, apr_pool_t *pool,
                                    const char *fmt, ...);

/** Read an item from the network into @c item. */
svn_error_t *svn_ra_svn_read_item(svn_ra_svn_conn_t *conn, apr_pool_t *pool,
                                  svn_ra_svn_item_t **item);

/** Scan data on @c conn until we find something which looks like the
 * beginning of an svn server greeting (an open paren followed by a
 * whitespace character).  This function is appropriate for beginning
 * a client connection opened in tunnel mode, since people's dotfiles
 * sometimes write output to stdout.  It may only be called at the
 * beginning of a client connection.
 */
svn_error_t *svn_ra_svn_skip_leading_garbage(svn_ra_svn_conn_t *conn,
                                             apr_pool_t *pool);

/** Parse an array of @c svn_sort__item_t structures as a tuple, using a
 * printf-like interface.  The format string @a fmt may contain:
 *
 *<pre>
 *   Spec  Argument type          Item type
 *   ----  --------------------   ---------
 *   n     apr_uint64_t *         Number
 *   r     svn_revnum_t *         Number
 *   s     svn_string_t **        String
 *   c     const char **          String
 *   w     const char **          Word
 *   b     svn_boolean_t *        Word ("true" or "false")
 *   l     apr_array_header_t **  List
 *   (                            Begin tuple
 *   )                            End tuple
 *   ?                            Tuple is allowed to end here
 *</pre>
 *
 * Note that a tuple is only allowed to end precisely at a '?', or at
 * the end of the specification, of course.  So if @a fmt is "c?cc"
 * and @a list contains two elements, an error will result.
 *
 * If an optional part of a tuple contains no data, 'r' values will be
 * set to @c SVN_INVALID_REVNUM and 's', 'c', 'w', and 'l' values will
 * be set to @c NULL.  'n' and 'b' may not appear inside an optional
 * tuple specification.
 */
svn_error_t *svn_ra_svn_parse_tuple(apr_array_header_t *list,
                                    apr_pool_t *pool,
                                    const char *fmt, ...);

/** Read a tuple from the network and parse it as a tuple, using the
 * format string notation from @c svn_ra_svn_parse_tuple.
 */
svn_error_t *svn_ra_svn_read_tuple(svn_ra_svn_conn_t *conn, apr_pool_t *pool,
                                   const char *fmt, ...);

/** Read a command response from the network and parse it as a tuple, using 
 * the format string notation from @c svn_ra_svn_parse_tuple.
 */
svn_error_t *svn_ra_svn_read_cmd_response(svn_ra_svn_conn_t *conn,
                                          apr_pool_t *pool,
                                          const char *fmt, ...);

/** Accept commands over the network and handle them according to @a
 * commands.  Command handlers will be passed @a conn, a subpool of @a
 * pool (cleared after each command is handled), the parameters of the
 * command, and @a baton.  Commands will be accepted until a
 * terminating command is received (a command with "terminate" set in
 * the command table).  If a command handler returns an errors wrapped
 * in SVN_RA_SVN_CMD_ERR (see the SVN_CMD_ERR macro above), the error
 * will be reported to the other side of the connection and the
 * command loop will continue; any other kind of error (typically a
 * network or protocol error) is passed through to the caller.
 */
svn_error_t *svn_ra_svn_handle_commands(svn_ra_svn_conn_t *conn,
                                        apr_pool_t *pool,
                                        const svn_ra_svn_cmd_entry_t *commands,
                                        void *baton);

/** Write a command over the network, using the same format string notation 
 * as svn_ra_svn_write_tuple.
 */
svn_error_t *svn_ra_svn_write_cmd(svn_ra_svn_conn_t *conn, apr_pool_t *pool,
                                  const char *cmdname, const char *fmt, ...);

/** Write a successful command response over the network, using the
 * same format string notation as svn_ra_svn_write_tuple.  Do not use
 * partial tuples with this function; if you need to use partial
 * tuples, just write out the "success" and argument tuple by hand.
 */
svn_error_t *svn_ra_svn_write_cmd_response(svn_ra_svn_conn_t *conn,
                                           apr_pool_t *pool,
                                           const char *fmt, ...);

/** Write an unsuccessful command response over the network. */
svn_error_t *svn_ra_svn_write_cmd_failure(svn_ra_svn_conn_t *conn,
                                          apr_pool_t *pool, svn_error_t *err);

/** Set @a *editor and @a *edit_baton to an editor which will pass editing
 * operations over the network, using @a conn and @a pool.
 *
 * Upon successful completion of the edit, the editor will invoke @a callback 
 * with @a callback_baton as an argument.
 */
void svn_ra_svn_get_editor(const svn_delta_editor_t **editor,
                           void **edit_baton, svn_ra_svn_conn_t *conn,
                           apr_pool_t *pool, svn_ra_svn_edit_callback callback,
                           void *callback_baton);

/** Receive edit commands over the network and use them to drive @a editor
 * with @a edit_baton.  On return, @a *aborted will be set if the edit was
 * aborted.
 */
svn_error_t *svn_ra_svn_drive_editor(svn_ra_svn_conn_t *conn, apr_pool_t *pool,
                                     const svn_delta_editor_t *editor,
                                     void *edit_baton,
                                     svn_boolean_t *aborted);

/** This function is only intended for use by svnserve.
 *
 * Perform CRAM-MD5 password authentication.  On success, return
 * SVN_NO_ERROR with *user set to the username and *success set to
 * TRUE.  On an error which can be reported to the client, report the
 * error and return SVN_NO_ERROR with *success set to FALSE.  On
 * communications failure, return an error.
 */
svn_error_t *svn_ra_svn_cram_server(svn_ra_svn_conn_t *conn, apr_pool_t *pool,
                                    svn_config_t *pwdb, const char **user,
                                    svn_boolean_t *success);

/**
 * Get libsvn_ra_svn version information.
 * @since New in 1.1.
 */
const svn_version_t *svn_ra_svn_version (void);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif  /* SVN_RA_SVN_H */
