/* ==================================================== ======== ======= *
 *
 *  uview.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _uview_hpp_
#define	_uview_hpp_
//pragma ident	"@(#)uview.hpp	ubit:03.05.05"
#include <ubit/uctrl.hpp>
#include <ubit/uevent.hpp>

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Specifies the View Style of an UBox.
 */
class UViewStyle : public UMode {
public:
  UViewStyle(UView* (*)(class UBoxLink*, UView*, UWinGraph*), u_modes = 0);
  virtual ~UViewStyle() {destructs();}
  virtual void update();

  class UView* (*makeView)(class UBoxLink*, UView* parview, UWinGraph*);
  ///< pointer to the corresponding UView::makeView "static" constructor.

  virtual void addingTo(class ULink *selflink, UGroup *parent);
  virtual void removingFrom(class ULink *selflink, UGroup *parent);
  ///< NOTE that this function require a specific destructor.
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
/** Box View.
 *  UBox objects (and subclasses except UWin) can have several views.
 *  these views can be retrieved by methods UBox::getView(), UBox::getViews(), etc.
 *  
 *  Note: UWin objects (and subclasses) have only onse (shared) view that
 *  is called the "window view".
 */
class UView : public URegion {
public:

  UView(class UBoxLink*, UView* parview, UWinGraph*);
  virtual ~UView();

  static UView* makeView(class UBoxLink*, UView* parview, UWinGraph*);
  ///< makeView() is a "static creator" used by UViewStyle to make a new view.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  class UView* getParentView() const {return parview;}
  ///< returns the parent view.

  class UDisp*  getDisp()  const;
  class UAppli* getAppli() const;
  ///< return the corresponding Display and Application contexts.

  class UBox *getBox() const;
  ///< returns the UBox that controls this view.

  class UBox* getBoxParent() const;
  /**< returns the parent of the UBox that controls this view.
   * Note: this function is meaningless if this view is a WINDOW view
   * (= if the box that controls this view is a UWin)
   */

  bool isRealized() const;
  /**< returns true if this view has been realized.
   * a view is realized when physical (= X window) resources have been created 
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // size

  u_dim getWidth()  const {return width;}
  u_dim getHeight() const {return height;}
  void  getSize(u_dim& w, u_dim& h) const {w = width; h = height;}
  ///< returns the size of this View.

  void setWidth(u_dim width);
  void setHeight(u_dim height);
  void resize(u_dim width, u_dim height);
  /**< changes the size of this view (when applicable).
   * Note: these functions do no redisplay the view (the new size will be
   * taken into account when the view is updated)
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // position

  u_pos getX() const;
  u_pos getY() const;
  bool where(u_pos &x, u_pos &y) const;
  /**< returns the location of this view relatively to its <b>parent</b>.
   * Note: returns 0 if this view is a WINDOW View.
   */

  u_pos getXwin() const {return x;}
  u_pos getYwin() const {return y;}
  bool whereInWin(u_pos &x, u_pos &y) const;
  /**< returns the location of this view relatively to its enclosing <b>window</b>.
   * Notes: 
   * - returns 0 if this view is a WINDOW View.
   * - as layout is performed automatically, there is no method for changing
   *   the x,y position directly (use the 'UPos' object for this purpose) 
   */

  u_pos getXscreen() const;
  u_pos getYscreen() const;
  bool whereOnScreen(u_pos& x, u_pos& y) const;
  ///< returns the location of this view in screen coordinate.
 
  u_pos getXappli() const;
  u_pos getYappli() const;
  bool whereOnAppli(u_pos& x, u_pos& y) const;
  ///< returns the location of this view in the UAppli coordinate (= relatively to the main frame).

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // conversions

  static bool convertPos(UView* to_view, u_pos& to_x, u_pos& to_y,
			 UView* from_view, u_pos from_x, u_pos from_y);
  ///< converts position in 'from_view' to position in 'to_view'.

  u_pos XToXwin(u_pos x_in_view) const {return x_in_view + x;}
  u_pos YToYwin(u_pos y_in_view) const {return y_in_view + y;}
  ///< converts a position in view coordinates to window coordinates.

  u_pos XwinToX(u_pos x_in_win) const {return x_in_win - x;}
  u_pos YwinToY(u_pos y_in_win) const {return y_in_win - y;}
  ///< convert a position in window coordinates to view coordinates.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  static  UViewStyle style;  // renderer
  virtual UViewStyle* getViewStyle() {return &style;}
  ///< [impl] returns the current view style.

  class UNatWin* getNatWin() const;
  class UWin*    getHardwin() const;
  class UView*   getHardwinView() const;
  ///< [impl] return native counterparts.

  class UWinGraph* getWinGraph() const {return wingraph;}
  class UWinGraph& wg() const {return *wingraph;}
  /**< [impl] return Window Graphics.
   * CAUTION: 
   * - clients should never use this data for drawing (nor try to delete it)!
   *  => use an UGraph object instead.
   */

#ifndef NO_DOC
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  friend class UViewLayoutImpl;
  friend class UViewUpdateImpl;
  friend class UBox;
  friend class UBoxLink;

  enum VMODES {
    /// INITIALIZING is set at the 1st update, INITIALIZED at the 2nd update
    /// (a combination of layout+update+layout is necessary for correct init)
    INITIALIZING = 1<<0,
    INITIALIZED  = 1<<1,

    /// this view has been damaged and must be repaint
    DAMAGED      = 1<<2,

    /// this view has a fixed width (resp. height)
    FIXED_WIDTH  = 1<<4,
    FIXED_HEIGHT = 1<<5,

    /// the children of this view have been realized (for win views only)
    REALIZED_CHILDREN = 1<<6,

    /// coordinates are obsolete because this (floating) view is being
    /// moved (NB: coords are updated by UView::doUpdate())
    OBS_COORDS = 1<<7

    // !BEWARE: no comma after last item!
  };

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  bool isVmode(int _vmodes) const {return ((vmodes & _vmodes) != 0);}
  ///< [impl] is ONE of these View modes verified?.

  bool allVmodes(int _vmodes) const {return ((vmodes & _vmodes) == _vmodes);}
  ///< [impl] are ALL these vmodes verified ?.

  int  getVmodes() const {return vmodes;}
  ///< [impl] returns an ORed combination of current modes.

  void setVmodes(int _vmodes, bool state);
  ///< [impl] sets View modes according to state.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  class UMargins* getBorderMargins() {return border_margins;}
  void setBorderMargins(class UMargins* f) {border_margins = f;}  
  ///< [impl] get/set the border frame of this view (NB: can be null).

  class UView* getNext() {return nextview;}
  void setNext(UView* v) {nextview = v;}
  ///< [impl] get/set the next view in the list (NB: can be null).

  class UViewExt* getExt() {return pext;}
  ///< [impl] get extensions (NB: can be null).

  class UBoxLink *getBoxLink() {return boxlink;}
  ///< [impl] get the corresponding box link.

  void setXwin(u_pos x_in_win);
  void setYwin(u_pos y_in_win);
  void setParentView(UView* _parview) {parview = _parview;}

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  // update view paint in VIEW coordinates:
  // - 'localRegion' is relative to View coordinates
  //   a null value means: "repaint the whole view"
  // - 'doubleBuffering' : paints in double buffering

  void updatePaint(URegion* localRegion = null, bool double_buffering = false);

  // update view paint in WINDOW coordinates:
  // - 'windowRegion' is relative to Window coordinates
  // - 'doubleBuffering' : paints in double buffering
  // - 'updateAllLayers' : paints all layers located beneath the view
  //   (this in necessary in case of transparent views)
  // - otherwise, if updateAllLayers is false : paints from 'firstPaintedLayer' 
  //   (if it is not null) then (always) paints 'this' view

  void updateWinPaint(const URegion& window_region, UView* firstPaintedLayer,
		      bool double_buffering, bool update_all_layers);

  // update internal data but don't repaint (must be called after scrolling
  // and in various cases for initializing the views' content)

  int updateWinData(const URegion& region);

  //Note: contains() modifies the mofifies clip
  virtual UView* contains(u_pos xmouse, u_pos y_mouse, URegion& clip);
  virtual UView* contains(UView* searchv, URegion& clip);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual bool doLayout(class UContext&, class UViewLayout&);
  virtual void doUpdate(class UContext&, URegion r, URegion clip, 
			class UViewUpdate&);

  // locateSource either searches:
  // - 'searchedView'
  // - the visible view where (e->xmouse, e->ymouse) is located
  // this function sets:  e->{source, sourceView, sourceProps, redrawnClip}

  virtual UView* locateSource(const UContext& parp, URegion clip, 
			      class UEvent&, const UView* searchedView,
			      class USourceProps);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void doLayout2(class UViewLayoutImpl&, UGroup*, UContext&,
			 class UViewLayout&);

  virtual void doUpdate2(class UViewUpdateImpl&, UGroup*, UContext&,
			 URegion& r, URegion& clip, class UViewUpdate&);

  virtual UView* locateSource2(UGroup*, const class UContext& parp, URegion clip,  
			       class UEvent&, const UView* searchedView,
			       class USourceProps&);

  virtual bool locateElemV(UContext&, class ULink*, class UWinGraph&, 
			   const URegion&, class UViewUpdate&);
  virtual bool locateElemH(UContext&, class ULink*, class UWinGraph&, 
			   const URegion&, class UViewUpdate&);
  virtual bool locateElemRef(UContext&, class ULink*, class UWinGraph&, 
			     const URegion&, class UViewUpdate&);

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

protected:
  class UView     *parview; 	   // parent view
  class UBoxLink  *boxlink;	   // corresponding Box Link
  class UWinGraph *wingraph;	   // corresponding Window Graphics
  class UMargins  *border_margins; // border margins of this view (if any)
  class UViewExt  *pext;	   // view extensions (if any)
  class UView     *nextview;	   // next view in the list
  int vmodes;			   // modes of this view
public:
  int edit_shift;
  u_dim chwidth, chheight;	// size occupied by children
  unsigned short hflex_count, vflex_count; // number of horiz and vert flexible children
#endif
};

// ==================================================== [Elc:03] ======= //
// ==================================================== ======== ======= //

class UFlowView: public UView {
public:
  static  UViewStyle style;  // renderer
  virtual UViewStyle* getViewStyle() {return &style;}
  
  // "static" constructor used by UViewStyle to make a new view
  static UView* makeView(UBoxLink*, UView* parview, UWinGraph*);
  UFlowView(UBoxLink*, UView* parview, UWinGraph*);

  virtual ~UFlowView();

#ifndef NO_DOC
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  class UFlowLine *lines;
  class UFlowCell *cells;
  int line_count, cell_count, lastline_strcell;
  int alloc_line_count, alloc_cell_count;

  virtual void getHints(UContext &parctx, class UViewLayout &vl);
  virtual bool doLayout(UContext&, class UViewLayout &vl);
  virtual void doUpdate(UContext&, URegion r, URegion clip, class UViewUpdate&);
  //NB: NOT virtual:
  bool locateElemPos(UContext&, ULink*, class UFlowCell*,
		     class UWinGraph&, const URegion&, 
		     class UViewUpdate &uc);
  bool locateElemPtr(UContext&, ULink*, class UFlowCell*,
		     class UWinGraph&, const URegion&, 
		     class UViewUpdate &uc);
#endif
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */

