# command.tcl
# $Id: command.tcl,v 1.16 1999/07/31 00:33:37 chris Exp $
# Handles the various ed commands.
#
#
# XED --- an X-enabled implementation of ed, the standard editor.
# Copyright (C) 1999  Chris Laas
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# Chris Laas
# golem@mit.edu
# 22 Pratt Court
# Cohasset, MA 02025 USA


set default_filename ""
set current_addr 0
set last_addr 0
set dirty 0
set input_mode command
set confirm ""
set last_error ""
array set subst {global 0 pflags {}}


# Line printing subsystem.
for {set i 1} {$i < 256} {incr i} {
	set p [format {\%03o} $i]
	set unambiguous([subst -nocommands -novariables $p]) $p
}
for {set i 32} {$i < 127} {incr i} {
	set c [subst -nocommands -novariables [format {\%03o} $i]]
	set unambiguous($c) $c
}
foreach special {a b f n r t v} {
	set unambiguous([subst -nocommands -novariables \\$special]) \\$special
}
set unambiguous(\\) "\\\\"
proc print-lines {addr1 addr2 flags} {
	global unambiguous current_addr

	if {$addr1 == {}} { set addr1 $current_addr }
	if {$addr2 == {}} { set addr2 $addr1 }

	set f_p 0
	set f_l 0
	set f_n 0
	while {$flags != {}} {
		switch -exact [string index $flags 0] \
				p { set f_p 1 } l { set f_l 1 } n { set f_n 1 } \
				default { error "Invalid command suffix" }
		set flags [string range $flags 1 end]
	}
	if {!$f_p && !$f_l && !$f_n} { return }

	set out {}
	set n [expr $addr1-1]

	set lines [buffer-get $addr1 $addr2]
	if {$lines == {}} { set lines {{}} } else { set lines [split $lines \n] }
	foreach line $lines {
		if $f_l {
			set newline {}
			foreach c [split [string trimright $line \n] {}] {
				append newline $unambiguous($c)
			}
			set line $newline\$
		}
		if $f_n {
			append out [format "%-7d %s\n" [incr n] $line]
		} else {
			append out $line\n
		}
	}
	puts-response [string trimright $out \n]
}



# The a command.
proc command-append {addr1 addr2 command line} {
	global input_mode current_addr
	if {[string tolower $line] == "bout"} { easter-egg ; return }
	set current_addr $addr2
	set input_mode "text $input_mode"
	vwait input_mode
	print-lines {} {} $line
}
proc easter-egg {} {
	global M1 M2 G L K y1 y2 v1 v2 xe entry_line old_entry_line VERSION \
		line_source
	set about_message "Xed version $VERSION    by Chris Laas <golem@mit.edu>    Copyright (c) 1999"

	if {$line_source == "stdin"} { puts-response $about_message ; return }

	set M1 10
	set M2 0.01
	set G -0.0003
	set L 0.8
	set K 0.01
	
	set M1 1
	set M2 0.2
	set G -0.0003
	set L 0.6
	set K 0.002

	set y1 1.0
	set y2 [expr $y1 - $L]
	set v1 0.0
	set v2 0.0
	
	set xe 0.0

	proc egg-step {} {
		global M1 M2 G L K y1 y2 v1 v2 xe
		set new_y1 [expr $y1 + $v1]
		set new_y2 [expr $y2 + $v2]
		set new_y1 [expr $new_y1 >= 1.0 ? 1.0 : $new_y1 <= 0.0 ? 0.0 : $new_y1]
		set new_y2 [expr $new_y2 >= 1.0 ? 1.0 : $new_y2 <= 0.0 ? 0.0 : $new_y2]
		set y1 $new_y1
		set y2 $new_y2
		if {$new_y1 == 0.0 || $new_y1 == 1.0} { set v1 0.0 }
		if {$new_y2 == 0.0 || $new_y2 == 1.0} { set v2 0.0 }
		set springf [expr $K * ($L - abs($y2 - $y1)) * ($y1 > $y2 ? 1.0 : -1.0)]
		set v1 [expr $v1 + $G + $springf / $M1]
		set v2 [expr $v2 + $G - $springf / $M2]
	
		.ysb set [expr 1.0 - $y1] [expr 1.0 - $y2]

		set xe [expr $xe - ($v1 < 0.0 ? $v1 / 8.0 : $v1 / 16.0)]
		.e xview moveto $xe
	}

	foreach child [winfo children .] {
		bindtags $child [concat egg_wait [bindtags $child]]
	}
	
	set old_entry_line $entry_line
	set entry_line {                                                                                                                                                                }
	set l [.e xview]
	set entry_line [string range $entry_line 0 [expr int(([lindex $l 1] - [lindex $l 0]) * [string length $entry_line])]]
	append entry_line $about_message$entry_line

	proc egg-tick {} { egg-step ; global id ; set id [after 10 egg-tick] }
	egg-tick

	set script {
		after cancel $id
		set entry_line $old_entry_line
		eval ".ysb set [.t yview]"
		foreach child [winfo children .] {
			bindtags $child [lrange [bindtags $child] 1 end]
		}
	}
	bind egg_wait <Key>    $script
	bind egg_wait <Button> $script
}

# The c command.
proc command-change {addr1 addr2 command line} {
	command-delete $addr1 $addr2 d {}
	command-append [expr $addr1-1] [expr $addr1-1] a $line
}

# The d command.
proc command-delete {addr1 addr2 command line} {
	global current_addr last_addr cut_buffer dirty
	if {$addr1 == 0} { error "Address out of range, cannot delete address 0" }

	set cut_buffer [buffer-get $addr1 $addr2]

	buffer-delete $addr1 $addr2
	incr last_addr [expr $addr1-$addr2-1]
	set current_addr $addr1
	if {$current_addr > $last_addr} { set current_addr $last_addr }
	set dirty 1
	print-lines {} {} $line
}

# The e, E, and r commands.
proc command-edit-or-read-file {addr1 addr2 command line} {
	global is_tclet default_filename current_addr last_addr dirty confirm

	if $is_tclet { error "Reading files is not permitted in a Tclet" }

	if {$dirty && $command == "e"} {
		if {$confirm == "command-edit-or-read-file"} {
			set confirm ""
		} else {
			set confirm "command-edit-or-read-file"
			error "Warning: file modified"
		}
	}

	if {$line == ""} {
		set file $default_filename
	} elseif {[string match |* $line]} {
		set file \\$line
	} elseif {[string match !* $line]} {
		set file [list |sh -c [string range $line 1 end]]
	} else {
		set file $line
	}

	set f [open $file r]
	if {$command == "e" || $command == "E"} {
		buffer-clear
		set addr2 0
		set last_addr 0
	} elseif {$command == "r"} {
		# Insert at address.
	} else {
		error "Internal error"
	}
	set addr $addr2
	set bytecount 0
	while {![eof $f]} {
		set fileline [gets $f]
		# Ignore a trailing empty line.
		if {$fileline == "" && [eof $f]} { break }
		incr bytecount [string length $fileline]
		# A byte for the newline.
		incr bytecount
		buffer-put $addr $fileline
		incr addr
	}
	close $f

	set current_addr $addr
	incr last_addr [expr $addr - $addr2]
	if {$command == "e" || $command == "E"} {
		set dirty 0
		if {![string match !* $line]} { set default_filename $file }
	} elseif {$command == "r"} {
		set dirty 1
		if {![string match !* $line] && $default_filename == ""} {
			set default_filename $file
		}
	} else {
		error "Internal error"
	}

	puts-response $bytecount
}

# The f command.
proc command-default-filename {addr1 addr2 command line} {
	global default_filename
	if {[string match |* $line]} { set line \\$line }
	if {$line != ""} { set default_filename $line }
	puts-response $default_filename
}

# The g, v, G, and V commands.
proc command-match {addr1 addr2 command line} {
	global current_addr input_mode continuation
	if {$addr1 == 0} {
		error "Address out of range, cannot search on address 0"
	}

	if {[string length $line] == 0} { error "Invalid pattern delimiter" }
	set regexp [regexp-compile [parse-pattern line delimiter]]

	if {$command == "g" || $command == "v"} {
		while {[regsub {\\$} $line "\n" line]} {
			set input_mode "continuation $input_mode"
			vwait input_mode
			append line $continuation
		}
		set commands [split $line \n]
	}

	set count 0
	if {$command == "g" || $command == "G"} {
		while {[set addr [buffer-search-forwards $regexp $addr1 $addr2]]!={}} {
			buffer-mark-set _[incr count] $addr
			set addr1 [expr $addr + 1]
		}
	} else {
		set next_match [buffer-search-forwards $regexp $addr1 $addr2]
		for {set addr $addr1} {$addr <= $addr2} {set addr $addr1} {
			if {$addr == $next_match} {
				set next_match [buffer-search-forwards $regexp $addr1 $addr2]
			} else {
				buffer-mark-set _[incr count] $addr
			}
			set addr1 [expr $addr + 1]
		}
	}

	if {$command == "g" || $command == "v"} {
		set input_mode "batchcommand $input_mode"
	}

	for {set i 1} {$i <= $count} {incr i} {
		if [catch { set current_addr [buffer-mark-get _$i] } error] {continue}
		if {$command == "g" || $command == "v"} {
			batch-process-lines $commands
		} else {
			print-lines {} {} p
			set input_mode "subcommand $input_mode"
			vwait input_mode
		}
	}

	if {$command == "g" || $command == "v"} {
		set input_mode [lrange $input_mode 1 end]
	}
}

# The H command.
proc command-toggle-errors {addr1 addr2 command line} {
	global verbose last_error
	if {$verbose} {
		set verbose 0
	} else {
		set verbose 1
		puts-error $last_error
	}
	print-lines {} {} $line
}

# The h command.
proc command-last-error {addr1 addr2 command line} {
	global last_error
	puts-error $last_error
	print-lines {} {} $line
}

# The i command.
proc command-insert {addr1 addr2 command line} {
	if {$addr2 == 0} {
		error "Address out of range, cannot insert before address 0"
	}
	incr addr2 -1
	command-append $addr2 $addr2 a $line
}

# The j command.
proc command-join {addr1 addr2 command line} {
	global current_addr last_addr cut_buffer dirty
	if {$addr1 == 0} { error "Address out of range, cannot join address 0" }
	set cut_buffer [buffer-get $addr1 $addr2]
	regsub -all \n $cut_buffer {} new_line
	buffer-delete $addr1 $addr2
	buffer-put [expr $addr1-1] $new_line
	incr last_addr [expr $addr1-$addr2]
	set current_addr $addr1
	set dirty 1
	print-lines {} {} $line
}

# The k command.
proc command-mark {addr1 addr2 command line} {
	buffer-mark-set [string index $line 0] $addr2
}

# The m command.
proc command-move {addr1 addr2 command line} {
	global current_addr last_addr dirty
	if {$addr1 == 0} {
		error "Address out of range, cannot move from address 0"
	}
	read-addr-range $line line daddr1 daddr2
	if {$daddr2 == {}} { set daddr2 $current_addr }
	set text [buffer-get $addr1 $addr2]
	buffer-delete $addr1 $addr2
	if {$daddr2 < $addr1} {
		buffer-put $daddr2 $text
		set current_addr [expr $daddr2+1+$addr2-$addr1]
	} elseif {$daddr2 >= $addr2} {
		buffer-put [expr $daddr2-1-$addr2+$addr1] $text
		set current_addr $daddr2
	} else {
		error "Invalid destination, lies in source range"
	}
	set dirty 1
	print-lines {} {} $line
}

# The p, l, and n commands.
proc command-print {addr1 addr2 command line} {
	global current_addr
	print-lines $addr1 $addr2 $command$line
	set current_addr $addr2
}

# The P command.
proc command-toggle-prompt {addr1 addr2 command line} {
	global prompt
	if {$prompt == ""} {
		set prompt *
	} else {
		set prompt ""
	}
	print-lines {} {} $line
}

# The q and Q commands.
proc command-quit {addr1 addr2 command line} {
	global dirty confirm
	if {$dirty && $command == "q"} {
		if {$confirm == "command-quit"} {
			set confirm ""
		} else {
			set confirm "command-quit"
			error "Warning: file modified"
		}
	}
	exit
}

# The s command.
proc command-substitute {addr1 addr2 command line} {
	global current_addr last_addr cut_buffer dirty
	global subst
	if {$addr1 == 0} {
		error "Address out of range, cannot substitute on address 0"
	}

	# Detect the second (repeat last substitution) form of the command.
	if {$line == {} || [regexp {^[gprln0-9]} $line]} {
		if {![info exists subst(pat)]} { error "No previous substitution" }
		set sgnum 0
		while {$line != {}} {
			if {![regexp {^([gprln]|[0-9]+)(.*)} $line junk flag line]} {
				error "Invalid command suffix"
			}
			switch $flag {
				g {
					set sgnum 0
					set subst(global) [expr !$subst(global)]
				}
				r { error "r suffix unimplemented --- can't figure out what it's supposed to do." }
				p { set subst(pflags) [expr {$subst(pflags)=={} ? "p" : {}}] }
				l - n { append subst(pflags) p$flag }
				default { set sgnum $flag }
			}
		}
	} else {
		if {[string length $line] == 0} { error "Invalid pattern delimiter" }
		set subst(pat) [regexp-compile [parse-pattern line delimiter]]
		set subst(template) [parse-subst-template line $delimiter]
		set sgnum 0
		set subst(pflags) {}
		set subst(global) 0
		if {$line == {}} {
			set subst(pflags) p
		} else {
			set line [string range $line 1 end]
			while {$line != {}} {
				if {![regexp {^([gpln]|[0-9]+)(.*)} $line junk flag line]} {
					error "Invalid command suffix"
				}
				switch $flag {
					g {
						set sgnum 0
						set subst(global) 1
					}
					p - l - n { append subst(pflags) $flag }
					default { set sgnum $flag }
				}
			}
		}
	}

	while {[set addr [buffer-search-forwards $subst(pat) $addr1 $addr2]]!={}} {
		set text [buffer-get $addr $addr]
		set bufr $text
		set bufl {}
		for {set i 1} {$i < $sgnum} {incr i} {
			if ![regexp -indices -- $subst(pat) $bufr indices] { break }
			append bufl [string range $bufr 0 [lindex $indices 1]]
			set bufr [string range $bufr [expr [lindex $indices 1]+1] end]
		}
		if {$i < $sgnum} { continue }

		if {$bufr == {}} {
			# The regsub command deals incorrectly with an empty input:
			# for example, the pattern ^ doesn't match the empty input.
			# Thus, we special-case it.
			set result [regexp -- $subst(pat) {}]
			if $result {
				# It matched, as we expected.  Set $bufr to $subst(template),
				# replacing "&" and "\n" (where n==0..9) with the empty string.
				regsub -all -- {&|\\[0-9]} $subst(template) {} bufr
			}
		} else {
			if $subst(global) {
				set result [regsub -all -- $subst(pat) $bufr $subst(template) bufr]
			} else {
				set result [regsub      -- $subst(pat) $bufr $subst(template) bufr]
			}
		}
		if {$sgnum == 0 && $result == 0} {
			error "Internal error: false regexp match"
		}

		set buffer $bufl$bufr
		buffer-delete $addr $addr
		buffer-put [expr $addr-1] $buffer

		if {$buffer == {}} {
			set newlines 0
		} else {
			set newlines [expr [llength [split $buffer \n]] - 1]
		}
		incr addr1 $newlines
		incr addr2 $newlines
		incr last_addr $newlines

		set cut_buffer $text
		set dirty 1
		set current_addr $addr

		set addr1 [expr $addr + 1]
	}

	print-lines {} {} $subst(pflags)
}

# The t command.
proc command-transfer {addr1 addr2 command line} {
	global current_addr last_addr dirty
	if {$addr1 == 0} {
		error "Address out of range, cannot transfer from address 0"
	}
	read-addr-range $line line daddr1 daddr2
	if {$daddr2 == {}} { set daddr2 $current_addr }
	buffer-put $daddr2 [buffer-get $addr1 $addr2]
	set lines [expr $addr2-$addr1+1]
	incr last_addr $lines
	set current_addr [expr $daddr2+$lines]
	set dirty 1
	print-lines {} {} $line
}

# The u command.
proc command-undo {addr1 addr2 command line} {
	global last_undo_commands input_mode

	set input_mode "batchcommand $input_mode"
	batch-process-lines $last_undo_commands
	set input_mode [lrange $input_mode 1 end]

	print-lines {} {} $line
}

# The w and W commands.
proc command-write-file {addr1 addr2 command line} {
	global is_tclet default_filename current_addr last_addr confirm dirty
	global backup_files

	if $is_tclet { error "Writing files is not permitted in a Tclet" }

	set quit 0
	if {$line == "q" || [string match "q\[ \t\n\]*" $line]} {
		set quit 1
		set line [string trim [string range $line 2 end]]
	}

	if {$line == ""} {
		set file $default_filename
	} elseif {[string match |* $line]} {
		set file \\$line
	} elseif {[string match !* $line]} {
		set file [list |sh -c [string range $line 1 end]]
	} else {
		set file $line
	}

	if {$command == "W"} {
		set mode a
	} elseif {$command == "w"} {
		set mode w
	} else {
		error "Internal error"
	}

	catch {
		if {$backup_files && [file isfile $file]} {
			if {![file exists $file~]   && [file writable [file dirname $file]]
				|| [file isfile $file~] && [file writable $file~]} {
				file rename -force -- $file $file~
				file copy   -force -- $file~ $file
			}
		}
	}

	set f [open $file $mode]
	set text [buffer-get $addr1 $addr2]
	puts $f $text
	close $f

	if {$default_filename == ""} {
		if {[string match |* $line]} { set line \\$line }
		if {$line != ""} { set default_filename $line }
	}

	set dirty 0

	puts-response [string length $text]

	if $quit { exit }
}

# The x command.
proc command-put-cut-buffer {addr1 addr2 command line} {
	global cut_buffer current_addr last_addr dirty
	if {![info exists cut_buffer]} { error "Nothing to put" }
	buffer-put $addr2 $cut_buffer
	set lines [llength [split $cut_buffer \n]]
	if {$lines == 0} { set lines 1 }
	set current_addr [expr $addr2+$lines]
	incr last_addr $lines
	set dirty 1
	print-lines {} {} $line
}

# The y command.
proc command-yank-cut-buffer {addr1 addr2 command line} {
	global cut_buffer
	set cut_buffer [buffer-get $addr1 $addr2]
	print-lines {} {} $line
}

# The z command.
proc command-scroll {addr1 addr2 command line} {
	global line_source current_addr last_addr
	if {![regexp {^([0-9]+)(.*)} $line junk count line]} {
		switch $line_source {
			stdin {
				set stty [exec stty -a]
				if {![regexp {rows[ \t]*=[ \t]*([0-9]+)} $stty junk count]} {
					set count 24
				}
				# Take into account the last line of the terminal will be blank.
				set count [expr $count - 1]
			}
			X {
				set count 1
			}
		}
	}
	set addr1 $addr2
	if {$addr2+$count-1 > $last_addr} {
		set current_addr $last_addr
	} else {
		set current_addr [expr $addr2+$count-1]
	}
	print-lines $addr1 $current_addr p$line
}

# The ! command.
proc command-shell {addr1 addr2 command line} {
	global is_tclet last_shell_command default_filename

	if $is_tclet { error "Executing programs is not permitted in a Tclet" }

	if {[string index $line 0] == "!"} {
		if [catch {set line $last_shell_command[string range $line 1 end]}] {
			error "No previous shell command"
		}
	}
	regsub {%} $line $default_filename line
	exec sh -c $line <@stdin >@stdout 2>@stderr
	puts-response !
}

# The # command.
proc command-comment {addr1 addr2 command line} {
	# Does absolutely nothing.
}

# The = command.
proc command-print-line-number {addr1 addr2 command line} {
	puts-response $addr2
	print-lines {} {} $line
}

# The null command.  (When nothing but an address is typed.)
proc command-print-line {addr1 addr2 command line} {
	global current_addr last_addr
	if {$addr2 == 0} { error "Address out of range, cannot print address 0" }
	set current_addr $addr2
	puts-response [buffer-get $current_addr $current_addr]
}

# The ~ command, which is used in the undo system, and which I
# use for debugging.
proc command-eval-tcl {addr1 addr2 command line} {
	puts-response [uplevel #0 $line]
}
