(******************************************************************************)
(*                               OCamlPro                                     *)
(*                                                                            *)
(* Copyright 2013-2015 OCamlPro                                               *)
(* All rights reserved. See accompanying files for the terms under            *)
(* which this file is distributed. In doubt, contact us at                    *)
(* contact@ocamlpro.com (http://www.ocamlpro.com/)                            *)
(*                                                                            *)
(******************************************************************************)

open Format

module Timer = struct
  let counter = ref 0.0
  let go () = counter := (Unix.times()).Unix.tms_cutime
  let value () = (Unix.times()).Unix.tms_cutime -. !counter
end

let debug = false
let silent = true

let sigs =
  [
    Sys.sigabrt;
    Sys.sigalrm;
    Sys.sigfpe;
    Sys.sighup;
    Sys.sigill;
    Sys.sigint;
    Sys.sigkill;
    Sys.sigpipe;
    Sys.sigquit;
    Sys.sigsegv;
    Sys.sigterm;
    Sys.sigusr1;
    Sys.sigusr2;
    Sys.sigcont;
    Sys.sigstop;
    Sys.sigtstp;
    Sys.sigttin;
    Sys.sigttou;
    Sys.sigvtalrm;
    Sys.sigprof
  ]

(* let _ =  *)
(*   List.iter *)
(*     (fun sign -> *)
(*       try *)
(*         Sys.set_signal sign *)
(*           (Sys.Signal_handle  *)
(*              (fun _ -> eprintf "Here: %d @." sign)) *)
(*       with Sys_error("Invalid argument") -> () *)
(*     )sigs *)


let timeout = ref 0

let satML_plugin = ref ""

let usage code =
  printf
    "usage: a2exp <%s> <%s> <%s> <%s>@."
    "timeout"
    "path-to-satML-plugin (or \"\" to deactivate it)"
    "path-to-alt-ergo"
    "options to alt-ergo";
  exit code

let timeout =
  try
    match Sys.argv.(1) with
      | "-help" -> usage 0
      | _ ->
        try
          let v = int_of_string Sys.argv.(1) in
          if v <= 0 then
            begin
              eprintf "Error: <timeout> should be a positive integer@.";
              exit 1
            end;
          v
        with Failure "int_of_string" ->
          eprintf
            "Failed to parse the value \"%s\" of timeout@." Sys.argv.(1);
          usage 1;
  with Invalid_argument("index out of bounds") ->
    eprintf "Timeout not given@.";
    usage 1

let satML_plugin =
  try match Sys.argv.(2) with
    | "-help" -> usage 0
    | s -> s
  with Invalid_argument("index out of bounds") ->
    eprintf "Path to satML not given@.";
    usage 1

let command =
  let tmp = ref "" in
  for i = 3 to Array.length Sys.argv - 1 do
    tmp := sprintf "%s %s" !tmp Sys.argv.(i)
  done;
  !tmp

let file_out =
  let len = Array.length Sys.argv in
  let f = ref None in
  let i = ref 0 in
  while !i < len do
    let s = Sys.argv.(!i) in
    if Filename.check_suffix s ".mlw" || Filename.check_suffix s ".why"  then
      begin
        f := Some (Filename.basename s);
        i := len
      end
    else incr i
  done;
  match !f with
  | None -> eprintf "No .mlw or .why file detected!@."; usage 1
  | Some f -> sprintf "/tmp/.%s_____.stdout" f


let strategies =
  if satML_plugin = "" then
    [
      1, "-nb-triggers 1 -no-Ematching";
      2, "-nb-triggers 1";
      3, "-nb-triggers 1 -greedy";
      4, "-nb-triggers 2 -greedy";
      5, "-nb-triggers 10"
    ]
  else
    [
      1, "-nb-triggers 1 -no-Ematching";
      2, "-nb-triggers 1";
      3, "-nb-triggers 1 -no-Ematching -sat-plugin " ^ satML_plugin;
      4, "-nb-triggers 1 -sat-plugin " ^ satML_plugin;
      5, "-nb-triggers 1 -greedy";
      6, "-nb-triggers 2 -greedy";
      7, "-nb-triggers 10"
    ]


exception Timeout of string
exception Unknown of string
exception Valid of int * string * string

let valid_reg = Str.regexp "Valid"
let timeout_reg = Str.regexp "Timeout"
let unknown_reg = Str.regexp "I don't know"

let check_valid s id cmd strat =
  try
    ignore (Str.search_forward valid_reg s 0);
    if debug then eprintf "OUT says valid : %s@." s;
    if silent then raise (Valid (id, strat, s));
    raise (Valid (id, cmd, s))
  with Not_found -> ()


let check_timeout s =
  try
    ignore (Str.search_forward timeout_reg s 0);
    if debug then eprintf "OUT says timeout : %s@." s;
    raise (Timeout s)
  with Not_found -> ()

let check_unknown s =
  try
    ignore (Str.search_forward unknown_reg s 0);
    if debug then eprintf "OUT says unknown : %s@." s;
    raise (Unknown s)
  with Not_found -> ()

let parse_result id cmd strat result =
  if debug then begin
    match result with
      | Unix.WEXITED n -> eprintf "EXITED %d : " n;
      | Unix.WSIGNALED n -> eprintf "SIGNALED %d : " n;
      | Unix.WSTOPPED n -> eprintf "STOPPED %d : " n
  end;
  let cin = open_in file_out in
  begin
    try
      let s = input_line cin in
      check_valid s id cmd strat;
      check_unknown s;
      check_timeout s;
      if debug then eprintf "OUT: %s@." s;
      close_in cin
    with End_of_file -> close_in cin
    | e -> close_in cin; raise e

  end

let try_strategy timeout id strat =
  let cmd0 = sprintf "%s %s" command strat in
  let cmd = sprintf "%s -timelimit %d > %s" cmd0 timeout file_out in
  if debug then eprintf "@.I'll try: %s@." cmd;
  try parse_result id cmd0 strat (Unix.system cmd)
  with Unix.Unix_error(_, "waitpid", "") -> eprintf "LA1@."; exit 124

let main () =
  try
    let acc = ref 0. in
    let nb_lstrats = List.length strategies in
    let split_to = timeout / (nb_lstrats + 1) in
    if split_to > 0 then
      begin
        List.iter
          (fun (id,st) ->
            begin
              try
                Timer.go ();
                try_strategy split_to id st;
                acc := !acc +. (Timer.value ())
              with
                | Timeout _ -> acc := !acc +. (float_of_int split_to)
                | Unknown _ -> acc := !acc +. (Timer.value ())
            end;
          ) strategies;
        if debug then eprintf "used to %2.4f@." !acc;
        let unused_to = (float_of_int (split_to * nb_lstrats)) -. !acc in
        if debug then eprintf "unused to %2.4f@." unused_to;
        let unused_to = int_of_float unused_to  in
        if debug then eprintf "%d@." unused_to;
        try_strategy
          (timeout - nb_lstrats * split_to + unused_to)
          (1 + nb_lstrats)
          ""
      end
    else
      begin
        eprintf "Warning: timeout very small. No strategy will be tried@.";
        try_strategy timeout (1 + nb_lstrats) ""
      end
  with

    | Unknown s ->
      printf "%s@." s;
      exit 0

    | Timeout s ->
      printf "%s@." s;
      exit 124

    | Valid (id, cmd, s) ->
      if not debug then eprintf "@.success with %d: %s@." id cmd;
      printf "%s@." s;
      exit 0

let _ = main ()

