/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Seasstat   seasrange       Seasonal range
      Seasstat   seasmin         Seasonal minimum
      Seasstat   seasmax         Seasonal maximum
      Seasstat   seassum         Seasonal sum
      Seasstat   seasmean        Seasonal mean
      Seasstat   seasavg         Seasonal average
      Seasstat   seasvar         Seasonal variance
      Seasstat   seasvar1        Seasonal variance [Normalize by (n-1)]
      Seasstat   seasstd         Seasonal standard deviation
      Seasstat   seasstd1        Seasonal standard deviation [Normalize by (n-1)]
*/

#include <cdi.h>

#include "cdo_int.h"
#include "pstream_int.h"
#include "datetime.h"
#include "printinfo.h"
#include "cdo_season.h"

void *
Seasstat(void *process)
{
  TimeStat timestat_date = TimeStat::MEAN;
  int64_t vdate0 = 0, vdate1 = 0;
  int vtime0 = 0, vtime1 = 0;
  int nrecs;
  int varID, levelID;
  int year, month, day, seas0 = 0;
  size_t nmiss;
  int oldmon = 0;
  int nseason = 0;
  const char *seas_name[4];

  cdoInitialize(process);

  // clang-format off
  cdoOperatorAdd("seasrange", func_range, 0, nullptr);
  cdoOperatorAdd("seasmin",   func_min,   0, nullptr);
  cdoOperatorAdd("seasmax",   func_max,   0, nullptr);
  cdoOperatorAdd("seassum",   func_sum,   0, nullptr);
  cdoOperatorAdd("seasmean",  func_mean,  0, nullptr);
  cdoOperatorAdd("seasavg",   func_avg,   0, nullptr);
  cdoOperatorAdd("seasvar",   func_var,   0, nullptr);
  cdoOperatorAdd("seasvar1",  func_var1,  0, nullptr);
  cdoOperatorAdd("seasstd",   func_std,   0, nullptr);
  cdoOperatorAdd("seasstd1",  func_std1,  0, nullptr);

  const int operatorID = cdoOperatorID();
  const int operfunc = cdoOperatorF1(operatorID);

  const bool lrange  = operfunc == func_range;
  const bool lmean   = operfunc == func_mean || operfunc == func_avg;
  const bool lstd    = operfunc == func_std || operfunc == func_std1;
  const bool lvarstd = operfunc == func_std || operfunc == func_var || operfunc == func_std1 || operfunc == func_var1;
  const int  divisor = operfunc == func_std1 || operfunc == func_var1;
  // clang-format on
  const bool lvars2 = lvarstd || lrange;

  const int season_start = get_season_start();
  get_season_name(seas_name);

  const int streamID1 = cdoStreamOpenRead(cdoStreamName(0));

  const int vlistID1 = cdoStreamInqVlist(streamID1);
  const int vlistID2 = vlistDuplicate(vlistID1);

  const int taxisID1 = vlistInqTaxis(vlistID1);
  const int taxisID2 = taxisDuplicate(taxisID1);
  taxisWithBounds(taxisID2);
  if (taxisInqType(taxisID2) == TAXIS_FORECAST) taxisDefType(taxisID2, TAXIS_RELATIVE);
  vlistDefTaxis(vlistID2, taxisID2);

  const int streamID2 = cdoStreamOpenWrite(cdoStreamName(1));
  pstreamDefVlist(streamID2, vlistID2);

  const int maxrecs = vlistNrecs(vlistID1);
  std::vector<RecordInfo> recinfo(maxrecs);

  DateTimeList dtlist;
  dtlist.setStat(timestat_date);
  dtlist.setCalendar(taxisInqCalendar(taxisID1));

  const size_t gridsizemax = vlistGridsizeMax(vlistID1);

  Field field;
  fieldInit(field);
  field.ptr = (double *) Malloc(gridsizemax * sizeof(double));

  FieldVector2D samp1, vars1, vars2;
  fieldsFromVlist(vlistID1, samp1, FIELD_NONE);
  fieldsFromVlist(vlistID1, vars1, FIELD_PTR);
  if (lvars2) fieldsFromVlist(vlistID1, vars2, FIELD_PTR);

  int tsID = 0;
  int otsID = 0;
  while (true)
    {
      long nsets = 0;
      bool newseas = false;
      while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
        {
          dtlist.taxisInqTimestep(taxisID1, nsets);
          int64_t vdate = dtlist.getVdate(nsets);
          int vtime = dtlist.getVtime(nsets);

          cdiDecodeDate(vdate, &year, &month, &day);

          int newmon = month;

          if (season_start == START_DEC && newmon == 12) newmon = 0;

          int seas = month_to_season(month);

          if (nsets == 0)
            {
              nseason++;
              vdate0 = vdate;
              vtime0 = vtime;
              seas0 = seas;
              oldmon = newmon;
            }

          if (newmon < oldmon) newseas = true;

          if ((seas != seas0) || newseas) break;

          oldmon = newmon;

          for (int recID = 0; recID < nrecs; recID++)
            {
              pstreamInqRecord(streamID1, &varID, &levelID);

              if (tsID == 0)
                {
                  recinfo[recID].varID = varID;
                  recinfo[recID].levelID = levelID;
                  recinfo[recID].lconst = vlistInqVarTimetype(vlistID1, varID) == TIME_CONSTANT;
                }

              Field &rsamp1 = samp1[varID][levelID];
              Field &rvars1 = vars1[varID][levelID];

              const size_t gridsize = rvars1.size;

              if (nsets == 0)
                {
                  pstreamReadRecord(streamID1, rvars1.ptr, &nmiss);
                  rvars1.nmiss = nmiss;
                  if (lrange)
                    {
                      vars2[varID][levelID].nmiss = rvars1.nmiss;
                      for (size_t i = 0; i < gridsize; i++) vars2[varID][levelID].ptr[i] = rvars1.ptr[i];
                    }

                  if (nmiss > 0 || rsamp1.ptr)
                    {
                      if (rsamp1.ptr == nullptr) rsamp1.ptr = (double *) Malloc(gridsize * sizeof(double));

                      for (size_t i = 0; i < gridsize; i++) rsamp1.ptr[i] = !DBL_IS_EQUAL(rvars1.ptr[i], rvars1.missval);
                    }
                }
              else
                {
                  pstreamReadRecord(streamID1, field.ptr, &nmiss);
                  field.nmiss = nmiss;
                  field.grid = rvars1.grid;
                  field.missval = rvars1.missval;

                  if (field.nmiss > 0 || rsamp1.ptr)
                    {
                      if (rsamp1.ptr == nullptr)
                        {
                          rsamp1.ptr = (double *) Malloc(gridsize * sizeof(double));
                          for (size_t i = 0; i < gridsize; i++) rsamp1.ptr[i] = nsets;
                        }

                      for (size_t i = 0; i < gridsize; i++)
                        if (!DBL_IS_EQUAL(field.ptr[i], rvars1.missval)) rsamp1.ptr[i]++;
                    }

                  if (lvarstd)
                    {
                      farsumq(vars2[varID][levelID], field);
                      farsum(rvars1, field);
                    }
                  else if (lrange)
                    {
                      farmin(vars2[varID][levelID], field);
                      farmax(rvars1, field);
                    }
                  else
                    {
                      farfun(rvars1, field, operfunc);
                    }
                }
            }

          if (nsets == 0 && lvarstd)
            for (int recID = 0; recID < maxrecs; recID++)
              {
                if (recinfo[recID].lconst) continue;

                const int varID = recinfo[recID].varID;
                const int levelID = recinfo[recID].levelID;

                farmoq(vars2[varID][levelID], vars1[varID][levelID]);
              }

          vdate1 = vdate;
          vtime1 = vtime;
          nsets++;
          tsID++;
        }

      if (nrecs == 0 && nsets == 0) break;

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (recinfo[recID].lconst) continue;

          const int varID = recinfo[recID].varID;
          const int levelID = recinfo[recID].levelID;
          Field &rsamp1 = samp1[varID][levelID];
          Field &rvars1 = vars1[varID][levelID];

          if (lmean)
            {
              if (rsamp1.ptr)
                fardiv(rvars1, rsamp1);
              else
                farcdiv(rvars1, (double) nsets);
            }
          else if (lvarstd)
            {
              Field &rvars2 = vars2[varID][levelID];
              if (rsamp1.ptr)
                {
                  if (lstd)
                    farstd(rvars1, rvars2, rsamp1, divisor);
                  else
                    farvar(rvars1, rvars2, rsamp1, divisor);
                }
              else
                {
                  if (lstd)
                    farcstd(rvars1, rvars2, nsets, divisor);
                  else
                    farcvar(rvars1, rvars2, nsets, divisor);
                }
            }
          else if (lrange)
            {
              Field &rvars2 = vars2[varID][levelID];
              farsub(rvars1, rvars2);
            }
        }

      if (Options::cdoVerbose)
        {
          char vdatestr0[32], vtimestr0[32];
          char vdatestr1[32], vtimestr1[32];
          date2str(vdate0, vdatestr0, sizeof(vdatestr0));
          time2str(vtime0, vtimestr0, sizeof(vtimestr0));
          date2str(vdate1, vdatestr1, sizeof(vdatestr1));
          time2str(vtime1, vtimestr1, sizeof(vtimestr1));
          cdoPrint("season: %3d %3s  start: %s %s  end: %s %s ntimesteps: %ld", nseason, seas_name[seas0], vdatestr0, vtimestr0,
                   vdatestr1, vtimestr1, nsets);
        }

      dtlist.statTaxisDefTimestep(taxisID2, nsets);
      pstreamDefTimestep(streamID2, otsID);

      if (nsets < 3)
        {
          char vdatestr[32];
          date2str(vdate0, vdatestr, sizeof(vdatestr));
          cdoWarning("Season %3d (%s) has only %d input time step%s!", otsID + 1, vdatestr, nsets, nsets == 1 ? "" : "s");
        }

      for (int recID = 0; recID < maxrecs; recID++)
        {
          if (otsID && recinfo[recID].lconst) continue;

          const int varID = recinfo[recID].varID;
          const int levelID = recinfo[recID].levelID;
          Field &rvars1 = vars1[varID][levelID];

          pstreamDefRecord(streamID2, varID, levelID);
          pstreamWriteRecord(streamID2, rvars1.ptr, rvars1.nmiss);
        }

      if (nrecs == 0) break;
      otsID++;
    }

  fieldsFree(vlistID1, samp1);
  fieldsFree(vlistID1, vars1);
  if (lvars2) fieldsFree(vlistID1, vars2);

  if (field.ptr) Free(field.ptr);

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
