/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/util/record_vectors.ts.cpp
 *
 * $Id$
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta]
 */


#include <diagnostics/util/record_vectors.ts.hpp>

#include <diagnostics/unittest/test_exception.hpp>

#define PUSH_RECORD(STR_WHAT,LEVEL,TYPE) result.push_back(CSTR_RECORD(STR_WHAT,LEVEL,TYPE))

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;

#define PUSH_RECORD(STR_WHAT,LEVEL,TYPE) result.push_back(CSTR_RECORD(STR_WHAT,LEVEL,TYPE))

////////////////////////////////////////////////////////////////////////////////

Record_Vector* Empty_Trace_0sec_0usec::clone() const
{
    return new Empty_Trace_0sec_0usec;
}

::std::vector< ::diagnostics::Record> Empty_Trace_0sec_0usec::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    PUSH_RECORD("empty_test_case_0sec_0usec",LEVEL_TEST,TYPE_TESTCASE_ENTER);
    PUSH_RECORD("empty_test_case_0sec_0usec",LEVEL_TEST,TYPE_TESTCASE_EXIT);
    return result;
}

void Empty_Trace_0sec_0usec::check(::diagnostics::unittest::Test_Run_Result const & t,
				   bool const logging,
				   Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_COMPLETE);
    TEST_ASSERT(t.failure_count()==0);
    TEST_ASSERT(t.invalidation_count()==0);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(t.sec()==0);
    TEST_ASSERT(t.usec()==0);
#endif
    TEST_ASSERT(t.records().size()==2);
}

    
////////////////////////////////////////////////////////////////////////////////


Record_Vector* Empty_Trace_1sec_1usec_No_Carry::clone() const
{
    return new Empty_Trace_1sec_1usec_No_Carry;
}

::std::vector< ::diagnostics::Record> Empty_Trace_1sec_1usec_No_Carry::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    
    result.push_back(Record(LEVEL_TEST,
			    TYPE_TESTCASE_ENTER,
			    0,
			    "empty_test_case_1sec_1usec_no_carry","base","file",1
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
			    , \
				::diagnostics::testing::Dummy_Values_For_Pid::value2,\
				::diagnostics::testing::Dummy_Values_For_Tid::value3,\
                1000,10000,"host"
#endif
			    ));
    
    result.push_back(Record(LEVEL_TEST,
			    TYPE_TESTCASE_EXIT,
			    0,
			    "empty_test_case_1sec_1usec_no_carry","base","file",1
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
				,\
			    ::diagnostics::testing::Dummy_Values_For_Pid::value2,\
				::diagnostics::testing::Dummy_Values_For_Tid::value3,\
                1001,10001,
			    "host"
#endif
			    ));
    return result;
}

void Empty_Trace_1sec_1usec_No_Carry::check(::diagnostics::unittest::Test_Run_Result const & t,
					    bool const logging,
					    Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_COMPLETE);
    TEST_ASSERT(t.failure_count()==0);
    TEST_ASSERT(t.invalidation_count()==0);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(t.sec()==1);
    TEST_ASSERT(t.usec()==1);
#endif
    TEST_ASSERT(t.records().size()==2);
}


////////////////////////////////////////////////////////////////////////////////


Record_Vector* Empty_Trace_1sec_1usec_With_Carry::clone() const
{
    return new Empty_Trace_1sec_1usec_With_Carry;
}

::std::vector< ::diagnostics::Record> Empty_Trace_1sec_1usec_With_Carry::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    
    result.push_back(Record(LEVEL_TEST,
			    TYPE_TESTCASE_ENTER,
			    0,
			    "empty_test_case_1sec_1usec_with_carry","base","file",1
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
				,\
			    ::diagnostics::testing::Dummy_Values_For_Pid::value2,\
				::diagnostics::testing::Dummy_Values_For_Tid::value3,\
                1001,1000*1000-1,"host"
#endif
			    ));
    
    result.push_back(Record(LEVEL_TEST,
			    TYPE_TESTCASE_EXIT,
			    0,
			    "empty_test_case_1sec_1usec_with_carry","base","file",1
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
				,\
			    ::diagnostics::testing::Dummy_Values_For_Pid::value2,\
				::diagnostics::testing::Dummy_Values_For_Tid::value3,\
                1003,0,"host"
#endif
			    ));
    return result;
}

void Empty_Trace_1sec_1usec_With_Carry::check(::diagnostics::unittest::Test_Run_Result const & t,
					      bool const logging,
					      Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_COMPLETE);
    TEST_ASSERT(t.failure_count()==0);
    TEST_ASSERT(t.invalidation_count()==0);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(t.sec()==1);
    TEST_ASSERT(t.usec()==1);
#endif
    TEST_ASSERT(t.records().size()==2);
}


////////////////////////////////////////////////////////////////////////////////



Record_Vector*  Trace_With_One_Failure::clone() const
{
    return new Trace_With_One_Failure;
}

::std::vector< ::diagnostics::Record> Trace_With_One_Failure::records()
{
    using namespace diagnostics;
    ::std::vector<Record> result;
    PUSH_RECORD("trace_with_one_failure",LEVEL_TEST,TYPE_TESTCASE_ENTER);
    PUSH_RECORD("failure",LEVEL_TEST,TYPE_FAILED_ASSERTION);
    PUSH_RECORD("trace_with_one_failure",LEVEL_TEST,TYPE_TESTCASE_EXIT);
    return result;
}
    
void Trace_With_One_Failure::check(::diagnostics::unittest::Test_Run_Result const & t,
				   bool const logging,
				   Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_COMPLETE);
    TEST_ASSERT(t.failure_count()==1);
    TEST_ASSERT(t.invalidation_count()==0);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(t.sec()==0);
    TEST_ASSERT(t.usec()==0);
#endif
    TEST_ASSERT(t.records().size()==3);
}

////////////////////////////////////////////////////////////////////////////////



Record_Vector* Trace_With_One_Invalidation::clone() const
{
    return new Trace_With_One_Invalidation;
}

::std::vector< ::diagnostics::Record> Trace_With_One_Invalidation::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    PUSH_RECORD("trace_with_one_failure",LEVEL_TEST,TYPE_TESTCASE_ENTER);
    PUSH_RECORD("failed check -- invalidation",LEVEL_TEST,TYPE_FAILED_CHECK);
    PUSH_RECORD("trace_with_one_failure",LEVEL_TEST,TYPE_TESTCASE_EXIT);
    return result;
}

void Trace_With_One_Invalidation::check(::diagnostics::unittest::Test_Run_Result const & t,
					bool const logging,
					Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_COMPLETE);
    TEST_ASSERT(t.failure_count()==0);
    TEST_ASSERT(t.invalidation_count()==1);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(t.sec()==0);
    TEST_ASSERT(t.usec()==0);
#endif
    TEST_ASSERT(t.records().size()==3);
}


////////////////////////////////////////////////////////////////////////////////


Record_Vector* Trace_With_One_Failed_Check::clone() const
{
    return new Trace_With_One_Failed_Check;
}

::std::vector< ::diagnostics::Record> Trace_With_One_Failed_Check::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    PUSH_RECORD("trace_with_one_failure",LEVEL_TEST,TYPE_TESTCASE_ENTER);
    PUSH_RECORD("failed check",LEVEL_AUDIT,TYPE_FAILED_CHECK);
    PUSH_RECORD("trace_with_one_failure",LEVEL_TEST,TYPE_TESTCASE_EXIT);
    return result;
}

void Trace_With_One_Failed_Check::check(::diagnostics::unittest::Test_Run_Result const & t,
					bool const logging,
					Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_COMPLETE);
    TEST_ASSERT(t.failure_count()==(target_level==LEVEL_AUDIT ? 0 : 1));
    TEST_ASSERT(t.invalidation_count()==0);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(t.sec()==0);
    TEST_ASSERT(t.usec()==0);
#endif
    TEST_ASSERT(t.records().size()== (logging ? (target_level!=LEVEL_AUDIT ? 3 : 2) : 3));
}

////////////////////////////////////////////////////////////////////////////////


Record_Vector* Trace_With_All_Level_Type_Combinations::clone() const
{
    return new Trace_With_All_Level_Type_Combinations;
}


::std::vector< ::diagnostics::Record> Trace_With_All_Level_Type_Combinations::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    PUSH_RECORD("trace_with_all_level_type_combinations",LEVEL_TEST,TYPE_TESTCASE_ENTER);
    inner_records(result);
    PUSH_RECORD("trace_with_all_level_type_combinations",LEVEL_TEST,TYPE_TESTCASE_EXIT);
    return result;
}

void Trace_With_All_Level_Type_Combinations::check(::diagnostics::unittest::Test_Run_Result const & t,
						   bool const logging,
						   Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_COMPLETE);
    TEST_ASSERT(t.failure_count()==(target_level==LEVEL_AUDIT 
						   ? 10
						   : (target_level==LEVEL_DEBUG ? 11 : 12)));
    TEST_ASSERT(t.invalidation_count()==1);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(t.sec()==0);
    TEST_ASSERT(t.usec()==0);
#endif
    int level_ajust=0; // LEVEL_AUDIT
    if(target_level==LEVEL_DEBUG) level_ajust=1;
    else if(target_level==LEVEL_PROD) level_ajust=2;
    
    TEST_ASSERT(static_cast<int>(t.records().size())==
				(logging ? (19+2+level_ajust) : (13*4+2)));
}

void Trace_With_All_Level_Type_Combinations::inner_records(::std::vector< ::diagnostics::Record> & result)
{
    using namespace diagnostics;
    // reporting problems
    PUSH_RECORD("failed check",LEVEL_AUDIT,TYPE_FAILED_CHECK);
    PUSH_RECORD("failed check",LEVEL_DEBUG,TYPE_FAILED_CHECK);
    PUSH_RECORD("failed check",LEVEL_PROD,TYPE_FAILED_CHECK);
    PUSH_RECORD("failed check -- INVALIDATION",LEVEL_TEST,TYPE_FAILED_CHECK);

    PUSH_RECORD("failed assertion -- FAILURE",LEVEL_AUDIT,TYPE_FAILED_ASSERTION);
    PUSH_RECORD("failed assertion -- FAILURE",LEVEL_DEBUG,TYPE_FAILED_ASSERTION);
    PUSH_RECORD("failed assertion -- FAILURE",LEVEL_PROD,TYPE_FAILED_ASSERTION);
    PUSH_RECORD("failed assertion -- FAILURE",LEVEL_TEST,TYPE_FAILED_ASSERTION);

    PUSH_RECORD("unexpected exception -- FAILURE",LEVEL_AUDIT,TYPE_UNEXPECTED_EXCEPTION);
    PUSH_RECORD("unexpected exception -- FAILURE",LEVEL_DEBUG,TYPE_UNEXPECTED_EXCEPTION);
    PUSH_RECORD("unexpected exception -- FAILURE",LEVEL_PROD,TYPE_UNEXPECTED_EXCEPTION);
    PUSH_RECORD("unexpected exception -- FAILURE",LEVEL_TEST,TYPE_UNEXPECTED_EXCEPTION);

    PUSH_RECORD("wrong exception",LEVEL_AUDIT,TYPE_WRONG_EXCEPTION);
    PUSH_RECORD("wrong exception",LEVEL_DEBUG,TYPE_WRONG_EXCEPTION);
    PUSH_RECORD("wrong exception",LEVEL_PROD,TYPE_WRONG_EXCEPTION);
    PUSH_RECORD("wrong exception -- FAILURE",LEVEL_TEST,TYPE_WRONG_EXCEPTION);

    PUSH_RECORD("missing exception",LEVEL_AUDIT,TYPE_MISSING_EXCEPTION);
    PUSH_RECORD("missing exception",LEVEL_DEBUG,TYPE_MISSING_EXCEPTION);
    PUSH_RECORD("missing exception",LEVEL_PROD,TYPE_MISSING_EXCEPTION);
    PUSH_RECORD("missing exception -- FAILURE",LEVEL_TEST,TYPE_MISSING_EXCEPTION);

    // just traces
    PUSH_RECORD("TYPE_TRACE_BINARY",LEVEL_AUDIT,TYPE_TRACE_BINARY);
    PUSH_RECORD("TYPE_TRACE_BINARY",LEVEL_DEBUG,TYPE_TRACE_BINARY);
    PUSH_RECORD("TYPE_TRACE_BINARY",LEVEL_PROD,TYPE_TRACE_BINARY);
    PUSH_RECORD("TYPE_TRACE_BINARY",LEVEL_TEST,TYPE_TRACE_BINARY);

    PUSH_RECORD("TYPE_TRACE",LEVEL_AUDIT,TYPE_TRACE);
    PUSH_RECORD("TYPE_TRACE",LEVEL_DEBUG,TYPE_TRACE);
    PUSH_RECORD("TYPE_TRACE",LEVEL_PROD,TYPE_TRACE);
    PUSH_RECORD("TYPE_TRACE",LEVEL_TEST,TYPE_TRACE);


    // blocks
    PUSH_RECORD("TYPE_METHOD_ENTER",LEVEL_AUDIT,TYPE_METHOD_ENTER);
    PUSH_RECORD("TYPE_METHOD_ENTER",LEVEL_DEBUG,TYPE_METHOD_ENTER);
    PUSH_RECORD("TYPE_METHOD_ENTER",LEVEL_PROD,TYPE_METHOD_ENTER);
    PUSH_RECORD("TYPE_METHOD_ENTER",LEVEL_TEST,TYPE_METHOD_ENTER);

    PUSH_RECORD("TYPE_METHOD_EXIT",LEVEL_TEST,TYPE_METHOD_EXIT);
    PUSH_RECORD("TYPE_METHOD_EXIT",LEVEL_PROD,TYPE_METHOD_EXIT);
    PUSH_RECORD("TYPE_METHOD_EXIT",LEVEL_DEBUG,TYPE_METHOD_EXIT);
    PUSH_RECORD("TYPE_METHOD_EXIT",LEVEL_AUDIT,TYPE_METHOD_EXIT);



    PUSH_RECORD("TYPE_BLOCK_ENTER",LEVEL_AUDIT,TYPE_BLOCK_ENTER);
    PUSH_RECORD("TYPE_BLOCK_ENTER",LEVEL_DEBUG,TYPE_BLOCK_ENTER);
    PUSH_RECORD("TYPE_BLOCK_ENTER",LEVEL_PROD,TYPE_BLOCK_ENTER);
    PUSH_RECORD("TYPE_BLOCK_ENTER",LEVEL_TEST,TYPE_BLOCK_ENTER);

    PUSH_RECORD("TYPE_BLOCK_EXIT",LEVEL_TEST,TYPE_BLOCK_EXIT);
    PUSH_RECORD("TYPE_BLOCK_EXIT",LEVEL_PROD,TYPE_BLOCK_EXIT);
    PUSH_RECORD("TYPE_BLOCK_EXIT",LEVEL_DEBUG,TYPE_BLOCK_EXIT);
    PUSH_RECORD("TYPE_BLOCK_EXIT",LEVEL_AUDIT,TYPE_BLOCK_EXIT);


    PUSH_RECORD("TYPE_PROCEDURE_ENTER",LEVEL_AUDIT,TYPE_PROCEDURE_ENTER);
    PUSH_RECORD("TYPE_PROCEDURE_ENTER",LEVEL_DEBUG,TYPE_PROCEDURE_ENTER);
    PUSH_RECORD("TYPE_PROCEDURE_ENTER",LEVEL_PROD,TYPE_PROCEDURE_ENTER);
    PUSH_RECORD("TYPE_PROCEDURE_ENTER",LEVEL_TEST,TYPE_PROCEDURE_ENTER);

    PUSH_RECORD("TYPE_PROCEDURE_EXIT",LEVEL_TEST,TYPE_PROCEDURE_EXIT);
    PUSH_RECORD("TYPE_PROCEDURE_EXIT",LEVEL_PROD,TYPE_PROCEDURE_EXIT);
    PUSH_RECORD("TYPE_PROCEDURE_EXIT",LEVEL_DEBUG,TYPE_PROCEDURE_EXIT);
    PUSH_RECORD("TYPE_PROCEDURE_EXIT",LEVEL_AUDIT,TYPE_PROCEDURE_EXIT);


    PUSH_RECORD("TYPE_LOG_OPEN",LEVEL_AUDIT,TYPE_LOG_OPEN);
    PUSH_RECORD("TYPE_LOG_OPEN",LEVEL_DEBUG,TYPE_LOG_OPEN);
    PUSH_RECORD("TYPE_LOG_OPEN",LEVEL_PROD,TYPE_LOG_OPEN);
    PUSH_RECORD("TYPE_LOG_OPEN",LEVEL_TEST,TYPE_LOG_OPEN);

    PUSH_RECORD("TYPE_LOG_CLOSE",LEVEL_TEST,TYPE_LOG_CLOSE);
    PUSH_RECORD("TYPE_LOG_CLOSE",LEVEL_PROD,TYPE_LOG_CLOSE);
    PUSH_RECORD("TYPE_LOG_CLOSE",LEVEL_DEBUG,TYPE_LOG_CLOSE);
    PUSH_RECORD("TYPE_LOG_CLOSE",LEVEL_AUDIT,TYPE_LOG_CLOSE);
}
    


////////////////////////////////////////////////////////////////////////////////

Record_Vector* Trace_With_All_Level_Type_Combinations_Times_2::clone() const
{
    return new Trace_With_All_Level_Type_Combinations_Times_2;
}

::std::vector< ::diagnostics::Record> Trace_With_All_Level_Type_Combinations_Times_2::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    PUSH_RECORD("trace_with_all_level_type_combinations_times_2",LEVEL_TEST,TYPE_TESTCASE_ENTER);
    Trace_With_All_Level_Type_Combinations::inner_records(result);
    Trace_With_All_Level_Type_Combinations::inner_records(result);
    PUSH_RECORD("trace_with_all_level_type_combinations_times_2",LEVEL_TEST,TYPE_TESTCASE_EXIT);
    
    return result;
}

void Trace_With_All_Level_Type_Combinations_Times_2::check(::diagnostics::unittest::Test_Run_Result const & t,
							   bool const logging,
							   Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_COMPLETE);
    TEST_ASSERT(t.failure_count()==(target_level==LEVEL_AUDIT 
						   ? 10
						   : (target_level==LEVEL_DEBUG ? 11 : 12))*2);
    TEST_ASSERT(t.invalidation_count()==1*2);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(t.sec()==0);
    TEST_ASSERT(t.usec()==0);
#endif
    int level_ajust=0; // LEVEL_AUDIT
    if(target_level==LEVEL_DEBUG) level_ajust=2;
    else if(target_level==LEVEL_PROD) level_ajust=4;

    TEST_ASSERT(static_cast<int>(t.records().size())==
				(logging ? (38+2+level_ajust) : ((13*4)*2+2)));
}


////////////////////////////////////////////////////////////////////////////////

Record_Vector* Trace_With_Two_Test_Case_Enter_Msgs::clone() const
{
    return new Trace_With_Two_Test_Case_Enter_Msgs;
}

::std::vector< ::diagnostics::Record> Trace_With_Two_Test_Case_Enter_Msgs::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    PUSH_RECORD("trace_with_two_test_case_enter_msgs",LEVEL_TEST,TYPE_TESTCASE_ENTER);
    PUSH_RECORD("some trace",LEVEL_TEST,TYPE_TRACE);
    PUSH_RECORD("trace_with_two_test_case_enter_msgs",LEVEL_TEST,TYPE_TESTCASE_ENTER);
    return result;
}

void Trace_With_Two_Test_Case_Enter_Msgs::check(::diagnostics::unittest::Test_Run_Result const & t,
						bool const logging,
						Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_INCOMPLETE);
    TEST_ASSERT(t.failure_count()==0);
    TEST_ASSERT(t.invalidation_count()==0);
    TEST_ASSERT(t.records().size()==2);
}

////////////////////////////////////////////////////////////////////////////////

Record_Vector* Trace_With_No_Test_Case_Enter::clone() const
{
    return new Trace_With_No_Test_Case_Enter;
}

::std::vector< ::diagnostics::Record> Trace_With_No_Test_Case_Enter::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    PUSH_RECORD("some trace",LEVEL_TEST,TYPE_TRACE);
    return result;
}

void Trace_With_No_Test_Case_Enter::check(::diagnostics::unittest::Test_Run_Result const & t,
					  bool const logging,
					  Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_EMPTY);
    TEST_ASSERT(t.failure_count()==0);
    TEST_ASSERT(t.invalidation_count()==0);
    TEST_ASSERT(t.records().size()==0);
}


////////////////////////////////////////////////////////////////////////////////

Record_Vector* Trace_With_Two_Test_Case_Exits::clone() const
{
    return new Trace_With_Two_Test_Case_Exits;
}

::std::vector< ::diagnostics::Record> Trace_With_Two_Test_Case_Exits::records()
{
    using namespace diagnostics;
    
    ::std::vector<Record> result;
    PUSH_RECORD("trace_with_all_level_type_combinations",LEVEL_TEST,TYPE_TESTCASE_ENTER);
    Trace_With_All_Level_Type_Combinations::inner_records(result);
    PUSH_RECORD("trace_with_all_level_type_combinations",LEVEL_TEST,TYPE_TESTCASE_EXIT);
    PUSH_RECORD("trace_with_all_level_type_combinations",LEVEL_TEST,TYPE_TESTCASE_EXIT);
    return result;
}

void Trace_With_Two_Test_Case_Exits::check(::diagnostics::unittest::Test_Run_Result const & t,
					   bool const logging,
					   Level_t const target_level)
{
    using namespace diagnostics;
    using namespace unittest;
    
    TEST_ASSERT(t.abstract_state()==Test_Run_Result::STATE_COMPLETE);
    TEST_ASSERT(t.failure_count()==(target_level==LEVEL_AUDIT 
						   ? 10
						   : (target_level==LEVEL_DEBUG ? 11 : 12)));
    TEST_ASSERT(t.invalidation_count()==1);
#if DIAGNOSTICS_SWITCH_SYSTEM_CALLS_ENABLED == 1
    TEST_ASSERT(t.sec()==0);
    TEST_ASSERT(t.usec()==0);
#endif
    TEST_ASSERT(t.records().size()==(logging ? 19 : (13*4+2)));
}


TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

// vim:ts=4:sw=4
