/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __INSPECTOR_H_INCLUDED
#define __INSPECTOR_H_INCLUDED

#include <stdlib.h>
#include "lib/glue-log.h"
#include "lib/system_node_tree.h"

/* Set this to 1 to activate inspection feature */
#define ENABLE_INSPECTION 1

extern volatile int insp_stopped;

typedef void (*insert_text_t)(void *ctx, const char *text, int len);
typedef const char* (*get_prompt_text_t)(void *ctx);
typedef void (*pop_from_prompt_t)(void *ctx);
typedef void (*push_to_prompt_t)(void *ctx, const char *name, size_t len);
typedef struct {
	void *ctx;
	insert_text_t insert_text_cb;
	get_prompt_text_t get_prompt_text_cb;
	pop_from_prompt_t pop_from_prompt_cb;
	push_to_prompt_t push_to_prompt_cb;
} console_interface;

typedef struct inspector_context_t inspector_context;
inspector_context* inspector_context_new(void);
void inspector_set_console(inspector_context *ctx, console_interface *ci);
console_interface* inspector_get_console(inspector_context *ctx);
void
inspector_evaluate_breaks(inspector_context *ctx,
                          system_node* node,
                          system_node_field* field);
void inspector_exec_command(inspector_context *ctx, const char* cmd, int len);

extern inspector_context *private_inspector_ctx;
static inline void inspector_set_global_inspector(inspector_context *ctx) {
	private_inspector_ctx = ctx;
}

static inline inspector_context* inspector_get_global_inspector(void) {
	return private_inspector_ctx;
}

static inline void
system_check_break(system_node *node, size_t offset)
{
#if ENABLE_INSPECTION
        system_node_field *field;

	if (! node->has_active_breakpoints) {
		return;
	}
	field = system_get_node_field(node, offset);
	if (! field->is_used_in_breakpoints) {
		return;
	}
	inspector_evaluate_breaks(inspector_get_global_inspector(), node, field);
#endif
}

#define STATE_GET(var) \
	cpssp->NAME.STATE_ ## var

#if ! ENABLE_INSPECTION

#define STATE_DECL(name, var, type)

#define STATE_SET(var, value) \
	cpssp->NAME.STATE_ ## var = value

#else

#define STATE_DECL(name, var, type) \
	system_node_register_field(cpssp->node, name, offsetof(struct cpssp, NAME.STATE_ ## var), type)

#define STATE_SET(var, value) \
	do { \
		cpssp->NAME.STATE_ ## var = value; \
		system_check_break(cpssp->node, offsetof(struct cpssp, NAME.STATE_ ## var)); \
	} while (0)

#endif

#endif /*__INSPECTOR_H_INCLUDED */
